/**
 * @file EvolutionGeniviDbusDMCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback Interface for DM.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback Interface for DM.
 */

#include "EvolutionGeniviDbusDMCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"

#include "FwAssert.h"

using namespace ::org::alps::evo::DM;
using namespace ::asf::core;

namespace btstackif {
namespace genivi {

EvolutionGeniviDbusDMCallbackIf::EvolutionGeniviDbusDMCallbackIf()
{
}

EvolutionGeniviDbusDMCallbackIf::EvolutionGeniviDbusDMCallbackIf(IDbusRecHandler* recHandler) : DbusCallbackIf(recHandler)
{
}

EvolutionGeniviDbusDMCallbackIf::~EvolutionGeniviDbusDMCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviDbusDMCallbackIf::onProxyAvailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_DM);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviDbusDMCallbackIf::onProxyUnavailable(const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(objPath);

   BTSDbusServiceAvailability availabilityEvent = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   if(ServiceState__Available == currentState)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_DM);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// org/alps/evo/DMProxy implementation --- start
void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplDeviceConnectReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplDeviceConnectReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplDeviceDisconnectReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplDeviceDisconnectReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplCancelConnectReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplCancelConnectReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplCancelConnectReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplCancelConnectReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplCancelConnectReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplCancelConnectReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplGetHwVersionReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplGetHwVersionReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplGetHwVersionReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplGetHwVersionReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplGetHwVersionReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplGetHwVersionReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplSetConfigurationReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplSetConfigurationReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplSetConfigurationReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplSetConfigurationReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplSetConfigurationReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplSetConfigurationReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplReadConfigurationReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplReadConfigurationReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplReadConfigurationReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplReadConfigurationReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplReadConfigurationReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplReadConfigurationReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplSspDebugModeReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplSspDebugModeReqError >& error)
{
   (void)(proxy);
   (void)(error);
   // not used
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplSspDebugModeReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplSspDebugModeReqResponse >& response)
{
   (void)(proxy);
   (void)(response);
   // not used
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplTestModeReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplTestModeReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeLinkQualityReqError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeLinkQualityReqError >& error)
{
   (void)(proxy);
   onError(ptrNew_Ipc2Bts_BtApplTestModeLinkQualityReqResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeLinkQualityReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeLinkQualityReqResponse >& response)
{
   (void)(proxy);
   onResponse(ptrNew_Ipc2Bts_BtApplTestModeLinkQualityReqResult(), response);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplDeviceConnectCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplDeviceConnectCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectIndError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectIndError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectIndSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectIndSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplDeviceConnectInd* ptrResultMsg = ptrNew_Ipc2Bts_BtApplDeviceConnectInd();
   if(NULL != ptrResultMsg)
   {
      convertBdAddressBinary2String(ptrResultMsg->getBDAddressMutable(), signal->getAddress());
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      ptrResultMsg->getServiceFunctionMutable().setData(signal->getFunction());
      ptrResultMsg->setInstance(signal->getInstance());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplDeviceDisconnectCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplDeviceDisconnectCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectIndError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectIndError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectIndSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectIndSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplDeviceDisconnectInd* ptrResultMsg = ptrNew_Ipc2Bts_BtApplDeviceDisconnectInd();
   if(NULL != ptrResultMsg)
   {
      convertBdAddressBinary2String(ptrResultMsg->getBDAddressMutable(), signal->getAddress());
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      ptrResultMsg->setReason(convertBtApplDisconnectReason2InternalValue(signal->getReason()));
      ptrResultMsg->getServiceFunctionMutable().setData(signal->getFunction());
      ptrResultMsg->setInstance(signal->getInstance());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectCompIndError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectCompIndError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceDisconnectCompIndSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceDisconnectCompIndSignal >& signal)
{
   (void)(proxy);
   onSignal(ptrNew_Ipc2Bts_BtApplDeviceDisconnectCompInd(), signal);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectCompIndError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectCompIndError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplDeviceConnectCompIndSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplDeviceConnectCompIndSignal >& signal)
{
   (void)(proxy);
   onSignal(ptrNew_Ipc2Bts_BtApplDeviceConnectCompInd(), signal);
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplCancelConnectCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplCancelConnectCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplCancelConnectCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplCancelConnectCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplCancelConnectCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplCancelConnectCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplStatusNotificationIndError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplStatusNotificationIndError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplStatusNotificationIndSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplStatusNotificationIndSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplStatusNotificationInd* ptrResultMsg = ptrNew_Ipc2Bts_BtApplStatusNotificationInd();
   if(NULL != ptrResultMsg)
   {
      convertStringBinary2String(ptrResultMsg->getDeviceNameMutable(), signal->getData());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplGetHwVersionCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplGetHwVersionCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplGetHwVersionCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplGetHwVersionCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplGetHwVersionCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplGetHwVersionCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      ptrResultMsg->setBtCoreVersion((BTSBtCoreVersion)signal->getBt_version()); // cast OK because enum and Evolution value match
      ptrResultMsg->setManufacturer(signal->getManufactory());
      ptrResultMsg->setHwSubVersion(signal->getHw_subversion());
      BTSExtensionInfo& info = ptrResultMsg->getExtInfoMutable();
      info = signal->getExt_info();
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplSetConfigurationCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplSetConfigurationCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplSetConfigurationCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplSetConfigurationCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplSetConfigurationCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplSetConfigurationCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      ptrResultMsg->setInstance(signal->getInstance());
      ptrResultMsg->setConfigId((BTSEvoConfigIdEnum)signal->getConfig_id()); // cast OK because enum and Evolution value match
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplReadConfigurationCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplReadConfigurationCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplReadConfigurationCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplReadConfigurationCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplReadConfigurationCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplReadConfigurationCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      ptrResultMsg->setInstance(signal->getInstance());
      ptrResultMsg->setConfigId((BTSEvoConfigIdEnum)signal->getConfig_id()); // cast OK because enum and Evolution value match
      ptrResultMsg->setData(signal->getData());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplTestModeCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplTestModeCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeCompIndError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeCompIndError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeCompIndSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeCompIndSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplTestModeCompInd* ptrResultMsg = ptrNew_Ipc2Bts_BtApplTestModeCompInd();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setEnable(0x01 == signal->getNotiType());
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeLinkQualityCfmError(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeLinkQualityCfmError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviDbusDMCallbackIf::onBtApplTestModeLinkQualityCfmSignal(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplTestModeLinkQualityCfmSignal >& signal)
{
   (void)(proxy);
   Ipc2Bts_BtApplTestModeLinkQualityCfm* ptrResultMsg = ptrNew_Ipc2Bts_BtApplTestModeLinkQualityCfm();
   if(NULL != ptrResultMsg)
   {
      ptrResultMsg->setStatus(convertStatusCode2InternalValue(signal->getStatus()));
      ptrResultMsg->setRssi((BTSRssi)signal->getRssi());
      ptrResultMsg->setLinkQuality(signal->getLinkquality());
      onSignal(ptrResultMsg, signal->getAct());
   }
}
// org/alps/evo/DMProxy implementation --- end

} //genivi
} //btstackif
