/**
 * @file DbusAccess_GEN.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS GENIVI handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of DBUS GENIVI handling.
 */

#include "cc_dbus_if/CcDbusIfControllerFactory.h"
#include "cc_dbus_if/CcDbusIfLaunchType.h"
#include "cc_dbus_if/ICcDbusIfController.h"
#include "cc_dbus_if/IWorkItem.h"
#include "cc_dbus_if/ICcDbusIfControllerClient.h"
#include "cc_dbus_if/IOrgFreedesktopDbusObjectManagerSendRequestIf.h"
#include "cc_dbus_if/ICcDbusIfControllerObjectManager.h"
#include "cc_dbus_if/IIntrospectableDbusSendRequestIf.h"
#include "cc_dbus_if/ICcDbusIfControllerIntrospectable.h"
#include "cc_dbus_if/IPropertiesGetAllDbusSendRequestIf.h"
#include "cc_dbus_if/ICcDbusIfControllerPropertiesGetAll.h"
#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#include "cc_dbus_if/ICcDbusIfControllerConnMan.h"
#include "cc_dbus_if/ICcDbusIfControllerEvolutionGenivi.h"

#include "cc_dbus_if/IConnManDbusManagerSendRequestIf.h"
#include "cc_dbus_if/IConnManDbusServiceSendRequestIf.h"
#include "cc_dbus_if/IConnManDbusTechnologySendRequestIf.h"

#include "cc_dbus_if/IEvolutionGeniviDbusAdapterSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusAgentSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusAgentManagerSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusDeviceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusDMSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusServiceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusTraceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusGeniviTraceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusDidSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusSppSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusObexClientSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusObexSessionSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusObexTransferSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpCallVolumeSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpHandsfreeSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpManagerSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpModemSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpNetworkOperatorSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpNetworkRegistrationSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpSiriSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusHfpVoiceCallSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusSerialSendRequestIf.h"

#include "DbusAccess_GEN.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviStackTypes.h"
#include "EvolutionGeniviUtils.h"
#include "DbusWorkItem.h"
#include "BtsUtils.h"

#include "IConnectionOrgFreedesktopObjectManagerCallbackIf.h"
#include "ConnectionOrgFreedesktopObjectManagerFactory.h"
#include "IntrospectableDbusCallbackIf.h"
#include "PropertiesGetAllDbusCallbackIf.h"
#include "PropertiesGetAllDbusFactory.h"

#include "ConnManDbusManagerFactory.h"
#include "ConnManDbusServiceFactory.h"
#include "ConnManDbusTechnologyFactory.h"

#include "IEvolutionGeniviAdapterCallbackIf.h"
#include "EvolutionGeniviAdapterFactory.h"
#include "IEvolutionGeniviAgentCallbackIf.h"
#include "EvolutionGeniviAgentFactory.h"
#include "EvolutionGeniviDbusAgentManagerFactory.h"
#include "EvolutionGeniviDbusDeviceFactory.h"
#include "EvolutionGeniviDbusDMFactory.h"
#include "EvolutionGeniviDbusServiceFactory.h"
#include "EvolutionGeniviDbusTraceFactory.h"
#include "EvolutionGeniviDbusGeniviTraceFactory.h"
#include "EvolutionGeniviDbusDidFactory.h"
#include "EvolutionGeniviDbusSppFactory.h"
#include "EvolutionGeniviDbusObexClientFactory.h"
#include "EvolutionGeniviDbusObexSessionFactory.h"
#include "EvolutionGeniviDbusObexTransferFactory.h"
#include "EvolutionGeniviDbusHfpCallVolumeFactory.h"
#include "EvolutionGeniviDbusHfpHandsfreeFactory.h"
#include "EvolutionGeniviDbusHfpManagerFactory.h"
#include "EvolutionGeniviDbusHfpModemFactory.h"
#include "EvolutionGeniviDbusHfpNetworkOperatorFactory.h"
#include "EvolutionGeniviDbusHfpNetworkRegistrationFactory.h"
#include "EvolutionGeniviDbusHfpSiriFactory.h"
#include "EvolutionGeniviDbusHfpVoiceCallManagerFactory.h"
#include "EvolutionGeniviDbusHfpVoiceCallFactory.h"
#include "IEvolutionGeniviSerialCallbackIf.h"
#include "EvolutionGeniviSerialFactory.h"

#include "FwBluetoothStringUtils.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
// #include "TraceBase.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DbusAccess_GEN.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

//#define BT_ON_OFF_SWITCH_FAILED
#undef BT_ON_OFF_SWITCH_FAILED
#ifdef BT_ON_OFF_SWITCH_FAILED
// debug section start
static int toggle = 1;
// debug section end
#endif

DbusAccess::DbusAccess() :
_getUpdateTimeout(2000U),
_count(20U),
_systemBusName("org.bluez"),
_sessionBusName("org.bluez.obex"),
_systemObjPath("/"),
_sessionObjPath("/" /*"/org/bluez/obex"*/), // TODO: both seems to working => check during testing
_objectManagerIf(0),
_introspectableIf(0),
_propertiesGetAllIf(0),
_connManManagerIf(0),
_connManServiceIf(0),
_connManTechnologyIf(0),
_adapterIf(0),
_agentIf(0),
_agentManagerIf(0),
_deviceIf(0),
_dmIf(0),
_serviceIf(0),
_traceIf(0),
_geniviTraceIf(0),
_didIf(0),
_sppIf(0),
_obexClientIf(0),
_obexSessionIf(0),
_obexTransferIf(0),
_hfpCallVolumeIf(0),
_hfpHandsfreeIf(0),
_hfpManagerIf(0),
_hfpModemIf(0),
_hfpNetworkOperatorIf(0),
_hfpNetworkRegistrationIf(0),
_hfpSiriIf(0),
_hfpVoiceCallManagerIf(0),
_hfpVoiceCallIf(0),
_serialIf(0),
_objectManagerCb(0),
_introspectableCb(0),
_propertiesGetAllCb(0),
_connManManagerCb(0),
_connManServiceCb(0),
_connManTechnologyCb(0),
_adapterCb(0),
_agentCb(0),
_agentManagerCb(0),
_deviceCb(0),
_dmCb(0),
_serviceCb(0),
_traceCb(0),
_geniviTraceCb(0),
_didCb(0),
_sppCb(0),
_obexClientCb(0),
_obexSessionCb(0),
_obexTransferCb(0),
_hfpCallVolumeCb(0),
_hfpHandsfreeCb(0),
_hfpManagerCb(0),
_hfpModemCb(0),
_hfpNetworkOperatorCb(0),
_hfpNetworkRegistrationCb(0),
_hfpSiriCb(0),
_hfpVoiceCallManagerCb(0),
_hfpVoiceCallCb(0),
_serialCb(0)
{
   setResponseTimeoutHandler(this);
   createTimeoutControlList(_count);
}

DbusAccess::DbusAccess(const DbusAccess& ref) :
DbusBase(ref),
_getUpdateTimeout(2000U),
_count(20U),
_systemBusName("org.bluez"),
_sessionBusName("org.bluez.obex"),
_systemObjPath("/"),
_sessionObjPath("/" /*"/org/bluez/obex"*/), // TODO: both seems to working => check during testing
_objectManagerIf(0),
_introspectableIf(0),
_propertiesGetAllIf(0),
_connManManagerIf(0),
_connManServiceIf(0),
_connManTechnologyIf(0),
_adapterIf(0),
_agentIf(0),
_agentManagerIf(0),
_deviceIf(0),
_dmIf(0),
_serviceIf(0),
_traceIf(0),
_geniviTraceIf(0),
_didIf(0),
_sppIf(0),
_obexClientIf(0),
_obexSessionIf(0),
_obexTransferIf(0),
_hfpCallVolumeIf(0),
_hfpHandsfreeIf(0),
_hfpManagerIf(0),
_hfpModemIf(0),
_hfpNetworkOperatorIf(0),
_hfpNetworkRegistrationIf(0),
_hfpSiriIf(0),
_hfpVoiceCallManagerIf(0),
_hfpVoiceCallIf(0),
_serialIf(0),
_objectManagerCb(0),
_introspectableCb(0),
_propertiesGetAllCb(0),
_connManManagerCb(0),
_connManServiceCb(0),
_connManTechnologyCb(0),
_adapterCb(0),
_agentCb(0),
_agentManagerCb(0),
_deviceCb(0),
_dmCb(0),
_serviceCb(0),
_traceCb(0),
_geniviTraceCb(0),
_didCb(0),
_sppCb(0),
_obexClientCb(0),
_obexSessionCb(0),
_obexTransferCb(0),
_hfpCallVolumeCb(0),
_hfpHandsfreeCb(0),
_hfpManagerCb(0),
_hfpModemCb(0),
_hfpNetworkOperatorCb(0),
_hfpNetworkRegistrationCb(0),
_hfpSiriCb(0),
_hfpVoiceCallManagerCb(0),
_hfpVoiceCallCb(0),
_serialCb(0)
{
   // ignore given parameter
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

DbusAccess& DbusAccess::operator=(const DbusAccess& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   DbusBase::operator=(ref);

   // ignore given parameter

   return *this;
}

DbusAccess::~DbusAccess()
{
   _objectManagerIf = 0;
   _introspectableIf = 0;
   _propertiesGetAllIf = 0;
   _connManManagerIf = 0;
   _connManServiceIf = 0;
   _connManTechnologyIf = 0;
   _adapterIf = 0;
   _agentIf = 0;
   _agentManagerIf = 0;
   _deviceIf = 0;
   _dmIf = 0;
   _serviceIf = 0;
   _traceIf = 0;
   _geniviTraceIf = 0;
   _didIf = 0;
   _sppIf = 0;
   _obexClientIf = 0;
   _obexSessionIf = 0;
   _obexTransferIf = 0;
   _hfpCallVolumeIf = 0;
   _hfpHandsfreeIf = 0;
   _hfpManagerIf = 0;
   _hfpModemIf = 0;
   _hfpNetworkOperatorIf = 0;
   _hfpNetworkRegistrationIf = 0;
   _hfpSiriIf = 0;
   _hfpVoiceCallManagerIf = 0;
   _hfpVoiceCallIf = 0;
   _serialIf = 0;

   ConnectionOrgFreedesktopObjectManagerFactory::getInstance().destroyCb();
   _objectManagerCb = 0;
   if(NULL != _introspectableCb)
   {
      delete _introspectableCb;
   }
   PropertiesGetAllDbusFactory::getInstance().destroyCb();
   _propertiesGetAllCb = 0;

   ConnManDbusManagerFactory::getInstance().destroyCb();
   _connManManagerCb = 0;
   ConnManDbusServiceFactory::getInstance().destroyCb();
   _connManServiceCb = 0;
   ConnManDbusTechnologyFactory::getInstance().destroyCb();
   _connManTechnologyCb = 0;

   EvolutionGeniviAdapterFactory::getInstance().destroyCb();
   _adapterCb = 0;
   EvolutionGeniviAgentFactory::getInstance().destroyCb();
   _agentCb = 0;
   EvolutionGeniviDbusAgentManagerFactory::getInstance().destroyCb();
   _agentManagerCb = 0;
   EvolutionGeniviDbusDeviceFactory::getInstance().destroyCb();
   _deviceCb = 0;
   EvolutionGeniviDbusDMFactory::getInstance().destroyCb();
   _dmCb = 0;
   EvolutionGeniviDbusServiceFactory::getInstance().destroyCb();
   _serviceCb = 0;
   EvolutionGeniviDbusTraceFactory::getInstance().destroyCb();
   _traceCb = 0;
   EvolutionGeniviDbusGeniviTraceFactory::getInstance().destroyCb();
   _geniviTraceCb = 0;
   EvolutionGeniviDbusDidFactory::getInstance().destroyCb();
   _didCb = 0;
   EvolutionGeniviDbusSppFactory::getInstance().destroyCb();
   _sppCb = 0;
   EvolutionGeniviDbusObexClientFactory::getInstance().destroyCb();
   _obexClientCb = 0;
   EvolutionGeniviDbusObexSessionFactory::getInstance().destroyCb();
   _obexSessionCb = 0;
   EvolutionGeniviDbusObexTransferFactory::getInstance().destroyCb();
   _obexTransferCb = 0;
   EvolutionGeniviDbusHfpCallVolumeFactory::getInstance().destroyCb();
   _hfpCallVolumeCb = 0;
   EvolutionGeniviDbusHfpHandsfreeFactory::getInstance().destroyCb();
   _hfpHandsfreeCb = 0;
   EvolutionGeniviDbusHfpManagerFactory::getInstance().destroyCb();
   _hfpManagerCb = 0;
   EvolutionGeniviDbusHfpModemFactory::getInstance().destroyCb();
   _hfpModemCb = 0;
   EvolutionGeniviDbusHfpNetworkOperatorFactory::getInstance().destroyCb();
   _hfpNetworkOperatorCb = 0;
   EvolutionGeniviDbusHfpNetworkRegistrationFactory::getInstance().destroyCb();
   _hfpNetworkRegistrationCb = 0;
   EvolutionGeniviDbusHfpSiriFactory::getInstance().destroyCb();
   _hfpSiriCb = 0;
   EvolutionGeniviDbusHfpVoiceCallManagerFactory::getInstance().destroyCb();
   _hfpVoiceCallManagerCb = 0;
   EvolutionGeniviDbusHfpVoiceCallFactory::getInstance().destroyCb();
   _hfpVoiceCallCb = 0;
   EvolutionGeniviSerialFactory::getInstance().destroyCb();
   _serialCb = 0;

   destroyTimeoutControlList();
}

void DbusAccess::sendBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true != _dbusIfAvailable)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   if(NULL == _controllerClient)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   BTSFunctionBlock functionBlock = (BTSFunctionBlock)ptrMessage->getFunctionBlock();

   if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
   {
      // push to waiting queue
      _bts2IpcWaitingQueue.push(ptrMessage);
      // trigger ASF component thread
      _controllerClient->pushWorkItem(new DbusWorkItemTransmit(this));
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

void DbusAccess::stop(void)
{
   _terminateWorkerThread = true;
}

void DbusAccess::run(void)
{
   _terminateWorkerThread = false;
}

void DbusAccess::resetHandler(void)
{
   FW_IF_NULL_PTR_RETURN(_controllerClient);

   ETG_TRACE_USR1((" resetHandler(): start"));

   // clear waiting queue
   _bts2IpcWaitingQueue.empty();

   // clear working queue and reset timeout control list (create event for ASF component thread)
   _controllerClient->pushWorkItem(new DbusWorkItemResetTransmit(this));

   ETG_TRACE_USR1((" resetHandler(): end"));
}

BTSErrorCode DbusAccess::setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   ETG_TRACE_USR1((" setCcDbusIfControllerIf(): [enter]"));

   if(NULL == controller)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return BTS_INVALID_PARAM;
   }

   if(BTS_USER_MODE_LAST <= userMode)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return BTS_INVALID_PARAM;
   }

   // check if interfaces are already set (checking _controllerClient is enough)
   if(NULL != _controllerClient)
   {
      // already set
      return BTS_OK;
   }

   BTSErrorCode errCode = BTS_OK;

   _controllerClient = controller->getCcDbusIfControllerClient();

   ::ccdbusif::ICcDbusIfControllerConnMan* connManIf = controller->getCcDbusIfControllerConnMan();
   ::ccdbusif::ICcDbusIfControllerEvolutionGenivi* evolutionIf = controller->getCcDbusIfControllerEvolutionGenivi();
   ::ccdbusif::ICcDbusIfControllerObjectManager* objectManagerMainIf = controller->getCcDbusIfControllerObjectManager();
   ::ccdbusif::ICcDbusIfControllerIntrospectable* introspectableMainIf = controller->getCcDbusIfControllerIntrospectable();
   ::ccdbusif::ICcDbusIfControllerPropertiesGetAll* propertiesGetAllMainIf = controller->getCcDbusIfControllerPropertiesGetAll();

   if((NULL == _controllerClient) ||
      (NULL == connManIf) ||
      (NULL == evolutionIf) ||
      (NULL == objectManagerMainIf) ||
      (NULL == introspectableMainIf) ||
      (NULL == propertiesGetAllMainIf))
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      errCode = BTS_INVALID_PARAM;
   }
   else
   {
      if((BTS_USER_MODE_CONNECTION == userMode) ||
         (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
         (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
         (BTS_USER_MODE_ALL_WO_AGENT == userMode))
      {
         //---------------------------------------------------------------------
         _objectManagerIf = objectManagerMainIf->getObjectManagerProxy();
         if(0 == _objectManagerIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(0 == _objectManagerCb)
         {
            _objectManagerCb = ConnectionOrgFreedesktopObjectManagerFactory::getInstance().getCb(this);
         }
         if(0 == _objectManagerCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // get callback id
         if(0 != _objectManagerCb)
         {
            if(0 == _objectManagerCb->getCallbackId())
            {
               _objectManagerCb->setCallbackId(controller->getCallbackId());
            }
            if(0 == _objectManagerCb->getCallbackId())
            {
               FW_NORMAL_ASSERT_ALWAYS();
               errCode = BTS_INVALID_PARAM;
            }
         }
         //---------------------------------------------------------------------
         _introspectableIf = introspectableMainIf->getIntrospectableProxy();
         if(NULL == _introspectableIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _introspectableCb)
         {
            _introspectableCb = new IntrospectableDbusCallbackIf(this);
         }
         if(NULL == _introspectableCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _propertiesGetAllIf = propertiesGetAllMainIf->getPropertiesGetAllProxy();
         if(NULL == _propertiesGetAllIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _propertiesGetAllCb)
         {
            _propertiesGetAllCb = PropertiesGetAllDbusFactory::getInstance().getCb(this);
         }
         if(NULL == _propertiesGetAllCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _connManManagerIf = connManIf->getConnManManagerProxy();
         if(NULL == _connManManagerIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _connManManagerCb)
         {
            _connManManagerCb = ConnManDbusManagerFactory::getInstance().getCb(this);
         }
         if(NULL == _connManManagerCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _connManServiceIf = connManIf->getConnManServiceProxy();
         if(NULL == _connManServiceIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _connManServiceCb)
         {
            _connManServiceCb = ConnManDbusServiceFactory::getInstance().getCb(this);
         }
         if(NULL == _connManServiceCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _connManTechnologyIf = connManIf->getConnManTechnologyProxy();
         if(NULL == _connManTechnologyIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _connManTechnologyCb)
         {
            _connManTechnologyCb = ConnManDbusTechnologyFactory::getInstance().getCb(this);
         }
         if(NULL == _connManTechnologyCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _adapterIf = evolutionIf->getEvolutionGeniviAdapterProxy();
         if(NULL == _adapterIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _adapterCb)
         {
            _adapterCb = EvolutionGeniviAdapterFactory::getInstance().getCb(this);
         }
         if(NULL == _adapterCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // get callback id
         if(0 != _adapterCb)
         {
            if(0 == _adapterCb->getCallbackId())
            {
               _adapterCb->setCallbackId(controller->getCallbackId());
            }
            if(0 == _adapterCb->getCallbackId())
            {
               FW_NORMAL_ASSERT_ALWAYS();
               errCode = BTS_INVALID_PARAM;
            }
         }
         //---------------------------------------------------------------------
         _agentManagerIf = evolutionIf->getEvolutionGeniviAgentManagerProxy();
         if(NULL == _agentManagerIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _agentManagerCb)
         {
            _agentManagerCb = EvolutionGeniviDbusAgentManagerFactory::getInstance().getCb(this);
         }
         if(NULL == _agentManagerCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _deviceIf = evolutionIf->getEvolutionGeniviDeviceProxy();
         if(NULL == _deviceIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _deviceCb)
         {
            _deviceCb = EvolutionGeniviDbusDeviceFactory::getInstance().getCb(this);
         }
         if(NULL == _deviceCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _dmIf = evolutionIf->getEvolutionGeniviDMProxy();
         if(NULL == _dmIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _dmCb)
         {
            _dmCb = EvolutionGeniviDbusDMFactory::getInstance().getCb(this);
         }
         if(NULL == _dmCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _serviceIf = evolutionIf->getEvolutionGeniviServiceProxy();
         if(NULL == _serviceIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _serviceCb)
         {
            _serviceCb = EvolutionGeniviDbusServiceFactory::getInstance().getCb(this);
         }
         if(NULL == _serviceCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _traceIf = evolutionIf->getEvolutionGeniviTraceProxy();
         if(NULL == _traceIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _traceCb)
         {
            _traceCb = EvolutionGeniviDbusTraceFactory::getInstance().getCb(this);
         }
         if(NULL == _traceCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _geniviTraceIf = evolutionIf->getEvolutionGeniviGeniviTraceProxy();
         if(NULL == _geniviTraceIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _geniviTraceCb)
         {
            _geniviTraceCb = EvolutionGeniviDbusGeniviTraceFactory::getInstance().getCb(this);
         }
         if(NULL == _geniviTraceCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _didIf = evolutionIf->getEvolutionGeniviDidProxy();
         if(NULL == _didIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _didCb)
         {
            _didCb = EvolutionGeniviDbusDidFactory::getInstance().getCb(this);
         }
         if(NULL == _didCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _sppIf = evolutionIf->getEvolutionGeniviSppProxy();
         if(NULL == _sppIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _sppCb)
         {
            _sppCb = EvolutionGeniviDbusSppFactory::getInstance().getCb(this);
         }
         if(NULL == _sppCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _obexClientIf = evolutionIf->getEvolutionGeniviObexClientProxy();
         if(NULL == _obexClientIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _obexClientCb)
         {
            _obexClientCb = EvolutionGeniviDbusObexClientFactory::getInstance().getCb(this);
         }
         if(NULL == _obexClientCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _hfpHandsfreeIf = evolutionIf->getEvolutionGeniviHfpHandsfreeProxy();
         if(NULL == _hfpHandsfreeIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _hfpHandsfreeCb)
         {
            _hfpHandsfreeCb = EvolutionGeniviDbusHfpHandsfreeFactory::getInstance().getCb(this);
         }
         if(NULL == _hfpHandsfreeCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _hfpManagerIf = evolutionIf->getEvolutionGeniviHfpManagerProxy();
         if(NULL == _hfpManagerIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _hfpManagerCb)
         {
            _hfpManagerCb = EvolutionGeniviDbusHfpManagerFactory::getInstance().getCb(this);
         }
         if(NULL == _hfpManagerCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _hfpModemIf = evolutionIf->getEvolutionGeniviHfpModemProxy();
         if(NULL == _hfpModemIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _hfpModemCb)
         {
            _hfpModemCb = EvolutionGeniviDbusHfpModemFactory::getInstance().getCb(this);
         }
         if(NULL == _hfpModemCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         //---------------------------------------------------------------------
         _serialIf = evolutionIf->getEvolutionGeniviSerialProxy();
         if(NULL == _serialIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _serialCb)
         {
            _serialCb = EvolutionGeniviSerialFactory::getInstance().getCb(this);
         }
         if(NULL == _serialCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // get callback id
         if(0 != _serialCb)
         {
            if(0 == _serialCb->getCallbackId())
            {
               _serialCb->setCallbackId(controller->getCallbackId());
            }
            if(0 == _serialCb->getCallbackId())
            {
               FW_NORMAL_ASSERT_ALWAYS();
               errCode = BTS_INVALID_PARAM;
            }
         }
         //---------------------------------------------------------------------
      }

      if((BTS_USER_MODE_CONNECTION == userMode) ||
         (BTS_USER_MODE_ALL_WI_AGENT == userMode))
      {
         //---------------------------------------------------------------------
         _agentIf = evolutionIf->getEvolutionGeniviAgentStub();
         if(NULL == _agentIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // create callback handler
         if(NULL == _agentCb)
         {
            _agentCb = EvolutionGeniviAgentFactory::getInstance().getCb(this);
         }
         if(NULL == _agentCb)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            errCode = BTS_INVALID_PARAM;
         }
         // get callback id
         if(0 != _agentCb)
         {
            if(0 == _agentCb->getCallbackId())
            {
               _agentCb->setCallbackId(controller->getCallbackId());
            }
            if(0 == _agentCb->getCallbackId())
            {
               FW_NORMAL_ASSERT_ALWAYS();
               errCode = BTS_INVALID_PARAM;
            }
         }
         //---------------------------------------------------------------------
      }
   }

   if(BTS_OK != errCode)
   {
      // reset all
      _controllerClient = 0;
      _objectManagerIf = 0;
      _introspectableIf = 0;
      _propertiesGetAllIf = 0;
      _connManManagerIf = 0;
      _connManServiceIf = 0;
      _connManTechnologyIf = 0;
      _adapterIf = 0;
      _agentIf = 0;
      _agentManagerIf = 0;
      _deviceIf = 0;
      _dmIf = 0;
      _serviceIf = 0;
      _traceIf = 0;
      _geniviTraceIf = 0;
      _didIf = 0;
      _sppIf = 0;
      _obexClientIf = 0;
      _obexSessionIf = 0;
      _obexTransferIf = 0;
      _hfpCallVolumeIf = 0;
      _hfpHandsfreeIf = 0;
      _hfpManagerIf = 0;
      _hfpModemIf = 0;
      _hfpNetworkOperatorIf = 0;
      _hfpNetworkRegistrationIf = 0;
      _hfpSiriIf = 0;
      _hfpVoiceCallManagerIf = 0;
      _hfpVoiceCallIf = 0;
      _serialIf = 0;

      // do not delete callback handler
   }
   else
   {
      // set marker that D-Bus interface is available
      _dbusIfAvailable = true;

      // set callback handler (if no send interface and no callback handler were created nothing will happen during next step)
      for(size_t i = 0; i < dbusInterfaces.size(); i++)
      {
         switch(dbusInterfaces[i].dbusInterface)
         {
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM:
               if((0 != _objectManagerIf) && (0 != _objectManagerCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _objectManagerIf->setCallbackIf(_objectManagerCb, true, _objectManagerCb->getCallbackId(), dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);
                  _objectManagerIf->createProxyIf(_objectManagerCb->getCallbackId(), false, dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);

                  // set device interface for handle created and removed devices
                  _objectManagerCb->setDeviceSendIf(_deviceIf);
                  // set service interface for handle created and removed services
                  _objectManagerCb->setServiceSendIf(_serviceIf);
                  // set serial interface for handle created and removed serials
                  _objectManagerCb->setSerialSendIf(_serialIf);
                  _objectManagerCb->setSerialCb(_serialCb);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION:
               if((0 != _objectManagerIf) && (0 != _objectManagerCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _objectManagerIf->setCallbackIf(_objectManagerCb, true, _objectManagerCb->getCallbackId(), dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);
                  _objectManagerIf->createProxyIf(_objectManagerCb->getCallbackId(), false, dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_INTROSPECTABLE:
               if((NULL != _introspectableIf) && (NULL != _introspectableCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING, dbusInterfaces[i].busName, dbusInterfaces[i].objPath, dbusInterfaces[i].busType);
                  _introspectableIf->setCallbackIf(_introspectableCb, true, dbusInterfaces[i].busName, dbusInterfaces[i].objPath, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType, NULL);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL:
               if((NULL != _propertiesGetAllIf) && (NULL != _propertiesGetAllCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING, dbusInterfaces[i].busName, dbusInterfaces[i].objPath, dbusInterfaces[i].busType);
                  _propertiesGetAllIf->setCallbackIf(_propertiesGetAllCb, true, dbusInterfaces[i].busName, dbusInterfaces[i].objPath, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType, NULL);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_ADAPTER:
               if((0 != _adapterIf) && (0 != _adapterCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _adapterIf->setCallbackIf(_adapterCb, true, _adapterCb->getCallbackId());
                  _adapterIf->createProxyIf(_adapterCb->getCallbackId());
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT:
               if((0 != _agentIf) && (0 != _agentCb))
               {
                  // no service availability message
                  _agentIf->setCallbackIf(_agentCb, true, _agentCb->getCallbackId());
                  _agentIf->createStubIf(_agentCb->getCallbackId());
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_AGENT_MANAGER:
               if((NULL != _agentManagerIf) && (NULL != _agentManagerCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _agentManagerIf->setCallbackIf(_agentManagerCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DEVICE:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DM:
               if((NULL != _dmIf) && (NULL != _dmCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _dmIf->setCallbackIf(_dmCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_SERVICE:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_TRACE:
               if((NULL != _traceIf) && (NULL != _traceCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _traceIf->setCallbackIf(_traceCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_GENIVI_TRACE:
               if((NULL != _geniviTraceIf) && (NULL != _geniviTraceCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _geniviTraceIf->setCallbackIf(_geniviTraceCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DID:
               if((NULL != _didIf) && (NULL != _didCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _didIf->setCallbackIf(_didCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_SPP:
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_OBEX_CLIENT:
               if((NULL != _obexClientIf) && (NULL != _obexClientCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _obexClientIf->setCallbackIf(_obexClientCb, true);
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_SERIAL:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER:
               if((NULL != _connManManagerIf) && (NULL != _connManManagerCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _connManManagerIf->setCallbackIf(_connManManagerCb, true);

                  if(NULL != _connManServiceIf)
                  {
                       ConnManDbusManagerFactory::getInstance().setServiceIf(_connManServiceIf);
                  }
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_CONNMAN_TECHNOLOGY:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_HANDSFREE:
               // created during runtime
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_MANAGER:
               if((NULL != _hfpManagerIf) && (NULL != _hfpManagerCb))
               {
                  createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
                  _hfpManagerIf->setCallbackIf((::ccdbusif::IEvolutionGeniviDbusHfpManagerCallbackIf*)_hfpManagerCb, true);

                  // Manager needs interfaces for Handsfree + Modem
                  if((NULL != _hfpHandsfreeIf) && (NULL != _hfpModemIf))
                  {
                     EvolutionGeniviDbusHfpManagerFactory::getInstance().setHfpHandsfreeIf(_hfpHandsfreeIf);
                     EvolutionGeniviDbusHfpManagerFactory::getInstance().setHfpModemIf(_hfpModemIf);
                  }
               }
               break;
            case (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_MODEM:
               // created during runtime
               break;
            default:
               break;
         }
      }

      // set callback for proxies created during runtime TODO: handle in for-switch-block before
      if((NULL != _deviceIf) && (NULL != _deviceCb))
      {
         _deviceIf->setCallbackIf(_deviceCb, true);
      }
      if((NULL != _serviceIf) && (NULL != _serviceCb))
      {
         _serviceIf->setCallbackIf(_serviceCb, true);
      }
      if((NULL != _connManManagerIf) && (NULL != _connManManagerCb))
      {
         _connManManagerIf->setCallbackIf(_connManManagerCb, true);
         // Service needs interface for connamnService
         if(NULL != _connManServiceIf)
         {
            ConnManDbusManagerFactory::getInstance().setServiceIf(_connManServiceIf);
         }
      }
      if((NULL != _connManServiceIf) && (NULL != _connManServiceCb))
      {
         _connManServiceIf->setCallbackIf(_connManServiceCb, true);
      }
      if((NULL != _connManTechnologyIf) && (NULL != _connManTechnologyCb))
      {
         _connManTechnologyIf->setCallbackIf(_connManTechnologyCb, true);
      }
      if((NULL != _hfpHandsfreeIf) && (NULL != _hfpHandsfreeCb))
      {
         _hfpHandsfreeIf->setCallbackIf(_hfpHandsfreeCb, true);
      }
      if((NULL != _hfpModemIf) && (NULL != _hfpModemCb))
      {
         _hfpModemIf->setCallbackIf((::ccdbusif::IEvolutionGeniviDbusHfpModemCallbackIf*)_hfpModemCb, true);

         // Modem needs interface for Handsfree
         if(NULL != _hfpHandsfreeIf)
         {
            EvolutionGeniviDbusHfpModemFactory::getInstance().setHfpHandsfreeIf(_hfpHandsfreeIf);
         }
      }
      if((0 != _serialIf) && (0 != _serialCb))
      {
         // created during runtime - do not create service availability message
         _serialIf->setCallbackIf(_serialCb, true, _serialCb->getCallbackId());
         // created during runtime - do not create proxy
      }

      // check for Genivi compliant stacks and create extensions
      switch(stackInterface)
      {
         case BTS_IF_ALPS_EVOLUTION_GENIVI:
            createAlpsEvolutionGeniviExtensions(configuration.connectionConfiguration.configuration, userMode, stackInterface);
            break;
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }

      // initialize extensions
      initializeExtensions(controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
   }

   ::ccdbusif::evolution::printTypeSizes();

   ETG_TRACE_USR1((" setCcDbusIfControllerIf(): [exit] _dbusIfAvailable=%d", _dbusIfAvailable));

   return errCode;
}

void DbusAccess::resetCcDbusIfControllerIf(void)
{
   // check if interfaces are already reset (checking _controllerClient is enough)
   if(NULL == _controllerClient)
   {
      // already reset
      return;
   }

   // reset all
   _controllerClient = 0;
   _objectManagerIf = 0;
   _introspectableIf = 0;
   _propertiesGetAllIf = 0;
   _connManManagerIf = 0;
   _connManServiceIf = 0;
   _connManTechnologyIf = 0;
   _adapterIf = 0;
   _agentIf = 0;
   _agentManagerIf = 0;
   _deviceIf = 0;
   _dmIf = 0;
   _serviceIf = 0;
   _traceIf = 0;
   _geniviTraceIf = 0;
   _didIf = 0;
   _sppIf = 0;
   _obexClientIf = 0;
   _obexSessionIf = 0;
   _obexTransferIf = 0;
   _hfpCallVolumeIf = 0;
   _hfpHandsfreeIf = 0;
   _hfpManagerIf = 0;
   _hfpModemIf = 0;
   _hfpNetworkOperatorIf = 0;
   _hfpNetworkRegistrationIf = 0;
   _hfpSiriIf = 0;
   _hfpVoiceCallManagerIf = 0;
   _hfpVoiceCallIf = 0;
   _serialIf = 0;

   // destroy extensions
   destroyExtensions();

   // reset marker that D-Bus interface is available
   _dbusIfAvailable = false;

   // do not delete callback handler

   ETG_TRACE_USR1((" resetCcDbusIfControllerIf(): _dbusIfAvailable=%d", _dbusIfAvailable));
}

void DbusAccess::handleTimerTick(void)
{
   if(false == _timerHandlingEnabled)
   {
      return;
   }

   if(0 == _controllerClient)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // trigger ASF component thread
   _controllerClient->pushWorkItem(new DbusWorkItemTimeout(this));
}

void DbusAccess::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(0 == testCommand)
   {
      return;
   }

   if(0 == strcmp(BTS_SIMULATE_REQUEST_CONFIRMATION, testCommand))
   {
      if(0 != _objectManagerCb)
      {
         if(1 == testData)
         {
            _objectManagerCb->setSimulateRequestConfirmation(true);
         }
         else if(0 == testData)
         {
            _objectManagerCb->setSimulateRequestConfirmation(false);
         }
      }
   }
   else
   {
      // check extension
      FW_NORMAL_ASSERT(true == setExtensionsSimulationTestCommand(testCommand, testData));
   }
}

void DbusAccess::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)testData;

   if(0 == testCommand)
   {
      return;
   }

   {
      // check extension
      FW_NORMAL_ASSERT(true == setExtensionsSimulationTestCommand(testCommand, testData));
   }
}

void DbusAccess::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   if(0 == testCommand)
   {
      return;
   }

   if(0 == strcmp(BTS_SEND_VIRTUAL_IPC2BTS_MSG, testCommand))
   {
      // push to receive queue
      pushIpc2BtsMessage(testData.clone());
   }
   else
   {
      // check extension
      FW_NORMAL_ASSERT(true == setExtensionsSimulationTestCommand(testCommand, testData));
   }
}

void DbusAccess::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   BTSGenDbusServiceInterface serviceInterface = (BTSGenDbusServiceInterface)interface;

   if(BTS_GEN_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   pushIpc2BtsMessage(ptrMsg, true);
}

void DbusAccess::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   BTSGenDbusServiceInterface serviceInterface = (BTSGenDbusServiceInterface)interface;

   if(BTS_GEN_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SYSTEM != busType) && ((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SESSION != busType))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(NULL != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   pushIpc2BtsMessage(ptrMsg, true);
}

void DbusAccess::threadFunction(void* arguments)
{
   (void)arguments;
}

void DbusAccess::setTerminate(void* arguments)
{
   (void)arguments;
}

void DbusAccess::handleDbusTrmQueue(void)
{
   // HINT: This function is called within context of ASF component thread.

   int counter = 0;
   Bts2Ipc_BaseMessage* ptrMessage;
   BTSFunctionBlock functionBlock;

   while(counter < MAX_NUMBER_BTS2IPC_MGS_PROCESSING)
   {
      ptrMessage = _bts2IpcWaitingQueue.pop();
      if(NULL != ptrMessage)
      {
         // do trace
         ptrMessage->doOutputTrace();

         functionBlock = (BTSFunctionBlock)ptrMessage->getFunctionBlock();

         if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
         {
            switch(functionBlock)
            {
               case BTS_FB_CONFIG:
                  handleConfigurationMessage(ptrMessage);
                  break;
               case BTS_FB_CONNECTION:
                  handleConnectionMessage(ptrMessage);
                  break;
               case BTS_FB_TELEPHONY:
                  handlePhonecallMessage(ptrMessage);
                  break;
               case BTS_FB_PHONEBOOK:
                  handlePhonebookMessage(ptrMessage);
                  break;
               case BTS_FB_MESSAGING:
                  handleMessagingMessage(ptrMessage);
                  break;
               case BTS_FB_MEDIAPLAYER:
                  handleMediaPlayerMessage(ptrMessage);
                  break;
               default:
                  // should never happen else you have programmed something wrong
                  // #error_indication
                  FW_NORMAL_ASSERT_ALWAYS();
                  delete ptrMessage;
                  break;
            }
         }
         else
         {
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
            delete ptrMessage;
         }

         counter++;
      }
      else
      {
         counter = MAX_NUMBER_BTS2IPC_MGS_PROCESSING;
      }
   }
}

void DbusAccess::handleDbusResetTrmQueue(void)
{
   // HINT: This function is called within context of ASF component thread.

   // ThreadInfo threadInfo(this);
   // ETG_TRACE_USR1((" handleDbusResetTrmQueue(): thread=%s", threadInfo.getInfo()));

   FW_IF_NULL_PTR_RETURN(_controllerClient);

   ETG_TRACE_USR1((" handleDbusResetTrmQueue(): start"));

   // clear working queue
   _bts2IpcWorkingQueue.empty(false);

   // reset timeout control list
   releaseAllTimeoutControlEntries();

   // reset all Bluetooth related runtime proxies
   _controllerClient->destroyAllBluetoothRuntimeProxies();

   ETG_TRACE_USR1((" handleDbusResetTrmQueue(): end"));
}

void DbusAccess::handleReceivedDbusMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   // HINT: This function is called within context of ASF component thread.

   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // no check for _dbusIfAvailable necessary because this is receiving direction

   if(NULL == _controllerClient)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // check for high priority message; only signals are allowed to be high priority messages; no explicit check is implemented; programmer has to ensure correct implementation/usage
   if(true == highPrio)
   {
      // for signals no matching Bts2Ipc message is available in working queue therefore no check is necessary
      pushIpc2BtsMessage(ptrMessage, highPrio);
   }
   else
   {
      // push to callback queue and process directly
      _ipc2BtsCallbackQueue.push(ptrMessage, false);
      handleDbusCallbackMessage();
   }
}

void DbusAccess::handleDbusCallbackMessage(void)
{
   // HINT: This function is called within context of ASF component thread.

   int counter = 0;
   Ipc2Bts_BaseMessage* ptrIpc2BtsMessage;
   Bts2Ipc_BaseMessage* ptrBts2IpcQueueMessage;

   while(counter < MAX_NUMBER_IPC2BTS_MGS_PROCESSING)
   {
      ptrIpc2BtsMessage = _ipc2BtsCallbackQueue.pop(false);
      if(0 != ptrIpc2BtsMessage)
      {
         // find matching request message
         ptrBts2IpcQueueMessage = findAndRemoveBts2IpcWorkingMessage(ptrIpc2BtsMessage, false);

         if((0 < ptrIpc2BtsMessage->getErrorName().length()) || (0 < ptrIpc2BtsMessage->getErrorMessage().length()))
         {
            ETG_TRACE_USR1((" handleDbusCallbackMessage(): error name   =%s", ptrIpc2BtsMessage->getErrorName().c_str()));
            ETG_TRACE_USR1((" handleDbusCallbackMessage(): error message=%s", ptrIpc2BtsMessage->getErrorMessage().c_str()));
         }

         if(0 != ptrBts2IpcQueueMessage)
         {
            // remove timer entry
            releaseTimeoutControlEntry(ptrBts2IpcQueueMessage);

            ETG_TRACE_USR3((" handleDbusCallbackMessage(): Bts2Ipc 0x%08X found in working queue", ptrBts2IpcQueueMessage->getMessageTraceId()));

            // check dbus if handler list
            if(false == transferMessageDataLocal(ptrIpc2BtsMessage, ptrBts2IpcQueueMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               ETG_TRACE_FATAL((" handleDbusCallbackMessage(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
            }

            // copy meta data
            ptrIpc2BtsMessage->setBtsDestinationFunctionBlock(ptrBts2IpcQueueMessage->getBtsSourceFunctionBlock());
            ptrIpc2BtsMessage->setApp2BtsCompareItem(ptrBts2IpcQueueMessage->getApp2BtsCompareItem());

            // set response flag
            ptrIpc2BtsMessage->setResponseMessageFlag(true);

            delete ptrBts2IpcQueueMessage;
         }
         else
         {
            ETG_TRACE_USR3((" handleDbusCallbackMessage(): Bts2Ipc message not available => signal/indication message received"));
         }

         //check and remove Bts2Ipc working messages if object path was removed
         removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(ptrIpc2BtsMessage);

         pushIpc2BtsMessage(ptrIpc2BtsMessage);

         counter++;
      }
      else
      {
         counter = MAX_NUMBER_IPC2BTS_MGS_PROCESSING;
      }
   }
}

void DbusAccess::handleDbusTimeoutEvent(void)
{
   // HINT: This function is called within context of ASF component thread.

   _timerMaster.requestCurrentTime();
   _timerMaster.checkElapsedTimer();
}

void DbusAccess::handleDbusResponseTimeout(Bts2Ipc_BaseMessage* ptrMessage)
{
   // HINT: This function is called within context of ASF component thread.

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   Bts2Ipc_BaseMessage* ptrBts2IpcQueueMessage(ptrMessage);

   // remove request message
   removeBts2IpcWorkingMessage(ptrBts2IpcQueueMessage, false);

   // remove timer entry
   releaseTimeoutControlEntry(ptrBts2IpcQueueMessage);

   ETG_TRACE_ERR((" handleDbusResponseTimeout(): Bts2Ipc 0x%08X found in working queue", ptrBts2IpcQueueMessage->getMessageTraceId()));

   Ipc2Bts_BaseMessage* ipc2BtsMessage(0);

   // check dbus if handler list
   if(false == createErrorMessageLocal(&ipc2BtsMessage, ptrBts2IpcQueueMessage, BTS_IPC_METHOD_RETURN_TIMEOUT))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" handleDbusResponseTimeout(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
   }
   else
   {
      if(0 == ipc2BtsMessage)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" handleDbusResponseTimeout(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
      }
      else
      {
         // check dbus if handler list
         if(false == transferMessageDataLocal(ipc2BtsMessage, ptrBts2IpcQueueMessage))
         {
            FW_NORMAL_ASSERT_ALWAYS();
            ETG_TRACE_FATAL((" handleDbusResponseTimeout(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
         }

         // copy meta data
         ipc2BtsMessage->setBtsDestinationFunctionBlock(ptrBts2IpcQueueMessage->getBtsSourceFunctionBlock());
         ipc2BtsMessage->setApp2BtsCompareItem(ptrBts2IpcQueueMessage->getApp2BtsCompareItem());

         // set response flag
         ipc2BtsMessage->setResponseMessageFlag(true);

         pushIpc2BtsMessage(ipc2BtsMessage);
      }
   }

   delete ptrBts2IpcQueueMessage;
}

void DbusAccess::removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(Ipc2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if (0 < ptrMessage->getObjectId().size())
   {
      Bts2Ipc_BaseMessage* ptrBts2IpcQueueMessage = NULL;
      ::std::vector<Bts2Ipc_BaseMessage*> bts2IpcMsgList;

      // take messages from the queue for processing and remove them from the queue
      findAndRemoveBts2IpcWorkingMessages(bts2IpcMsgList, ptrMessage, false);

      for(size_t i = 0; i < bts2IpcMsgList.size(); i++)
      {
         ptrBts2IpcQueueMessage = bts2IpcMsgList[i];

         if (NULL != ptrBts2IpcQueueMessage)
         {
            // remove timer entry
            releaseTimeoutControlEntry(ptrBts2IpcQueueMessage);

            ETG_TRACE_ERR((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%08X found in working queue", ptrBts2IpcQueueMessage->getMessageTraceId()));

            Ipc2Bts_BaseMessage* ipc2BtsMessage(0);

            // check dbus if handler list
            if (false == createErrorMessageLocal(&ipc2BtsMessage, ptrBts2IpcQueueMessage, BTS_IPC_DBUS_ERROR_NO_SERVER))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               ETG_TRACE_FATAL((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
            }
            else
            {
               if (0 == ipc2BtsMessage)
               {
                  FW_NORMAL_ASSERT_ALWAYS();
                  ETG_TRACE_FATAL((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
               }
               else
               {
                  // check dbus if handler list
                  if(false == transferMessageDataLocal(ipc2BtsMessage, ptrBts2IpcQueueMessage))
                  {
                     FW_NORMAL_ASSERT_ALWAYS();
                     ETG_TRACE_FATAL((" removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(): Bts2Ipc 0x%04X not processed", ptrBts2IpcQueueMessage->getTraceOpCode()));
                  }

                  // copy meta data
                  ipc2BtsMessage->setBtsDestinationFunctionBlock(ptrBts2IpcQueueMessage->getBtsSourceFunctionBlock());
                  ipc2BtsMessage->setApp2BtsCompareItem(ptrBts2IpcQueueMessage->getApp2BtsCompareItem());

                  // set response flag
                  ipc2BtsMessage->setResponseMessageFlag(true);

                  pushIpc2BtsMessage(ipc2BtsMessage);
               }
            }

            delete ptrBts2IpcQueueMessage;
         }
      }
   }
}

void DbusAccess::sendErrorCode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, Bts2Ipc_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   Ipc2Bts_BaseMessage* ipc2BtsMessage(0);

   // check dbus if handler list
   if(false == createErrorMessageLocal(&ipc2BtsMessage, ptrMessage, BTS_IPC_SENDING_FAILED))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_FATAL((" sendErrorCode(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
   }
   else
   {
      if(0 == ipc2BtsMessage)
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" sendErrorCode(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
      }
      else
      {
         // check dbus if handler list
         if(false == transferMessageDataLocal(ipc2BtsMessage, ptrMessage))
         {
            FW_NORMAL_ASSERT_ALWAYS();
            ETG_TRACE_FATAL((" sendErrorCode(): Bts2Ipc 0x%04X not processed", ptrMessage->getTraceOpCode()));
         }

         // copy meta data
         ipc2BtsMessage->setBtsDestinationFunctionBlock(ptrMessage->getBtsSourceFunctionBlock());
         ipc2BtsMessage->setApp2BtsCompareItem(ptrMessage->getApp2BtsCompareItem());

         // set response flag
         ipc2BtsMessage->setResponseMessageFlag(true);

         sendIpc2BtsMsgList.push_back(ipc2BtsMessage);
      }
   }
}

bool DbusAccess::createErrorMessageLocal(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN_FALSE(bts2IpcMessage);
   const Bts2Ipc_BaseMessage& messageRef(*bts2IpcMessage);

   bool handled(true);
   const BTSFunctionBlock functionBlock((BTSFunctionBlock)messageRef.getFunctionBlock());

   if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
   {
      switch(functionBlock)
      {
         case BTS_FB_CONFIG:
            *ipc2BtsMessage = createIpc2BtsConfigurationErrorMessage(messageRef);
            break;
         case BTS_FB_CONNECTION:
            *ipc2BtsMessage = createIpc2BtsConnectionErrorMessage(messageRef);
            break;
         case BTS_FB_TELEPHONY:
            *ipc2BtsMessage = createIpc2BtsPhonecallErrorMessage(messageRef);
            break;
         case BTS_FB_PHONEBOOK:
            *ipc2BtsMessage = createIpc2BtsPhonebookErrorMessage(messageRef);
            break;
         case BTS_FB_MESSAGING:
            *ipc2BtsMessage = createIpc2BtsMessagingErrorMessage(messageRef);
            break;
         case BTS_FB_MEDIAPLAYER:
            *ipc2BtsMessage = createIpc2BtsMediaPlayerErrorMessage(messageRef);
            break;
         default:
            handled = false;
            break;
      }
   }
   else
   {
      handled = false;
   }

   if(0 != (*ipc2BtsMessage))
   {
      (*ipc2BtsMessage)->setIpcCommonErrorCode(errorCode);
   }

   return handled;
}

bool DbusAccess::transferMessageDataLocal(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN_FALSE(bts2IpcMessage);
   // Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   bool handled(true);
   const BTSFunctionBlock functionBlock((BTSFunctionBlock)bts2IpcMessageRef.getFunctionBlock());

   if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
   {
      switch(functionBlock)
      {
         case BTS_FB_CONFIG:
            transferMessageDataConfiguration(ipc2BtsMessage, bts2IpcMessage);
            break;
         case BTS_FB_CONNECTION:
            transferMessageDataConnection(ipc2BtsMessage, bts2IpcMessage);
            break;
         case BTS_FB_TELEPHONY:
            transferMessageDataPhonecall(ipc2BtsMessage, bts2IpcMessage);
            break;
         case BTS_FB_PHONEBOOK:
            transferMessageDataPhonebook(ipc2BtsMessage, bts2IpcMessage);
            break;
         case BTS_FB_MESSAGING:
            transferMessageDataMessaging(ipc2BtsMessage, bts2IpcMessage);
            break;
         case BTS_FB_MEDIAPLAYER:
            transferMessageDataMediaPlayer(ipc2BtsMessage, bts2IpcMessage);
            break;
         default:
            handled = false;
            break;
      }
   }
   else
   {
      handled = false;
   }

   return handled;
}

void DbusAccess::handleConfigurationMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   (void)(ptrMessage);
}

void DbusAccess::handleConnectionMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSGenBts2IpcOpcode opCode = (BTSGenBts2IpcOpcode)ptrMessage->getOpCode();

   if((Bts2IpcOC_ConnectionBlockStart < opCode) && (opCode < Bts2IpcOC_ConnectionBlockEnd))
   {
      ::std::vector<Ipc2Bts_BaseMessage*> sendIpc2BtsMsgList;
      sendIpc2BtsMsgList.reserve(10);
      bool deleteBts2IpcMsg = false;

      switch(opCode)
      {
         //------------------------------------------------------------------------------
         // class Bts2Ipc_GetAllPropertiesConnection;
         // '         class Bts2Ipc_(.*);' => '         case Bts2IpcOC_$1:\R            handleBts2Ipc$1(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_$1*>(ptrMessage));\R            break;'
         //------------------------------------------------------------------------------
         case Bts2IpcOC_GetManagedObjectsConnection:
            handleBts2IpcGetManagedObjectsConnection(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetManagedObjectsConnection*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAllPropertiesConnection:
            handleBts2IpcGetAllPropertiesConnection(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAllPropertiesConnection*>(ptrMessage));
            break;
         case Bts2IpcOC_GetIntrospectionConnection:
            handleBts2IpcGetIntrospectionConnection(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetIntrospectionConnection*>(ptrMessage));
            break;
         case Bts2IpcOC_ReleaseRes:
            handleBts2IpcReleaseRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_ReleaseRes*>(ptrMessage));
            break;
         case Bts2IpcOC_RequestPinCodeRes:
            handleBts2IpcRequestPinCodeRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RequestPinCodeRes*>(ptrMessage));
            break;
         case Bts2IpcOC_DisplayPinCodeRes:
            handleBts2IpcDisplayPinCodeRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_DisplayPinCodeRes*>(ptrMessage));
            break;
         case Bts2IpcOC_RequestPasskeyRes:
            handleBts2IpcRequestPasskeyRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RequestPasskeyRes*>(ptrMessage));
            break;
         case Bts2IpcOC_RequestConfirmationRes:
            handleBts2IpcRequestConfirmationRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RequestConfirmationRes*>(ptrMessage));
            break;
         case Bts2IpcOC_CancelRes:
            handleBts2IpcCancelRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CancelRes*>(ptrMessage));
            break;
         case Bts2IpcOC_CancelRequestRes:
            handleBts2IpcCancelRequestRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CancelRequestRes*>(ptrMessage));
            break;
         case Bts2IpcOC_AuthorizeServiceRes:
            handleBts2IpcAuthorizeServiceRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_AuthorizeServiceRes*>(ptrMessage));
            break;
         case Bts2IpcOC_RequestAuthorizationRes:
            handleBts2IpcRequestAuthorizationRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RequestAuthorizationRes*>(ptrMessage));
            break;
         case Bts2IpcOC_RequestLinkkeyRes:
            handleBts2IpcRequestLinkkeyRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RequestLinkkeyRes*>(ptrMessage));
            break;
         case Bts2IpcOC_DisplayPasskeyRes:
            handleBts2IpcDisplayPasskeyRes(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_DisplayPasskeyRes*>(ptrMessage));
            break;
         case Bts2IpcOC_StartDiscovery:
            handleBts2IpcStartDiscovery(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_StartDiscovery*>(ptrMessage));
            break;
         case Bts2IpcOC_StopDiscovery:
            handleBts2IpcStopDiscovery(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_StopDiscovery*>(ptrMessage));
            break;
         case Bts2IpcOC_RemoveDevice:
            handleBts2IpcRemoveDevice(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RemoveDevice*>(ptrMessage));
            break;
         case Bts2IpcOC_CreateDevice:
            handleBts2IpcCreateDevice(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CreateDevice*>(ptrMessage));
            break;
         case Bts2IpcOC_SendHCICmd:
            handleBts2IpcSendHCICmd(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SendHCICmd*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterAddress:
            handleBts2IpcGetAdapterAddress(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterAddress*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterClass:
            handleBts2IpcGetAdapterClass(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterClass*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterAlias:
            handleBts2IpcGetAdapterAlias(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterAlias*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterAlias:
            handleBts2IpcSetAdapterAlias(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterAlias*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterPowered:
            handleBts2IpcGetAdapterPowered(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterPowered*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterPowered:
            handleBts2IpcSetAdapterPowered(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterPowered*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterDiscoverable:
            handleBts2IpcGetAdapterDiscoverable(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterDiscoverable*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterDiscoverable:
            handleBts2IpcSetAdapterDiscoverable(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterDiscoverable*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterDiscoverabletimeout:
            handleBts2IpcGetAdapterDiscoverabletimeout(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterDiscoverabletimeout*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterDiscoverabletimeout:
            handleBts2IpcSetAdapterDiscoverabletimeout(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterDiscoverabletimeout*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterPairable:
            handleBts2IpcGetAdapterPairable(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterPairable*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterPairable:
            handleBts2IpcSetAdapterPairable(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterPairable*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterPairabletimeout:
            handleBts2IpcGetAdapterPairabletimeout(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterPairabletimeout*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterPairabletimeout:
            handleBts2IpcSetAdapterPairabletimeout(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterPairabletimeout*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterDiscovering:
            handleBts2IpcGetAdapterDiscovering(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterDiscovering*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterUuids:
            handleBts2IpcGetAdapterUuids(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterUuids*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterUuids:
            handleBts2IpcSetAdapterUuids(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterUuids*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterWbsMode:
            handleBts2IpcGetAdapterWbsMode(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterWbsMode*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterWbsMode:
            handleBts2IpcSetAdapterWbsMode(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterWbsMode*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterConnectable:
            handleBts2IpcGetAdapterConnectable(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterConnectable*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterConnectable:
            handleBts2IpcSetAdapterConnectable(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterConnectable*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterHcimode:
            handleBts2IpcGetAdapterHcimode(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterHcimode*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterHcimode:
            handleBts2IpcSetAdapterHcimode(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterHcimode*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterVersion:
            handleBts2IpcGetAdapterVersion(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterVersion*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterTraceLevel:
            handleBts2IpcGetAdapterTraceLevel(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterTraceLevel*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterTraceLevel:
            handleBts2IpcSetAdapterTraceLevel(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterTraceLevel*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterTraceCategory:
            handleBts2IpcGetAdapterTraceCategory(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterTraceCategory*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterTraceCategory:
            handleBts2IpcSetAdapterTraceCategory(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterTraceCategory*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAdapterAvailableSppServers:
            handleBts2IpcGetAdapterAvailableSppServers(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAdapterAvailableSppServers*>(ptrMessage));
            break;
         case Bts2IpcOC_SetAdapterAvailableSppServers:
            handleBts2IpcSetAdapterAvailableSppServers(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetAdapterAvailableSppServers*>(ptrMessage));
            break;
         case Bts2IpcOC_RegisterAgent:
            handleBts2IpcRegisterAgent(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RegisterAgent*>(ptrMessage));
            break;
         case Bts2IpcOC_UnregisterAgent:
            handleBts2IpcUnregisterAgent(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_UnregisterAgent*>(ptrMessage));
            break;
         case Bts2IpcOC_RequestDefaultAgent:
            handleBts2IpcRequestDefaultAgent(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RequestDefaultAgent*>(ptrMessage));
            break;
         case Bts2IpcOC_DeviceConnect:
            handleBts2IpcDeviceConnect(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_DeviceConnect*>(ptrMessage));
            break;
         case Bts2IpcOC_DeviceDisconnect:
            handleBts2IpcDeviceDisconnect(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_DeviceDisconnect*>(ptrMessage));
            break;
         case Bts2IpcOC_Pair:
            handleBts2IpcPair(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_Pair*>(ptrMessage));
            break;
         case Bts2IpcOC_CancelPairing:
            handleBts2IpcCancelPairing(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CancelPairing*>(ptrMessage));
            break;
         case Bts2IpcOC_DiscoverServices:
            handleBts2IpcDiscoverServices(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_DiscoverServices*>(ptrMessage));
            break;
         case Bts2IpcOC_CancelDiscovery:
            handleBts2IpcCancelDiscovery(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CancelDiscovery*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceAddress:
            handleBts2IpcGetDeviceAddress(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceAddress*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceClass:
            handleBts2IpcGetDeviceClass(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceClass*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceIcon:
            handleBts2IpcGetDeviceIcon(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceIcon*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceUuids:
            handleBts2IpcGetDeviceUuids(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceUuids*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDevicePaired:
            handleBts2IpcGetDevicePaired(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDevicePaired*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceConnected:
            handleBts2IpcGetDeviceConnected(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceConnected*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceTrusted:
            handleBts2IpcGetDeviceTrusted(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceTrusted*>(ptrMessage));
            break;
         case Bts2IpcOC_SetDeviceTrusted:
            handleBts2IpcSetDeviceTrusted(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetDeviceTrusted*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceBlocked:
            handleBts2IpcGetDeviceBlocked(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceBlocked*>(ptrMessage));
            break;
         case Bts2IpcOC_SetDeviceBlocked:
            handleBts2IpcSetDeviceBlocked(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetDeviceBlocked*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceAlias:
            handleBts2IpcGetDeviceAlias(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceAlias*>(ptrMessage));
            break;
         case Bts2IpcOC_SetDeviceAlias:
            handleBts2IpcSetDeviceAlias(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetDeviceAlias*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceAdapter:
            handleBts2IpcGetDeviceAdapter(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceAdapter*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceModalias:
            handleBts2IpcGetDeviceModalias(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceModalias*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceRssi:
            handleBts2IpcGetDeviceRssi(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceRssi*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceVendorId:
            handleBts2IpcGetDeviceVendorId(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceVendorId*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceVendorIdSource:
            handleBts2IpcGetDeviceVendorIdSource(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceVendorIdSource*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceProductId:
            handleBts2IpcGetDeviceProductId(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceProductId*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceMapInfo:
            handleBts2IpcGetDeviceMapInfo(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceMapInfo*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceConnectedService:
            handleBts2IpcGetDeviceConnectedService(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceConnectedService*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceVersion:
            handleBts2IpcGetDeviceVersion(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceVersion*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDevicePbapInfo:
            handleBts2IpcGetDevicePbapInfo(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDevicePbapInfo*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceMode:
            handleBts2IpcGetDeviceMode(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceMode*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceRole:
            handleBts2IpcGetDeviceRole(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceRole*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDeviceEnableAvpPause:
            handleBts2IpcGetDeviceEnableAvpPause(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDeviceEnableAvpPause*>(ptrMessage));
            break;
         case Bts2IpcOC_SetDeviceEnableAvpPause:
            handleBts2IpcSetDeviceEnableAvpPause(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetDeviceEnableAvpPause*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplDeviceConnectReq:
            handleBts2IpcBtApplDeviceConnectReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplDeviceConnectReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplDeviceDisconnectReq:
            handleBts2IpcBtApplDeviceDisconnectReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplDeviceDisconnectReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplCancelConnectReq:
            handleBts2IpcBtApplCancelConnectReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplCancelConnectReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplGetHwVersionReq:
            handleBts2IpcBtApplGetHwVersionReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplGetHwVersionReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplSetConfigurationReq:
            handleBts2IpcBtApplSetConfigurationReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplSetConfigurationReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplReadConfigurationReq:
            handleBts2IpcBtApplReadConfigurationReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplReadConfigurationReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplTestModeReq:
            handleBts2IpcBtApplTestModeReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplTestModeReq*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplTestModeLinkQualityReq:
            handleBts2IpcBtApplTestModeLinkQualityReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplTestModeLinkQualityReq*>(ptrMessage));
            break;
         case Bts2IpcOC_Connect:
            handleBts2IpcConnect(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_Connect*>(ptrMessage));
            break;
         case Bts2IpcOC_Disconnect:
            handleBts2IpcDisconnect(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_Disconnect*>(ptrMessage));
            break;
         case Bts2IpcOC_GetServiceRemoteUuid:
            handleBts2IpcGetServiceRemoteUuid(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetServiceRemoteUuid*>(ptrMessage));
            break;
         case Bts2IpcOC_GetServiceState:
            handleBts2IpcGetServiceState(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetServiceState*>(ptrMessage));
            break;
         case Bts2IpcOC_GetServiceDevice:
            handleBts2IpcGetServiceDevice(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetServiceDevice*>(ptrMessage));
            break;
         case Bts2IpcOC_GetServiceVersion:
            handleBts2IpcGetServiceVersion(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetServiceVersion*>(ptrMessage));
            break;
         case Bts2IpcOC_DebugTraceConfig:
            handleBts2IpcDebugTraceConfig(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_DebugTraceConfig*>(ptrMessage));
            break;
         case Bts2IpcOC_ProtocolTraceConfig:
            handleBts2IpcProtocolTraceConfig(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_ProtocolTraceConfig*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplDidSetReq:
            handleBts2IpcBtApplDidSetReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplDidSetReq*>(ptrMessage));
            break;
         case Bts2IpcOC_GetDunDeviceFilePath:
            handleBts2IpcGetDunDeviceFilePath(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetDunDeviceFilePath*>(ptrMessage));
            break;
         case Bts2IpcOC_BtApplSppSetUuidReq:
            handleBts2IpcBtApplSppSetUuidReq(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_BtApplSppSetUuidReq*>(ptrMessage));
            break;
         case Bts2IpcOC_CreateSession:
            handleBts2IpcCreateSession(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CreateSession*>(ptrMessage));
            break;
         case Bts2IpcOC_RemoveSession:
            handleBts2IpcRemoveSession(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_RemoveSession*>(ptrMessage));
            break;
         case Bts2IpcOC_CancelSession:
            handleBts2IpcCancelSession(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_CancelSession*>(ptrMessage));
            break;
         case Bts2IpcOC_GetConnManManagerProperties:
            handleBts2IpcGetConnManManagerProperties(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetConnManManagerProperties*>(ptrMessage));
            break;
         case Bts2IpcOC_GetConnManTechnologies:
            handleBts2IpcGetConnManTechnologies(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetConnManTechnologies*>(ptrMessage));
            break;
         case Bts2IpcOC_GetConnManServices:
            handleBts2IpcGetConnManServices(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetConnManServices*>(ptrMessage));
            break;
         case Bts2IpcOC_GetConnManServiceProperties:
            handleBts2IpcGetConnManServiceProperties(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetConnManServiceProperties*>(ptrMessage));
            break;
         case Bts2IpcOC_ConnManServiceConnect:
            handleBts2IpcConnManServiceConnect(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_ConnManServiceConnect*>(ptrMessage));
            break;
         case Bts2IpcOC_ConnManServiceDisconnect:
            handleBts2IpcConnManServiceDisconnect(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_ConnManServiceDisconnect*>(ptrMessage));
            break;
         default:
         {
            bool sendErrorIpc2BtsMsg(false); // TODO: correct implementation
            // check extension
            if(false == sendExtensionConnectionMessage(deleteBts2IpcMsg, sendErrorIpc2BtsMsg, ptrMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               deleteBts2IpcMsg = true;
            }
            break;
         }
      }

      if(true == deleteBts2IpcMsg)
      {
         // error case
         delete ptrMessage;
      }
      else
      {
         handleSendingSuccess(ptrMessage);
      }

      sendIpc2BtsMessageList(sendIpc2BtsMsgList);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

void DbusAccess::handlePhonecallMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSGenBts2IpcOpcode opCode = (BTSGenBts2IpcOpcode)ptrMessage->getOpCode();

   if((Bts2IpcOC_TelephonyBlockStart < opCode) && (opCode < Bts2IpcOC_TelephonyBlockEnd))
   {
      ::std::vector<Ipc2Bts_BaseMessage*> sendIpc2BtsMsgList;
      sendIpc2BtsMsgList.reserve(10);
      bool deleteBts2IpcMsg = false;

      switch(opCode)
      {
         case Bts2IpcOC_GetManagedObjectsTelephony:
            handleBts2IpcGetManagedObjectsTelephony(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetManagedObjectsTelephony*>(ptrMessage));
            break;
         case Bts2IpcOC_GetAllPropertiesTelephony:
            handleBts2IpcGetAllPropertiesTelephony(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetAllPropertiesTelephony*>(ptrMessage));
            break;
         case Bts2IpcOC_GetIntrospectionTelephony:
            handleBts2IpcGetIntrospectionTelephony(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetIntrospectionTelephony*>(ptrMessage));
            break;
         case Bts2IpcOC_GetHandsfreeProperties:
            handleBts2IpcGetHandsfreeProperties(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetHandsfreeProperties*>(ptrMessage));
            break;
         case Bts2IpcOC_GetModems:
            handleBts2IpcGetModems(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetModems*>(ptrMessage));
            break;
         case Bts2IpcOC_GetModemProperties:
            handleBts2IpcGetModemProperties(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_GetModemProperties*>(ptrMessage));
            break;
         default:
         {
            bool sendErrorIpc2BtsMsg(false); // TODO: correct implementation
            // check extension
            if(false == sendExtensionPhonecallMessage(deleteBts2IpcMsg, sendErrorIpc2BtsMsg, ptrMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               deleteBts2IpcMsg = true;
            }
            break;
         }
      }

      if(true == deleteBts2IpcMsg)
      {
         // error case
         delete ptrMessage;
      }
      else
      {
         handleSendingSuccess(ptrMessage);
      }

      sendIpc2BtsMessageList(sendIpc2BtsMsgList);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

void DbusAccess::handlePhonebookMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSGenBts2IpcOpcode opCode = (BTSGenBts2IpcOpcode)ptrMessage->getOpCode();

   if((Bts2IpcOC_PhonebookBlockStart < opCode) && (opCode < Bts2IpcOC_PhonebookBlockEnd))
   {
      // TODO
#if 0
      ::std::vector<Ipc2Bts_BaseMessage*> sendIpc2BtsMsgList;
      sendIpc2BtsMsgList.reserve(10);
      bool deleteBts2IpcMsg = false;

      switch(opCode)
      {
         // TODO: example
         //case Bts2IpcOC_SetManagerConfiguration:
         //   handleBts2IpcSetManagerConfiguration(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetManagerConfiguration*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         //   break;
         default:
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
            deleteBts2IpcMsg = true;
            break;
      }

      if(true == deleteBts2IpcMsg)
      {
         // error case
         delete ptrMessage;
      }
      else
      {
         handleSendingSuccess(ptrMessage);
      }

      sendIpc2BtsMessageList(sendIpc2BtsMsgList);
#endif
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

void DbusAccess::handleMessagingMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSGenBts2IpcOpcode opCode = (BTSGenBts2IpcOpcode)ptrMessage->getOpCode();

   if((Bts2IpcOC_MessagingBlockStart < opCode) && (opCode < Bts2IpcOC_MessagingBlockEnd))
   {
      // TODO
#if 0
      ::std::vector<Ipc2Bts_BaseMessage*> sendIpc2BtsMsgList;
      sendIpc2BtsMsgList.reserve(10);
      bool deleteBts2IpcMsg = false;

      switch(opCode)
      {
         // TODO: example
         //case Bts2IpcOC_SetManagerConfiguration:
         //   handleBts2IpcSetManagerConfiguration(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetManagerConfiguration*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         //   break;
         default:
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
            deleteBts2IpcMsg = true;
            break;
      }

      if(true == deleteBts2IpcMsg)
      {
         // error case
         delete ptrMessage;
      }
      else
      {
         handleSendingSuccess(ptrMessage);
      }

      sendIpc2BtsMessageList(sendIpc2BtsMsgList);
#endif
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

void DbusAccess::handleMediaPlayerMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSGenBts2IpcOpcode opCode = (BTSGenBts2IpcOpcode)ptrMessage->getOpCode();

   if((Bts2IpcOC_MediaPlayerBlockStart < opCode) && (opCode < Bts2IpcOC_MediaPlayerBlockEnd))
   {
      // TODO
#if 0
      ::std::vector<Ipc2Bts_BaseMessage*> sendIpc2BtsMsgList;
      sendIpc2BtsMsgList.reserve(10);
      bool deleteBts2IpcMsg = false;

      switch(opCode)
      {
         // TODO: example
         //case Bts2IpcOC_SetManagerConfiguration:
         //   handleBts2IpcSetManagerConfiguration(sendIpc2BtsMsgList, deleteBts2IpcMsg, static_cast<Bts2Ipc_SetManagerConfiguration*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
         //   break;
         default:
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
            deleteBts2IpcMsg = true;
            break;
      }

      if(true == deleteBts2IpcMsg)
      {
         // error case
         delete ptrMessage;
      }
      else
      {
         handleSendingSuccess(ptrMessage);
      }

      sendIpc2BtsMessageList(sendIpc2BtsMsgList);
#endif
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }
}

Ipc2Bts_BaseMessage* DbusAccess::createIpc2BtsConfigurationErrorMessage(IN const Bts2Ipc_BaseMessage& message) const
{
   (void)(message);

   return 0;
}

Ipc2Bts_BaseMessage* DbusAccess::createIpc2BtsConnectionErrorMessage(IN const Bts2Ipc_BaseMessage& message) const
{
   Ipc2Bts_BaseMessage* returnMsg(0);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)message.getOpCode());

   if((Bts2IpcOC_ConnectionBlockStart < opCode) && (opCode < Bts2IpcOC_ConnectionBlockEnd))
   {
      switch(opCode)
      {
         case Bts2IpcOC_GetManagedObjectsConnection:
            // no result message
            break;
         case Bts2IpcOC_GetAllPropertiesConnection:
            returnMsg = ptrNew_Ipc2Bts_GetAllPropertiesConnection();
            break;
         case Bts2IpcOC_GetIntrospectionConnection:
            // no result message
            break;
         case Bts2IpcOC_ReleaseRes:
            // no result message
            break;
         case Bts2IpcOC_RequestPinCodeRes:
            // no result message
            break;
         case Bts2IpcOC_DisplayPinCodeRes:
            // no result message
            break;
         case Bts2IpcOC_RequestPasskeyRes:
            // no result message
            break;
         case Bts2IpcOC_RequestConfirmationRes:
            // no result message
            break;
         case Bts2IpcOC_CancelRes:
            // no result message
            break;
         case Bts2IpcOC_CancelRequestRes:
            // no result message
            break;
         case Bts2IpcOC_AuthorizeServiceRes:
            // no result message
            break;
         case Bts2IpcOC_RequestAuthorizationRes:
            // no result message
            break;
         case Bts2IpcOC_RequestLinkkeyRes:
            // no result message
            break;
         case Bts2IpcOC_DisplayPasskeyRes:
            // no result message
            break;
         case Bts2IpcOC_StartDiscovery:
            returnMsg = ptrNew_Ipc2Bts_StartDiscoveryResult();
            break;
         case Bts2IpcOC_StopDiscovery:
            returnMsg = ptrNew_Ipc2Bts_StopDiscoveryResult();
            break;
         case Bts2IpcOC_RemoveDevice:
            returnMsg = ptrNew_Ipc2Bts_RemoveDeviceResult();
            break;
         case Bts2IpcOC_CreateDevice:
            returnMsg = ptrNew_Ipc2Bts_CreateDeviceResult();
            break;
         case Bts2IpcOC_SendHCICmd:
            returnMsg = ptrNew_Ipc2Bts_SendHCICmdResult();
            break;
         case Bts2IpcOC_GetAdapterAddress:
            returnMsg = ptrNew_Ipc2Bts_AdapterAddressUpdate();
            break;
         case Bts2IpcOC_GetAdapterClass:
            returnMsg = ptrNew_Ipc2Bts_AdapterClassUpdate();
            break;
         case Bts2IpcOC_GetAdapterAlias:
            returnMsg = ptrNew_Ipc2Bts_AdapterAliasUpdate();
            break;
         case Bts2IpcOC_SetAdapterAlias:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterPowered:
            returnMsg = ptrNew_Ipc2Bts_AdapterPoweredUpdate();
            break;
         case Bts2IpcOC_SetAdapterPowered:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterDiscoverable:
            returnMsg = ptrNew_Ipc2Bts_AdapterDiscoverableUpdate();
            break;
         case Bts2IpcOC_SetAdapterDiscoverable:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterDiscoverabletimeout:
            returnMsg = ptrNew_Ipc2Bts_AdapterDiscoverabletimeoutUpdate();
            break;
         case Bts2IpcOC_SetAdapterDiscoverabletimeout:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterPairable:
            returnMsg = ptrNew_Ipc2Bts_AdapterPairableUpdate();
            break;
         case Bts2IpcOC_SetAdapterPairable:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterPairabletimeout:
            returnMsg = ptrNew_Ipc2Bts_AdapterPairabletimeoutUpdate();
            break;
         case Bts2IpcOC_SetAdapterPairabletimeout:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterDiscovering:
            returnMsg = ptrNew_Ipc2Bts_AdapterDiscoveringUpdate();
            break;
         case Bts2IpcOC_GetAdapterUuids:
            returnMsg = ptrNew_Ipc2Bts_AdapterUuidsUpdate();
            break;
         case Bts2IpcOC_SetAdapterUuids:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterWbsMode:
            returnMsg = ptrNew_Ipc2Bts_AdapterWbsModeUpdate();
            break;
         case Bts2IpcOC_SetAdapterWbsMode:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterConnectable:
            returnMsg = ptrNew_Ipc2Bts_AdapterConnectableUpdate();
            break;
         case Bts2IpcOC_SetAdapterConnectable:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterHcimode:
            returnMsg = ptrNew_Ipc2Bts_AdapterHcimodeUpdate();
            break;
         case Bts2IpcOC_SetAdapterHcimode:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterVersion:
            returnMsg = ptrNew_Ipc2Bts_AdapterVersionUpdate();
            break;
         case Bts2IpcOC_GetAdapterTraceLevel:
            returnMsg = ptrNew_Ipc2Bts_AdapterTraceLevelUpdate();
            break;
         case Bts2IpcOC_SetAdapterTraceLevel:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterTraceCategory:
            returnMsg = ptrNew_Ipc2Bts_AdapterTraceCategoryUpdate();
            break;
         case Bts2IpcOC_SetAdapterTraceCategory:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetAdapterAvailableSppServers:
            returnMsg = ptrNew_Ipc2Bts_AdapterAvailableSppServersUpdate();
            break;
         case Bts2IpcOC_SetAdapterAvailableSppServers:
            // no method return on ASF level
            break;
         case Bts2IpcOC_RegisterAgent:
            returnMsg = ptrNew_Ipc2Bts_RegisterAgentResult();
            break;
         case Bts2IpcOC_UnregisterAgent:
            returnMsg = ptrNew_Ipc2Bts_UnregisterAgentResult();
            break;
         case Bts2IpcOC_RequestDefaultAgent:
            returnMsg = ptrNew_Ipc2Bts_RequestDefaultAgentResult();
            break;
         case Bts2IpcOC_DeviceConnect:
            returnMsg = ptrNew_Ipc2Bts_DeviceConnectResult();
            break;
         case Bts2IpcOC_DeviceDisconnect:
            returnMsg = ptrNew_Ipc2Bts_DeviceDisconnectResult();
            break;
         case Bts2IpcOC_Pair:
            returnMsg = ptrNew_Ipc2Bts_PairResult();
            break;
         case Bts2IpcOC_CancelPairing:
            returnMsg = ptrNew_Ipc2Bts_CancelPairingResult();
            break;
         case Bts2IpcOC_DiscoverServices:
            returnMsg = ptrNew_Ipc2Bts_DiscoverServicesResult();
            break;
         case Bts2IpcOC_CancelDiscovery:
            returnMsg = ptrNew_Ipc2Bts_CancelDiscoveryResult();
            break;
         case Bts2IpcOC_GetDeviceAddress:
            returnMsg = ptrNew_Ipc2Bts_DeviceAddressUpdate();
            break;
         case Bts2IpcOC_GetDeviceClass:
            returnMsg = ptrNew_Ipc2Bts_DeviceClassUpdate();
            break;
         case Bts2IpcOC_GetDeviceIcon:
            returnMsg = ptrNew_Ipc2Bts_DeviceIconUpdate();
            break;
         case Bts2IpcOC_GetDeviceUuids:
            returnMsg = ptrNew_Ipc2Bts_DeviceUuidsUpdate();
            break;
         case Bts2IpcOC_GetDevicePaired:
            returnMsg = ptrNew_Ipc2Bts_DevicePairedUpdate();
            break;
         case Bts2IpcOC_GetDeviceConnected:
            returnMsg = ptrNew_Ipc2Bts_DeviceConnectedUpdate();
            break;
         case Bts2IpcOC_GetDeviceTrusted:
            returnMsg = ptrNew_Ipc2Bts_DeviceTrustedUpdate();
            break;
         case Bts2IpcOC_SetDeviceTrusted:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetDeviceBlocked:
            returnMsg = ptrNew_Ipc2Bts_DeviceBlockedUpdate();
            break;
         case Bts2IpcOC_SetDeviceBlocked:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetDeviceAlias:
            returnMsg = ptrNew_Ipc2Bts_DeviceAliasUpdate();
            break;
         case Bts2IpcOC_SetDeviceAlias:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetDeviceAdapter:
            returnMsg = ptrNew_Ipc2Bts_DeviceAdapterUpdate();
            break;
         case Bts2IpcOC_GetDeviceModalias:
            returnMsg = ptrNew_Ipc2Bts_DeviceModaliasUpdate();
            break;
         case Bts2IpcOC_GetDeviceRssi:
            returnMsg = ptrNew_Ipc2Bts_DeviceRssiUpdate();
            break;
         case Bts2IpcOC_GetDeviceVendorId:
            returnMsg = ptrNew_Ipc2Bts_DeviceVendorIdUpdate();
            break;
         case Bts2IpcOC_GetDeviceVendorIdSource:
            returnMsg = ptrNew_Ipc2Bts_DeviceVendorIdSourceUpdate();
            break;
         case Bts2IpcOC_GetDeviceProductId:
            returnMsg = ptrNew_Ipc2Bts_DeviceProductIdUpdate();
            break;
         case Bts2IpcOC_GetDeviceMapInfo:
            returnMsg = ptrNew_Ipc2Bts_DeviceMapInfoUpdate();
            break;
         case Bts2IpcOC_GetDeviceConnectedService:
            returnMsg = ptrNew_Ipc2Bts_DeviceConnectedServiceUpdate();
            break;
         case Bts2IpcOC_GetDeviceVersion:
            returnMsg = ptrNew_Ipc2Bts_DeviceVersionUpdate();
            break;
         case Bts2IpcOC_GetDevicePbapInfo:
            returnMsg = ptrNew_Ipc2Bts_DevicePbapInfoUpdate();
            break;
         case Bts2IpcOC_GetDeviceMode:
            returnMsg = ptrNew_Ipc2Bts_DeviceModeUpdate();
            break;
         case Bts2IpcOC_GetDeviceRole:
            returnMsg = ptrNew_Ipc2Bts_DeviceRoleUpdate();
            break;
         case Bts2IpcOC_GetDeviceEnableAvpPause:
            returnMsg = ptrNew_Ipc2Bts_DeviceEnableAvpPauseUpdate();
            break;
         case Bts2IpcOC_SetDeviceEnableAvpPause:
            // no method return on ASF level
            break;
         case Bts2IpcOC_BtApplDeviceConnectReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplDeviceConnectReqResult();
            break;
         case Bts2IpcOC_BtApplDeviceDisconnectReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplDeviceDisconnectReqResult();
            break;
         case Bts2IpcOC_BtApplCancelConnectReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplCancelConnectReqResult();
            break;
         case Bts2IpcOC_BtApplGetHwVersionReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplGetHwVersionReqResult();
            break;
         case Bts2IpcOC_BtApplSetConfigurationReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplSetConfigurationReqResult();
            break;
         case Bts2IpcOC_BtApplReadConfigurationReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplReadConfigurationReqResult();
            break;
         case Bts2IpcOC_BtApplTestModeReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplTestModeReqResult();
            break;
         case Bts2IpcOC_BtApplTestModeLinkQualityReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplTestModeLinkQualityReqResult();
            break;
         case Bts2IpcOC_Connect:
            returnMsg = ptrNew_Ipc2Bts_ConnectResult();
            break;
         case Bts2IpcOC_Disconnect:
            returnMsg = ptrNew_Ipc2Bts_DisconnectResult();
            break;
         case Bts2IpcOC_GetServiceRemoteUuid:
            returnMsg = ptrNew_Ipc2Bts_ServiceRemoteUuidUpdate();
            break;
         case Bts2IpcOC_GetServiceState:
            returnMsg = ptrNew_Ipc2Bts_ServiceStateUpdate();
            break;
         case Bts2IpcOC_GetServiceDevice:
            returnMsg = ptrNew_Ipc2Bts_ServiceDeviceUpdate();
            break;
         case Bts2IpcOC_GetServiceVersion:
            returnMsg = ptrNew_Ipc2Bts_ServiceVersionUpdate();
            break;
         case Bts2IpcOC_DebugTraceConfig:
            returnMsg = ptrNew_Ipc2Bts_DebugTraceConfigResult();
            break;
         case Bts2IpcOC_ProtocolTraceConfig:
            returnMsg = ptrNew_Ipc2Bts_ProtocolTraceConfigResult();
            break;
         case Bts2IpcOC_BtApplDidSetReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplDidSetReqResult();
            break;
         case Bts2IpcOC_GetDunDeviceFilePath:
            returnMsg = ptrNew_Ipc2Bts_DunDeviceFilePathUpdate();
            break;
         case Bts2IpcOC_BtApplSppSetUuidReq:
            returnMsg = ptrNew_Ipc2Bts_BtApplSppSetUuidReqResult();
            break;
         case Bts2IpcOC_CreateSession:
            returnMsg = ptrNew_Ipc2Bts_CreateSessionResult();
            break;
         case Bts2IpcOC_RemoveSession:
            returnMsg = ptrNew_Ipc2Bts_RemoveSessionResult();
            break;
         case Bts2IpcOC_CancelSession:
            returnMsg = ptrNew_Ipc2Bts_CancelSessionResult();
            break;
         case Bts2IpcOC_GetConnManManagerProperties:
            returnMsg = ptrNew_Ipc2Bts_GetConnManManagerPropertiesResult();
            break;
         case Bts2IpcOC_GetConnManTechnologies:
            returnMsg = ptrNew_Ipc2Bts_GetConnManTechnologiesResult();
            break;
         case Bts2IpcOC_GetConnManServices:
            returnMsg = ptrNew_Ipc2Bts_GetConnManServicesResult();
            break;
         case Bts2IpcOC_GetConnManServiceProperties:
            returnMsg = ptrNew_Ipc2Bts_GetConnManServicePropertiesResult();
            break;
         case Bts2IpcOC_ConnManServiceConnect:
            returnMsg = ptrNew_Ipc2Bts_ConnManServiceConnectResult();
            break;
         case Bts2IpcOC_ConnManServiceDisconnect:
            returnMsg = ptrNew_Ipc2Bts_ConnManServiceDisconnectResult();
            break;
         case Bts2IpcOC_CreateLeDevice:
            returnMsg = ptrNew_Ipc2Bts_CreateLeDeviceResult();
            break;
         case Bts2IpcOC_RemoveLeDevice:
            returnMsg = ptrNew_Ipc2Bts_RemoveLeDeviceResult();
            break;
         case Bts2IpcOC_GetLeAdapterAddressType:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterAddressTypeUpdate();
            break;
         case Bts2IpcOC_GetLeAdapterAddress:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterAddressUpdate();
            break;
         case Bts2IpcOC_GetLeAdapterAlias:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterAliasUpdate();
            break;
         case Bts2IpcOC_SetLeAdapterAlias:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetLeAdapterAdvertisingTimeout:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterAdvertisingTimeoutUpdate();
            break;
         case Bts2IpcOC_SetLeAdapterAdvertisingTimeout:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetLeAdapterDiscoverableMode:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterDiscoverableModeUpdate();
            break;
         case Bts2IpcOC_SetLeAdapterDiscoverableMode:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetLeAdapterAdvertisingData:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterAdvertisingDataUpdate();
            break;
         case Bts2IpcOC_SetLeAdapterAdvertisingData:
            // no method return on ASF level
            break;
         case Bts2IpcOC_GetLeAdapterAdvertiseEnable:
            returnMsg = ptrNew_Ipc2Bts_LeAdapterAdvertiseEnableUpdate();
            break;
         case Bts2IpcOC_SetLeAdapterAdvertiseEnable:
            // no method return on ASF level
            break;
         case Bts2IpcOC_LeDeviceDisconnect:
            returnMsg = ptrNew_Ipc2Bts_LeDeviceDisconnectResult();
            break;
         case Bts2IpcOC_GetLeDeviceAddressType:
            returnMsg = ptrNew_Ipc2Bts_LeDeviceAddressTypeUpdate();
            break;
         case Bts2IpcOC_GetLeDeviceAddress:
            returnMsg = ptrNew_Ipc2Bts_LeDeviceAddressUpdate();
            break;
         case Bts2IpcOC_GetLeDeviceConnected:
            returnMsg = ptrNew_Ipc2Bts_LeDeviceConnectedUpdate();
            break;
         case Bts2IpcOC_SendSeamlessPairingErrorCode:
            returnMsg = ptrNew_Ipc2Bts_SendSeamlessPairingErrorCodeResult();
            break;
         case Bts2IpcOC_SeamlessPairingAllowAccess:
            returnMsg = ptrNew_Ipc2Bts_SeamlessPairingAllowAccessResult();
            break;
         case Bts2IpcOC_GetSeamlessPairingPairingData:
            returnMsg = ptrNew_Ipc2Bts_SeamlessPairingPairingDataUpdate();
            break;
         case Bts2IpcOC_SetSeamlessPairingPairingData:
            // no method return on ASF level
            break;
         default:
            // check extension
            FW_NORMAL_ASSERT(true == createExtensionConnectionErrorMessage(&returnMsg, &message));
            break;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return returnMsg;
}

Ipc2Bts_BaseMessage* DbusAccess::createIpc2BtsPhonecallErrorMessage(IN const Bts2Ipc_BaseMessage& message) const
{
   Ipc2Bts_BaseMessage* returnMsg(0);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)message.getOpCode());

   if((Bts2IpcOC_TelephonyBlockStart < opCode) && (opCode < Bts2IpcOC_TelephonyBlockEnd))
   {
      switch(opCode)
      {
         case Bts2IpcOC_GetManagedObjectsTelephony:
            // no result message
            break;
         case Bts2IpcOC_GetAllPropertiesTelephony:
            // no result message
            break;
         case Bts2IpcOC_GetIntrospectionTelephony:
            // no result message
            break;
         case Bts2IpcOC_GetHandsfreeProperties:
            returnMsg = ptrNew_Ipc2Bts_GetHandsfreeProperties();
            break;
         case Bts2IpcOC_GetModems:
            returnMsg = ptrNew_Ipc2Bts_GetModems();
            break;
         case Bts2IpcOC_GetModemProperties:
            returnMsg = ptrNew_Ipc2Bts_GetModemProperties();
            break;
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return returnMsg;
}

Ipc2Bts_BaseMessage* DbusAccess::createIpc2BtsPhonebookErrorMessage(IN const Bts2Ipc_BaseMessage& message) const
{
   Ipc2Bts_BaseMessage* returnMsg(0);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)message.getOpCode());

   if((Bts2IpcOC_PhonebookBlockStart < opCode) && (opCode < Bts2IpcOC_PhonebookBlockEnd))
   {
      switch(opCode)
      {
         // add if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return returnMsg;
}

Ipc2Bts_BaseMessage* DbusAccess::createIpc2BtsMessagingErrorMessage(IN const Bts2Ipc_BaseMessage& message) const
{
   Ipc2Bts_BaseMessage* returnMsg(0);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)message.getOpCode());

   if((Bts2IpcOC_MessagingBlockStart < opCode) && (opCode < Bts2IpcOC_MessagingBlockEnd))
   {
      switch(opCode)
      {
         // add if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return returnMsg;
}

Ipc2Bts_BaseMessage* DbusAccess::createIpc2BtsMediaPlayerErrorMessage(IN const Bts2Ipc_BaseMessage& message) const
{
   Ipc2Bts_BaseMessage* returnMsg(0);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)message.getOpCode());

   if((Bts2IpcOC_MediaPlayerBlockStart < opCode) && (opCode < Bts2IpcOC_MediaPlayerBlockEnd))
   {
      switch(opCode)
      {
         // add if necessary
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return returnMsg;
}


void DbusAccess::transferMessageDataConfiguration(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   (void)(ipc2BtsMessage);
   (void)(bts2IpcMessage);
}

void DbusAccess::transferMessageDataConnection(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN(bts2IpcMessage);
   Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)bts2IpcMessageRef.getOpCode());

   if((Bts2IpcOC_ConnectionBlockStart < opCode) && (opCode < Bts2IpcOC_ConnectionBlockEnd))
   {
      switch(opCode)
      {
         case Bts2IpcOC_GetManagedObjectsConnection:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAllPropertiesConnection:
         {
            // copy interface
            Ipc2Bts_GetAllPropertiesConnection& outMsg = static_cast< Ipc2Bts_GetAllPropertiesConnection& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetAllPropertiesConnection& inMsg = static_cast< const Bts2Ipc_GetAllPropertiesConnection& >(bts2IpcMessageRef);
            outMsg.setInterface(inMsg.getInterface());
            break;
         }
         case Bts2IpcOC_GetIntrospectionConnection:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_ReleaseRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_RequestPinCodeRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_DisplayPinCodeRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_RequestPasskeyRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_RequestConfirmationRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_CancelRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_CancelRequestRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_AuthorizeServiceRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_RequestAuthorizationRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_RequestLinkkeyRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_DisplayPasskeyRes:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_StartDiscovery:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_StopDiscovery:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_RemoveDevice:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_CreateDevice:
         {
            // copy address
            Ipc2Bts_CreateDeviceResult& outMsg = static_cast< Ipc2Bts_CreateDeviceResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_CreateDevice& inMsg = static_cast< const Bts2Ipc_CreateDevice& >(bts2IpcMessageRef);
            outMsg.setBDAddress(inMsg.getBDAddress());
            break;
         }
         case Bts2IpcOC_SendHCICmd:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetAdapterAddress:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetAdapterClass:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetAdapterAlias:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterAlias:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterPowered:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterPowered:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterDiscoverable:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterDiscoverable:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterDiscoverabletimeout:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterDiscoverabletimeout:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterPairable:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterPairable:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterPairabletimeout:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterPairabletimeout:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterDiscovering:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetAdapterUuids:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterUuids:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterWbsMode:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterWbsMode:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterConnectable:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterConnectable:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterHcimode:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterHcimode:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterVersion:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetAdapterTraceLevel:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterTraceLevel:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterTraceCategory:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterTraceCategory:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAdapterAvailableSppServers:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_SetAdapterAvailableSppServers:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_RegisterAgent:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_UnregisterAgent:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_RequestDefaultAgent:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_DeviceConnect:
         {
            // copy object path
            Ipc2Bts_DeviceConnectResult& outMsg = static_cast< Ipc2Bts_DeviceConnectResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_DeviceConnect& inMsg = static_cast< const Bts2Ipc_DeviceConnect& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_DeviceDisconnect:
         {
            // copy object path
            Ipc2Bts_DeviceDisconnectResult& outMsg = static_cast< Ipc2Bts_DeviceDisconnectResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_DeviceDisconnect& inMsg = static_cast< const Bts2Ipc_DeviceDisconnect& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_Pair:
         {
            // copy object path
            Ipc2Bts_PairResult& outMsg = static_cast< Ipc2Bts_PairResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_Pair& inMsg = static_cast< const Bts2Ipc_Pair& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_CancelPairing:
         {
            // copy object path
            Ipc2Bts_CancelPairingResult& outMsg = static_cast< Ipc2Bts_CancelPairingResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_CancelPairing& inMsg = static_cast< const Bts2Ipc_CancelPairing& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_DiscoverServices:
         {
            // copy object path
            Ipc2Bts_DiscoverServicesResult& outMsg = static_cast< Ipc2Bts_DiscoverServicesResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_DiscoverServices& inMsg = static_cast< const Bts2Ipc_DiscoverServices& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_CancelDiscovery:
         {
            // copy object path
            Ipc2Bts_CancelDiscoveryResult& outMsg = static_cast< Ipc2Bts_CancelDiscoveryResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_CancelDiscovery& inMsg = static_cast< const Bts2Ipc_CancelDiscovery& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceAddress:
         {
            // copy object path
            Ipc2Bts_DeviceAddressUpdate& outMsg = static_cast< Ipc2Bts_DeviceAddressUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceAddress& inMsg = static_cast< const Bts2Ipc_GetDeviceAddress& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceClass:
         {
            // copy object path
            Ipc2Bts_DeviceClassUpdate& outMsg = static_cast< Ipc2Bts_DeviceClassUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceClass& inMsg = static_cast< const Bts2Ipc_GetDeviceClass& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceIcon:
         {
            // copy object path
            Ipc2Bts_DeviceIconUpdate& outMsg = static_cast< Ipc2Bts_DeviceIconUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceIcon& inMsg = static_cast< const Bts2Ipc_GetDeviceIcon& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceUuids:
         {
            // copy object path
            Ipc2Bts_DeviceUuidsUpdate& outMsg = static_cast< Ipc2Bts_DeviceUuidsUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceUuids& inMsg = static_cast< const Bts2Ipc_GetDeviceUuids& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDevicePaired:
         {
            // copy object path
            Ipc2Bts_DevicePairedUpdate& outMsg = static_cast< Ipc2Bts_DevicePairedUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDevicePaired& inMsg = static_cast< const Bts2Ipc_GetDevicePaired& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceConnected:
         {
            // copy object path
            Ipc2Bts_DeviceConnectedUpdate& outMsg = static_cast< Ipc2Bts_DeviceConnectedUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceConnected& inMsg = static_cast< const Bts2Ipc_GetDeviceConnected& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceTrusted:
         {
            // copy object path
            Ipc2Bts_DeviceTrustedUpdate& outMsg = static_cast< Ipc2Bts_DeviceTrustedUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceTrusted& inMsg = static_cast< const Bts2Ipc_GetDeviceTrusted& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_SetDeviceTrusted:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetDeviceBlocked:
         {
            // copy object path
            Ipc2Bts_DeviceBlockedUpdate& outMsg = static_cast< Ipc2Bts_DeviceBlockedUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceBlocked& inMsg = static_cast< const Bts2Ipc_GetDeviceBlocked& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_SetDeviceBlocked:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetDeviceAlias:
         {
            // copy object path
            Ipc2Bts_DeviceAliasUpdate& outMsg = static_cast< Ipc2Bts_DeviceAliasUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceAlias& inMsg = static_cast< const Bts2Ipc_GetDeviceAlias& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_SetDeviceAlias:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetDeviceAdapter:
         {
            // copy object path
            Ipc2Bts_DeviceAdapterUpdate& outMsg = static_cast< Ipc2Bts_DeviceAdapterUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceAdapter& inMsg = static_cast< const Bts2Ipc_GetDeviceAdapter& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceModalias:
         {
            // copy object path
            Ipc2Bts_DeviceModaliasUpdate& outMsg = static_cast< Ipc2Bts_DeviceModaliasUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceModalias& inMsg = static_cast< const Bts2Ipc_GetDeviceModalias& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceRssi:
         {
            // copy object path
            Ipc2Bts_DeviceRssiUpdate& outMsg = static_cast< Ipc2Bts_DeviceRssiUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceRssi& inMsg = static_cast< const Bts2Ipc_GetDeviceRssi& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceVendorId:
         {
            // copy object path
            Ipc2Bts_DeviceVendorIdUpdate& outMsg = static_cast< Ipc2Bts_DeviceVendorIdUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceVendorId& inMsg = static_cast< const Bts2Ipc_GetDeviceVendorId& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceVendorIdSource:
         {
            // copy object path
            Ipc2Bts_DeviceVendorIdSourceUpdate& outMsg = static_cast< Ipc2Bts_DeviceVendorIdSourceUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceVendorIdSource& inMsg = static_cast< const Bts2Ipc_GetDeviceVendorIdSource& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceProductId:
         {
            // copy object path
            Ipc2Bts_DeviceProductIdUpdate& outMsg = static_cast< Ipc2Bts_DeviceProductIdUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceProductId& inMsg = static_cast< const Bts2Ipc_GetDeviceProductId& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceMapInfo:
         {
            // copy object path
            Ipc2Bts_DeviceMapInfoUpdate& outMsg = static_cast< Ipc2Bts_DeviceMapInfoUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceMapInfo& inMsg = static_cast< const Bts2Ipc_GetDeviceMapInfo& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceConnectedService:
         {
            // copy object path
            Ipc2Bts_DeviceConnectedServiceUpdate& outMsg = static_cast< Ipc2Bts_DeviceConnectedServiceUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceConnectedService& inMsg = static_cast< const Bts2Ipc_GetDeviceConnectedService& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceVersion:
         {
            // copy object path
            Ipc2Bts_DeviceVersionUpdate& outMsg = static_cast< Ipc2Bts_DeviceVersionUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceVersion& inMsg = static_cast< const Bts2Ipc_GetDeviceVersion& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDevicePbapInfo:
         {
            // copy object path
            Ipc2Bts_DevicePbapInfoUpdate& outMsg = static_cast< Ipc2Bts_DevicePbapInfoUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDevicePbapInfo& inMsg = static_cast< const Bts2Ipc_GetDevicePbapInfo& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceMode:
         {
            // copy object path
            Ipc2Bts_DeviceModeUpdate& outMsg = static_cast< Ipc2Bts_DeviceModeUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceMode& inMsg = static_cast< const Bts2Ipc_GetDeviceMode& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceRole:
         {
            // copy object path
            Ipc2Bts_DeviceRoleUpdate& outMsg = static_cast< Ipc2Bts_DeviceRoleUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceRole& inMsg = static_cast< const Bts2Ipc_GetDeviceRole& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_GetDeviceEnableAvpPause:
         {
            // copy object path
            Ipc2Bts_DeviceEnableAvpPauseUpdate& outMsg = static_cast< Ipc2Bts_DeviceEnableAvpPauseUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetDeviceEnableAvpPause& inMsg = static_cast< const Bts2Ipc_GetDeviceEnableAvpPause& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_SetDeviceEnableAvpPause:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_BtApplDeviceConnectReq:
         {
            // copy address, service function, instance
            Ipc2Bts_BtApplDeviceConnectReqResult& outMsg = static_cast< Ipc2Bts_BtApplDeviceConnectReqResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_BtApplDeviceConnectReq& inMsg = static_cast< const Bts2Ipc_BtApplDeviceConnectReq& >(bts2IpcMessageRef);
            outMsg.setBDAddress(inMsg.getBDAddress());
            outMsg.setServiceFunction(inMsg.getServiceFunction());
            outMsg.setInstance(inMsg.getInstance());
            break;
         }
         case Bts2IpcOC_BtApplDeviceDisconnectReq:
         {
            // copy address, service function, instance
            Ipc2Bts_BtApplDeviceDisconnectReqResult& outMsg = static_cast< Ipc2Bts_BtApplDeviceDisconnectReqResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_BtApplDeviceDisconnectReq& inMsg = static_cast< const Bts2Ipc_BtApplDeviceDisconnectReq& >(bts2IpcMessageRef);
            outMsg.setBDAddress(inMsg.getBDAddress());
            outMsg.setServiceFunction(inMsg.getServiceFunction());
            outMsg.setInstance(inMsg.getInstance());
            break;
         }
         case Bts2IpcOC_BtApplCancelConnectReq:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_BtApplGetHwVersionReq:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_BtApplSetConfigurationReq:
         {
            // copy instance and config id
            Ipc2Bts_BtApplSetConfigurationReqResult& outMsg = static_cast< Ipc2Bts_BtApplSetConfigurationReqResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_BtApplSetConfigurationReq& inMsg = static_cast< const Bts2Ipc_BtApplSetConfigurationReq& >(bts2IpcMessageRef);
            outMsg.setInstance(inMsg.getInstance());
            outMsg.setConfigId(inMsg.getConfigId());
            break;
         }
         case Bts2IpcOC_BtApplReadConfigurationReq:
         {
            // copy instance and config id
            Ipc2Bts_BtApplReadConfigurationReqResult& outMsg = static_cast< Ipc2Bts_BtApplReadConfigurationReqResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_BtApplReadConfigurationReq& inMsg = static_cast< const Bts2Ipc_BtApplReadConfigurationReq& >(bts2IpcMessageRef);
            outMsg.setInstance(inMsg.getInstance());
            outMsg.setConfigId(inMsg.getConfigId());
            break;
         }
         case Bts2IpcOC_BtApplTestModeReq:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_BtApplTestModeLinkQualityReq:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_Connect:
         {
            // copy object path
            Ipc2Bts_ConnectResult& outMsg = static_cast< Ipc2Bts_ConnectResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_Connect& inMsg = static_cast< const Bts2Ipc_Connect& >(bts2IpcMessageRef);
            outMsg.setService(inMsg.getService());
            break;
         }
         case Bts2IpcOC_Disconnect:
         {
            // copy object path
            Ipc2Bts_DisconnectResult& outMsg = static_cast< Ipc2Bts_DisconnectResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_Disconnect& inMsg = static_cast< const Bts2Ipc_Disconnect& >(bts2IpcMessageRef);
            outMsg.setService(inMsg.getService());
            break;
         }
         case Bts2IpcOC_GetServiceRemoteUuid:
         {
            // copy object path
            Ipc2Bts_ServiceRemoteUuidUpdate& outMsg = static_cast< Ipc2Bts_ServiceRemoteUuidUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetServiceRemoteUuid& inMsg = static_cast< const Bts2Ipc_GetServiceRemoteUuid& >(bts2IpcMessageRef);
            outMsg.setService(inMsg.getService());
            break;
         }
         case Bts2IpcOC_GetServiceState:
         {
            // copy object path
            Ipc2Bts_ServiceStateUpdate& outMsg = static_cast< Ipc2Bts_ServiceStateUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetServiceState& inMsg = static_cast< const Bts2Ipc_GetServiceState& >(bts2IpcMessageRef);
            outMsg.setService(inMsg.getService());
            break;
         }
         case Bts2IpcOC_GetServiceDevice:
         {
            // copy object path
            Ipc2Bts_ServiceDeviceUpdate& outMsg = static_cast< Ipc2Bts_ServiceDeviceUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetServiceDevice& inMsg = static_cast< const Bts2Ipc_GetServiceDevice& >(bts2IpcMessageRef);
            outMsg.setService(inMsg.getService());
            break;
         }
         case Bts2IpcOC_GetServiceVersion:
         {
            // copy object path
            Ipc2Bts_ServiceVersionUpdate& outMsg = static_cast< Ipc2Bts_ServiceVersionUpdate& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetServiceVersion& inMsg = static_cast< const Bts2Ipc_GetServiceVersion& >(bts2IpcMessageRef);
            outMsg.setService(inMsg.getService());
            break;
         }
         case Bts2IpcOC_DebugTraceConfig:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_ProtocolTraceConfig:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_BtApplDidSetReq:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetDunDeviceFilePath:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_BtApplSppSetUuidReq:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_CreateSession:
         {
            // copy address, protocol + instance
            Ipc2Bts_CreateSessionResult& outMsg = static_cast< Ipc2Bts_CreateSessionResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_CreateSession& inMsg = static_cast< const Bts2Ipc_CreateSession& >(bts2IpcMessageRef);
            outMsg.setBDAddress(inMsg.getBDAddress());
            outMsg.setProtocolId(inMsg.getProtocolId());
            outMsg.setInstanceId(inMsg.getInstanceId());
            break;
         }
         case Bts2IpcOC_RemoveSession:
         {
            // copy address, protocol + instance
            Ipc2Bts_RemoveSessionResult& outMsg = static_cast< Ipc2Bts_RemoveSessionResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_RemoveSession& inMsg = static_cast< const Bts2Ipc_RemoveSession& >(bts2IpcMessageRef);
            outMsg.setBDAddress(inMsg.getBDAddress());
            outMsg.setProtocolId(inMsg.getProtocolId());
            outMsg.setInstanceId(inMsg.getInstanceId());
            break;
         }
         case Bts2IpcOC_CancelSession:
         {
            // copy address, protocol + instance
            Ipc2Bts_CancelSessionResult& outMsg = static_cast< Ipc2Bts_CancelSessionResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_CancelSession& inMsg = static_cast< const Bts2Ipc_CancelSession& >(bts2IpcMessageRef);
            outMsg.setBDAddress(inMsg.getBDAddress());
            outMsg.setProtocolId(inMsg.getProtocolId());
            outMsg.setInstanceId(inMsg.getInstanceId());
            break;
         }
         case Bts2IpcOC_GetConnManManagerProperties:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetConnManTechnologies:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetConnManServices:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetConnManServiceProperties:
         {
            // copy object path
            Ipc2Bts_GetConnManServicePropertiesResult& outMsg = static_cast< Ipc2Bts_GetConnManServicePropertiesResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetConnManServiceProperties& inMsg = static_cast< const Bts2Ipc_GetConnManServiceProperties& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_ConnManServiceConnect:
         {
            // copy object path
            Ipc2Bts_ConnManServiceConnectResult& outMsg = static_cast< Ipc2Bts_ConnManServiceConnectResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_ConnManServiceConnect& inMsg = static_cast< const Bts2Ipc_ConnManServiceConnect& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_ConnManServiceDisconnect:
         {
            // copy object path
            Ipc2Bts_ConnManServiceDisconnectResult& outMsg = static_cast< Ipc2Bts_ConnManServiceDisconnectResult& >(ipc2BtsMessageRef);
            const Bts2Ipc_ConnManServiceDisconnect& inMsg = static_cast< const Bts2Ipc_ConnManServiceDisconnect& >(bts2IpcMessageRef);
            outMsg.setDevice(inMsg.getDevice());
            break;
         }
         case Bts2IpcOC_CreateLeDevice:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_RemoveLeDevice:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetLeAdapterAddressType:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetLeAdapterAddress:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetLeAdapterAlias:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SetLeAdapterAlias:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetLeAdapterAdvertisingTimeout:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SetLeAdapterAdvertisingTimeout:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetLeAdapterDiscoverableMode:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SetLeAdapterDiscoverableMode:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetLeAdapterAdvertisingData:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SetLeAdapterAdvertisingData:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetLeAdapterAdvertiseEnable:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SetLeAdapterAdvertiseEnable:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_LeDeviceDisconnect:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetLeDeviceAddressType:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetLeDeviceAddress:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetLeDeviceConnected:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SendSeamlessPairingErrorCode:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SeamlessPairingAllowAccess:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_GetSeamlessPairingPairingData:
         {
            // TODO
            break;
         }
         case Bts2IpcOC_SetSeamlessPairingPairingData:
         {
            // no Ipc2Bts message available
            break;
         }
         default:
         {
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DbusAccess::transferMessageDataPhonecall(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN(bts2IpcMessage);
   Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)bts2IpcMessageRef.getOpCode());

   if((Bts2IpcOC_TelephonyBlockStart < opCode) && (opCode < Bts2IpcOC_TelephonyBlockEnd))
   {
      switch(opCode)
      {
         case Bts2IpcOC_GetManagedObjectsTelephony:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetAllPropertiesTelephony:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetIntrospectionTelephony:
         {
            // no Ipc2Bts message available
            break;
         }
         case Bts2IpcOC_GetHandsfreeProperties:
         {
            // copy object path
            Ipc2Bts_GetHandsfreeProperties& outMsg = static_cast< Ipc2Bts_GetHandsfreeProperties& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetHandsfreeProperties& inMsg = static_cast< const Bts2Ipc_GetHandsfreeProperties& >(bts2IpcMessageRef);
            outMsg.setModem(inMsg.getModem());
            break;
         }
         case Bts2IpcOC_GetModems:
         {
            // nothing to transfer
            break;
         }
         case Bts2IpcOC_GetModemProperties:
         {
            // copy object path
            Ipc2Bts_GetModemProperties& outMsg = static_cast< Ipc2Bts_GetModemProperties& >(ipc2BtsMessageRef);
            const Bts2Ipc_GetModemProperties& inMsg = static_cast< const Bts2Ipc_GetModemProperties& >(bts2IpcMessageRef);
            outMsg.setModem(inMsg.getModem());
            break;
         }
         default:
         {
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DbusAccess::transferMessageDataPhonebook(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN(bts2IpcMessage);
   // Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)bts2IpcMessageRef.getOpCode());

   if((Bts2IpcOC_PhonebookBlockStart < opCode) && (opCode < Bts2IpcOC_PhonebookBlockEnd))
   {
      switch(opCode)
      {
         // add if necessary
         default:
         {
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DbusAccess::transferMessageDataMessaging(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN(bts2IpcMessage);
   // Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)bts2IpcMessageRef.getOpCode());

   if((Bts2IpcOC_MessagingBlockStart < opCode) && (opCode < Bts2IpcOC_MessagingBlockEnd))
   {
      switch(opCode)
      {
         // add if necessary
         default:
         {
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DbusAccess::transferMessageDataMediaPlayer(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN(bts2IpcMessage);
   // Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   const BTSGenBts2IpcOpcode opCode((BTSGenBts2IpcOpcode)bts2IpcMessageRef.getOpCode());

   if((Bts2IpcOC_MediaPlayerBlockStart < opCode) && (opCode < Bts2IpcOC_MediaPlayerBlockEnd))
   {
      switch(opCode)
      {
         // add if necessary
         default:
         {
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

//------------------------------------------------------------------------------
// class Bts2Ipc_GetManagedObjectsConnection;
// 'class Bts2Ipc_(.*);' => 'void DbusAccess::handleBts2Ipc$1(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_$1* ptrMessage)\R{\R   if(NULL == ptrMessage)\R   {\R      FW_NORMAL_ASSERT_ALWAYS();\R      return;\R   }\R\R   // TODO\R}\R'
//------------------------------------------------------------------------------
void DbusAccess::handleBts2IpcGetManagedObjectsConnection(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetManagedObjectsConnection* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);
}

void DbusAccess::handleBts2IpcGetAllPropertiesConnection(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAllPropertiesConnection* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _propertiesGetAllIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      act_t token = _propertiesGetAllIf->sendGetAllRequest(ptrMessage->getBusName(), ptrMessage->getObjPath(), (::ccdbusif::DbusBusType)ptrMessage->getBusType(), NULL, parser.getInterface2String(ptrMessage->getInterface()));

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAllPropertiesConnection(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetIntrospectionConnection(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetIntrospectionConnection* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _introspectableIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _introspectableIf->sendIntrospectRequest(ptrMessage->getBusName(), ptrMessage->getObjPath(), (::ccdbusif::DbusBusType)ptrMessage->getBusType(), NULL);

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetIntrospectionConnection(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      // nothing to do
   }
}

void DbusAccess::handleBts2IpcReleaseRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ReleaseRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: none
      _agentIf->sendReleaseResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcReleaseRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcRequestPinCodeRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestPinCodeRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: org.bluez.Error.Rejected, org.bluez.Error.Canceled
      if(0 == ptrMessage->getPinCode().size())
      {
          ::std::string errorName = "org.bluez.Error.Rejected";
          ::std::string errorMessage;
         _agentIf->sendRequestPinCodeError(_agentCb->getCallbackId(), errorName, errorMessage, ptrMessage->getDbusToken());
      }
      else
      {
         _agentIf->sendRequestPinCodeResponse(_agentCb->getCallbackId(), ptrMessage->getPinCode(), ptrMessage->getDbusToken());
      }

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcRequestPinCodeRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing except waiting for pairing finished
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcDisplayPinCodeRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DisplayPinCodeRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: org.bluez.Error.Rejected, org.bluez.Error.Canceled => but description says: empty reply should be returned; here it also makes no sense to send a reject because no user input possible
      _agentIf->sendDisplayPinCodeResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcDisplayPinCodeRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing except waiting for pairing finished
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcRequestPasskeyRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestPasskeyRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: org.bluez.Error.Rejected, org.bluez.Error.Canceled
      if(0 == ptrMessage->getPasskey().size())
      {
          ::std::string errorName = "org.bluez.Error.Rejected";
          ::std::string errorMessage;
         _agentIf->sendRequestPasskeyError(_agentCb->getCallbackId(), errorName, errorMessage, ptrMessage->getDbusToken());
      }
      else
      {
         uint32 passkey;
         convertNumericValueString2Binary(passkey, ptrMessage->getPasskey());
         _agentIf->sendRequestPasskeyResponse(_agentCb->getCallbackId(), passkey, ptrMessage->getDbusToken());
      }

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcRequestPasskeyRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing except waiting for pairing finished
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcRequestConfirmationRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestConfirmationRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: org.bluez.Error.Rejected, org.bluez.Error.Canceled
      if(BTS_CONFIRM_ACCEPT == ptrMessage->getAccept())
      {
         _agentIf->sendRequestConfirmationResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());
      }
      else
      {
         ::std::string errorName = "org.bluez.Error.Rejected";
         ::std::string errorMessage;
         _agentIf->sendRequestConfirmationError(_agentCb->getCallbackId(), errorName, errorMessage, ptrMessage->getDbusToken());
      }

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcRequestConfirmationRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing except waiting for pairing finished
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcCancelRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: none
      _agentIf->sendCancelResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcCancelRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcCancelRequestRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelRequestRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: none
      _agentIf->sendCancelRequestResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcCancelRequestRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcAuthorizeServiceRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_AuthorizeServiceRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: org.bluez.Error.Rejected, org.bluez.Error.Canceled
      if(BTS_CONFIRM_ACCEPT == ptrMessage->getAccept())
      {
         _agentIf->sendAuthorizeServiceResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());
      }
      else
      {
         ::std::string errorName = "org.bluez.Error.Rejected";
         ::std::string errorMessage;
         _agentIf->sendAuthorizeServiceError(_agentCb->getCallbackId(), errorName, errorMessage, ptrMessage->getDbusToken());
      }

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcAuthorizeServiceRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcRequestAuthorizationRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestAuthorizationRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: none but also in case of SSP just works it is possible to reject
      if(BTS_CONFIRM_ACCEPT == ptrMessage->getAccept())
      {
         _agentIf->sendRequestAuthorizationResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());
      }
      else
      {
         ::std::string errorName = "org.bluez.Error.Rejected"; // TODO: put error strings to ccdbusif parser
         ::std::string errorMessage;
         _agentIf->sendRequestAuthorizationError(_agentCb->getCallbackId(), errorName, errorMessage, ptrMessage->getDbusToken());
      }

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcRequestAuthorizationRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcRequestLinkkeyRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestLinkkeyRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: none

      ::std::vector<uint8> linkkey;
      convertLinkKeyString2Binary(linkkey, ptrMessage->getLinkKey(), true); // link key can be empty
      _agentIf->sendRequestLinkkeyResponse(_agentCb->getCallbackId(), linkkey, ptrMessage->getDbusToken());

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcRequestLinkkeyRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcDisplayPasskeyRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DisplayPasskeyRes* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _agentIf) || (0 == _agentCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // possible errors: org.bluez.Error.Rejected, org.bluez.Error.Canceled => but description says: empty reply should be returned; here it also makes no sense to send a reject because no user input possible
      _agentIf->sendDisplayPasskeyResponse(_agentCb->getCallbackId(), ptrMessage->getDbusToken());

      ::ccdbusif::ActInfo info(ptrMessage->getDbusToken());
      ETG_TRACE_USR3((" handleBts2IpcDisplayPasskeyRes(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
   }

   // if sending fails due to any reason we will do nothing except waiting for pairing finished
   // delete in every case because there will be no result message
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcStartDiscovery(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_StartDiscovery* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendStartDiscoveryRequest(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcStartDiscovery(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcStopDiscovery(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_StopDiscovery* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendStopDiscoveryRequest(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcStopDiscovery(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcRemoveDevice(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RemoveDevice* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendRemoveDeviceRequest(_adapterCb->getCallbackId(), ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcRemoveDevice(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcCreateDevice(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CreateDevice* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      ::std::string address;
      convertBdAddress2ExternalValue(address, ptrMessage->getBDAddress());
      act_t token = _adapterIf->sendCreateDeviceRequest(_adapterCb->getCallbackId(), address);

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcCreateDevice(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSendHCICmd(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SendHCICmd* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterAddress(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterAddress* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterClass(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterClass* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterAlias* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterAlias* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendAliasSet(_adapterCb->getCallbackId(), ptrMessage->getAlias());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterAlias(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterPowered(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterPowered* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterPowered(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterPowered* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
#ifdef BT_ON_OFF_SWITCH_FAILED
      // debug section start
      if(1 == toggle)
      {
         toggle = 0;
      }
      else
      {
         toggle = 1;
      }
      if(0 == toggle)
      {
         ETG_TRACE_USR3((" handleBts2IpcSetAdapterPowered(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
         deleteBts2IpcMsg = true;
         return;
      }
      // debug section end
#endif

      _adapterIf->sendPoweredSet(_adapterCb->getCallbackId(), ptrMessage->getPowered());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterPowered(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterDiscoverable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterDiscoverable* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendDiscoverableGet(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAdapterDiscoverable(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetAdapterDiscoverable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterDiscoverable* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendDiscoverableSet(_adapterCb->getCallbackId(), ptrMessage->getDiscoverable());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterDiscoverable(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterDiscoverabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterDiscoverabletimeout* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterDiscoverabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterDiscoverabletimeout* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendDiscoverableTimeoutSet(_adapterCb->getCallbackId(), ptrMessage->getDiscoverableTimeout());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterDiscoverabletimeout(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterPairable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterPairable* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendPairableGet(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAdapterPairable(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetAdapterPairable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterPairable* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendPairableSet(_adapterCb->getCallbackId(), ptrMessage->getPairable());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterPairable(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterPairabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterPairabletimeout* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterPairabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterPairabletimeout* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendPairableTimeoutSet(_adapterCb->getCallbackId(), ptrMessage->getPairableTimeout());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterPairabletimeout(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterDiscovering(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterDiscovering* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterUuids(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterUuids* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendUUIDsGet(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAdapterUuids(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetAdapterUuids(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterUuids* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendUUIDsSet(_adapterCb->getCallbackId(), ptrMessage->getUuids());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterUuids(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterWbsMode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterWbsMode* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterWbsMode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterWbsMode* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterConnectable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterConnectable* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendConnectableGet(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAdapterConnectable(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetAdapterConnectable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterConnectable* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendConnectableSet(_adapterCb->getCallbackId(), ptrMessage->getConnectable());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterConnectable(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterHcimode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterHcimode* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterHcimode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterHcimode* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _adapterIf->sendHCIModeSet(_adapterCb->getCallbackId(), ptrMessage->getHciMode());

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterHcimode(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetAdapterVersion(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterVersion* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterTraceLevel(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterTraceLevel* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterTraceLevel(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterTraceLevel* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterTraceCategory(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterTraceCategory* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcSetAdapterTraceCategory(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterTraceCategory* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO
}

void DbusAccess::handleBts2IpcGetAdapterAvailableSppServers(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterAvailableSppServers* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _adapterIf->sendAvailableSPPServersGet(_adapterCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAdapterAvailableSppServers(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetAdapterAvailableSppServers(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterAvailableSppServers* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if((0 == _adapterIf) || (0 == _adapterCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      // convert UUIDs to values with hyphen
      const BTSUuidList& availableSPPServers(ptrMessage->getAvailableSppServers());
      ::std::vector< ::std::string > tmpAvailableSPPServers;
      tmpAvailableSPPServers.reserve(availableSPPServers.size());
      ::std::string convertedUuid;

      for(size_t i = 0; i < availableSPPServers.size(); i++)
      {
         convertUuid2ExternalValue(convertedUuid, availableSPPServers[i]);
         tmpAvailableSPPServers.push_back(convertedUuid);
         convertedUuid.clear();
      }

      _adapterIf->sendAvailableSPPServersSet(_adapterCb->getCallbackId(), tmpAvailableSPPServers);

      ETG_TRACE_USR3((" handleBts2IpcSetAdapterAvailableSppServers(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcRegisterAgent(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RegisterAgent* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _agentManagerIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      ::std::string capability;
      convertAgentCapability2String(capability, ptrMessage->getCapability());

      act_t token = _agentManagerIf->sendRegisterAgentRequest(ptrMessage->getAgent(), capability);

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcRegisterAgent(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcUnregisterAgent(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_UnregisterAgent* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcRequestDefaultAgent(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestDefaultAgent* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcDeviceConnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DeviceConnect* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendConnectRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcDeviceConnect(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcDeviceDisconnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DeviceDisconnect* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendDisconnectRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcDeviceDisconnect(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcPair(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_Pair* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendPairRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcPair(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcCancelPairing(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelPairing* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendCancelPairingRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcCancelPairing(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcDiscoverServices(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DiscoverServices* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendDiscoverServicesRequest(ptrMessage->getDevice(), ptrMessage->getUuid());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcDiscoverServices(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcCancelDiscovery(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelDiscovery* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendCancelDiscoveryRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcCancelDiscovery(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceAddress(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceAddress* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendAddressGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendAddressGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceAddress(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceClass(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceClass* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendClassGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendClassGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceClass(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceIcon(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceIcon* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendIconGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendIconGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceIcon(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceUuids(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceUuids* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendUUIDsGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendUUIDsGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceUuids(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDevicePaired(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDevicePaired* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendPairedGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendPairedGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDevicePaired(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceConnected(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceConnected* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendConnectedGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendConnectedGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceConnected(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceTrusted(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceTrusted* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendTrustedGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendTrustedGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceTrusted(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetDeviceTrusted(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceTrusted* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _deviceIf->sendTrustedSet(ptrMessage->getDevice(), ptrMessage->getTrusted());

      ETG_TRACE_USR3((" handleBts2IpcSetDeviceTrusted(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcGetDeviceBlocked(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceBlocked* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendBlockedGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendBlockedGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceBlocked(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetDeviceBlocked(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceBlocked* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcGetDeviceAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceAlias* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendAliasGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendAliasGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceAlias(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetDeviceAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceAlias* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcGetDeviceAdapter(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceAdapter* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendAdapterGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendAdapterGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceAdapter(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceModalias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceModalias* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendModaliasGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendModaliasGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceModalias(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceRssi(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceRssi* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendRSSIGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendRSSIGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceRssi(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceVendorId(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceVendorId* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendVendorGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendVendorGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceVendorId(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceVendorIdSource(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceVendorIdSource* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendVendorSourceGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendVendorSourceGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceVendorIdSource(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceProductId(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceProductId* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendProductGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendProductGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceProductId(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceMapInfo(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceMapInfo* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendMapInfoGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendMapInfoGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceMapInfo(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceConnectedService(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceConnectedService* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendConnectedServiceGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendConnectedServiceGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceConnectedService(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceVersion(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceVersion* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendVersionGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendVersionGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceVersion(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDevicePbapInfo(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDevicePbapInfo* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendPBAPInfoGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendPBAPInfoGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDevicePbapInfo(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceMode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceMode* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendModeGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendModeGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceMode(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceRole(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceRole* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendRoleGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendRoleGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceRole(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDeviceEnableAvpPause(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceEnableAvpPause* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _deviceIf->sendEnableAvpPauseGet(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendEnableAvpPauseGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetDeviceEnableAvpPause(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcSetDeviceEnableAvpPause(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceEnableAvpPause* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _deviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      _deviceIf->sendEnableAvpPauseSet(ptrMessage->getDevice(), ptrMessage->getEnableAvpPause());

      ETG_TRACE_USR3((" handleBts2IpcSetDeviceEnableAvpPause(): Bts2Ipc 0x%08X success: token=<none>", ptrMessage->getMessageTraceId()));
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }

   // delete because there is no method return on ASF level
   deleteBts2IpcMsg = true;
}

void DbusAccess::handleBts2IpcBtApplDeviceConnectReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplDeviceConnectReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      BTSBinaryBDAddress binaryAddr;
      convertBdAddressString2Binary(binaryAddr, ptrMessage->getBDAddress());
      act_t token = _dmIf->sendBtApplDeviceConnectReqRequest(binaryAddr.getData(), ptrMessage->getServiceFunction().getData(), ptrMessage->getInstance());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplDeviceConnectReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplDeviceDisconnectReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplDeviceDisconnectReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _dmIf->sendBtApplDeviceDisconnectReqRequest(ptrMessage->getServiceFunction().getData(), ptrMessage->getInstance());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplDeviceDisconnectReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplCancelConnectReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplCancelConnectReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _dmIf->sendBtApplCancelConnectReqRequest();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplCancelConnectReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplGetHwVersionReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplGetHwVersionReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _dmIf->sendBtApplGetHwVersionReqRequest();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplGetHwVersionReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplSetConfigurationReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplSetConfigurationReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _dmIf->sendBtApplSetConfigurationReqRequest(ptrMessage->getInstance(), (uint16)ptrMessage->getConfigId(), ptrMessage->getData());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplSetConfigurationReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplReadConfigurationReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplReadConfigurationReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _dmIf->sendBtApplReadConfigurationReqRequest(ptrMessage->getInstance(), (uint16)ptrMessage->getConfigId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplReadConfigurationReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplTestModeReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplTestModeReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      uint8 enable = 0;
      if(true == ptrMessage->getEnable())
      {
         enable = 1;
      }
      BTSBinaryBDAddress binaryAddr;
      convertBdAddressString2Binary(binaryAddr, ptrMessage->getBDAddress());

      act_t token = _dmIf->sendBtApplTestModeReqRequest(enable, (uint8)ptrMessage->getMode(), (uint8)ptrMessage->getRole(), binaryAddr.getData());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" sendBtApplTestModeReqRequest(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcBtApplTestModeLinkQualityReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplTestModeLinkQualityReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _dmIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _dmIf->sendBtApplTestModeLinkQualityReqRequest((uint8)ptrMessage->getMode());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" sendBtApplTestModeLinkQualityReqRequest(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcConnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_Connect* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _serviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _serviceIf->sendConnectRequest(ptrMessage->getService());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcConnect(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getService());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcDisconnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_Disconnect* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _serviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _serviceIf->sendDisconnectRequest(ptrMessage->getService());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcDisconnect(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getService());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetServiceRemoteUuid(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceRemoteUuid* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _serviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _serviceIf->sendRemoteUUIDGet(ptrMessage->getService());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendRemoteUUIDGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetServiceRemoteUuid(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getService());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetServiceState(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceState* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _serviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _serviceIf->sendStateGet(ptrMessage->getService());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendStateGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetServiceState(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getService());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetServiceDevice(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceDevice* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _serviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _serviceIf->sendDeviceGet(ptrMessage->getService());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendDeviceGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetServiceDevice(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getService());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetServiceVersion(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceVersion* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _serviceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _serviceIf->sendVersionGet(ptrMessage->getService());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         ETG_TRACE_FATAL(("sendVersionGet sending failed"));
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetServiceVersion(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getService());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcDebugTraceConfig(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DebugTraceConfig* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcProtocolTraceConfig(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ProtocolTraceConfig* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcBtApplDidSetReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplDidSetReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _didIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      uint8 index = 0;
      ::std::vector< uint8 > record;
      const BTSDeviceIdServiceRecord& serviceRecord = ptrMessage->getServiceRecord();

      record.reserve(17 + serviceRecord.clientExecutableURL.size() + serviceRecord.serviceDescription.size() + serviceRecord.clientDocumentationURL.size());

      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, serviceRecord.specificationID);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, serviceRecord.vendorID);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, serviceRecord.productID);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, serviceRecord.version);
      ::ccdbusif::evolution_genivi_utility::writeUInt8(record, (uint8_t)serviceRecord.primaryRecord);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, serviceRecord.vendorIDSource);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, (uint16_t)serviceRecord.clientExecutableURL.size());
      ::ccdbusif::evolution_genivi_utility::writeString(record, serviceRecord.clientExecutableURL);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, (uint16_t)serviceRecord.serviceDescription.size());
      ::ccdbusif::evolution_genivi_utility::writeString(record, serviceRecord.serviceDescription);
      ::ccdbusif::evolution_genivi_utility::writeUInt16(record, (uint16_t)serviceRecord.clientDocumentationURL.size());
      ::ccdbusif::evolution_genivi_utility::writeString(record, serviceRecord.clientDocumentationURL);

      act_t token = _didIf->sendBtApplDidSetReqRequest(index, record);

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplDidSetReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetDunDeviceFilePath(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDunDeviceFilePath* ptrMessage)
{
   // not used
   FW_NORMAL_ASSERT_ALWAYS();

   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);
   (void)(ptrMessage);
}

void DbusAccess::handleBts2IpcBtApplSppSetUuidReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplSppSetUuidReq* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _sppIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else if((false == ::fw::isValid128BitUuid(ptrMessage->getOutgoingUuid())) ||
           ((false == ptrMessage->getIncomingUuid().empty()) && (false == ::fw::isValid128BitUuid(ptrMessage->getIncomingUuid()))))
   {
      // check must be done before
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      ::std::vector< uint8 > data;
      uint8_t type = 0x03; // 128 bit UUID
      ::std::vector< uint8 > binaryOutgoingUuid;
      ::std::vector< uint8 > binaryIncomingUuid;
      convertUuidString2Binary(binaryOutgoingUuid, ptrMessage->getOutgoingUuid());

      data.reserve(34);

      ::ccdbusif::evolution_genivi_utility::writeUInt8(data, type);
      ::ccdbusif::evolution_genivi_utility::writeArray(data, binaryOutgoingUuid);

      if(false == ptrMessage->getIncomingUuid().empty())
      {
         // set also incoming UUID
         convertUuidString2Binary(binaryIncomingUuid, ptrMessage->getIncomingUuid());
         ::ccdbusif::evolution_genivi_utility::writeUInt8(data, type);
         ::ccdbusif::evolution_genivi_utility::writeArray(data, binaryIncomingUuid);
      }
      else
      {
         // empty incoming UUID
         const BTSUuid incomingUuid("0000");
         convertUuidString2Binary(binaryIncomingUuid, incomingUuid);
         ::ccdbusif::evolution_genivi_utility::writeUInt8(data, 0x01); // 16 bit UUID to be used with 0x0000 UUID value TODO: to be tested
         ::ccdbusif::evolution_genivi_utility::writeArray(data, binaryIncomingUuid);
      }

      act_t token = _sppIf->sendBtApplSppSetUuidReqRequest(ptrMessage->getInstance(), data);

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcBtApplSppSetUuidReq(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcCreateSession(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CreateSession* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _obexClientIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      ::std::string destination;
      ::std::string target;
      ::std::string source;
      const uint8 channel = 0;
      const int32 instance = (int32)ptrMessage->getInstanceId();
      act_t token;
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;

      convertBdAddress2ExternalValue(destination, ptrMessage->getBDAddress());
      convertBdAddress2ExternalValue(source, ptrMessage->getSourceAddress());

      if(BTS_PROTO_PIM == ptrMessage->getProtocolId())
      {
         target = parser.getObexClientType2String(::ccdbusif::evolution::OBEX_CLIENT_ARGS_PBAP);
         token = _obexClientIf->sendCreateSessionRequest(destination, target, source, channel);
      }
      else
      {
         target = parser.getObexClientType2String(::ccdbusif::evolution::OBEX_CLIENT_ARGS_MAP);
         token = _obexClientIf->sendCreateSessionRequest(destination, target, source, channel, instance);
      }

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcCreateSession(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcRemoveSession(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RemoveSession* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _obexClientIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _obexClientIf->sendRemoveSessionRequest(ptrMessage->getSession());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcRemoveSession(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcCancelSession(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelSession* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _obexClientIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      ::std::string destination;
      ::std::string target;
      ::std::string source;
      const uint8 channel = 0;
      const int32 instance = (int32)ptrMessage->getInstanceId();
      act_t token;
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;

      convertBdAddress2ExternalValue(destination, ptrMessage->getBDAddress());
      convertBdAddress2ExternalValue(source, ptrMessage->getSourceAddress());

      if(BTS_PROTO_PIM == ptrMessage->getProtocolId())
      {
         target = parser.getObexClientType2String(::ccdbusif::evolution::OBEX_CLIENT_ARGS_PBAP);
         token = _obexClientIf->sendCancelSessionRequest(destination, target, source, channel);
      }
      else
      {
         target = parser.getObexClientType2String(::ccdbusif::evolution::OBEX_CLIENT_ARGS_MAP);
         token = _obexClientIf->sendCancelSessionRequest(destination, target, source, channel, instance);
      }

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcCancelSession(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetConnManManagerProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManManagerProperties* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _connManManagerIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _connManManagerIf->sendGetPropertiesRequest();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetConnManServices(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetConnManTechnologies(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManTechnologies* ptrMessage)
{
   // not used
   FW_NORMAL_ASSERT_ALWAYS();

   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);
   (void)(ptrMessage);
}

void DbusAccess::handleBts2IpcGetConnManServices(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManServices* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _connManManagerIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _connManManagerIf->sendGetServicesRequest();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetConnManServices(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetConnManServiceProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManServiceProperties* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _connManServiceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _connManServiceIf->sendGetPropertiesRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetConnManServiceProperties(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setTimeout(_getUpdateTimeout);
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcConnManServiceConnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ConnManServiceConnect* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _connManServiceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _connManServiceIf->sendConnectRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcConnManServiceConnect(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcConnManServiceDisconnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ConnManServiceDisconnect* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _connManServiceIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _connManServiceIf->sendDisconnectRequest(ptrMessage->getDevice());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcConnManServiceDisconnect(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getDevice());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetManagedObjectsTelephony(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetManagedObjectsTelephony* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcGetAllPropertiesTelephony(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAllPropertiesTelephony* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcGetIntrospectionTelephony(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetIntrospectionTelephony* ptrMessage)
{
   (void)(sendIpc2BtsMsgList);
   (void)(deleteBts2IpcMsg);

   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // TODO
}

void DbusAccess::handleBts2IpcGetHandsfreeProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetHandsfreeProperties* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _hfpHandsfreeIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _hfpHandsfreeIf->sendGetPropertiesRequest(ptrMessage->getModem());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetHfpConnectionProperties(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getModem());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetModems(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetModems* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _hfpManagerIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _hfpManagerIf->sendGetModemsRequest();

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetModemConnections(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

void DbusAccess::handleBts2IpcGetModemProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetModemProperties* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(NULL == _hfpModemIf)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteBts2IpcMsg = true;
   }
   else
   {
      act_t token = _hfpModemIf->sendGetPropertiesRequest(ptrMessage->getModem());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteBts2IpcMsg = true;
      }
      else
      {
         // sending success
         ptrMessage->setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetModemConnectionProperties(): Bts2Ipc 0x%08X success: token=%s", ptrMessage->getMessageTraceId(), info.getInfo()));
         ptrMessage->setObjectId(ptrMessage->getModem());
      }
   }

   if(true == deleteBts2IpcMsg)
   {
      sendErrorCode(sendIpc2BtsMsgList, ptrMessage);
   }
}

//---------------------------------------------------------------------------

} //genivi
} //btstackif
