/**
 * @file DbusAccessExtensions.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief Implementation of DBUS functionality for extensions.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of DBUS functionality for extensions.
 */

#include "DbusAccess_GEN.h"
#include "IDbusIfHandler.h"

namespace btstackif {
namespace genivi {

void DbusAccess::initializeExtensions(IN ::ccdbusif::ICcDbusIfController* controller, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   // add general extensions here if needed

   initializeSingleExtension(_configurationExtensions, controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
   initializeSingleExtension(_connectionExtensions, controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
   initializeSingleExtension(_phonecallExtensions, controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
   initializeSingleExtension(_phonebookExtensions, controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
   initializeSingleExtension(_messagingExtensions, controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
   initializeSingleExtension(_mediaPlayerExtensions, controller, component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
}

void DbusAccess::initializeSingleExtension(INOUT ::std::vector< IDbusIfHandler* >& extension, IN ::ccdbusif::ICcDbusIfController* controller, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         // set all interfaces before calling initialize
         extension[i]->setDbusBaseIf(this);
         extension[i]->setDbusRecHandlerIf(this);
         extension[i]->initialize();
         extension[i]->setCcDbusIfControllerIf(component, stackInterface, subComponent, userMode, controller, dbusInterfaces, configuration);
      }
   }
}

void DbusAccess::destroyExtensions(void)
{
   destroySingleExtension(_configurationExtensions);
   destroySingleExtension(_connectionExtensions);
   destroySingleExtension(_phonecallExtensions);
   destroySingleExtension(_phonebookExtensions);
   destroySingleExtension(_messagingExtensions);
   destroySingleExtension(_mediaPlayerExtensions);
}

void DbusAccess::destroySingleExtension(INOUT ::std::vector< IDbusIfHandler* >& extension)
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         extension[i]->resetCcDbusIfControllerIf();
         extension[i]->destroy();
         delete extension[i];
      }
   }
   extension.clear();
}

bool DbusAccess::setExtensionsSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(true == setSingleExtensionSimulationTestCommand(_configurationExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_connectionExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_phonecallExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_phonebookExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_messagingExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_mediaPlayerExtensions, testCommand, testData))
   {
      return true;
   }

   return false;
}

bool DbusAccess::setSingleExtensionSimulationTestCommand(IN ::std::vector< IDbusIfHandler* >& extension, IN const char* testCommand, IN const unsigned int testData)
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         if(true == extension[i]->setSimulationTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusAccess::setExtensionsSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   if(true == setSingleExtensionSimulationTestCommand(_configurationExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_connectionExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_phonecallExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_phonebookExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_messagingExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_mediaPlayerExtensions, testCommand, testData))
   {
      return true;
   }

   return false;
}

bool DbusAccess::setSingleExtensionSimulationTestCommand(IN ::std::vector< IDbusIfHandler* >& extension, IN const char* testCommand, IN const unsigned char* testData)
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         if(true == extension[i]->setSimulationTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusAccess::setExtensionsSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   if(true == setSingleExtensionSimulationTestCommand(_configurationExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_connectionExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_phonecallExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_phonebookExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_messagingExtensions, testCommand, testData))
   {
      return true;
   }

   if(true == setSingleExtensionSimulationTestCommand(_mediaPlayerExtensions, testCommand, testData))
   {
      return true;
   }

   return false;
}

bool DbusAccess::setSingleExtensionSimulationTestCommand(IN ::std::vector< IDbusIfHandler* >& extension, IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         if(true == extension[i]->setSimulationTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusAccess::sendExtensionMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   if(true == sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _configurationExtensions, message))
   {
      return true;
   }

   if(true == sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _connectionExtensions, message))
   {
      return true;
   }

   if(true == sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _phonecallExtensions, message))
   {
      return true;
   }

   if(true == sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _phonebookExtensions, message))
   {
      return true;
   }

   if(true == sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _messagingExtensions, message))
   {
      return true;
   }

   if(true == sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _mediaPlayerExtensions, message))
   {
      return true;
   }

   return false;
}

bool DbusAccess::sendExtensionConnectionMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   return sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _connectionExtensions, message);
}

bool DbusAccess::sendExtensionPhonecallMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   return sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _phonecallExtensions, message);
}

bool DbusAccess::sendExtensionPhonebookMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   return sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _phonebookExtensions, message);
}

bool DbusAccess::sendExtensionMessagingMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   return sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _messagingExtensions, message);
}

bool DbusAccess::sendExtensionMediaPlayerMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   return sendSingleExtensionMessage(deleteMsg, sendErrorMsg, _mediaPlayerExtensions, message);
}

bool DbusAccess::sendSingleExtensionMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN ::std::vector< IDbusIfHandler* >& extension, IN Bts2Ipc_BaseMessage* message)
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         if(true == extension[i]->sendMessage(deleteMsg, sendErrorMsg, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusAccess::createExtensionErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage)
{
   if(true == createSingleExtensionErrorMessage(ipc2BtsMessage, _configurationExtensions, bts2IpcMessage))
   {
      return true;
   }

   if(true == createSingleExtensionErrorMessage(ipc2BtsMessage, _connectionExtensions, bts2IpcMessage))
   {
      return true;
   }

   if(true == createSingleExtensionErrorMessage(ipc2BtsMessage, _phonecallExtensions, bts2IpcMessage))
   {
      return true;
   }

   if(true == createSingleExtensionErrorMessage(ipc2BtsMessage, _phonebookExtensions, bts2IpcMessage))
   {
      return true;
   }

   if(true == createSingleExtensionErrorMessage(ipc2BtsMessage, _messagingExtensions, bts2IpcMessage))
   {
      return true;
   }

   if(true == createSingleExtensionErrorMessage(ipc2BtsMessage, _mediaPlayerExtensions, bts2IpcMessage))
   {
      return true;
   }

   return false;
}

bool DbusAccess::createExtensionConnectionErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   return createSingleExtensionErrorMessage(ipc2BtsMessage, _connectionExtensions, bts2IpcMessage);
}

bool DbusAccess::createExtensionPhonecallErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   return createSingleExtensionErrorMessage(ipc2BtsMessage, _phonecallExtensions, bts2IpcMessage);
}

bool DbusAccess::createExtensionPhonebookErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   return createSingleExtensionErrorMessage(ipc2BtsMessage, _phonebookExtensions, bts2IpcMessage);
}

bool DbusAccess::createExtensionMessagingErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   return createSingleExtensionErrorMessage(ipc2BtsMessage, _messagingExtensions, bts2IpcMessage);
}

bool DbusAccess::createExtensionMediaPlayerErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   return createSingleExtensionErrorMessage(ipc2BtsMessage, _mediaPlayerExtensions, bts2IpcMessage);
}

bool DbusAccess::createSingleExtensionErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const ::std::vector< IDbusIfHandler* >& extension, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   for(size_t i = 0; i < extension.size(); i++)
   {
      if(0 != extension[i])
      {
         if(true == extension[i]->createErrorMessage(ipc2BtsMessage, bts2IpcMessage, BTS_IPC_SENDING_FAILED))
         {
            return true;
         }
      }
   }

   return false;
}

} //genivi
} //btstackif
