/**
 * @file ConnManDbusManagerFactory.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief ConnMan instance factory for Manager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan instance factory for Manager.
 */

#include "ConnManDbusManagerFactory.h"
#include "ConnManDbusManagerCallbackIf.h"

namespace btstackif {
namespace genivi {

ConnManDbusManagerFactory::ConnManDbusManagerFactory()
{
   _createdLocally = true;
   _cb = NULL;
}

ConnManDbusManagerFactory::~ConnManDbusManagerFactory()
{
   _cb = NULL;
}

ConnManDbusManagerFactory& ConnManDbusManagerFactory::getInstance(void)
{
   static ConnManDbusManagerFactory factory;
   return factory;
}

::ccdbusif::IConnManDbusManagerCallbackIf* ConnManDbusManagerFactory::getCb(IDbusRecHandler* handler)
{
   if(NULL != _cb)
   {
      // already created; set receive handler
      _cb->setReceiveHandler(handler);
   }
   else
   {
      _cb = new ConnManDbusManagerCallbackIf(handler);
      _createdLocally = true;
   }

   return _cb;
}

void ConnManDbusManagerFactory::destroyCb(void)
{
   if((NULL != _cb) && (true == _createdLocally))
   {
      delete _cb;
      _cb = NULL;
   }
}

void ConnManDbusManagerFactory::setCbForTesting(ConnManDbusManagerCallbackIf* testCallback)
{
   if(NULL != _cb)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _cb = testCallback;
      _createdLocally = false;
   }
}

void ConnManDbusManagerFactory::setReceiveHandler(IDbusRecHandler* handler)
{
   if(NULL != _cb)
   {
      _cb->setReceiveHandler(handler);
   }
}

void ConnManDbusManagerFactory::setServiceIf(::ccdbusif::IConnManDbusServiceSendRequestIf* handler)
{
   if(NULL != _cb)
   {
      _cb->setServiceIf(handler);
   }
}

} //genivi
} //btstackif
