/**
 * @file DbusAccess_GEN.h
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS GENIVI handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for DBUS GENIVI handling.
 */

#ifndef _DBUS_ACCESS_GENIVI_H_
#define _DBUS_ACCESS_GENIVI_H_

#include "BtStackInternalTypes.h"
#include "DbusBase.h"
#include "IDbusTrmHandler.h"
#include "IDbusResetTrmHandler.h"
#include "IDbusRecHandler.h"
#include "IDbusCallbackHandler.h"
#include "IDbusTimeoutHandler.h"
#include "IDbusResponseTimeoutHandler.h"

// class forward declarations
namespace ccdbusif {
class IOrgFreedesktopDbusObjectManagerSendRequestIf;
class IIntrospectableDbusSendRequestIf;
class IPropertiesGetAllDbusSendRequestIf;
// ConnMan sending
class IConnManDbusManagerSendRequestIf;
class IConnManDbusServiceSendRequestIf;
class IConnManDbusTechnologySendRequestIf;
// ConnMan callback
class IConnManDbusManagerCallbackIf;
class IConnManDbusServiceCallbackIf;
class IConnManDbusTechnologyCallbackIf;
// Evolution sending
class IEvolutionGeniviDbusAdapterSendRequestIf;
class IEvolutionGeniviDbusAgentSendRequestIf;
class IEvolutionGeniviDbusAgentManagerSendRequestIf;
class IEvolutionGeniviDbusDeviceSendRequestIf;
class IEvolutionGeniviDbusDMSendRequestIf;
class IEvolutionGeniviDbusServiceSendRequestIf;
class IEvolutionGeniviDbusTraceSendRequestIf;
class IEvolutionGeniviDbusGeniviTraceSendRequestIf;
class IEvolutionGeniviDbusDidSendRequestIf;
class IEvolutionGeniviDbusSppSendRequestIf;
class IEvolutionGeniviDbusObexClientSendRequestIf;
class IEvolutionGeniviDbusObexSessionSendRequestIf;
class IEvolutionGeniviDbusObexTransferSendRequestIf;
class IEvolutionGeniviDbusHfpCallVolumeSendRequestIf;
class IEvolutionGeniviDbusHfpHandsfreeSendRequestIf;
class IEvolutionGeniviDbusHfpManagerSendRequestIf;
class IEvolutionGeniviDbusHfpModemSendRequestIf;
class IEvolutionGeniviDbusHfpNetworkOperatorSendRequestIf;
class IEvolutionGeniviDbusHfpNetworkRegistrationSendRequestIf;
class IEvolutionGeniviDbusHfpSiriSendRequestIf;
class IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf;
class IEvolutionGeniviDbusHfpVoiceCallSendRequestIf;
class IEvolutionGeniviDbusSerialSendRequestIf;
// Evolution callback
class IEvolutionGeniviDbusAgentManagerCallbackIf;
class IEvolutionGeniviDbusDeviceCallbackIf;
class IEvolutionGeniviDbusDMCallbackIf;
class IEvolutionGeniviDbusServiceCallbackIf;
class IEvolutionGeniviDbusTraceCallbackIf;
class IEvolutionGeniviDbusGeniviTraceCallbackIf;
class IEvolutionGeniviDbusDidCallbackIf;
class IEvolutionGeniviDbusSppCallbackIf;
class IEvolutionGeniviDbusObexClientCallbackIf;
class IEvolutionGeniviDbusObexSessionCallbackIf;
class IEvolutionGeniviDbusObexTransferCallbackIf;
class IEvolutionGeniviDbusHfpCallVolumeCallbackIf;
class IEvolutionGeniviDbusHfpHandsfreeCallbackIf;
class IEvolutionGeniviDbusHfpNetworkOperatorCallbackIf;
class IEvolutionGeniviDbusHfpNetworkRegistrationCallbackIf;
class IEvolutionGeniviDbusHfpSiriCallbackIf;
class IEvolutionGeniviDbusHfpVoiceCallManagerCallbackIf;
class IEvolutionGeniviDbusHfpVoiceCallCallbackIf;
} //ccdbusif

namespace btstackif {
namespace genivi {

// class forward declarations
class IConnectionOrgFreedesktopObjectManagerCallbackIf;
class IntrospectableDbusCallbackIf;
class PropertiesGetAllDbusCallbackIf;
// Evolution callback
class IEvolutionGeniviAdapterCallbackIf;
class IEvolutionGeniviAgentCallbackIf;
class IEvolutionGeniviHfpManagerCallbackIf; // TODO: change all other too
class IEvolutionGeniviHfpModemCallbackIf;
class IEvolutionGeniviSerialCallbackIf;

// Bts2IPC messages
//------------------------------------------------------------------------------
// Line 23: class Bts2Ipc_GetManagedObjectsConnection : public Bts2Ipc_BaseMessage
// 'Line .*: class Bts2Ipc_(.*) : public Bts2Ipc_BaseMessage' => 'class Bts2Ipc_$1;'
//------------------------------------------------------------------------------
class Bts2Ipc_GetManagedObjectsConnection;
class Bts2Ipc_GetAllPropertiesConnection;
class Bts2Ipc_GetIntrospectionConnection;
class Bts2Ipc_ReleaseRes;
class Bts2Ipc_RequestPinCodeRes;
class Bts2Ipc_DisplayPinCodeRes;
class Bts2Ipc_RequestPasskeyRes;
class Bts2Ipc_RequestConfirmationRes;
class Bts2Ipc_CancelRes;
class Bts2Ipc_CancelRequestRes;
class Bts2Ipc_AuthorizeServiceRes;
class Bts2Ipc_RequestAuthorizationRes;
class Bts2Ipc_RequestLinkkeyRes;
class Bts2Ipc_DisplayPasskeyRes;
class Bts2Ipc_StartDiscovery;
class Bts2Ipc_StopDiscovery;
class Bts2Ipc_RemoveDevice;
class Bts2Ipc_CreateDevice;
class Bts2Ipc_SendHCICmd;
class Bts2Ipc_GetAdapterAddress;
class Bts2Ipc_GetAdapterClass;
class Bts2Ipc_GetAdapterAlias;
class Bts2Ipc_SetAdapterAlias;
class Bts2Ipc_GetAdapterPowered;
class Bts2Ipc_SetAdapterPowered;
class Bts2Ipc_GetAdapterDiscoverable;
class Bts2Ipc_SetAdapterDiscoverable;
class Bts2Ipc_GetAdapterDiscoverabletimeout;
class Bts2Ipc_SetAdapterDiscoverabletimeout;
class Bts2Ipc_GetAdapterPairable;
class Bts2Ipc_SetAdapterPairable;
class Bts2Ipc_GetAdapterPairabletimeout;
class Bts2Ipc_SetAdapterPairabletimeout;
class Bts2Ipc_GetAdapterDiscovering;
class Bts2Ipc_GetAdapterUuids;
class Bts2Ipc_SetAdapterUuids;
class Bts2Ipc_GetAdapterWbsMode;
class Bts2Ipc_SetAdapterWbsMode;
class Bts2Ipc_GetAdapterConnectable;
class Bts2Ipc_SetAdapterConnectable;
class Bts2Ipc_GetAdapterHcimode;
class Bts2Ipc_SetAdapterHcimode;
class Bts2Ipc_GetAdapterVersion;
class Bts2Ipc_GetAdapterTraceLevel;
class Bts2Ipc_SetAdapterTraceLevel;
class Bts2Ipc_GetAdapterTraceCategory;
class Bts2Ipc_SetAdapterTraceCategory;
class Bts2Ipc_GetAdapterAvailableSppServers;
class Bts2Ipc_SetAdapterAvailableSppServers;
class Bts2Ipc_RegisterAgent;
class Bts2Ipc_UnregisterAgent;
class Bts2Ipc_RequestDefaultAgent;
class Bts2Ipc_DeviceConnect;
class Bts2Ipc_DeviceDisconnect;
class Bts2Ipc_Pair;
class Bts2Ipc_CancelPairing;
class Bts2Ipc_DiscoverServices;
class Bts2Ipc_CancelDiscovery;
class Bts2Ipc_GetDeviceAddress;
class Bts2Ipc_GetDeviceClass;
class Bts2Ipc_GetDeviceIcon;
class Bts2Ipc_GetDeviceUuids;
class Bts2Ipc_GetDevicePaired;
class Bts2Ipc_GetDeviceConnected;
class Bts2Ipc_GetDeviceTrusted;
class Bts2Ipc_SetDeviceTrusted;
class Bts2Ipc_GetDeviceBlocked;
class Bts2Ipc_SetDeviceBlocked;
class Bts2Ipc_GetDeviceAlias;
class Bts2Ipc_SetDeviceAlias;
class Bts2Ipc_GetDeviceAdapter;
class Bts2Ipc_GetDeviceModalias;
class Bts2Ipc_GetDeviceRssi;
class Bts2Ipc_GetDeviceVendorId;
class Bts2Ipc_GetDeviceVendorIdSource;
class Bts2Ipc_GetDeviceProductId;
class Bts2Ipc_GetDeviceMapInfo;
class Bts2Ipc_GetDeviceConnectedService;
class Bts2Ipc_GetDeviceVersion;
class Bts2Ipc_GetDevicePbapInfo;
class Bts2Ipc_GetDeviceMode;
class Bts2Ipc_GetDeviceRole;
class Bts2Ipc_GetDeviceEnableAvpPause;
class Bts2Ipc_SetDeviceEnableAvpPause;
class Bts2Ipc_BtApplDeviceConnectReq;
class Bts2Ipc_BtApplDeviceDisconnectReq;
class Bts2Ipc_BtApplCancelConnectReq;
class Bts2Ipc_BtApplGetHwVersionReq;
class Bts2Ipc_BtApplSetConfigurationReq;
class Bts2Ipc_BtApplReadConfigurationReq;
class Bts2Ipc_BtApplTestModeReq;
class Bts2Ipc_BtApplTestModeLinkQualityReq;
class Bts2Ipc_Connect;
class Bts2Ipc_Disconnect;
class Bts2Ipc_GetServiceRemoteUuid;
class Bts2Ipc_GetServiceState;
class Bts2Ipc_GetServiceDevice;
class Bts2Ipc_GetServiceVersion;
class Bts2Ipc_DebugTraceConfig;
class Bts2Ipc_ProtocolTraceConfig;
class Bts2Ipc_BtApplDidSetReq;
class Bts2Ipc_GetDunDeviceFilePath;
class Bts2Ipc_BtApplSppSetUuidReq;
class Bts2Ipc_CreateSession;
class Bts2Ipc_RemoveSession;
class Bts2Ipc_CancelSession;
class Bts2Ipc_GetConnManManagerProperties;
class Bts2Ipc_GetConnManTechnologies;
class Bts2Ipc_GetConnManServices;
class Bts2Ipc_GetConnManServiceProperties;
class Bts2Ipc_ConnManServiceConnect;
class Bts2Ipc_ConnManServiceDisconnect;
class Bts2Ipc_GetManagedObjectsTelephony;
class Bts2Ipc_GetAllPropertiesTelephony;
class Bts2Ipc_GetIntrospectionTelephony;
class Bts2Ipc_GetHandsfreeProperties;
class Bts2Ipc_GetModems;
class Bts2Ipc_GetModemProperties;
//------------------------------------------------------------------------------

/**
 *
 */
class DbusAccess : public DbusBase, public IDbusTrmHandler, public IDbusResetTrmHandler, public IDbusRecHandler, public IDbusCallbackHandler, public IDbusTimeoutHandler, public IDbusResponseTimeoutHandler
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Default constructor.
    */
   DbusAccess();

private:
   /**
    * Copy constructor.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be copied
    */
   DbusAccess(const DbusAccess& ref);

   /**
    * Assignment operator.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be assigned
    */
   DbusAccess& operator=(const DbusAccess& ref);

public:
   /**
    * Destructor.
    */
   virtual ~DbusAccess();

   virtual void sendBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage);

   virtual void stop(void);

   virtual void run(void);

   virtual void resetHandler(void);

   virtual BTSErrorCode setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   virtual void resetCcDbusIfControllerIf(void);

   virtual void handleTimerTick(void);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType);

   virtual void threadFunction(void* arguments);

   virtual void setTerminate(void* arguments);

   virtual void handleDbusTrmQueue(void);

   virtual void handleDbusResetTrmQueue(void);

   virtual void handleReceivedDbusMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false);

   virtual void handleDbusCallbackMessage(void);

   virtual void handleDbusTimeoutEvent(void);

   virtual void handleDbusResponseTimeout(Bts2Ipc_BaseMessage* ptrMessage);

private:
   const BTSTimeValue _getUpdateTimeout; /**< timeout value in ms for receiving update message for a get request */
   const unsigned int _count; /**< initial number of timer entries */
   const ::std::string _systemBusName; // TODO: check for exporting to separate file
   const ::std::string _sessionBusName; // TODO: check if needed
   const ::std::string _systemObjPath; // TODO: check if needed
   const ::std::string _sessionObjPath; // TODO: check if needed

   // general services
   ::ccdbusif::IOrgFreedesktopDbusObjectManagerSendRequestIf* _objectManagerIf;
   ::ccdbusif::IIntrospectableDbusSendRequestIf* _introspectableIf;
   ::ccdbusif::IPropertiesGetAllDbusSendRequestIf* _propertiesGetAllIf;

   // ConnMan services
   ::ccdbusif::IConnManDbusManagerSendRequestIf* _connManManagerIf;
   ::ccdbusif::IConnManDbusServiceSendRequestIf* _connManServiceIf;
   ::ccdbusif::IConnManDbusTechnologySendRequestIf* _connManTechnologyIf;

   // specific services
   ::ccdbusif::IEvolutionGeniviDbusAdapterSendRequestIf* _adapterIf;
   ::ccdbusif::IEvolutionGeniviDbusAgentSendRequestIf* _agentIf;
   ::ccdbusif::IEvolutionGeniviDbusAgentManagerSendRequestIf* _agentManagerIf;
   ::ccdbusif::IEvolutionGeniviDbusDeviceSendRequestIf* _deviceIf;
   ::ccdbusif::IEvolutionGeniviDbusDMSendRequestIf* _dmIf;
   ::ccdbusif::IEvolutionGeniviDbusServiceSendRequestIf* _serviceIf;
   ::ccdbusif::IEvolutionGeniviDbusTraceSendRequestIf* _traceIf;
   ::ccdbusif::IEvolutionGeniviDbusGeniviTraceSendRequestIf* _geniviTraceIf;
   ::ccdbusif::IEvolutionGeniviDbusDidSendRequestIf* _didIf;
   ::ccdbusif::IEvolutionGeniviDbusSppSendRequestIf* _sppIf;
   ::ccdbusif::IEvolutionGeniviDbusObexClientSendRequestIf* _obexClientIf;
   ::ccdbusif::IEvolutionGeniviDbusObexSessionSendRequestIf* _obexSessionIf;
   ::ccdbusif::IEvolutionGeniviDbusObexTransferSendRequestIf* _obexTransferIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpCallVolumeSendRequestIf* _hfpCallVolumeIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpHandsfreeSendRequestIf* _hfpHandsfreeIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpManagerSendRequestIf* _hfpManagerIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpModemSendRequestIf* _hfpModemIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpNetworkOperatorSendRequestIf* _hfpNetworkOperatorIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpNetworkRegistrationSendRequestIf* _hfpNetworkRegistrationIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpSiriSendRequestIf* _hfpSiriIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf* _hfpVoiceCallManagerIf;
   ::ccdbusif::IEvolutionGeniviDbusHfpVoiceCallSendRequestIf* _hfpVoiceCallIf;
   ::ccdbusif::IEvolutionGeniviDbusSerialSendRequestIf* _serialIf;

   // general services
   IConnectionOrgFreedesktopObjectManagerCallbackIf* _objectManagerCb;
   IntrospectableDbusCallbackIf* _introspectableCb;
   PropertiesGetAllDbusCallbackIf* _propertiesGetAllCb;

   // ConnMan services
   ::ccdbusif::IConnManDbusManagerCallbackIf* _connManManagerCb;
   ::ccdbusif::IConnManDbusServiceCallbackIf* _connManServiceCb;
   ::ccdbusif::IConnManDbusTechnologyCallbackIf* _connManTechnologyCb;

   // specific services
   IEvolutionGeniviAdapterCallbackIf* _adapterCb;
   IEvolutionGeniviAgentCallbackIf* _agentCb;
   ::ccdbusif::IEvolutionGeniviDbusAgentManagerCallbackIf* _agentManagerCb;
   ::ccdbusif::IEvolutionGeniviDbusDeviceCallbackIf* _deviceCb;
   ::ccdbusif::IEvolutionGeniviDbusDMCallbackIf* _dmCb;
   ::ccdbusif::IEvolutionGeniviDbusServiceCallbackIf* _serviceCb;
   ::ccdbusif::IEvolutionGeniviDbusTraceCallbackIf* _traceCb;
   ::ccdbusif::IEvolutionGeniviDbusGeniviTraceCallbackIf* _geniviTraceCb;
   ::ccdbusif::IEvolutionGeniviDbusDidCallbackIf* _didCb;
   ::ccdbusif::IEvolutionGeniviDbusSppCallbackIf* _sppCb;
   ::ccdbusif::IEvolutionGeniviDbusObexClientCallbackIf* _obexClientCb;
   ::ccdbusif::IEvolutionGeniviDbusObexSessionCallbackIf* _obexSessionCb;
   ::ccdbusif::IEvolutionGeniviDbusObexTransferCallbackIf* _obexTransferCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpCallVolumeCallbackIf* _hfpCallVolumeCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpHandsfreeCallbackIf* _hfpHandsfreeCb;
   IEvolutionGeniviHfpManagerCallbackIf* _hfpManagerCb;
   IEvolutionGeniviHfpModemCallbackIf* _hfpModemCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpNetworkOperatorCallbackIf* _hfpNetworkOperatorCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpNetworkRegistrationCallbackIf* _hfpNetworkRegistrationCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpSiriCallbackIf* _hfpSiriCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpVoiceCallManagerCallbackIf* _hfpVoiceCallManagerCb;
   ::ccdbusif::IEvolutionGeniviDbusHfpVoiceCallCallbackIf* _hfpVoiceCallCb;
   IEvolutionGeniviSerialCallbackIf* _serialCb;

   //---------------------------------------------------------------------------

   /**
    * Create ALPS EVOLUTION GENIVI specific extensions.
    *
    * @param[in] configuration: local stack configuration data
    * @param[in] userMode: user mode
    * @param[in] interface: stack interface type
    */
   void createAlpsEvolutionGeniviExtensions(IN const BTSLocalStackConfiguration& configuration, IN const BTSUserMode userMode, IN const BTSInterfaceType interface);

   /**
    * Initialize general extensions (can be related to other functionality e.g. WiFi access).
    *
    * @param[in] controller: CcDbusIf controller (pointer)
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[in] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   void initializeExtensions(IN ::ccdbusif::ICcDbusIfController* controller, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   /**
    * Initialize single general extension.
    *
    * @param[in,out] extension: extension
    * @param[in] controller: CcDbusIf controller (pointer)
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[in] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   void initializeSingleExtension(INOUT ::std::vector< IDbusIfHandler* >& extension, IN ::ccdbusif::ICcDbusIfController* controller, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   /**
    * Destroy general extensions (can be related to other functionality e.g. WiFi access).
    */
   void destroyExtensions(void);

   /**
    * Destroy single general extension.
    *
    * @param[in,out] extension: extension
    */
   void destroySingleExtension(INOUT ::std::vector< IDbusIfHandler* >& extension);

   /**
    * Set a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setExtensionsSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData);

   /**
    * Set a simulation test command for single extension.
    *
    * @param[in] extension: extension
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setSingleExtensionSimulationTestCommand(IN ::std::vector< IDbusIfHandler* >& extension, IN const char* testCommand, IN const unsigned int testData);

   /**
    * Set a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setExtensionsSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Set a simulation test command for single extension.
    *
    * @param[in] extension: extension
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setSingleExtensionSimulationTestCommand(IN ::std::vector< IDbusIfHandler* >& extension, IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Set a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setExtensionsSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData);

   /**
    * Set a simulation test command for single extension.
    *
    * @param[in] extension: extension
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool setSingleExtensionSimulationTestCommand(IN ::std::vector< IDbusIfHandler* >& extension, IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData);

   /**
    * Send given message (extension).
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] message: message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool sendExtensionMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);

   /**
    * Send given message (extension).
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] message: message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool sendExtensionConnectionMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);
   bool sendExtensionPhonecallMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);
   bool sendExtensionPhonebookMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);
   bool sendExtensionMessagingMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);
   bool sendExtensionMediaPlayerMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);

   /**
    * Send given message (extension).
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] extension: extension
    * @param[in] message: message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool sendSingleExtensionMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN ::std::vector< IDbusIfHandler* >& extension, IN Bts2Ipc_BaseMessage* message);

   /**
    * Create error message (extension).
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] message: message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool createExtensionErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage);

   /**
    * Create error message (extension).
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] message: message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool createExtensionConnectionErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   bool createExtensionPhonecallErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   bool createExtensionPhonebookErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   bool createExtensionMessagingErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   bool createExtensionMediaPlayerErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;

   /**
    * Create error message (extension).
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] extension: extension
    * @param[in] message: message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool createSingleExtensionErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const ::std::vector< IDbusIfHandler* >& extension, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;

   //---------------------------------------------------------------------------

   /** local functions*/
   void removeBts2IpcWorkingMessagesOnObjectPathRemovedSignal(Ipc2Bts_BaseMessage* ptrMessage);

   void sendErrorCode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, Bts2Ipc_BaseMessage* ptrMessage);

   bool createErrorMessageLocal(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const;

   bool transferMessageDataLocal(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;

   void handleConfigurationMessage(Bts2Ipc_BaseMessage* ptrMessage);
   void handleConnectionMessage(Bts2Ipc_BaseMessage* ptrMessage);
   void handlePhonecallMessage(Bts2Ipc_BaseMessage* ptrMessage);
   void handlePhonebookMessage(Bts2Ipc_BaseMessage* ptrMessage);
   void handleMessagingMessage(Bts2Ipc_BaseMessage* ptrMessage);
   void handleMediaPlayerMessage(Bts2Ipc_BaseMessage* ptrMessage);

   Ipc2Bts_BaseMessage* createIpc2BtsConfigurationErrorMessage(IN const Bts2Ipc_BaseMessage& message) const;
   Ipc2Bts_BaseMessage* createIpc2BtsConnectionErrorMessage(IN const Bts2Ipc_BaseMessage& message) const;
   Ipc2Bts_BaseMessage* createIpc2BtsPhonecallErrorMessage(IN const Bts2Ipc_BaseMessage& message) const;
   Ipc2Bts_BaseMessage* createIpc2BtsPhonebookErrorMessage(IN const Bts2Ipc_BaseMessage& message) const;
   Ipc2Bts_BaseMessage* createIpc2BtsMessagingErrorMessage(IN const Bts2Ipc_BaseMessage& message) const;
   Ipc2Bts_BaseMessage* createIpc2BtsMediaPlayerErrorMessage(IN const Bts2Ipc_BaseMessage& message) const;

   void transferMessageDataConfiguration(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   void transferMessageDataConnection(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   void transferMessageDataPhonecall(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   void transferMessageDataPhonebook(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   void transferMessageDataMessaging(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;
   void transferMessageDataMediaPlayer(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;

   //------------------------------------------------------------------------------
   // class Bts2Ipc_GetAllPropertiesConnection;
   // '   class Bts2Ipc_(.*);' => '   void handleBts2Ipc$1(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_$1* ptrMessage);'
   //------------------------------------------------------------------------------
   void handleBts2IpcGetManagedObjectsConnection(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetManagedObjectsConnection* ptrMessage);
   void handleBts2IpcGetAllPropertiesConnection(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAllPropertiesConnection* ptrMessage);
   void handleBts2IpcGetIntrospectionConnection(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetIntrospectionConnection* ptrMessage);
   void handleBts2IpcReleaseRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ReleaseRes* ptrMessage);
   void handleBts2IpcRequestPinCodeRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestPinCodeRes* ptrMessage);
   void handleBts2IpcDisplayPinCodeRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DisplayPinCodeRes* ptrMessage);
   void handleBts2IpcRequestPasskeyRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestPasskeyRes* ptrMessage);
   void handleBts2IpcRequestConfirmationRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestConfirmationRes* ptrMessage);
   void handleBts2IpcCancelRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelRes* ptrMessage);
   void handleBts2IpcCancelRequestRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelRequestRes* ptrMessage);
   void handleBts2IpcAuthorizeServiceRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_AuthorizeServiceRes* ptrMessage);
   void handleBts2IpcRequestAuthorizationRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestAuthorizationRes* ptrMessage);
   void handleBts2IpcRequestLinkkeyRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestLinkkeyRes* ptrMessage);
   void handleBts2IpcDisplayPasskeyRes(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DisplayPasskeyRes* ptrMessage);
   void handleBts2IpcStartDiscovery(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_StartDiscovery* ptrMessage);
   void handleBts2IpcStopDiscovery(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_StopDiscovery* ptrMessage);
   void handleBts2IpcRemoveDevice(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RemoveDevice* ptrMessage);
   void handleBts2IpcCreateDevice(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CreateDevice* ptrMessage);
   void handleBts2IpcSendHCICmd(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SendHCICmd* ptrMessage);
   void handleBts2IpcGetAdapterAddress(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterAddress* ptrMessage);
   void handleBts2IpcGetAdapterClass(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterClass* ptrMessage);
   void handleBts2IpcGetAdapterAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterAlias* ptrMessage);
   void handleBts2IpcSetAdapterAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterAlias* ptrMessage);
   void handleBts2IpcGetAdapterPowered(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterPowered* ptrMessage);
   void handleBts2IpcSetAdapterPowered(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterPowered* ptrMessage);
   void handleBts2IpcGetAdapterDiscoverable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterDiscoverable* ptrMessage);
   void handleBts2IpcSetAdapterDiscoverable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterDiscoverable* ptrMessage);
   void handleBts2IpcGetAdapterDiscoverabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterDiscoverabletimeout* ptrMessage);
   void handleBts2IpcSetAdapterDiscoverabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterDiscoverabletimeout* ptrMessage);
   void handleBts2IpcGetAdapterPairable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterPairable* ptrMessage);
   void handleBts2IpcSetAdapterPairable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterPairable* ptrMessage);
   void handleBts2IpcGetAdapterPairabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterPairabletimeout* ptrMessage);
   void handleBts2IpcSetAdapterPairabletimeout(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterPairabletimeout* ptrMessage);
   void handleBts2IpcGetAdapterDiscovering(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterDiscovering* ptrMessage);
   void handleBts2IpcGetAdapterUuids(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterUuids* ptrMessage);
   void handleBts2IpcSetAdapterUuids(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterUuids* ptrMessage);
   void handleBts2IpcGetAdapterWbsMode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterWbsMode* ptrMessage);
   void handleBts2IpcSetAdapterWbsMode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterWbsMode* ptrMessage);
   void handleBts2IpcGetAdapterConnectable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterConnectable* ptrMessage);
   void handleBts2IpcSetAdapterConnectable(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterConnectable* ptrMessage);
   void handleBts2IpcGetAdapterHcimode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterHcimode* ptrMessage);
   void handleBts2IpcSetAdapterHcimode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterHcimode* ptrMessage);
   void handleBts2IpcGetAdapterVersion(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterVersion* ptrMessage);
   void handleBts2IpcGetAdapterTraceLevel(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterTraceLevel* ptrMessage);
   void handleBts2IpcSetAdapterTraceLevel(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterTraceLevel* ptrMessage);
   void handleBts2IpcGetAdapterTraceCategory(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterTraceCategory* ptrMessage);
   void handleBts2IpcSetAdapterTraceCategory(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterTraceCategory* ptrMessage);
   void handleBts2IpcGetAdapterAvailableSppServers(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAdapterAvailableSppServers* ptrMessage);
   void handleBts2IpcSetAdapterAvailableSppServers(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetAdapterAvailableSppServers* ptrMessage);
   void handleBts2IpcRegisterAgent(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RegisterAgent* ptrMessage);
   void handleBts2IpcUnregisterAgent(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_UnregisterAgent* ptrMessage);
   void handleBts2IpcRequestDefaultAgent(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RequestDefaultAgent* ptrMessage);
   void handleBts2IpcDeviceConnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DeviceConnect* ptrMessage);
   void handleBts2IpcDeviceDisconnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DeviceDisconnect* ptrMessage);
   void handleBts2IpcPair(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_Pair* ptrMessage);
   void handleBts2IpcCancelPairing(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelPairing* ptrMessage);
   void handleBts2IpcDiscoverServices(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DiscoverServices* ptrMessage);
   void handleBts2IpcCancelDiscovery(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelDiscovery* ptrMessage);
   void handleBts2IpcGetDeviceAddress(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceAddress* ptrMessage);
   void handleBts2IpcGetDeviceClass(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceClass* ptrMessage);
   void handleBts2IpcGetDeviceIcon(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceIcon* ptrMessage);
   void handleBts2IpcGetDeviceUuids(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceUuids* ptrMessage);
   void handleBts2IpcGetDevicePaired(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDevicePaired* ptrMessage);
   void handleBts2IpcGetDeviceConnected(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceConnected* ptrMessage);
   void handleBts2IpcGetDeviceTrusted(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceTrusted* ptrMessage);
   void handleBts2IpcSetDeviceTrusted(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceTrusted* ptrMessage);
   void handleBts2IpcGetDeviceBlocked(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceBlocked* ptrMessage);
   void handleBts2IpcSetDeviceBlocked(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceBlocked* ptrMessage);
   void handleBts2IpcGetDeviceAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceAlias* ptrMessage);
   void handleBts2IpcSetDeviceAlias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceAlias* ptrMessage);
   void handleBts2IpcGetDeviceAdapter(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceAdapter* ptrMessage);
   void handleBts2IpcGetDeviceModalias(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceModalias* ptrMessage);
   void handleBts2IpcGetDeviceRssi(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceRssi* ptrMessage);
   void handleBts2IpcGetDeviceVendorId(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceVendorId* ptrMessage);
   void handleBts2IpcGetDeviceVendorIdSource(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceVendorIdSource* ptrMessage);
   void handleBts2IpcGetDeviceProductId(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceProductId* ptrMessage);
   void handleBts2IpcGetDeviceMapInfo(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceMapInfo* ptrMessage);
   void handleBts2IpcGetDeviceConnectedService(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceConnectedService* ptrMessage);
   void handleBts2IpcGetDeviceVersion(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceVersion* ptrMessage);
   void handleBts2IpcGetDevicePbapInfo(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDevicePbapInfo* ptrMessage);
   void handleBts2IpcGetDeviceMode(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceMode* ptrMessage);
   void handleBts2IpcGetDeviceRole(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceRole* ptrMessage);
   void handleBts2IpcGetDeviceEnableAvpPause(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDeviceEnableAvpPause* ptrMessage);
   void handleBts2IpcSetDeviceEnableAvpPause(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_SetDeviceEnableAvpPause* ptrMessage);
   void handleBts2IpcBtApplDeviceConnectReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplDeviceConnectReq* ptrMessage);
   void handleBts2IpcBtApplDeviceDisconnectReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplDeviceDisconnectReq* ptrMessage);
   void handleBts2IpcBtApplCancelConnectReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplCancelConnectReq* ptrMessage);
   void handleBts2IpcBtApplGetHwVersionReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplGetHwVersionReq* ptrMessage);
   void handleBts2IpcBtApplSetConfigurationReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplSetConfigurationReq* ptrMessage);
   void handleBts2IpcBtApplReadConfigurationReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplReadConfigurationReq* ptrMessage);
   void handleBts2IpcBtApplTestModeReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplTestModeReq* ptrMessage);
   void handleBts2IpcBtApplTestModeLinkQualityReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplTestModeLinkQualityReq* ptrMessage);
   void handleBts2IpcConnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_Connect* ptrMessage);
   void handleBts2IpcDisconnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_Disconnect* ptrMessage);
   void handleBts2IpcGetServiceRemoteUuid(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceRemoteUuid* ptrMessage);
   void handleBts2IpcGetServiceState(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceState* ptrMessage);
   void handleBts2IpcGetServiceDevice(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceDevice* ptrMessage);
   void handleBts2IpcGetServiceVersion(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetServiceVersion* ptrMessage);
   void handleBts2IpcDebugTraceConfig(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_DebugTraceConfig* ptrMessage);
   void handleBts2IpcProtocolTraceConfig(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ProtocolTraceConfig* ptrMessage);
   void handleBts2IpcBtApplDidSetReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplDidSetReq* ptrMessage);
   void handleBts2IpcGetDunDeviceFilePath(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetDunDeviceFilePath* ptrMessage);
   void handleBts2IpcBtApplSppSetUuidReq(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_BtApplSppSetUuidReq* ptrMessage);
   void handleBts2IpcCreateSession(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CreateSession* ptrMessage);
   void handleBts2IpcRemoveSession(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_RemoveSession* ptrMessage);
   void handleBts2IpcCancelSession(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_CancelSession* ptrMessage);
   void handleBts2IpcGetConnManManagerProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManManagerProperties* ptrMessage);
   void handleBts2IpcGetConnManTechnologies(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManTechnologies* ptrMessage);
   void handleBts2IpcGetConnManServices(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManServices* ptrMessage);
   void handleBts2IpcGetConnManServiceProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetConnManServiceProperties* ptrMessage);
   void handleBts2IpcConnManServiceConnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ConnManServiceConnect* ptrMessage);
   void handleBts2IpcConnManServiceDisconnect(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_ConnManServiceDisconnect* ptrMessage);
   void handleBts2IpcGetManagedObjectsTelephony(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetManagedObjectsTelephony* ptrMessage);
   void handleBts2IpcGetAllPropertiesTelephony(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetAllPropertiesTelephony* ptrMessage);
   void handleBts2IpcGetIntrospectionTelephony(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetIntrospectionTelephony* ptrMessage);
   void handleBts2IpcGetHandsfreeProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetHandsfreeProperties* ptrMessage);
   void handleBts2IpcGetModems(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetModems* ptrMessage);
   void handleBts2IpcGetModemProperties(OUT ::std::vector<Ipc2Bts_BaseMessage*>& sendIpc2BtsMsgList, OUT bool& deleteBts2IpcMsg, IN Bts2Ipc_GetModemProperties* ptrMessage);
   //---------------------------------------------------------------------------
};

} //genivi
} //btstackif

#endif //_DBUS_ACCESS_GENIVI_H_
