/**
 * @file EvolutionGeniviAgentCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback for Agent.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback implementation for Agent.
 */

#include "EvolutionGeniviAgentCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "BtsUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviAgentCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::org::bluez::Agent;

namespace btstackif {
namespace genivi {

EvolutionGeniviAgentCallbackIf::EvolutionGeniviAgentCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0)
{
}

EvolutionGeniviAgentCallbackIf::~EvolutionGeniviAgentCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
// --- not needed because Agent interface is provided by local side ---
// "ServiceAvailableIF" implementation --- end

// /org/bluez/AgentStub implementation --- start
void EvolutionGeniviAgentCallbackIf::onReleaseRequest(const ::boost::shared_ptr< ReleaseRequest >& request, const act_t act)
{
   (void)(request);

   Ipc2Bts_Release* ptrMsg = ptrNew_Ipc2Bts_Release();

   if(NULL != ptrMsg)
   {
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onRequestPinCodeRequest(const ::boost::shared_ptr< RequestPinCodeRequest >& request, const act_t act)
{
   Ipc2Bts_RequestPinCode* ptrMsg = ptrNew_Ipc2Bts_RequestPinCode();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onDisplayPinCodeRequest(const ::boost::shared_ptr< DisplayPinCodeRequest >& request, const act_t act)
{
   Ipc2Bts_DisplayPinCode* ptrMsg = ptrNew_Ipc2Bts_DisplayPinCode();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      ptrMsg->setPasskey(request->getPincode());
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onRequestPasskeyRequest(const ::boost::shared_ptr< RequestPasskeyRequest >& request, const act_t act)
{
   Ipc2Bts_RequestPasskey* ptrMsg = ptrNew_Ipc2Bts_RequestPasskey();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onRequestConfirmationRequest(const ::boost::shared_ptr< RequestConfirmationRequest >& request, const act_t act)
{
   Ipc2Bts_RequestConfirmation* ptrMsg = ptrNew_Ipc2Bts_RequestConfirmation();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      BTSNumericValue numericValue;
      convertNumericValueBinary2String(numericValue, request->getPasskey());
      ptrMsg->setPasskey(numericValue);
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onCancelRequest(const ::boost::shared_ptr< CancelRequest >& request, const act_t act)
{
   (void)(request);

   Ipc2Bts_Cancel* ptrMsg = ptrNew_Ipc2Bts_Cancel();

   if(NULL != ptrMsg)
   {
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onCancelRequestRequest(const ::boost::shared_ptr< CancelRequestRequest >& request, const act_t act)
{
   (void)(request);

   Ipc2Bts_CancelRequest* ptrMsg = ptrNew_Ipc2Bts_CancelRequest();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onAuthorizeServiceRequest(const ::boost::shared_ptr< AuthorizeServiceRequest >& request, const act_t act)
{
   // in case of enabled SPP server functionality we have to consider remote connect for SPP services
   bool processRequest(false);
   ::std::string convertedUuid;
   BTSSupportedServicesBit supportedServicesBit(convertUuid2SupportedService(convertedUuid, request->getUuid()));
   BTSProtocolId protocol(BTS_PROTO_HFP);

   switch(supportedServicesBit)
   {
      case BTS_SUPP_SRV_HFP:
         protocol = BTS_PROTO_HFP;
         processRequest = true;
         break;
      case BTS_SUPP_SRV_A2DP:
      case BTS_SUPP_SRV_AVRCP:
         protocol = BTS_PROTO_AVP;
         processRequest = true;
         break;
      case BTS_SUPP_SRV_SPP:
         protocol = BTS_PROTO_SPP;
         // convertedUuid was set during conversion
         processRequest = true;
         break;
      default:
         // for all other protocols we do not support server functionality (=> remote connect)
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   if(true == processRequest)
   {
      Ipc2Bts_AuthorizeService* ptrMsg = ptrNew_Ipc2Bts_AuthorizeService();

      if(NULL != ptrMsg)
      {
         ptrMsg->setDevice(request->getDevice());
         ptrMsg->setProtocol(protocol);
         ptrMsg->setUuid(convertedUuid);
         onRequest(ptrMsg, act);
      }
   }
}

void EvolutionGeniviAgentCallbackIf::onRequestAuthorizationRequest(const ::boost::shared_ptr< RequestAuthorizationRequest >& request, const act_t act)
{
   Ipc2Bts_RequestAuthorization* ptrMsg = ptrNew_Ipc2Bts_RequestAuthorization();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onRequestLinkkeyRequest(const ::boost::shared_ptr< RequestLinkkeyRequest >& request, const act_t act)
{
   Ipc2Bts_RequestLinkkey* ptrMsg = ptrNew_Ipc2Bts_RequestLinkkey();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      onRequest(ptrMsg, act);
   }
}

void EvolutionGeniviAgentCallbackIf::onDisplayPasskeyRequest(const ::boost::shared_ptr< DisplayPasskeyRequest >& request, const act_t act)
{
   Ipc2Bts_DisplayPasskey* ptrMsg = ptrNew_Ipc2Bts_DisplayPasskey();

   if(NULL != ptrMsg)
   {
      ptrMsg->setDevice(request->getDevice());
      BTSNumericValue numericValue;
      convertNumericValueBinary2String(numericValue, request->getPasskey());
      ptrMsg->setPasskey(numericValue);
      ptrMsg->setEntered(request->getEntered());
      onRequest(ptrMsg, act);
   }
}
// /org/bluez/AgentStub implementation --- end

} //genivi
} //btstackif
