/**
 * @file EvolutionGeniviAdapterCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief EvolutionGenivi DBUS Callback for Adapter.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Callback implementation for Adapter.
 */

#include "EvolutionGeniviAdapterCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "BtsUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviAdapterCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::org::bluez::Adapter1;

namespace btstackif {
namespace genivi {

EvolutionGeniviAdapterCallbackIf::EvolutionGeniviAdapterCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0)
{
}

EvolutionGeniviAdapterCallbackIf::~EvolutionGeniviAdapterCallbackIf()
{
}

// "ServiceAvailableIF" implementation --- start
void EvolutionGeniviAdapterCallbackIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* msg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != msg)
   {
      msg->setInterface(BTS_GEN_DBUS_SERVICE_ADAPTER);
      msg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}

void EvolutionGeniviAdapterCallbackIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);

   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* msg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != msg)
   {
      msg->setInterface(BTS_GEN_DBUS_SERVICE_ADAPTER);
      msg->setAvailabilityEvent(availabilityEvent);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// /org/bluez/Adapter1Proxy implementation --- start
void EvolutionGeniviAdapterCallbackIf::onStartDiscoveryErrorCb(const ::boost::shared_ptr< StartDiscoveryError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_StartDiscoveryResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onStartDiscoveryResponseCb(const ::boost::shared_ptr< StartDiscoveryResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onResponse(ptrNew_Ipc2Bts_StartDiscoveryResult(), response);
}

void EvolutionGeniviAdapterCallbackIf::onStopDiscoveryErrorCb(const ::boost::shared_ptr< StopDiscoveryError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_StopDiscoveryResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onStopDiscoveryResponseCb(const ::boost::shared_ptr< StopDiscoveryResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onResponse(ptrNew_Ipc2Bts_StopDiscoveryResult(), response);
}

void EvolutionGeniviAdapterCallbackIf::onRemoveDeviceErrorCb(const ::boost::shared_ptr< RemoveDeviceError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_RemoveDeviceResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onRemoveDeviceResponseCb(const ::boost::shared_ptr< RemoveDeviceResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onResponse(ptrNew_Ipc2Bts_RemoveDeviceResult(), response);
}

void EvolutionGeniviAdapterCallbackIf::onCreateDeviceErrorCb(const ::boost::shared_ptr< CreateDeviceError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_CreateDeviceResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onCreateDeviceResponseCb(const ::boost::shared_ptr< CreateDeviceResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_CreateDeviceResult* ptrResultMsg = ptrNew_Ipc2Bts_CreateDeviceResult();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDevice(response->getDevice());
      onResponse(ptrResultMsg, response->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onSendHCICmdErrorCb(const ::boost::shared_ptr< SendHCICmdError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_SendHCICmdResult(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onSendHCICmdResponseCb(const ::boost::shared_ptr< SendHCICmdResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onResponse(ptrNew_Ipc2Bts_SendHCICmdResult(), response->getAct());
}

void EvolutionGeniviAdapterCallbackIf::onAdapterInitializationErrorCb(const ::boost::shared_ptr< AdapterInitializationError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviAdapterCallbackIf::onAdapterInitializationSignalCb(const ::boost::shared_ptr< AdapterInitializationSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onSignal(ptrNew_Ipc2Bts_AdapterInitialization(), signal);
}

void EvolutionGeniviAdapterCallbackIf::onDeviceCreatedErrorCb(const ::boost::shared_ptr< DeviceCreatedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviAdapterCallbackIf::onDeviceCreatedSignalCb(const ::boost::shared_ptr< DeviceCreatedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_DeviceCreated* ptrResultMsg = ptrNew_Ipc2Bts_DeviceCreated();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDevice(signal->getDevice());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onDeviceRemovedErrorCb(const ::boost::shared_ptr< DeviceRemovedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviAdapterCallbackIf::onDeviceRemovedSignalCb(const ::boost::shared_ptr< DeviceRemovedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_DeviceRemoved* ptrResultMsg = ptrNew_Ipc2Bts_DeviceRemoved();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDevice(signal->getDevice());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onBtStackLogErrorCb(const ::boost::shared_ptr< BtStackLogError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviAdapterCallbackIf::onBtStackLogSignalCb(const ::boost::shared_ptr< BtStackLogSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_BtStackLog* ptrResultMsg = ptrNew_Ipc2Bts_BtStackLog();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setStatus((BTSLogStatus)signal->getStatus()); // cast OK because enum and Evolution value match
      ptrResultMsg->setLogOpcode(signal->getOpcode());
      ptrResultMsg->setLogLength((BTSLogLength)signal->getData().size());
      ptrResultMsg->setLogData(signal->getData());
      onSignal(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onHCIEventErrorCb(const ::boost::shared_ptr< HCIEventError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviAdapterCallbackIf::onHCIEventSignalCb(const ::boost::shared_ptr< HCIEventSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_HCIEvent* ptrResultMsg = ptrNew_Ipc2Bts_HCIEvent();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setLength((BTSHciLength)signal->getData().size());
      ptrResultMsg->setData(signal->getData());
      onUpdate(ptrResultMsg, signal->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onDeviceFoundErrorCb(const ::boost::shared_ptr< DeviceFoundError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // is never triggered
   FW_NORMAL_ASSERT_ALWAYS();
}

void EvolutionGeniviAdapterCallbackIf::onDeviceFoundSignalCb(const ::boost::shared_ptr< DeviceFoundSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   Ipc2Bts_DeviceFound* msg = ptrNew_Ipc2Bts_DeviceFound();

   if(0 != msg)
   {
      // first: parse dictionary
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      parser.setTraces(true);
      BTSDbusPropertyList ifProperties;
      ::std::vector<int> matchingInterfaces;
      matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_DEVICE); // keys matching device properties
      parser.parseProperties(ifProperties, matchingInterfaces, signal->getValuesMutable(), busType, busName, objPath, interfaceName);

      // second: find parsed entries
      BTSDiscoveredDeviceInfo& device = msg->getDeviceMutable();
      for(size_t i = 0; i < ifProperties.size(); i++)
      {
         switch((::ccdbusif::evolution::DeviceProperty)ifProperties[i].propEnum)
         {
            case ::ccdbusif::evolution::DEVICE_ADDRESS:
               convertBdAddress2InternalValue(device.address, ifProperties[i].propData.getString());
               break;
            case ::ccdbusif::evolution::DEVICE_ALIAS:
               device.name = ifProperties[i].propData.getString();
               break;
            case ::ccdbusif::evolution::DEVICE_RSSI:
               device.rssi = (BTSRssi)ifProperties[i].propData.getInt16();
               break;
            case ::ccdbusif::evolution::DEVICE_CLASS:
            {
               const uint32_t cod(ifProperties[i].propData.getUInt32());
               const unsigned char minor = (unsigned char)((cod >> 2) & 0x0000003F); // TODO: move cod conversion to FW
               const unsigned char major = (unsigned char)((cod >> 8) & 0x0000001F);
               const unsigned short int service = (unsigned short int)((cod >> 13) & 0x000007FF);

               device.majorServiceClass.setData(service);
               device.majorDeviceClass = (BTSMajorDeviceClass)major;
               device.minorDeviceClass = (BTSMinorDeviceClass)minor;
               break;
            }
            case ::ccdbusif::evolution::DEVICE_UUIDS:
            {
               BTSSupportedServices& serviceList = device.services;
               BTSSppCapabilityList& sppCapabilities = device.sppCapabilities;
               BTSSppCapability sppCapability;
               const ::std::vector< ::std::string >& uuids = ifProperties[i].propData.getStringArray();
               ::std::string convertedUuid;

               for(size_t i = 0; i < uuids.size(); i++)
               {
                  BTSSupportedServicesBit supportedServicesBit(convertUuid2SupportedService(convertedUuid, uuids[i]));

                  if(BTS_SUPP_SRV_LAST != supportedServicesBit)
                  {
                     serviceList.setBit(supportedServicesBit);

                     if(BTS_SUPP_SRV_SPP == supportedServicesBit)
                     {
                        sppCapability.uuidList.push_back(convertedUuid);
                        sppCapabilities.push_back(sppCapability);
                     }
                  }

                  convertedUuid.clear();
                  sppCapability.uuidList.clear();
               }
               break;
            }
            default:
               // ignore
               break;
         }
      }

      onUpdate(msg, signal->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onAddressErrorCb(const ::boost::shared_ptr< AddressError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterAddressUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onAddressUpdateCb(const ::boost::shared_ptr< AddressUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterAddressUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterAddressUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setBDAddress(update->getAddress());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onClassErrorCb(const ::boost::shared_ptr< ClassError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterClassUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onClassUpdateCb(const ::boost::shared_ptr< ClassUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterClassUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterClassUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setCod(update->getClass());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onAliasErrorCb(const ::boost::shared_ptr< AliasError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterAliasUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onAliasUpdateCb(const ::boost::shared_ptr< AliasUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterAliasUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterAliasUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setAlias(update->getAlias());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onPoweredErrorCb(const ::boost::shared_ptr< PoweredError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterPoweredUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onPoweredUpdateCb(const ::boost::shared_ptr< PoweredUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterPoweredUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterPoweredUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setPowered(update->getPowered());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onDiscoverableErrorCb(const ::boost::shared_ptr< DiscoverableError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterDiscoverableUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onDiscoverableUpdateCb(const ::boost::shared_ptr< DiscoverableUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterDiscoverableUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterDiscoverableUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDiscoverable(update->getDiscoverable());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onDiscoverableTimeoutErrorCb(const ::boost::shared_ptr< DiscoverableTimeoutError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterDiscoverabletimeoutUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onDiscoverableTimeoutUpdateCb(const ::boost::shared_ptr< DiscoverableTimeoutUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterDiscoverabletimeoutUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterDiscoverabletimeoutUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDiscoverableTimeout(update->getDiscoverableTimeout());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onPairableErrorCb(const ::boost::shared_ptr< PairableError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterPairableUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onPairableUpdateCb(const ::boost::shared_ptr< PairableUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterPairableUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterPairableUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setPairable(update->getPairable());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onPairableTimeoutErrorCb(const ::boost::shared_ptr< PairableTimeoutError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterPairabletimeoutUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onPairableTimeoutUpdateCb(const ::boost::shared_ptr< PairableTimeoutUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterPairabletimeoutUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterPairabletimeoutUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setPairableTimeout(update->getPairableTimeout());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onDiscoveringErrorCb(const ::boost::shared_ptr< DiscoveringError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterDiscoveringUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onDiscoveringUpdateCb(const ::boost::shared_ptr< DiscoveringUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterDiscoveringUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterDiscoveringUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setDiscovering(update->getDiscovering());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onUUIDsErrorCb(const ::boost::shared_ptr< UUIDsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterUuidsUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onUUIDsUpdateCb(const ::boost::shared_ptr< UUIDsUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterUuidsUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterUuidsUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setUuids(update->getUUIDs());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onWBSModeErrorCb(const ::boost::shared_ptr< WBSModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterWbsModeUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onWBSModeUpdateCb(const ::boost::shared_ptr< WBSModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterWbsModeUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterWbsModeUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setWbsMode(update->getWBSMode());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onConnectableErrorCb(const ::boost::shared_ptr< ConnectableError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterConnectableUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onConnectableUpdateCb(const ::boost::shared_ptr< ConnectableUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterConnectableUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterConnectableUpdate();

   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setConnectable(update->getConnectable());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onHCIModeErrorCb(const ::boost::shared_ptr< HCIModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterHcimodeUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onHCIModeUpdateCb(const ::boost::shared_ptr< HCIModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterHcimodeUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterHcimodeUpdate();
   if(0 != ptrResultMsg)
   {
      ptrResultMsg->setHciMode(update->getHCIMode());
      onUpdate(ptrResultMsg, update->getAct());
   }
}

void EvolutionGeniviAdapterCallbackIf::onVersionErrorCb(const ::boost::shared_ptr< VersionError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // TODO: implement
}

void EvolutionGeniviAdapterCallbackIf::onVersionUpdateCb(const ::boost::shared_ptr< VersionUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
#if 0
// TODO: move to Adapter
void EvolutionGeniviDbusDMCallbackIf::onBtApplGetVersionReqResponse(const ::boost::shared_ptr< DMProxy >& proxy, const ::boost::shared_ptr< BtApplGetVersionReqResponse >& response)
{
   (void)(proxy);
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   BTSDbusPropertyList ifProperties;
   ::std::vector<int> matchingInterfaces;

   matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_DM); // TODO: we need another way to bring the interface name to here
   parser.setTraces(true);
   parser.parseProperties(ifProperties, matchingInterfaces, response->getVersionMutable(), ::ccdbusif::BUS_TYPE_SYSTEM, proxy->getDBusBusName(), proxy->getDBusObjectPath(), proxy->getInterfaceName());

   Ipc2Bts_BtApplGetVersionReqResult* ptrResultMsg = ptrNew_Ipc2Bts_BtApplGetVersionReqResult();
   if(0 != ptrResultMsg)
   {
      BTSEvolutionVersion version;

      for(size_t i = 0; i < ifProperties.size(); i++)
      {
         switch(ifProperties[i].propEnum)
         {
            case ::ccdbusif::evolution::DM_CODE:
            {
               version.appCode = ifProperties[i].propData.getUInt16();
               break;
            }
            case ::ccdbusif::evolution::DM_VERSION:
            {
               version.appVersion = ifProperties[i].propData.getUInt16();
               break;
            }
            case ::ccdbusif::evolution::DM_CORE_VERSION:
            {
               version.coreVersion = ifProperties[i].propData.getUInt16();
               break;
            }
            case ::ccdbusif::evolution::DM_UNKNOWN:
            default:
            {
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            }
         }
      }

      ptrResultMsg->setStackVersion(version);
      onResponse(ptrResultMsg, response->getAct());
   }
#endif
}

void EvolutionGeniviAdapterCallbackIf::onTraceLevelErrorCb(const ::boost::shared_ptr< TraceLevelError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // TODO: implement
}

void EvolutionGeniviAdapterCallbackIf::onTraceLevelUpdateCb(const ::boost::shared_ptr< TraceLevelUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // TODO: implement
}

void EvolutionGeniviAdapterCallbackIf::onTraceCategoryErrorCb(const ::boost::shared_ptr< TraceCategoryError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // TODO: implement
}

void EvolutionGeniviAdapterCallbackIf::onTraceCategoryUpdateCb(const ::boost::shared_ptr< TraceCategoryUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(update);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   // TODO: implement
}

void EvolutionGeniviAdapterCallbackIf::onAvailableSPPServersErrorCb(const ::boost::shared_ptr< AvailableSPPServersError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
   onError(ptrNew_Ipc2Bts_AdapterAvailableSppServersUpdate(), error, convertErrorCode2InternalValue(error->getName(), error->getMessage()));
}

void EvolutionGeniviAdapterCallbackIf::onAvailableSPPServersUpdateCb(const ::boost::shared_ptr< AvailableSPPServersUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   Ipc2Bts_AdapterAvailableSppServersUpdate* ptrResultMsg = ptrNew_Ipc2Bts_AdapterAvailableSppServersUpdate();
   if(0 != ptrResultMsg)
   {
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      BTSUuidList& sppServers = ptrResultMsg->getAvailableSppServersMutable();
      const ::std::vector< ::std::string >& receivedSppServers = update->getAvailableSPPServers();
      ::std::string convertedUuid;
      sppServers.reserve(receivedSppServers.size());

      for(size_t i = 0; i < receivedSppServers.size(); i++)
      {
         if(true == parser.isValidUuid(receivedSppServers[i]))
         {
            // convert UUID
            convertUuid2InternalValue(convertedUuid, receivedSppServers[i]);
            // add to list
            sppServers.push_back(convertedUuid);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         convertedUuid.clear();
      }


      onUpdate(ptrResultMsg, update->getAct());
   }
}
// /org/bluez/Adapter1Proxy implementation --- end

} //genivi
} //btstackif
