/**
 * @file ConnectionOrgFreedesktopObjectManagerCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief OrgFreedesktop DBUS Callback for ObjectManager.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details OrgFreedesktop DBUS Callback implementation for ObjectManager.
 */

#include "ConnectionOrgFreedesktopObjectManagerCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"

#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "cc_dbus_if/IEvolutionGeniviDbusDeviceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusServiceSendRequestIf.h"
#include "cc_dbus_if/IEvolutionGeniviDbusSerialSendRequestIf.h"

#include "IEvolutionGeniviSerialCallbackIf.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnectionOrgFreedesktopObjectManagerCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::objmgr::org::freedesktop::DBus::ObjectManager;

namespace btstackif {
namespace genivi {

ConnectionOrgFreedesktopObjectManagerCallbackIf::ConnectionOrgFreedesktopObjectManagerCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0),
_deviceSendIf(0),
_serviceSendIf(0),
_serialSendIf(0),
_serialCb(0),
_simulateRequestConfirmation(false)
{
}

ConnectionOrgFreedesktopObjectManagerCallbackIf::~ConnectionOrgFreedesktopObjectManagerCallbackIf()
{
   _deviceSendIf = 0;
   _serviceSendIf = 0;
   _serialSendIf = 0;
   _serialCb = 0;
}

// "ServiceAvailableIF" implementation --- start
void ConnectionOrgFreedesktopObjectManagerCallbackIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      if(::ccdbusif::BUS_TYPE_SYSTEM == busType)
      {
         ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
      }
      else
      {
         ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION);
      }
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}

void ConnectionOrgFreedesktopObjectManagerCallbackIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      if(::ccdbusif::BUS_TYPE_SYSTEM == busType)
      {
         ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
      }
      else
      {
         ptrMsg->setInterface(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION);
      }
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
   }

   onSignal(ptrMsg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// /objmgr/org/freedesktop/DBus/ObjectManagerProxy implementation --- start
void ConnectionOrgFreedesktopObjectManagerCallbackIf::onGetManagedObjectsErrorCb(const ::boost::shared_ptr< GetManagedObjectsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void ConnectionOrgFreedesktopObjectManagerCallbackIf::onGetManagedObjectsResponseCb(const ::boost::shared_ptr< GetManagedObjectsResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(response);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void ConnectionOrgFreedesktopObjectManagerCallbackIf::onInterfacesAddedErrorCb(const ::boost::shared_ptr< InterfacesAddedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void ConnectionOrgFreedesktopObjectManagerCallbackIf::onInterfacesAddedSignalCb(const ::boost::shared_ptr< InterfacesAddedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   if(true == _simulateRequestConfirmation)
   {
      _simulateRequestConfirmation = false;
      Ipc2Bts_RequestConfirmation* ptrMsg = ptrNew_Ipc2Bts_RequestConfirmation();
      if(0 != ptrMsg)
      {
         ptrMsg->setDevice("/org/bluez/hci1/dev_00_09_DD_50_8F_C4");
         ptrMsg->setPasskey("123456");
         onSignal(ptrMsg, 1, true);
      }
   }

   // HINT: proxy object delivers following data:
   //    busName
   //    objPath
   //    interfaceName
   //    busType

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);
   // parser.parseInterfacesAdded(signal->getObject(), signal->getInterfacesMutable(), busType, busName, objPath, interfaceName);
   // parser.setTraces(false);

   ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces = signal->getInterfacesMutable();
   for(::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >::iterator it = inInterfaces.begin(); it != inInterfaces.end(); ++it)
   {
      if(parser.getInterface2String(::ccdbusif::evolution::IF_DEVICE) == it->first)
      {
         BTSDbusPropertyList ifProperties;
         ::std::vector<int> matchingInterfaces;
         matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_DEVICE);
         parser.parseProperties(ifProperties, matchingInterfaces, it->second, busType, busName, objPath, interfaceName);

         // send mapping
         BTSBDAddress address;
         for(size_t i = 0; i < ifProperties.size(); i++)
         {
            if(::ccdbusif::evolution::DEVICE_ADDRESS == (::ccdbusif::evolution::DeviceProperty)ifProperties[i].propEnum)
            {
               convertBdAddress2InternalValue(address, ifProperties[i].propData.getString());
               break;
            }
         }

         Ipc2Bts_AddDeviceObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_AddDeviceObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(address);
            ptrMsg->setObjPath(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct(), true);

         // create proxy
         if(0 != _deviceSendIf)
         {
            _deviceSendIf->deviceCreated(signal->getObject());
         }

         // send get all data
         Ipc2Bts_GetAllPropertiesConnection* ptrMsg1 = ptrNew_Ipc2Bts_GetAllPropertiesConnection();
         if(0 != ptrMsg1)
         {
            ptrMsg1->setInterface(::ccdbusif::evolution::IF_DEVICE);
            ptrMsg1->setProperties(ifProperties);
         }
         onSignal(ptrMsg1, signal->getAct());
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_SERVICE) == it->first)
      {
         BTSDbusPropertyList ifProperties;
         ::std::vector<int> matchingInterfaces;
         matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_SERVICE);
         parser.parseProperties(ifProperties, matchingInterfaces, it->second, busType, busName, objPath, interfaceName);

         // send mapping
         BTSObjectPath device;
         BTSProtocolId protocol = BTS_PROTO_LAST;
         ::std::string convertedUuid;

         for(size_t i = 0; i < ifProperties.size(); i++)
         {
            if(::ccdbusif::evolution::SERVICE_DEVICE == (::ccdbusif::evolution::ServiceProperty)ifProperties[i].propEnum)
            {
               device = ifProperties[i].propData.getString();
            }
            else if(::ccdbusif::evolution::SERVICE_REMOTEUUID == (::ccdbusif::evolution::ServiceProperty)ifProperties[i].propEnum)
            {
               // protocol = convertUuid2Protocol(parser.getUuid2Enum(ifProperties[i].propData.getString()));
               protocol = convertSupportedService2Protocol(convertUuid2SupportedService(convertedUuid, ifProperties[i].propData.getString()));
            }
         }

         Ipc2Bts_AddProtocolObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_AddProtocolObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setProtocol(protocol);
            ptrMsg->setDevice(device);
            ptrMsg->setUuid(convertedUuid);
            ptrMsg->setObjPath(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct(), true);

         // create proxy
         if(0 != _serviceSendIf)
         {
            _serviceSendIf->serviceCreated(signal->getObject());
         }

         // send get all data
         Ipc2Bts_GetAllPropertiesConnection* ptrMsg1 = ptrNew_Ipc2Bts_GetAllPropertiesConnection();
         if(0 != ptrMsg1)
         {
            ptrMsg1->setInterface(::ccdbusif::evolution::IF_SERVICE);
            ptrMsg1->setProperties(ifProperties);
         }
         onSignal(ptrMsg1, signal->getAct());
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_ADAPTER) == it->first)
      {
         BTSDbusPropertyList ifProperties;
         ::std::vector<int> matchingInterfaces;
         matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_ADAPTER);
         parser.parseProperties(ifProperties, matchingInterfaces, it->second, busType, busName, objPath, interfaceName);

         // send get all data
         Ipc2Bts_GetAllPropertiesConnection* ptrMsg1 = ptrNew_Ipc2Bts_GetAllPropertiesConnection();
         if(0 != ptrMsg1)
         {
            ptrMsg1->setInterface(::ccdbusif::evolution::IF_ADAPTER);
            ptrMsg1->setProperties(ifProperties);
         }
         onSignal(ptrMsg1, signal->getAct());
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_SESSION) == it->first)
      {
         BTSDbusPropertyList ifProperties;
         ::std::vector<int> matchingInterfaces;
         matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_SESSION);
         parser.parseProperties(ifProperties, matchingInterfaces, it->second, busType, busName, objPath, interfaceName);

         // send mapping
         BTSBDAddress address;
         BTSProtocolId protocol = BTS_PROTO_LAST;
         BTSMasInstanceId instance = 0;
         for(size_t i = 0; i < ifProperties.size(); i++)
         {
            if(::ccdbusif::evolution::OBEX_SESSION_DESTINATION == (::ccdbusif::evolution::ObexSessionProperty)ifProperties[i].propEnum)
            {
               convertBdAddress2InternalValue(address, ifProperties[i].propData.getString());
            }
            else if(::ccdbusif::evolution::OBEX_SESSION_TARGET == (::ccdbusif::evolution::ObexSessionProperty)ifProperties[i].propEnum)
            {
               protocol = convertUuid2Protocol(parser.getUuid2Enum(ifProperties[i].propData.getString()));
            }
            else if(::ccdbusif::evolution::OBEX_SESSION_INSTANCEID == (::ccdbusif::evolution::ObexSessionProperty)ifProperties[i].propEnum)
            {
               instance = (BTSMasInstanceId)ifProperties[i].propData.getInt32();
            }
         }

         Ipc2Bts_AddProtocolObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_AddProtocolObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(address);
            ptrMsg->setProtocol(protocol);
            ptrMsg->setInstanceId(instance);
            ptrMsg->setObjPath(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct(), true);

         // no proxy to be created

         // no data to be sent
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_SERIAL) == it->first)
      {
         // send mapping
         Ipc2Bts_AddSerialObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_AddSerialObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setObjPath(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct(), true);

         // create proxy
         if((0 != _serialSendIf) && (0 != _serialCb))
         {
            _serialSendIf->createProxyIf(_serialCb->getCallbackId(), true, signal->getObject());
         }

         // no data to be sent
      }
   }
}

void ConnectionOrgFreedesktopObjectManagerCallbackIf::onInterfacesRemovedErrorCb(const ::boost::shared_ptr< InterfacesRemovedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void ConnectionOrgFreedesktopObjectManagerCallbackIf::onInterfacesRemovedSignalCb(const ::boost::shared_ptr< InterfacesRemovedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   // HINT: proxy object delivers following data:
   //    objPath
   //    busName
   //    busType
   //    interfaceName

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   const ::std::vector< ::std::string >& removedInterfaces = signal->getInterfaces();

   parser.setTraces(true);
   // parser.parseInterfacesRemoved(signal->getObject(), removedInterfaces, busType, busName, objPath, interfaceName);
   // parser.setTraces(false);

   for(size_t i = 0; i < removedInterfaces.size(); ++i)
   {
      if(parser.getInterface2String(::ccdbusif::evolution::IF_DEVICE) == removedInterfaces[i])
      {
         // destroy proxy
         if(0 != _deviceSendIf)
         {
            _deviceSendIf->deviceRemoved(signal->getObject(), true);
         }

         // send removed mapping
         Ipc2Bts_DelDeviceObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_DelDeviceObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setObjPath(signal->getObject());
            ptrMsg->setObjectId(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct());
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_SERVICE) == removedInterfaces[i])
      {
         // destroy proxy
         if(0 != _serviceSendIf)
         {
            _serviceSendIf->serviceRemoved(signal->getObject(), true);
         }

         // send removed mapping
         Ipc2Bts_DelProtocolObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_DelProtocolObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setObjPath(signal->getObject());
            ptrMsg->setObjectId(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct());
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_SESSION) == removedInterfaces[i])
      {
         // no proxy to be destroyed

         // send removed mapping
         Ipc2Bts_DelProtocolObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_DelProtocolObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setObjPath(signal->getObject());
            ptrMsg->setObjectId(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct());
      }
      else if(parser.getInterface2String(::ccdbusif::evolution::IF_SERIAL) == removedInterfaces[i])
      {
         // destroy proxy
         if((0 != _serialSendIf) && (0 != _serialCb))
         {
            _serialSendIf->destroyProxyIf(_serialCb->getCallbackId(), true, signal->getObject());
         }

         // send removed mapping
         Ipc2Bts_DelSerialObjectPathMapping* ptrMsg = ptrNew_Ipc2Bts_DelSerialObjectPathMapping();
         if(0 != ptrMsg)
         {
            ptrMsg->setObjPath(signal->getObject());
            ptrMsg->setObjectId(signal->getObject());
         }
         onSignal(ptrMsg, signal->getAct());
      }
   }
}
// /objmgr/org/freedesktop/DBus/ObjectManagerProxy implementation --- end

} //genivi
} //btstackif
