/**
 * @file ConnectionControl_GEN.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Connection Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of connection control functionality.
 */

#include "ConnectionControl_GEN.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "EvolutionGeniviStackTypes.h"
#include "EvolutionGeniviUtils.h"
#include "ICheckWaitingQueue.h"
#include "IExtendedTimeoutHandler.h"
#include "BtsTimerPool.h"
#include "IConnectionSmExtension.h"
#include "FwUtils.h"

// generic SMs
#include "TestMode.h"
#include "Rssi.h"
#include "RestrictedPairingConnecting.h"
#include "DeviceManager.h"
#include "LinkKey.h"
#include "Inquiry.h"
#include "Pairing.h"
#include "ServiceSearch.h"
#include "Configuration.h"
#include "BasicConfiguration.h"
#include "LocalAdapterModes.h"
#include "SppPool.h"
#include "ProtocolManager.h"
#include "ConnectionPriority.h"
#include "LocalBluetoothName.h"
#include "DeviceIdServiceRecord.h"
#include "HfpDeviceCapabilities.h"
#include "BluetoothStackErrorHandling.h"
#include "Startup.h"
#include "SwitchBluetooth.h"

// Genivi SMs
#include "RssiGenivi.h"
#include "DeviceManagerGenivi.h"
#include "LinkKeyGenivi.h"
#include "InquiryGenivi.h"
#include "PairingGenivi.h"
#include "ServiceSearchGenivi.h"
#include "LocalAdapterModesGenivi.h"
#include "HfpAvpConnectDisconnectGenivi.h"
#include "ObexConnectDisconnectGenivi.h"
#include "PanConnectDisconnectGenivi.h"
#include "SppPoolGenivi.h"
#include "LocalBluetoothNameGenivi.h"
#include "ObjectPathManagerGenivi.h"
#include "BasicConfigurationGenivi.h"
#include "HfpDeviceCapabilitiesGenivi.h"
#include "StartupGenivi.h"
#include "SwitchBluetoothGenivi.h"

// Alps Evolution Genivi SMs
#include "IBasicConfigurationAlpsEvolutionExt.h"
#include "ISppConnectDisconnectAlpsEvolutionExt.h"
#include "IDeviceIdServiceRecordAlpsEvolutionExt.h"
#include "IBluetoothStackErrorHandlingAlpsEvolutionExt.h"

#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "cc_dbus_if/ConnManUtility.h"
#include "cc_dbus_if/EvolutionGeniviUtility.h"
#include "cc_dbus_if/EvolutionGeniviDbusTypes.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnectionControl_GEN.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

ConnectionControl::ConnectionControl() :
BasicControl(BTS_FB_CONNECTION),
_smExtension(0),
// generic SMs
_testMode(0),
_rssi(0),
_restrictedPairingConnecting(0),
_deviceManager(0),
_linkKey(0),
_inquiry(0),
_pairing(0),
_serviceSearch(0),
_configuration(0),
_basicConfiguration(0),
_localAdapterModes(0),
_sppPool(0),
_protocolManagerNew(0),
_connectionPriority(0),
_localBluetoothName(0),
_deviceIdServiceRecord(0),
_hfpDeviceCapabilities(0),
_bluetoothStackErrorHandling(0),
_startupNew(0),
_switchBluetooth(0),
// Genivi SMs
_geniviRssi(0),
_geniviDeviceManager(0),
_geniviLinkKey(0),
_geniviInquiry(0),
_geniviPairing(0),
_geniviServiceSearch(0),
_geniviLocalAdapterModes(0),
_geniviHfpAvpConnectDisconnect(0),
_geniviObexConnectDisconnect(0),
_geniviPanConnectDisconnect(0),
_geniviSppPool(0),
_geniviLocalBluetoothName(0),
_geniviBasicConfiguration(0),
_geniviHfpDeviceCapabilities(0),
_geniviStartup(0),
_geniviSwitchBluetooth(0),
// Alps Evolution Genivi SMs
_alpsEvolutionExtBasicConfiguration(0),
_alpsEvolutionExtSppConnectDisconnect(0),
_alpsEvolutionExtDeviceIdServiceRecord(0),
_alpsEvolutionExtBluetoothStackErrorHandling(0),
// test command
_testCommand()
{
   /*********** start here with specific class members ***************************/
   // create object manager
   (void)ObjectPathManager::getInstance();
   /*********** end here with specific class members *****************************/
}

ConnectionControl::ConnectionControl(const ConnectionControl& ref) :
BasicControl(BTS_FB_CONNECTION),
_smExtension(0),
// generic SMs
_testMode(0),
_rssi(0),
_restrictedPairingConnecting(0),
_deviceManager(0),
_linkKey(0),
_inquiry(0),
_pairing(0),
_serviceSearch(0),
_configuration(0),
_basicConfiguration(0),
_localAdapterModes(0),
_sppPool(0),
_protocolManagerNew(0),
_connectionPriority(0),
_localBluetoothName(0),
_deviceIdServiceRecord(0),
_hfpDeviceCapabilities(0),
_bluetoothStackErrorHandling(0),
_startupNew(0),
_switchBluetooth(0),
// Genivi SMs
_geniviRssi(0),
_geniviDeviceManager(0),
_geniviLinkKey(0),
_geniviInquiry(0),
_geniviPairing(0),
_geniviServiceSearch(0),
_geniviLocalAdapterModes(0),
_geniviHfpAvpConnectDisconnect(0),
_geniviObexConnectDisconnect(0),
_geniviPanConnectDisconnect(0),
_geniviSppPool(0),
_geniviLocalBluetoothName(0),
_geniviBasicConfiguration(0),
_geniviHfpDeviceCapabilities(0),
_geniviStartup(0),
_geniviSwitchBluetooth(0),
// Alps Evolution Genivi SMs
_alpsEvolutionExtBasicConfiguration(0),
_alpsEvolutionExtSppConnectDisconnect(0),
_alpsEvolutionExtDeviceIdServiceRecord(0),
_alpsEvolutionExtBluetoothStackErrorHandling(0),
// test command
_testCommand()
{
   // ignore given parameter
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

ConnectionControl& ConnectionControl::operator=(const ConnectionControl& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   BasicControl::operator=(ref);

   // ignore given parameter

   return *this;
}

ConnectionControl::~ConnectionControl()
{
   /*********** start here with specific class members ***************************/
   // reset object path manager
   ObjectPathManager::getInstance().reset();

   destroyGenericSMs();
   destroyGeniviSMs();

   switch(_stackInterface)
   {
      case BTS_IF_ALPS_EVOLUTION_GENIVI:
         destroyAlpsEvolutionGeniviSMs();
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   destroyExtensions();

   _smExtension = 0;

   _testMode = 0;
   _rssi = 0;
   _restrictedPairingConnecting = 0;
   _deviceManager = 0;
   _linkKey = 0;
   _inquiry = 0;
   _pairing = 0;
   _serviceSearch = 0;
   _configuration = 0;
   _basicConfiguration = 0;
   _localAdapterModes = 0;
   _sppPool = 0;
   _protocolManagerNew = 0;
   _connectionPriority = 0;
   _localBluetoothName = 0;
   _deviceIdServiceRecord = 0;
   _hfpDeviceCapabilities = 0;
   _bluetoothStackErrorHandling = 0;
   _startupNew = 0;
   _switchBluetooth = 0;

   _geniviRssi = 0;
   _geniviDeviceManager = 0;
   _geniviLinkKey = 0;
   _geniviInquiry = 0;
   _geniviPairing = 0;
   _geniviServiceSearch = 0;
   _geniviLocalAdapterModes = 0;
   _geniviHfpAvpConnectDisconnect = 0;
   _geniviObexConnectDisconnect = 0;
   _geniviPanConnectDisconnect = 0;
   _geniviSppPool = 0;
   _geniviLocalBluetoothName = 0;
   _geniviBasicConfiguration = 0;
   _geniviHfpDeviceCapabilities = 0;
   _geniviStartup = 0;
   _geniviSwitchBluetooth = 0;

   _alpsEvolutionExtBasicConfiguration = 0;
   _alpsEvolutionExtSppConnectDisconnect = 0;
   _alpsEvolutionExtDeviceIdServiceRecord = 0;
   _alpsEvolutionExtBluetoothStackErrorHandling = 0;
   /*********** end here with specific class members *****************************/
}

void ConnectionControl::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(0 == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   // do a general pre-check; e.g. check for DBUS service availability or ongoing startup
   bool rejectRequest = false;
   BTSCommonEnumClass resultCode = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   BTSCommonEnumClass statusCode = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   bool skipOpCodeCheck = false;

   if(false == doApp2BtsMsgPrecheck(rejectRequest, resultCode, statusCode, skipOpCodeCheck, ptrMessage))
   {
      // message was pushed to startup or switch bt queue
      return;
   }

   // check for rejecting request
   if(true == rejectRequest)
   {
      ::std::vector<App2Bts_BaseMessage*> msgList;
      msgList.push_back(ptrMessage);
      sendDirectAnswerForApp2BtsMessages(msgList, resultCode, statusCode);
      delete ptrMessage;
      return;
   }

   bool similarOpCodeInWorkingQueue = false;

   // check for skipping similar opcode check
   if(false == skipOpCodeCheck)
   {
      // get similar opcodes
      ::std::vector<BTSApp2BtsMessageMasterCompareItem> itemList;
      getSimilarOpCodes(itemList, ptrMessage);

      // check if opcode is in working queue
      similarOpCodeInWorkingQueue = isSimilarOpCodeInWorkingQueue(itemList);
   }

   // check if opcode is in working queue
   if(true == similarOpCodeInWorkingQueue)
   {
      // push to waiting queue
      pushApp2BtsMsgToWaitingQueue(ptrMessage, false); // single worker thread
      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X pushed to waiting queue", ptrMessage->getTraceOpCode()));
   }
   else
   {
      //===================================================================================================================
      // debug section start
      if(true == _testCommand.getTriggerSimulateStackResetDuringApp2BtsProcessing())
      {
         _testCommand.setTriggerSimulateStackResetDuringApp2BtsProcessing(false);

         (void)_testCommand.handleTestCommand("BTS_TRIGGER_SIMULATE_STACK_RESET", (const unsigned int)0);
      }
      // debug section end
      //===================================================================================================================

      BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
      ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
      sendBts2IpcMsgList.reserve(10);
      ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
      sendBts2AppMsgList.reserve(10);
      bool deleteApp2BtsMsg = false;
      BTSApp2BtsMessageCompareItem compareItem;
      ptrMessage->getCompareItem(compareItem);

      ETG_TRACE_USR3((" pushApp2BtsMessage(): App2Bts 0x%04X to be processed", ptrMessage->getTraceOpCode()));

      // handle message depending on opcode
      // - set marker to delete message in sub-handler function if necessary
      // - push message to working queue in sub-handler function if necessary
      // - create direct answer message (Bts2App) in sub-handler function if necessary
      // - create Bts2Ipc message in sub-handler function if necessary
      // - handle any error in sub-handler function because there is the best place to handle
      switch(opcode)
      {
         case App2BtsOC_StartInternalCommunication:
            handleApp2BtsStartInternalCommunication(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StartInternalCommunication*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_TriggerFbConnectionInitialized:
            handleApp2BtsTriggerFbConnectionInitialized(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_TriggerFbConnectionInitialized*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SwitchBluetoothOnOff:
            handleApp2BtsSwitchBluetoothOnOff(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SwitchBluetoothOnOff*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_GetBluetoothOnOffStatus:
            handleApp2BtsGetBluetoothOnOffStatus(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_GetBluetoothOnOffStatus*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SwitchBluetoothHardwareOnOff:
            handleApp2BtsSwitchBluetoothHardwareOnOff(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SwitchBluetoothHardwareOnOff*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SetLocalBtName:
            handleApp2BtsSetLocalBtName(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetLocalBtName*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_GetLocalBtName:
            handleApp2BtsGetLocalBtName(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_GetLocalBtName*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SetDeviceIdServiceRecord:
            handleApp2BtsSetDeviceIdServiceRecord(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetDeviceIdServiceRecord*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SetDeviceForRestrictedPairingConnecting:
            handleApp2BtsSetDeviceForRestrictedPairingConnecting(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetDeviceForRestrictedPairingConnecting*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SetLocalAdapterModes:
            handleApp2BtsSetLocalAdapterModes(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetLocalAdapterModes*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_GetLocalAdapterModes:
            handleApp2BtsGetLocalAdapterModes(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_GetLocalAdapterModes*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_StartDiscovery:
            handleApp2BtsStartDiscovery(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StartDiscovery*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_StopDiscovery:
            handleApp2BtsStopDiscovery(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StopDiscovery*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_GetDiscoveringStatus:
            handleApp2BtsGetDiscoveringStatus(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_GetDiscoveringStatus*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_StartPairing:
            handleApp2BtsStartPairing(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StartPairing*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_CancelPairing:
            handleApp2BtsCancelPairing(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_CancelPairing*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SetPinCode:
            handleApp2BtsSetPinCode(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetPinCode*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_ConfirmSecureSimplePairing:
            handleApp2BtsConfirmSecureSimplePairing(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_ConfirmSecureSimplePairing*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_SetLinkKey:
            handleApp2BtsSetLinkKey(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_SetLinkKey*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_ConnectProtocol:
            handleApp2BtsConnectProtocol(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_ConnectProtocol*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_AcceptRemoteProtocolConnect:
            handleApp2BtsAcceptRemoteProtocolConnect(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_AcceptRemoteProtocolConnect*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_RejectRemoteProtocolConnect:
            handleApp2BtsRejectRemoteProtocolConnect(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_RejectRemoteProtocolConnect*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_DisconnectProtocol:
            handleApp2BtsDisconnectProtocol(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_DisconnectProtocol*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_DisconnectDevice:
            handleApp2BtsDisconnectDevice(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_DisconnectDevice*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_GetProtocolConnectionStatus:
            handleApp2BtsGetProtocolConnectionStatus(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_GetProtocolConnectionStatus*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_GetDeviceConnectionStatus:
            handleApp2BtsGetDeviceConnectionStatus(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_GetDeviceConnectionStatus*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_StartRemoteServiceSearch:
            handleApp2BtsStartRemoteServiceSearch(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StartRemoteServiceSearch*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_CancelRemoteServiceSearch:
            handleApp2BtsCancelRemoteServiceSearch(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_CancelRemoteServiceSearch*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_QueryRemoteName:
            handleApp2BtsQueryRemoteName(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_QueryRemoteName*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_RequestLinkQuality:
            handleApp2BtsRequestLinkQuality(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_RequestLinkQuality*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_StartTestMode:
            handleApp2BtsStartTestMode(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StartTestMode*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_StopTestMode:
            handleApp2BtsStopTestMode(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_StopTestMode*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         case App2BtsOC_RequestTestModeLinkQuality:
            handleApp2BtsRequestTestModeLinkQuality(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, static_cast<App2Bts_RequestTestModeLinkQuality*>(ptrMessage)); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
            break;
         // all other
         default:
            // check extensions
            if(false == handleApp2BtsExtensionMessage(sendBts2IpcMsgList, sendBts2AppMsgList, deleteApp2BtsMsg, ptrMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
               deleteApp2BtsMsg = true;
            }
            break;
      }

      if(0 < sendBts2IpcMsgList.size())
      {
         _bts2IpcMessageWasSent = true; // this works only if single thread handling
      }

      sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);

      sendBts2AppMessageList(sendBts2AppMsgList);

      if(true == deleteApp2BtsMsg)
      {
         delete ptrMessage;
      }
   }
}

void ConnectionControl::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   if(0 == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(_functionBlock != ptrMessage->getBtsDestinationFunctionBlock())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return;
   }

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   BTSGenIpc2BtsOpcode opcode = (BTSGenIpc2BtsOpcode)ptrMessage->getOpCode();
   ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
   sendBts2IpcMsgList.reserve(10);
   ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);
   BTSHandleIpc2BtsMessageItem ipc2BtsMessageItem;

   // find related APP2BTS message
   // for req -> cfm sequence there must be a message in working queue
   // for ind sequence there can be a message in working queue
   ipc2BtsMessageItem.message = findApp2BtsWorkingMessageWrapper(ptrMessage->getApp2BtsCompareItem());

   // special handling is needed for e.g. connect/disconnect
   ipc2BtsMessageItem.item.opCode = App2BtsOC_None;

   // handle message depending on opcode
   // - do not delete Ipc2Bts message in sub-handler function because this done at the end of this function
   // - set marker to delete related App2Bts message in sub-handler function if necessary
   // - create Bts2App message in sub-handler function if necessary
   // - create Bts2Ipc message in sub-handler function if necessary
   // - handle any error in sub-handler function because there is the best place to handle
   switch(opcode)
   {
      case Ipc2BtsOC_ServiceAvailabilityConnection:
         handleIpc2BtsServiceAvailabilityConnection(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ServiceAvailabilityConnection& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetAllPropertiesConnection:
         handleIpc2BtsGetAllPropertiesConnection(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetAllPropertiesConnection& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AddDeviceObjectPathMapping:
         handleIpc2BtsAddDeviceObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AddDeviceObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DelDeviceObjectPathMapping:
         handleIpc2BtsDelDeviceObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DelDeviceObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AddProtocolObjectPathMapping:
         handleIpc2BtsAddProtocolObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AddProtocolObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DelProtocolObjectPathMapping:
         handleIpc2BtsDelProtocolObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DelProtocolObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AddSerialObjectPathMapping:
         handleIpc2BtsAddSerialObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AddSerialObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DelSerialObjectPathMapping:
         handleIpc2BtsDelSerialObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DelSerialObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_Release:
         handleIpc2BtsRelease(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_Release& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RequestPinCode:
         handleIpc2BtsRequestPinCode(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RequestPinCode& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DisplayPinCode:
         handleIpc2BtsDisplayPinCode(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DisplayPinCode& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RequestPasskey:
         handleIpc2BtsRequestPasskey(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RequestPasskey& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RequestConfirmation:
         handleIpc2BtsRequestConfirmation(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RequestConfirmation& >(*ptrMessage));
         break;
      case Ipc2BtsOC_Cancel:
         handleIpc2BtsCancel(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_Cancel& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CancelRequest:
         handleIpc2BtsCancelRequest(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CancelRequest& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AuthorizeService:
         handleIpc2BtsAuthorizeService(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AuthorizeService& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RequestAuthorization:
         handleIpc2BtsRequestAuthorization(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RequestAuthorization& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RequestLinkkey:
         handleIpc2BtsRequestLinkkey(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RequestLinkkey& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DisplayPasskey:
         handleIpc2BtsDisplayPasskey(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DisplayPasskey& >(*ptrMessage));
         break;
      case Ipc2BtsOC_StartDiscoveryResult:
         handleIpc2BtsStartDiscoveryResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_StartDiscoveryResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_StopDiscoveryResult:
         handleIpc2BtsStopDiscoveryResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_StopDiscoveryResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RemoveDeviceResult:
         handleIpc2BtsRemoveDeviceResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RemoveDeviceResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CreateDeviceResult:
         handleIpc2BtsCreateDeviceResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CreateDeviceResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_SendHCICmdResult:
         handleIpc2BtsSendHCICmdResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_SendHCICmdResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterInitialization:
         handleIpc2BtsAdapterInitialization(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterInitialization& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceCreated:
         handleIpc2BtsDeviceCreated(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceCreated& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceRemoved:
         handleIpc2BtsDeviceRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceRemoved& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtStackLog:
         handleIpc2BtsBtStackLog(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtStackLog& >(*ptrMessage));
         break;
      case Ipc2BtsOC_HCIEvent:
         handleIpc2BtsHCIEvent(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_HCIEvent& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceFound:
         handleIpc2BtsDeviceFound(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceFound& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterAddressUpdate:
         handleIpc2BtsAdapterAddressUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterAddressUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterClassUpdate:
         handleIpc2BtsAdapterClassUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterClassUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterAliasUpdate:
         handleIpc2BtsAdapterAliasUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterAliasUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterPoweredUpdate:
         handleIpc2BtsAdapterPoweredUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterPoweredUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterDiscoverableUpdate:
         handleIpc2BtsAdapterDiscoverableUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterDiscoverableUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterDiscoverabletimeoutUpdate:
         handleIpc2BtsAdapterDiscoverabletimeoutUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterDiscoverabletimeoutUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterPairableUpdate:
         handleIpc2BtsAdapterPairableUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterPairableUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterPairabletimeoutUpdate:
         handleIpc2BtsAdapterPairabletimeoutUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterPairabletimeoutUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterDiscoveringUpdate:
         handleIpc2BtsAdapterDiscoveringUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterDiscoveringUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterUuidsUpdate:
         handleIpc2BtsAdapterUuidsUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterUuidsUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterWbsModeUpdate:
         handleIpc2BtsAdapterWbsModeUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterWbsModeUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterConnectableUpdate:
         handleIpc2BtsAdapterConnectableUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterConnectableUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterHcimodeUpdate:
         handleIpc2BtsAdapterHcimodeUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterHcimodeUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterVersionUpdate:
         handleIpc2BtsAdapterVersionUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterVersionUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterTraceLevelUpdate:
         handleIpc2BtsAdapterTraceLevelUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterTraceLevelUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterTraceCategoryUpdate:
         handleIpc2BtsAdapterTraceCategoryUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterTraceCategoryUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AdapterAvailableSppServersUpdate:
         handleIpc2BtsAdapterAvailableSppServersUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AdapterAvailableSppServersUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RegisterAgentResult:
         handleIpc2BtsRegisterAgentResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RegisterAgentResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_UnregisterAgentResult:
         handleIpc2BtsUnregisterAgentResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_UnregisterAgentResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RequestDefaultAgentResult:
         handleIpc2BtsRequestDefaultAgentResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RequestDefaultAgentResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceConnectResult:
         handleIpc2BtsDeviceConnectResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceConnectResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceDisconnectResult:
         handleIpc2BtsDeviceDisconnectResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceDisconnectResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_PairResult:
         handleIpc2BtsPairResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_PairResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CancelPairingResult:
         handleIpc2BtsCancelPairingResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CancelPairingResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DiscoverServicesResult:
         handleIpc2BtsDiscoverServicesResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DiscoverServicesResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CancelDiscoveryResult:
         handleIpc2BtsCancelDiscoveryResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CancelDiscoveryResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ReportLinkkey:
         handleIpc2BtsReportLinkkey(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ReportLinkkey& >(*ptrMessage));
         break;
      case Ipc2BtsOC_IncomingPairingComp:
         handleIpc2BtsIncomingPairingComp(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_IncomingPairingComp& >(*ptrMessage));
         break;
      case Ipc2BtsOC_SDPProcessReport:
         handleIpc2BtsSDPProcessReport(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_SDPProcessReport& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceAddressUpdate:
         handleIpc2BtsDeviceAddressUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceAddressUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceClassUpdate:
         handleIpc2BtsDeviceClassUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceClassUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceIconUpdate:
         handleIpc2BtsDeviceIconUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceIconUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceUuidsUpdate:
         handleIpc2BtsDeviceUuidsUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceUuidsUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DevicePairedUpdate:
         handleIpc2BtsDevicePairedUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DevicePairedUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceConnectedUpdate:
         handleIpc2BtsDeviceConnectedUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceConnectedUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceTrustedUpdate:
         handleIpc2BtsDeviceTrustedUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceTrustedUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceBlockedUpdate:
         handleIpc2BtsDeviceBlockedUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceBlockedUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceAliasUpdate:
         handleIpc2BtsDeviceAliasUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceAliasUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceAdapterUpdate:
         handleIpc2BtsDeviceAdapterUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceAdapterUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceModaliasUpdate:
         handleIpc2BtsDeviceModaliasUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceModaliasUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceRssiUpdate:
         handleIpc2BtsDeviceRssiUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceRssiUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceVendorIdUpdate:
         handleIpc2BtsDeviceVendorIdUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceVendorIdUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceVendorIdSourceUpdate:
         handleIpc2BtsDeviceVendorIdSourceUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceVendorIdSourceUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceProductIdUpdate:
         handleIpc2BtsDeviceProductIdUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceProductIdUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceMapInfoUpdate:
         handleIpc2BtsDeviceMapInfoUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceMapInfoUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceConnectedServiceUpdate:
         handleIpc2BtsDeviceConnectedServiceUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceConnectedServiceUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceVersionUpdate:
         handleIpc2BtsDeviceVersionUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceVersionUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DevicePbapInfoUpdate:
         handleIpc2BtsDevicePbapInfoUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DevicePbapInfoUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceModeUpdate:
         handleIpc2BtsDeviceModeUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceModeUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceRoleUpdate:
         handleIpc2BtsDeviceRoleUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceRoleUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DeviceEnableAvpPauseUpdate:
         handleIpc2BtsDeviceEnableAvpPauseUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DeviceEnableAvpPauseUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceConnectReqResult:
         handleIpc2BtsBtApplDeviceConnectReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceConnectReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceDisconnectReqResult:
         handleIpc2BtsBtApplDeviceDisconnectReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceDisconnectReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplCancelConnectReqResult:
         handleIpc2BtsBtApplCancelConnectReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplCancelConnectReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplGetHwVersionReqResult:
         handleIpc2BtsBtApplGetHwVersionReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplGetHwVersionReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplSetConfigurationReqResult:
         handleIpc2BtsBtApplSetConfigurationReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplSetConfigurationReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplReadConfigurationReqResult:
         handleIpc2BtsBtApplReadConfigurationReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplReadConfigurationReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceConnectCfm:
         handleIpc2BtsBtApplDeviceConnectCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceConnectCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceConnectInd:
         handleIpc2BtsBtApplDeviceConnectInd(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceConnectInd& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceDisconnectCfm:
         handleIpc2BtsBtApplDeviceDisconnectCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceDisconnectCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceDisconnectInd:
         handleIpc2BtsBtApplDeviceDisconnectInd(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceDisconnectInd& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceDisconnectCompInd:
         handleIpc2BtsBtApplDeviceDisconnectCompInd(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceDisconnectCompInd& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDeviceConnectCompInd:
         handleIpc2BtsBtApplDeviceConnectCompInd(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDeviceConnectCompInd& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplCancelConnectCfm:
         handleIpc2BtsBtApplCancelConnectCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplCancelConnectCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplStatusNotificationInd:
         handleIpc2BtsBtApplStatusNotificationInd(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplStatusNotificationInd& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplGetHwVersionCfm:
         handleIpc2BtsBtApplGetHwVersionCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplGetHwVersionCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplSetConfigurationCfm:
         handleIpc2BtsBtApplSetConfigurationCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplSetConfigurationCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplReadConfigurationCfm:
         handleIpc2BtsBtApplReadConfigurationCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplReadConfigurationCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnectResult:
         handleIpc2BtsConnectResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnectResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DisconnectResult:
         handleIpc2BtsDisconnectResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DisconnectResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ServiceRemoteUuidUpdate:
         handleIpc2BtsServiceRemoteUuidUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ServiceRemoteUuidUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ServiceStateUpdate:
         handleIpc2BtsServiceStateUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ServiceStateUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ServiceDeviceUpdate:
         handleIpc2BtsServiceDeviceUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ServiceDeviceUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ServiceVersionUpdate:
         handleIpc2BtsServiceVersionUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ServiceVersionUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DebugTraceConfigResult:
         handleIpc2BtsDebugTraceConfigResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DebugTraceConfigResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ProtocolTraceConfigResult:
         handleIpc2BtsProtocolTraceConfigResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ProtocolTraceConfigResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ReportDebugTrace:
         handleIpc2BtsReportDebugTrace(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ReportDebugTrace& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ReportProtocolTrace:
         handleIpc2BtsReportProtocolTrace(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ReportProtocolTrace& >(*ptrMessage));
         break;
      case Ipc2BtsOC_FatalError:
         handleIpc2BtsFatalError(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_FatalError& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GeneralError:
         handleIpc2BtsGeneralError(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GeneralError& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ReportGeniviDebugTrace:
         handleIpc2BtsReportGeniviDebugTrace(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ReportGeniviDebugTrace& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GeniviFatalError:
         handleIpc2BtsGeniviFatalError(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GeniviFatalError& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDidSetReqResult:
         handleIpc2BtsBtApplDidSetReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDidSetReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplDidSetCfm:
         handleIpc2BtsBtApplDidSetCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplDidSetCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DunDeviceFilePathUpdate:
         handleIpc2BtsDunDeviceFilePathUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DunDeviceFilePathUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplSppSetUuidReqResult:
         handleIpc2BtsBtApplSppSetUuidReqResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplSppSetUuidReqResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplSppSetUuidCfm:
         handleIpc2BtsBtApplSppSetUuidCfm(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplSppSetUuidCfm& >(*ptrMessage));
         break;
      case Ipc2BtsOC_BtApplSppVirtualDeviceInd:
         handleIpc2BtsBtApplSppVirtualDeviceInd(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_BtApplSppVirtualDeviceInd& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CreateSessionResult:
         handleIpc2BtsCreateSessionResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CreateSessionResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_RemoveSessionResult:
         handleIpc2BtsRemoveSessionResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_RemoveSessionResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CancelSessionResult:
         handleIpc2BtsCancelSessionResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CancelSessionResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_SessionCreated:
         handleIpc2BtsSessionCreated(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_SessionCreated& >(*ptrMessage));
         break;
      case Ipc2BtsOC_SessionRemoved:
         handleIpc2BtsSessionRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_SessionRemoved& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AddTechnologyObjectPath:
         handleIpc2BtsAddTechnologyObjectPath(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AddTechnologyObjectPath& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DelTechnologyObjectPath:
         handleIpc2BtsDelTechnologyObjectPath(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DelTechnologyObjectPath& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AddNetworkObjectPathMapping:
         handleIpc2BtsAddNetworkObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AddNetworkObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DelNetworkObjectPathMapping:
         handleIpc2BtsDelNetworkObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DelNetworkObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetConnManManagerPropertiesResult:
         handleIpc2BtsGetConnManManagerPropertiesResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetConnManManagerPropertiesResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetConnManTechnologiesResult:
         handleIpc2BtsGetConnManTechnologiesResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetConnManTechnologiesResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetConnManServicesResult:
         handleIpc2BtsGetConnManServicesResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetConnManServicesResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManTechnologyAdded:
         handleIpc2BtsConnManTechnologyAdded(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManTechnologyAdded& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManTechnologyRemoved:
         handleIpc2BtsConnManTechnologyRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManTechnologyRemoved& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManServicesChanged:
         handleIpc2BtsConnManServicesChanged(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManServicesChanged& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManServicesRemoved:
         handleIpc2BtsConnManServicesRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManServicesRemoved& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetConnManServicePropertiesResult:
         handleIpc2BtsGetConnManServicePropertiesResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetConnManServicePropertiesResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManServiceConnectResult:
         handleIpc2BtsConnManServiceConnectResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManServiceConnectResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManServiceDisconnectResult:
         handleIpc2BtsConnManServiceDisconnectResult(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManServiceDisconnectResult& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManServiceStateUpdate:
         handleIpc2BtsConnManServiceStateUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManServiceStateUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ConnManServiceInterfaceUpdate:
         handleIpc2BtsConnManServiceInterfaceUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ConnManServiceInterfaceUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_SerialDeviceFilePath:
         handleIpc2BtsSerialDeviceFilePath(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_SerialDeviceFilePath& >(*ptrMessage));
         break;
      case Ipc2BtsOC_CharacterDeviceAddedRemoved:
         handleIpc2BtsCharacterDeviceAddedRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_CharacterDeviceAddedRemoved& >(*ptrMessage));
         break;
      case Ipc2BtsOC_AddHfpInterfaceObjectPathMapping:
         handleIpc2BtsAddHfpInterfaceObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_AddHfpInterfaceObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_DelHfpInterfaceObjectPathMapping:
         handleIpc2BtsDelHfpInterfaceObjectPathMapping(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_DelHfpInterfaceObjectPathMapping& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetHandsfreeProperties:
         handleIpc2BtsGetHandsfreeProperties(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetHandsfreeProperties& >(*ptrMessage));
         break;
      case Ipc2BtsOC_HandsfreeFeaturesUpdate:
         handleIpc2BtsHandsfreeFeaturesUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_HandsfreeFeaturesUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_HandsfreeInbandRingingUpdate:
         handleIpc2BtsHandsfreeInbandRingingUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_HandsfreeInbandRingingUpdate& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetModems:
         handleIpc2BtsGetModems(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetModems& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ModemAdded:
         handleIpc2BtsModemAdded(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ModemAdded& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ModemRemoved:
         handleIpc2BtsModemRemoved(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ModemRemoved& >(*ptrMessage));
         break;
      case Ipc2BtsOC_GetModemProperties:
         handleIpc2BtsGetModemProperties(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_GetModemProperties& >(*ptrMessage));
         break;
      case Ipc2BtsOC_ModemInterfacesUpdate:
         handleIpc2BtsModemInterfacesUpdate(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, static_cast< Ipc2Bts_ModemInterfacesUpdate& >(*ptrMessage));
         break;
      // all other
      default:
         // check extensions
         FW_NORMAL_ASSERT(true == handleIpc2BtsExtensionMessage(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, ptrMessage));
         break;
   }

   if(0 < sendBts2IpcMsgList.size())
   {
      if(0 != ipc2BtsMessageItem.message)
      {
         BTSApp2BtsMessageCompareItem compareItem;
         ipc2BtsMessageItem.message->getCompareItem(compareItem);
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
      else
      {
         ::std::vector<Bts2Ipc_BaseMessage*> respSendBts2IpcMsgList;
         respSendBts2IpcMsgList.reserve(sendBts2IpcMsgList.size());

         for(size_t i = 0; i < sendBts2IpcMsgList.size(); i++)
         {
            if(0 != sendBts2IpcMsgList[i])
            {
               if(true == sendBts2IpcMsgList[i]->getResponseMessageFlag())
               {
                  // response message: it could be possible that there is no related application message
                  respSendBts2IpcMsgList.push_back(sendBts2IpcMsgList[i]);
               }
               else
               {
                  // should never happen else you have programmed something wrong
                  // #error_indication
                  FW_NORMAL_ASSERT_ALWAYS();

                  delete sendBts2IpcMsgList[i];
               }
            }
         }

         sendBts2IpcMsgList.clear();

         if(0 < respSendBts2IpcMsgList.size())
         {
            BTSApp2BtsMessageCompareItem compareItem;
            sendBts2IpcMessageList(respSendBts2IpcMsgList, compareItem);
         }
      }
   }

   sendBts2AppMessageList(sendBts2AppMsgList);

   // delete received message
   delete ptrMessage;

   // remove related APP2BTS message and check waiting queue
   if((true == ipc2BtsMessageItem.deleteMessage) && (0 != ipc2BtsMessageItem.message))
   {
#if 0
      if(App2BtsOC_DisconnectDevice == ipc2BtsMessageItem.message->getOpCode())
      {
         ETG_TRACE_USR1((" ### delete App2BtsOC_DisconnectDevice: ipc2BtsMessageItem.item.opCode=%d", ipc2BtsMessageItem.item.opCode));
      }
#endif

      removeApp2BtsWorkingMessage(ipc2BtsMessageItem.message);

      if(App2BtsOC_None == ipc2BtsMessageItem.item.opCode)
      {
         checkWaitingQueue(ipc2BtsMessageItem.message);
      }

      delete ipc2BtsMessageItem.message;
   }

   // check waiting queue (special handling)
   if((App2BtsOC_None != ipc2BtsMessageItem.item.opCode) && (App2BtsOC_Ignore != ipc2BtsMessageItem.item.opCode))
   {
      if(0 != ipc2BtsMessageItem.checkIf)
      {
         ipc2BtsMessageItem.checkIf->checkWaitingQueue(ipc2BtsMessageItem.item);
      }
      else
      {
         checkWaitingQueueExtended(ipc2BtsMessageItem.item);
      }
   }
}

void ConnectionControl::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   ETG_TRACE_USR2((" setStackConfiguration()"));

   FW_NORMAL_ASSERT(component == _component); // already set
   FW_NORMAL_ASSERT(stackInterface == _stackInterface); // already set
   FW_NORMAL_ASSERT(subComponent == _functionBlock); // already set
   FW_NORMAL_ASSERT(true == configuration.connectionActive); // configuration for connection part is needed

   if(BTS_USER_MODE_LAST <= userMode)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   initializeGenericSMs();
   initializeGeniviSMs();

   // check for Genivi compliant stacks and do SM initialization
   switch(_stackInterface)
   {
      case BTS_IF_ALPS_EVOLUTION_GENIVI:
         initializeAlpsEvolutionGeniviSMs();
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         return;
   }

   initializeExtensions(configuration.connectionConfiguration.configuration, userMode, _stackInterface);

   // SMs are now available
   FW_IF_NULL_PTR_RETURN(_configuration);
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   FW_IF_NULL_PTR_RETURN(_geniviStartup);
   IConfiguration& configurationSm = *_configuration;
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;
   IStartupGenivi& geniviStartup = *_geniviStartup;

   // store configuration
   configurationSm.setConfiguration(configuration.connectionConfiguration.configuration);

   assignSMs();

#if ((defined GEN3X86) || (defined GEN4LSIM))
   checkForAvailableBtBox();
#endif

   // enable HFP device capabilities handler
   geniviHfpDeviceCapabilities.setEnabled(true, userMode);

   // add HFP device capabilities handler to service group
   geniviStartup.addToServiceGroup(userMode, geniviHfpDeviceCapabilities.getServiceEntry(), geniviHfpDeviceCapabilities.getEnabled());

   // add DBus interfaces
   geniviPanConnectDisconnect.addInterfaces(dbusInterfaces, userMode);
   geniviHfpDeviceCapabilities.addInterfaces(dbusInterfaces, userMode);

   // check extensions
   setExtensionStackConfiguration(_component, _stackInterface, _functionBlock, userMode, dbusInterfaces, configuration);

   // add DBus interfaces for startup handling as last step because group availability is checked by this call
   geniviStartup.addInterfaces(dbusInterfaces, userMode);
}

void ConnectionControl::triggerInitializedCallback(void)
{
   FW_IF_NULL_PTR_RETURN(_startupNew);
   IStartup& startup = *_startupNew;

   // forward request
   startup.handleTriggerInitializedCallback();
}

void ConnectionControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   BTSGenDbusServiceInterface serviceInterface = (BTSGenDbusServiceInterface)interface;

   if(BTS_GEN_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   sendInternalIpc2BtsMessage(ptrMsg, true);
}

void ConnectionControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   BTSGenDbusServiceInterface serviceInterface = (BTSGenDbusServiceInterface)interface;

   if(BTS_GEN_DBUS_SERVICE_LAST <= serviceInterface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availabilityEvent)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SYSTEM != busType) && ((BTSCommonEnumClass)::ccdbusif::BUS_TYPE_SESSION != busType))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceAvailabilityConnection* ptrMsg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();

   if(0 != ptrMsg)
   {
      ptrMsg->setInterface(serviceInterface);
      ptrMsg->setAvailabilityEvent(availabilityEvent);
      ptrMsg->setBusType(busType);
      ptrMsg->setBusName(busName);
      ptrMsg->setObjPath(objPath);
      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   sendInternalIpc2BtsMessage(ptrMsg, true);
}

void ConnectionControl::handleTimeout(IN IExtendedTimeoutHandler* handler, IN const BTSTimerId timerId)
{
   FW_IF_NULL_PTR_RETURN(handler);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // HINT: if any error happens now the error has to be handled within this function or within any called sub-function

   ::std::vector<Bts2Ipc_BaseMessage*> sendBts2IpcMsgList;
   sendBts2IpcMsgList.reserve(10);
   ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);
   BTSHandleIpc2BtsMessageItem ipc2BtsMessageItem;

   // find related APP2BTS message => not possible in timeout function

   // special handling is needed for e.g. connect/disconnect
   ipc2BtsMessageItem.item.opCode = App2BtsOC_None;

   // handle timeout
   handler->handleExtendedTimeout(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, timerId);

   // check for end of sequence
   connectionPriority.checkEndOfSequence(sendBts2IpcMsgList, sendBts2AppMsgList, ipc2BtsMessageItem, (true == ipc2BtsMessageItem.deleteMessage));

   // NOTE: next part is identical to pushIpc2BtsMessage() except deleting message
   if(0 < sendBts2IpcMsgList.size())
   {
      if(0 != ipc2BtsMessageItem.message)
      {
         BTSApp2BtsMessageCompareItem compareItem;
         ipc2BtsMessageItem.message->getCompareItem(compareItem);
         sendBts2IpcMessageList(sendBts2IpcMsgList, compareItem);
      }
      else
      {
         ::std::vector<Bts2Ipc_BaseMessage*> respSendBts2IpcMsgList;
         respSendBts2IpcMsgList.reserve(sendBts2IpcMsgList.size());

         for(size_t i = 0; i < sendBts2IpcMsgList.size(); i++)
         {
            if(0 != sendBts2IpcMsgList[i])
            {
               if(true == sendBts2IpcMsgList[i]->getResponseMessageFlag())
               {
                  // response message: it could be possible that there is no related application message
                  respSendBts2IpcMsgList.push_back(sendBts2IpcMsgList[i]);
               }
               else
               {
                  // should never happen else you have programmed something wrong
                  // #error_indication
                  FW_NORMAL_ASSERT_ALWAYS();

                  delete sendBts2IpcMsgList[i];
               }
            }
         }

         sendBts2IpcMsgList.clear();

         if(0 < respSendBts2IpcMsgList.size())
         {
            BTSApp2BtsMessageCompareItem compareItem;
            sendBts2IpcMessageList(respSendBts2IpcMsgList, compareItem);
         }
      }
   }

   sendBts2AppMessageList(sendBts2AppMsgList);

   // remove related APP2BTS message and check waiting queue
   if((true == ipc2BtsMessageItem.deleteMessage) && (0 != ipc2BtsMessageItem.message))
   {
#if 0
      if(App2BtsOC_DisconnectDevice == ipc2BtsMessageItem.message->getOpCode())
      {
         ETG_TRACE_USR1((" ### delete App2BtsOC_DisconnectDevice: ipc2BtsMessageItem.item.opCode=%d", ipc2BtsMessageItem.item.opCode));
      }
#endif

      removeApp2BtsWorkingMessage(ipc2BtsMessageItem.message);

      if(App2BtsOC_None == ipc2BtsMessageItem.item.opCode)
      {
         checkWaitingQueue(ipc2BtsMessageItem.message);
      }

      delete ipc2BtsMessageItem.message;
   }

   // check waiting queue (special handling)
   if((App2BtsOC_None != ipc2BtsMessageItem.item.opCode) && (App2BtsOC_Ignore != ipc2BtsMessageItem.item.opCode))
   {
      if(0 != ipc2BtsMessageItem.checkIf)
      {
         ipc2BtsMessageItem.checkIf->checkWaitingQueue(ipc2BtsMessageItem.item);
      }
      else
      {
         checkWaitingQueueExtended(ipc2BtsMessageItem.item);
      }
   }
}

void ConnectionControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(0 == testCommand)
   {
      return;
   }

   bool handled(false);

   if(true == _testCommand.handleTestCommand(testCommand, testData))
   {
      handled = true;
   }
   else if(true == setSubControlExtensionTestCommand(testCommand, testData))
   {
      handled = true;
   }

   FW_NORMAL_ASSERT(true == handled);
}

void ConnectionControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   if(0 == testCommand)
   {
      return;
   }

   if(0 == testData)
   {
      return;
   }

   bool handled(false);

   if(true == _testCommand.handleTestCommand(testCommand, testData))
   {
      handled = true;
   }
   else if(true == setSubControlExtensionTestCommand(testCommand, testData))
   {
      handled = true;
   }

   FW_NORMAL_ASSERT(true == handled);
}

void ConnectionControl::sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   if(0 == msgList.size())
   {
      // can happen
      // no assert --- FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
   sendBts2AppMsgList.reserve(10);

   sendDirectAnswerForApp2BtsMessagesWrapper(sendBts2AppMsgList, msgList, resultCode, statusCode);

   sendBts2AppMessageList(sendBts2AppMsgList);
}

void ConnectionControl::sendDirectAnswerForApp2BtsMessagesWrapper(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   if(0 == msgList.size())
   {
      // can happen
      // no assert --- FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;
   ::std::vector<Bts2App_BaseMessage*>& sendBts2AppMsgList(bts2AppMsgList);

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(0 == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      // if(true == _handleDoubledRequests) TODO: [low]: check after merging
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();

         ETG_TRACE_USR3((" sendDirectAnswerForApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // send direct answer depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done outside of this function
         switch(opcode)
         {
            case App2BtsOC_StartInternalCommunication:
               handleDoubledApp2BtsStartInternalCommunication(sendBts2AppMsgList, static_cast<App2Bts_StartInternalCommunication*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_TriggerFbConnectionInitialized:
               handleDoubledApp2BtsTriggerFbConnectionInitialized(sendBts2AppMsgList, static_cast<App2Bts_TriggerFbConnectionInitialized*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SwitchBluetoothOnOff:
               handleDoubledApp2BtsSwitchBluetoothOnOff(sendBts2AppMsgList, static_cast<App2Bts_SwitchBluetoothOnOff*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetBluetoothOnOffStatus:
               handleDoubledApp2BtsGetBluetoothOnOffStatus(sendBts2AppMsgList, static_cast<App2Bts_GetBluetoothOnOffStatus*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SwitchBluetoothHardwareOnOff:
               handleDoubledApp2BtsSwitchBluetoothHardwareOnOff(sendBts2AppMsgList, static_cast<App2Bts_SwitchBluetoothHardwareOnOff*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetLocalBtName:
               handleDoubledApp2BtsSetLocalBtName(sendBts2AppMsgList, static_cast<App2Bts_SetLocalBtName*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetLocalBtName:
               handleDoubledApp2BtsGetLocalBtName(sendBts2AppMsgList, static_cast<App2Bts_GetLocalBtName*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetDeviceIdServiceRecord:
               handleDoubledApp2BtsSetDeviceIdServiceRecord(sendBts2AppMsgList, static_cast<App2Bts_SetDeviceIdServiceRecord*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetDeviceForRestrictedPairingConnecting:
               handleDoubledApp2BtsSetDeviceForRestrictedPairingConnecting(sendBts2AppMsgList, static_cast<App2Bts_SetDeviceForRestrictedPairingConnecting*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetLocalAdapterModes:
               handleDoubledApp2BtsSetLocalAdapterModes(sendBts2AppMsgList, static_cast<App2Bts_SetLocalAdapterModes*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetLocalAdapterModes:
               handleDoubledApp2BtsGetLocalAdapterModes(sendBts2AppMsgList, static_cast<App2Bts_GetLocalAdapterModes*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartDiscovery:
               handleDoubledApp2BtsStartDiscovery(sendBts2AppMsgList, static_cast<App2Bts_StartDiscovery*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StopDiscovery:
               handleDoubledApp2BtsStopDiscovery(sendBts2AppMsgList, static_cast<App2Bts_StopDiscovery*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetDiscoveringStatus:
               handleDoubledApp2BtsGetDiscoveringStatus(sendBts2AppMsgList, static_cast<App2Bts_GetDiscoveringStatus*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartPairing:
               handleDoubledApp2BtsStartPairing(sendBts2AppMsgList, static_cast<App2Bts_StartPairing*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_CancelPairing:
               handleDoubledApp2BtsCancelPairing(sendBts2AppMsgList, static_cast<App2Bts_CancelPairing*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetPinCode:
               handleDoubledApp2BtsSetPinCode(sendBts2AppMsgList, static_cast<App2Bts_SetPinCode*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_ConfirmSecureSimplePairing:
               handleDoubledApp2BtsConfirmSecureSimplePairing(sendBts2AppMsgList, static_cast<App2Bts_ConfirmSecureSimplePairing*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetLinkKey:
               handleDoubledApp2BtsSetLinkKey(sendBts2AppMsgList, static_cast<App2Bts_SetLinkKey*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_ConnectProtocol:
               handleDoubledApp2BtsConnectProtocol(sendBts2AppMsgList, static_cast<App2Bts_ConnectProtocol*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_AcceptRemoteProtocolConnect:
               handleDoubledApp2BtsAcceptRemoteProtocolConnect(sendBts2AppMsgList, static_cast<App2Bts_AcceptRemoteProtocolConnect*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_RejectRemoteProtocolConnect:
               handleDoubledApp2BtsRejectRemoteProtocolConnect(sendBts2AppMsgList, static_cast<App2Bts_RejectRemoteProtocolConnect*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_DisconnectProtocol:
               handleDoubledApp2BtsDisconnectProtocol(sendBts2AppMsgList, static_cast<App2Bts_DisconnectProtocol*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_DisconnectDevice:
               handleDoubledApp2BtsDisconnectDevice(sendBts2AppMsgList, static_cast<App2Bts_DisconnectDevice*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetProtocolConnectionStatus:
               handleDoubledApp2BtsGetProtocolConnectionStatus(sendBts2AppMsgList, static_cast<App2Bts_GetProtocolConnectionStatus*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetDeviceConnectionStatus:
               handleDoubledApp2BtsGetDeviceConnectionStatus(sendBts2AppMsgList, static_cast<App2Bts_GetDeviceConnectionStatus*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartRemoteServiceSearch:
               handleDoubledApp2BtsStartRemoteServiceSearch(sendBts2AppMsgList, static_cast<App2Bts_StartRemoteServiceSearch*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_CancelRemoteServiceSearch:
               handleDoubledApp2BtsCancelRemoteServiceSearch(sendBts2AppMsgList, static_cast<App2Bts_CancelRemoteServiceSearch*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_QueryRemoteName:
               handleDoubledApp2BtsQueryRemoteName(sendBts2AppMsgList, static_cast<App2Bts_QueryRemoteName*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_RequestLinkQuality:
               handleDoubledApp2BtsRequestLinkQuality(sendBts2AppMsgList, static_cast<App2Bts_RequestLinkQuality*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartTestMode:
               handleDoubledApp2BtsStartTestMode(sendBts2AppMsgList, static_cast<App2Bts_StartTestMode*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StopTestMode:
               handleDoubledApp2BtsStopTestMode(sendBts2AppMsgList, static_cast<App2Bts_StopTestMode*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_RequestTestModeLinkQuality:
               handleDoubledApp2BtsRequestTestModeLinkQuality(sendBts2AppMsgList, static_cast<App2Bts_RequestTestModeLinkQuality*>(ptrMessage), resultCode, statusCode); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               // check extensions
               FW_NORMAL_ASSERT(true == handleDoubledApp2BtsExtensionMessages(sendBts2AppMsgList, ptrMessage, resultCode, statusCode));
               break;
         }
      }
   }
}

void ConnectionControl::reorderApp2BtsWaitingQueue(INOUT MessageQueue< App2Bts_BaseMessage >& waitingQueue)
{
   if(0 == waitingQueue.getSize())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(1 == waitingQueue.getSize())
   {
      // reorder does not make sense for 1 entry
      return;
   }

   // last pushed message has to be checked
   const App2Bts_BaseMessage* lastMsg(waitingQueue.getLast());
   FW_IF_NULL_PTR_RETURN(lastMsg);
   BTSApp2BtsMessageCompareItem lastItem;
   lastMsg->getCompareItem(lastItem);
   // reorder entries for PIM/MSG connect/disconnect; other requests have higher priority
   bool reorderConnect(false);
   // reorder entries for start/cancel service search; other requests have higher priority; PIM/MSG connect/disconnect requests have higher priority too
   bool reorderSearch(false);

   switch(lastItem.opCode)
   {
      case App2BtsOC_ConnectProtocol:
      case App2BtsOC_DisconnectProtocol:
         if((BTS_PROTO_PIM != lastItem.protocolId) && (BTS_PROTO_MSG != lastItem.protocolId))
         {
            reorderConnect = true;
         }
         reorderSearch = true;
         break;
      case App2BtsOC_DisconnectDevice:
         reorderConnect = true;
         reorderSearch = true;
         break;
      case App2BtsOC_StartRemoteServiceSearch:
      case App2BtsOC_CancelRemoteServiceSearch:
         // check internal flag because internal service search requests have higher priority too
         if(true == lastMsg->getInternalMessageFlag())
         {
            reorderConnect = true;
            reorderSearch = true;
         }
         break;
      case App2BtsOC_StartDiscovery:
      case App2BtsOC_StopDiscovery:
         reorderConnect = true;
         reorderSearch = true;
         break;
      case App2BtsOC_StartPairing:
      case App2BtsOC_CancelPairing:
         reorderConnect = true;
         reorderSearch = true;
         break;
      default:
         break;
   }

   if(true == reorderConnect)
   {
      reorderApp2BtsQueue(waitingQueue, true);

      // if reorder connect is executed then reorder search shall be started to ensure correct order
      reorderSearch = true;
   }

   if(true == reorderSearch)
   {
      reorderApp2BtsQueue(waitingQueue, false);
   }
}

bool ConnectionControl::doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN_FALSE(ptrMessage);
   FW_IF_NULL_PTR_RETURN_FALSE(_startupNew);
   FW_IF_NULL_PTR_RETURN_FALSE(_switchBluetooth);
   IStartup& startup = *_startupNew;
   ISwitchBluetooth& switchBluetooth = *_switchBluetooth;

   const BTSApp2BtsOpcode opCode(ptrMessage->getOpCode());

   if((App2BtsOC_ConnectionBlockStart < opCode) && (opCode < App2BtsOC_ConnectionBlockEnd))
   {
      if(BTS_DBUS_SERVICE_NOT_AVAILABLE == startup.getServiceAvailability())
      {
         rejectRequest = true;
      }
      else if(true == startup.getStartupOngoing())
      {
         startup.pushToStartupQueue(ptrMessage, false); // single worker thread
         return false;
      }
      else
      {
         // all needed DBUS services are available and startup sequence is finished => normal handling
      }

      // check opcode
      switch(opCode)
      {
         case App2BtsOC_StartInternalCommunication: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_TriggerFbConnectionInitialized: /*DONE*/
            // nothing to do
            // in case of ongoing startup the initialized callback is triggered automatically
            break;
         case App2BtsOC_SwitchBluetoothOnOff: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            break;
         case App2BtsOC_GetBluetoothOnOffStatus: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_SwitchBluetoothHardwareOnOff: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            break;
         case App2BtsOC_SetLocalBtName: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_GetLocalBtName: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_SetDeviceIdServiceRecord: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_SetDeviceForRestrictedPairingConnecting: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_SetLocalAdapterModes: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_GetLocalAdapterModes: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_StartDiscovery: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_StopDiscovery: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_GetDiscoveringStatus: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_StartPairing: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_CancelPairing: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_SetPinCode: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_ConfirmSecureSimplePairing: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_SetLinkKey: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_ConnectProtocol: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_AcceptRemoteProtocolConnect: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_RejectRemoteProtocolConnect: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_DisconnectProtocol: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_DisconnectDevice: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_GetProtocolConnectionStatus: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_GetDeviceConnectionStatus: /*DONE*/
            // nothing to do
            break;
         case App2BtsOC_StartRemoteServiceSearch: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_CancelRemoteServiceSearch: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = true; // conditions are checked within sub-handler function
            }
            break;
         case App2BtsOC_QueryRemoteName:
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            else
            {
               skipOpCodeCheck = false;
            }
            break;
         case App2BtsOC_RequestLinkQuality: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            else if(BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode())
            {
               rejectRequest = true;
               resultCode = (BTSCommonEnumClass)BTS_REQ_FAILED;
               // statusCode not needed
            }
            break;
         case App2BtsOC_StartTestMode: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            break;
         case App2BtsOC_StopTestMode: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            break;
         case App2BtsOC_RequestTestModeLinkQuality: /*DONE*/
            if(true == rejectRequest)
            {
               resultCode = (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR;
               // statusCode not needed
            }
            else if(true == switchBluetooth.getSwitchActive())
            {
               switchBluetooth.pushToSwitchBtQueue(ptrMessage, false); // single worker thread
               return false;
            }
            break;
         // add all other here if necessary
         default:
         {
            // check extensions
            bool continueProcessing(false);

            if(false == doApp2BtsExtensionMsgPrecheck(continueProcessing, rejectRequest, resultCode, statusCode, skipOpCodeCheck, ptrMessage))
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
            else
            {
               if(false == continueProcessing)
               {
                  return false;
               }
            }
            break;
         }
      }
   }
   else
   {
      // opcode not in range
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
      return false;
   }

   return true;
}

void ConnectionControl::getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   const BTSApp2BtsOpcode opCode(ptrMessage->getOpCode());
   BTSApp2BtsMessageMasterCompareItem item;

   if((App2BtsOC_ConnectionBlockStart < opCode) && (opCode < App2BtsOC_ConnectionBlockEnd))
   {
      size_t reserveSize = 0;

      switch(opCode)
      {
         case App2BtsOC_StartInternalCommunication: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_TriggerFbConnectionInitialized: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SwitchBluetoothOnOff: /*DONE*/
            reserveSize += 2;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_SwitchBluetoothHardwareOnOff;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetBluetoothOnOffStatus: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SwitchBluetoothHardwareOnOff: /*DONE*/
            reserveSize += 2;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_SwitchBluetoothOnOff;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetLocalBtName: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetLocalBtName: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetDeviceIdServiceRecord: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetDeviceForRestrictedPairingConnecting: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetLocalAdapterModes: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetLocalAdapterModes: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartDiscovery: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_StopDiscovery: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_GetDiscoveringStatus: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartPairing: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_CancelPairing: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_SetPinCode: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_ConfirmSecureSimplePairing: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetLinkKey: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_ConnectProtocol: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_AcceptRemoteProtocolConnect: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_RejectRemoteProtocolConnect: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_DisconnectProtocol: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_DisconnectDevice: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_GetProtocolConnectionStatus: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetDeviceConnectionStatus: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartRemoteServiceSearch: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_CancelRemoteServiceSearch: /*DONE*/
            connectionPriority.getSimilarOpCodes(itemList, *ptrMessage);
            break;
         case App2BtsOC_QueryRemoteName:
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_RequestLinkQuality: /*DONE*/
            reserveSize += 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartTestMode: /*DONE*/
            reserveSize += 3;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_StopTestMode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_RequestTestModeLinkQuality;
            itemList.push_back(item);
            break;
         case App2BtsOC_StopTestMode: /*DONE*/
            reserveSize += 3;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_StartTestMode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_RequestTestModeLinkQuality;
            itemList.push_back(item);
            break;
         case App2BtsOC_RequestTestModeLinkQuality: /*DONE*/
            reserveSize += 3;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_StartTestMode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_StopTestMode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            // check extensions
            FW_NORMAL_ASSERT(true == getSimilarExtensionOpCodes(itemList, ptrMessage));
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

void ConnectionControl::getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   const BTSApp2BtsOpcode opCode(ptrMessage->getOpCode());
   BTSApp2BtsMessageMasterCompareItem item;
   ptrMessage->getCompareItem(item);

   if((App2BtsOC_ConnectionBlockStart < opCode) && (opCode < App2BtsOC_ConnectionBlockEnd))
   {
      size_t reserveSize;

      switch(opCode)
      {
         case App2BtsOC_StartInternalCommunication: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_TriggerFbConnectionInitialized: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SwitchBluetoothOnOff: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetBluetoothOnOffStatus: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SwitchBluetoothHardwareOnOff: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetLocalBtName: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetLocalBtName: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetDeviceIdServiceRecord: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetDeviceForRestrictedPairingConnecting: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetLocalAdapterModes: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetLocalAdapterModes: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartDiscovery: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_StopDiscovery: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_GetDiscoveringStatus: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartPairing: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_CancelPairing: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_SetPinCode: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_ConfirmSecureSimplePairing: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_SetLinkKey: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_ConnectProtocol: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_AcceptRemoteProtocolConnect: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_RejectRemoteProtocolConnect: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_DisconnectProtocol: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_DisconnectDevice: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_GetProtocolConnectionStatus: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_GetDeviceConnectionStatus: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartRemoteServiceSearch: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_CancelRemoteServiceSearch: /*DONE*/
            connectionPriority.getMatchingOpCodes(itemList, highPrioItemList, *ptrMessage);
            break;
         case App2BtsOC_QueryRemoteName:
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_RequestLinkQuality: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.compareDeviceAddress = true;
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StartTestMode: /*DONE*/
            reserveSize = 2;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_StopTestMode;
            itemList.push_back(item);
            break;
         case App2BtsOC_StopTestMode: /*DONE*/
            reserveSize = 2;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            item.opCode = App2BtsOC_StartTestMode;
            itemList.push_back(item);
            break;
         case App2BtsOC_RequestTestModeLinkQuality: /*DONE*/
            reserveSize = 1;
            itemList.reserve(reserveSize);
            item.opCode = opCode;
            itemList.push_back(item);
            break;
         // add all other here if necessary
         default:
            // check extensions
            FW_NORMAL_ASSERT(true == getMatchingExtensionOpCodes(itemList, highPrioItemList, ptrMessage));
            break;
      }
   }
   else
   {
      // opcode in wrong range
      FW_NORMAL_ASSERT_ALWAYS();

      // add at least input opcode
      item.opCode = opCode;
      itemList.push_back(item);
   }
}

bool ConnectionControl::skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN_FALSE(ptrMessage);

   // function not used
   FW_NORMAL_ASSERT_ALWAYS();

   return false;
}

void ConnectionControl::checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem)
{
   // implement if necessary
   switch(compareItem.opCode)
   {
      case App2BtsOC_ConnectProtocol:
      {
         App2Bts_ConnectProtocol msg;
         msg.setBDAddress(compareItem.deviceAddress);
         msg.setProtocolId(compareItem.protocolId);
         msg.setUuid(compareItem.sppUuid);
         msg.setMasInstanceName(compareItem.masInstance);
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_DisconnectProtocol:
      {
         App2Bts_DisconnectProtocol msg;
         msg.setBDAddress(compareItem.deviceAddress);
         msg.setProtocolId(compareItem.protocolId);
         msg.setUuid(compareItem.sppUuid);
         msg.setMasInstanceName(compareItem.masInstance);
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_DisconnectDevice:
      {
         App2Bts_DisconnectDevice msg;
         msg.setBDAddress(compareItem.deviceAddress);
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_StartRemoteServiceSearch:
      {
         App2Bts_StartRemoteServiceSearch msg;
         msg.setBDAddress(compareItem.deviceAddress);
         msg.setSearchType(compareItem.searchType);
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_CancelRemoteServiceSearch:
      {
         App2Bts_CancelRemoteServiceSearch msg;
         msg.setBDAddress(compareItem.deviceAddress);
         msg.setSearchType(compareItem.searchType);
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_StartDiscovery:
      {
         App2Bts_StartDiscovery msg;
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_StopDiscovery:
      {
         App2Bts_StopDiscovery msg;
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_StartPairing:
      {
         App2Bts_StartPairing msg;
         msg.setBDAddress(compareItem.deviceAddress);
         checkWaitingQueue(&msg);
         break;
      }
      case App2BtsOC_CancelPairing:
      {
         App2Bts_CancelPairing msg;
         msg.setBDAddress(compareItem.deviceAddress);
         checkWaitingQueue(&msg);
         break;
      }
      default:
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }
}

void ConnectionControl::handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList)
{
   if(0 == msgList.size())
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(0 == ptrMessage)
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(_functionBlock != ptrMessage->getFunctionBlock())
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
         continue;
      }

      if(true == _handleDoubledRequests)
      {
         BTSApp2BtsOpcode opcode = ptrMessage->getOpCode();
         ::std::vector<Bts2App_BaseMessage*> sendBts2AppMsgList;
         sendBts2AppMsgList.reserve(10);

         ETG_TRACE_USR3((" handleDoubledApp2BtsMessages(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // handle doubled messages depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done at the end of this function
         switch(opcode)
         {
            case App2BtsOC_StartInternalCommunication:
               handleDoubledApp2BtsStartInternalCommunication(sendBts2AppMsgList, static_cast<App2Bts_StartInternalCommunication*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_TriggerFbConnectionInitialized:
               handleDoubledApp2BtsTriggerFbConnectionInitialized(sendBts2AppMsgList, static_cast<App2Bts_TriggerFbConnectionInitialized*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SwitchBluetoothOnOff:
               handleDoubledApp2BtsSwitchBluetoothOnOff(sendBts2AppMsgList, static_cast<App2Bts_SwitchBluetoothOnOff*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetBluetoothOnOffStatus:
               handleDoubledApp2BtsGetBluetoothOnOffStatus(sendBts2AppMsgList, static_cast<App2Bts_GetBluetoothOnOffStatus*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SwitchBluetoothHardwareOnOff:
               handleDoubledApp2BtsSwitchBluetoothHardwareOnOff(sendBts2AppMsgList, static_cast<App2Bts_SwitchBluetoothHardwareOnOff*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetLocalBtName:
               handleDoubledApp2BtsSetLocalBtName(sendBts2AppMsgList, static_cast<App2Bts_SetLocalBtName*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetLocalBtName:
               handleDoubledApp2BtsGetLocalBtName(sendBts2AppMsgList, static_cast<App2Bts_GetLocalBtName*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetDeviceIdServiceRecord:
               handleDoubledApp2BtsSetDeviceIdServiceRecord(sendBts2AppMsgList, static_cast<App2Bts_SetDeviceIdServiceRecord*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetDeviceForRestrictedPairingConnecting:
               handleDoubledApp2BtsSetDeviceForRestrictedPairingConnecting(sendBts2AppMsgList, static_cast<App2Bts_SetDeviceForRestrictedPairingConnecting*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetLocalAdapterModes:
               handleDoubledApp2BtsSetLocalAdapterModes(sendBts2AppMsgList, static_cast<App2Bts_SetLocalAdapterModes*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetLocalAdapterModes:
               handleDoubledApp2BtsGetLocalAdapterModes(sendBts2AppMsgList, static_cast<App2Bts_GetLocalAdapterModes*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartDiscovery:
               handleDoubledApp2BtsStartDiscovery(sendBts2AppMsgList, static_cast<App2Bts_StartDiscovery*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StopDiscovery:
               handleDoubledApp2BtsStopDiscovery(sendBts2AppMsgList, static_cast<App2Bts_StopDiscovery*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetDiscoveringStatus:
               handleDoubledApp2BtsGetDiscoveringStatus(sendBts2AppMsgList, static_cast<App2Bts_GetDiscoveringStatus*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartPairing:
               handleDoubledApp2BtsStartPairing(sendBts2AppMsgList, static_cast<App2Bts_StartPairing*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_CancelPairing:
               handleDoubledApp2BtsCancelPairing(sendBts2AppMsgList, static_cast<App2Bts_CancelPairing*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetPinCode:
               handleDoubledApp2BtsSetPinCode(sendBts2AppMsgList, static_cast<App2Bts_SetPinCode*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_ConfirmSecureSimplePairing:
               handleDoubledApp2BtsConfirmSecureSimplePairing(sendBts2AppMsgList, static_cast<App2Bts_ConfirmSecureSimplePairing*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_SetLinkKey:
               handleDoubledApp2BtsSetLinkKey(sendBts2AppMsgList, static_cast<App2Bts_SetLinkKey*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_ConnectProtocol:
               handleDoubledApp2BtsConnectProtocol(sendBts2AppMsgList, static_cast<App2Bts_ConnectProtocol*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_AcceptRemoteProtocolConnect:
               handleDoubledApp2BtsAcceptRemoteProtocolConnect(sendBts2AppMsgList, static_cast<App2Bts_AcceptRemoteProtocolConnect*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_RejectRemoteProtocolConnect:
               handleDoubledApp2BtsRejectRemoteProtocolConnect(sendBts2AppMsgList, static_cast<App2Bts_RejectRemoteProtocolConnect*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_DisconnectProtocol:
               handleDoubledApp2BtsDisconnectProtocol(sendBts2AppMsgList, static_cast<App2Bts_DisconnectProtocol*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_DisconnectDevice:
               handleDoubledApp2BtsDisconnectDevice(sendBts2AppMsgList, static_cast<App2Bts_DisconnectDevice*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetProtocolConnectionStatus:
               handleDoubledApp2BtsGetProtocolConnectionStatus(sendBts2AppMsgList, static_cast<App2Bts_GetProtocolConnectionStatus*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_GetDeviceConnectionStatus:
               handleDoubledApp2BtsGetDeviceConnectionStatus(sendBts2AppMsgList, static_cast<App2Bts_GetDeviceConnectionStatus*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartRemoteServiceSearch:
               handleDoubledApp2BtsStartRemoteServiceSearch(sendBts2AppMsgList, static_cast<App2Bts_StartRemoteServiceSearch*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_CancelRemoteServiceSearch:
               handleDoubledApp2BtsCancelRemoteServiceSearch(sendBts2AppMsgList, static_cast<App2Bts_CancelRemoteServiceSearch*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_QueryRemoteName:
               handleDoubledApp2BtsQueryRemoteName(sendBts2AppMsgList, static_cast<App2Bts_QueryRemoteName*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_RequestLinkQuality:
               handleDoubledApp2BtsRequestLinkQuality(sendBts2AppMsgList, static_cast<App2Bts_RequestLinkQuality*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StartTestMode:
               handleDoubledApp2BtsStartTestMode(sendBts2AppMsgList, static_cast<App2Bts_StartTestMode*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_StopTestMode:
               handleDoubledApp2BtsStopTestMode(sendBts2AppMsgList, static_cast<App2Bts_StopTestMode*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            case App2BtsOC_RequestTestModeLinkQuality:
               handleDoubledApp2BtsRequestTestModeLinkQuality(sendBts2AppMsgList, static_cast<App2Bts_RequestTestModeLinkQuality*>(ptrMessage), (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE); /*lint !e1774: Could use dynamic_cast to downcast polymorphic type*/ // no RTTI, use static_cast instead of dynamic_cast, we know the type of the message
               break;
            // all other
            default:
               // check extensions
               FW_NORMAL_ASSERT(true == handleDoubledApp2BtsExtensionMessages(sendBts2AppMsgList, ptrMessage, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE));
               break;
         }

         sendBts2AppMessageList(sendBts2AppMsgList);
      }
   }
}

/*********** start here with specific class members + methods *****************/
bool ConnectionControl::checkRequestResult(IN const BTSApp2BtsMessageHandlingType messageHandlingType, IN App2Bts_BaseMessage* ptrMessage)
{
   if(BTS_APP2BTS_PUSH_TO_WORKING == messageHandlingType)
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
      return false;
   }
   else if(BTS_APP2BTS_PUSH_TO_WAITING == messageHandlingType)
   {
      pushApp2BtsMsgToWaitingQueue(ptrMessage, false); // single worker thread
      return false;
   }
   else
   {
      return true;
   }
}

void ConnectionControl::doNextConfigurationStep(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   FW_IF_NULL_PTR_RETURN(_configuration);
   IConfiguration& configuration = *_configuration;

   configuration.continueSequence(bts2IpcMsgList, bts2AppMsgList, messageItem);
}

void ConnectionControl::initializeGenericSMs(void)
{
   if(0 == _testMode)
   {
      _testMode = new TestMode();
   }
   FW_NORMAL_ASSERT(0 != _testMode);

   if(0 == _rssi)
   {
      _rssi = new Rssi();
   }
   FW_NORMAL_ASSERT(0 != _rssi);

   if(0 == _restrictedPairingConnecting)
   {
      _restrictedPairingConnecting = new RestrictedPairingConnecting();
   }
   FW_NORMAL_ASSERT(0 != _restrictedPairingConnecting);

   if(0 == _deviceManager)
   {
      _deviceManager = new DeviceManager();
   }
   FW_NORMAL_ASSERT(0 != _deviceManager);

   if(0 == _linkKey)
   {
      _linkKey = new LinkKey();
   }
   FW_NORMAL_ASSERT(0 != _linkKey);

   if(0 == _inquiry)
   {
      _inquiry = new Inquiry();
   }
   FW_NORMAL_ASSERT(0 != _inquiry);

   if(0 == _pairing)
   {
      _pairing = new Pairing();
   }
   FW_NORMAL_ASSERT(0 != _pairing);

   if(0 == _serviceSearch)
   {
      _serviceSearch = new ServiceSearch();
   }
   FW_NORMAL_ASSERT(0 != _serviceSearch);

   if(0 == _configuration)
   {
      _configuration = new Configuration();
   }
   FW_NORMAL_ASSERT(0 != _configuration);

   if(0 == _basicConfiguration)
   {
      _basicConfiguration = new BasicConfiguration();
   }
   FW_NORMAL_ASSERT(0 != _basicConfiguration);

   if(0 == _localAdapterModes)
   {
      _localAdapterModes = new LocalAdapterModes();
   }
   FW_NORMAL_ASSERT(0 != _localAdapterModes);

   if(0 == _sppPool)
   {
      _sppPool = new SppPool();
   }
   FW_NORMAL_ASSERT(0 != _sppPool);

   if(0 == _protocolManagerNew)
   {
      _protocolManagerNew = new ProtocolManager();
   }
   FW_NORMAL_ASSERT(0 != _protocolManagerNew);

   if(0 == _connectionPriority)
   {
      _connectionPriority = new ConnectionPriority();
   }
   FW_NORMAL_ASSERT(0 != _connectionPriority);

   if(0 == _localBluetoothName)
   {
      _localBluetoothName = new LocalBluetoothName();
   }
   FW_NORMAL_ASSERT(0 != _localBluetoothName);

   if(0 == _deviceIdServiceRecord)
   {
      _deviceIdServiceRecord = new DeviceIdServiceRecord();
   }
   FW_NORMAL_ASSERT(0 != _deviceIdServiceRecord);

   if(0 == _hfpDeviceCapabilities)
   {
      _hfpDeviceCapabilities = new HfpDeviceCapabilities();
   }
   FW_NORMAL_ASSERT(0 != _hfpDeviceCapabilities);

   if(0 == _bluetoothStackErrorHandling)
   {
      _bluetoothStackErrorHandling = new BluetoothStackErrorHandling();
   }
   FW_NORMAL_ASSERT(0 != _bluetoothStackErrorHandling);

   if(0 == _startupNew)
   {
      _startupNew = new Startup();
   }
   FW_NORMAL_ASSERT(0 != _startupNew);

   if(0 == _switchBluetooth)
   {
      _switchBluetooth = new SwitchBluetooth();
   }
   FW_NORMAL_ASSERT(0 != _switchBluetooth);
}

void ConnectionControl::destroyGenericSMs(void)
{
   if(0 != _testMode)
   {
      delete _testMode;
      _testMode = 0;
   }

   if(0 != _rssi)
   {
      delete _rssi;
      _rssi = 0;
   }

   if(0 != _restrictedPairingConnecting)
   {
      delete _restrictedPairingConnecting;
      _restrictedPairingConnecting = 0;
   }

   if(0 != _deviceManager)
   {
      delete _deviceManager;
      _deviceManager = 0;
   }

   if(0 != _linkKey)
   {
      delete _linkKey;
      _linkKey = 0;
   }

   if(0 != _inquiry)
   {
      delete _inquiry;
      _inquiry = 0;
   }

   if(0 != _pairing)
   {
      delete _pairing;
      _pairing = 0;
   }

   if(0 != _serviceSearch)
   {
      delete _serviceSearch;
      _serviceSearch = 0;
   }

   if(0 != _configuration)
   {
      delete _configuration;
      _configuration = 0;
   }

   if(0 != _basicConfiguration)
   {
      delete _basicConfiguration;
      _basicConfiguration = 0;
   }

   if(0 != _localAdapterModes)
   {
      delete _localAdapterModes;
      _localAdapterModes = 0;
   }

   if(0 != _sppPool)
   {
      delete _sppPool;
      _sppPool = 0;
   }

   if(0 != _protocolManagerNew)
   {
      delete _protocolManagerNew;
      _protocolManagerNew = 0;
   }

   if(0 != _connectionPriority)
   {
      delete _connectionPriority;
      _connectionPriority = 0;
   }

   if(0 != _localBluetoothName)
   {
      delete _localBluetoothName;
      _localBluetoothName = 0;
   }

   if(0 != _deviceIdServiceRecord)
   {
      delete _deviceIdServiceRecord;
      _deviceIdServiceRecord = 0;
   }

   if(0 != _hfpDeviceCapabilities)
   {
      delete _hfpDeviceCapabilities;
      _hfpDeviceCapabilities = 0;
   }

   if(0 != _bluetoothStackErrorHandling)
   {
      delete _bluetoothStackErrorHandling;
      _bluetoothStackErrorHandling = 0;
   }

   if(0 != _startupNew)
   {
      delete _startupNew;
      _startupNew = 0;
   }

   if(0 != _switchBluetooth)
   {
      delete _switchBluetooth;
      _switchBluetooth = 0;
   }
}

void ConnectionControl::initializeGeniviSMs(void)
{
   if(0 == _geniviRssi)
   {
      _geniviRssi = new RssiGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviRssi);

   if(0 == _geniviDeviceManager)
   {
      _geniviDeviceManager = new DeviceManagerGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviDeviceManager);

   if(0 == _geniviLinkKey)
   {
      _geniviLinkKey = new LinkKeyGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviLinkKey);

   if(0 == _geniviInquiry)
   {
      _geniviInquiry = new InquiryGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviInquiry);

   if(0 == _geniviPairing)
   {
      _geniviPairing = new PairingGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviPairing);

   if(0 == _geniviServiceSearch)
   {
      _geniviServiceSearch = new ServiceSearchGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviServiceSearch);

   if(0 == _geniviLocalAdapterModes)
   {
      _geniviLocalAdapterModes = new LocalAdapterModesGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviLocalAdapterModes);

   if(0 == _geniviHfpAvpConnectDisconnect)
   {
      _geniviHfpAvpConnectDisconnect = new HfpAvpConnectDisconnectGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviHfpAvpConnectDisconnect);

   if(0 == _geniviObexConnectDisconnect)
   {
      _geniviObexConnectDisconnect = new ObexConnectDisconnectGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviObexConnectDisconnect);

   if(0 == _geniviPanConnectDisconnect)
   {
      _geniviPanConnectDisconnect = new PanConnectDisconnectGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviPanConnectDisconnect);

   if(0 == _geniviSppPool)
   {
      _geniviSppPool = new SppPoolGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviSppPool);

   if(0 == _geniviLocalBluetoothName)
   {
      _geniviLocalBluetoothName = new LocalBluetoothNameGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviLocalBluetoothName);

   if(0 == _geniviBasicConfiguration)
   {
      _geniviBasicConfiguration = new BasicConfigurationGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviBasicConfiguration);

   if(0 == _geniviHfpDeviceCapabilities)
   {
      _geniviHfpDeviceCapabilities = new HfpDeviceCapabilitiesGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviHfpDeviceCapabilities);

   if(0 == _geniviStartup)
   {
      _geniviStartup = new StartupGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviStartup);

   if(0 == _geniviSwitchBluetooth)
   {
      _geniviSwitchBluetooth = new SwitchBluetoothGenivi();
   }
   FW_NORMAL_ASSERT(0 != _geniviSwitchBluetooth);
}

void ConnectionControl::destroyGeniviSMs(void)
{
   if(0 != _geniviRssi)
   {
      delete _geniviRssi;
      _geniviRssi = 0;
   }

   if(0 != _geniviDeviceManager)
   {
      delete _geniviDeviceManager;
      _geniviDeviceManager = 0;
   }

   if(0 != _geniviLinkKey)
   {
      delete _geniviLinkKey;
      _geniviLinkKey = 0;
   }

   if(0 != _geniviInquiry)
   {
      delete _geniviInquiry;
      _geniviInquiry = 0;
   }

   if(0 != _geniviPairing)
   {
      delete _geniviPairing;
      _geniviPairing = 0;
   }

   if(0 != _geniviServiceSearch)
   {
      delete _geniviServiceSearch;
      _geniviServiceSearch = 0;
   }

   if(0 != _geniviLocalAdapterModes)
   {
      delete _geniviLocalAdapterModes;
      _geniviLocalAdapterModes = 0;
   }

   if(0 != _geniviHfpAvpConnectDisconnect)
   {
      delete _geniviHfpAvpConnectDisconnect;
      _geniviHfpAvpConnectDisconnect = 0;
   }

   if(0 != _geniviObexConnectDisconnect)
   {
      delete _geniviObexConnectDisconnect;
      _geniviObexConnectDisconnect = 0;
   }

   if(0 != _geniviPanConnectDisconnect)
   {
      delete _geniviPanConnectDisconnect;
      _geniviPanConnectDisconnect = 0;
   }

   if(0 != _geniviSppPool)
   {
      delete _geniviSppPool;
      _geniviSppPool = 0;
   }

   if(0 != _geniviLocalBluetoothName)
   {
      delete _geniviLocalBluetoothName;
      _geniviLocalBluetoothName = 0;
   }

   if(0 != _geniviBasicConfiguration)
   {
      delete _geniviBasicConfiguration;
      _geniviBasicConfiguration = 0;
   }

   if(0 != _geniviHfpDeviceCapabilities)
   {
      delete _geniviHfpDeviceCapabilities;
      _geniviHfpDeviceCapabilities = 0;
   }

   if(0 != _geniviStartup)
   {
      delete _geniviStartup;
      _geniviStartup = 0;
   }

   if(0 != _geniviSwitchBluetooth)
   {
      delete _geniviSwitchBluetooth;
      _geniviSwitchBluetooth = 0;
   }
}

void ConnectionControl::assignSMs(void)
{
   FW_IF_NULL_PTR_RETURN(_smExtension);

   // generic SMs
   FW_IF_NULL_PTR_RETURN(_testMode);
   FW_IF_NULL_PTR_RETURN(_rssi);
   FW_IF_NULL_PTR_RETURN(_restrictedPairingConnecting);
   FW_IF_NULL_PTR_RETURN(_deviceManager);
   FW_IF_NULL_PTR_RETURN(_linkKey);
   FW_IF_NULL_PTR_RETURN(_inquiry);
   FW_IF_NULL_PTR_RETURN(_pairing);
   FW_IF_NULL_PTR_RETURN(_serviceSearch);
   FW_IF_NULL_PTR_RETURN(_configuration);
   IConfigurationMaster* configurationMaster = _configuration->getConfigurationMasterHandler();
   FW_IF_NULL_PTR_RETURN(configurationMaster);
   FW_IF_NULL_PTR_RETURN(_basicConfiguration);
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   FW_IF_NULL_PTR_RETURN(_sppPool);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   FW_IF_NULL_PTR_RETURN(_localBluetoothName);
   FW_IF_NULL_PTR_RETURN(_deviceIdServiceRecord);
   FW_IF_NULL_PTR_RETURN(_hfpDeviceCapabilities);
   FW_IF_NULL_PTR_RETURN(_bluetoothStackErrorHandling);
   FW_IF_NULL_PTR_RETURN(_startupNew);
   FW_IF_NULL_PTR_RETURN(_switchBluetooth);

   // Genivi SMs
   FW_IF_NULL_PTR_RETURN(_geniviRssi);
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_geniviLinkKey);
   FW_IF_NULL_PTR_RETURN(_geniviInquiry);
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviSppPool);
   FW_IF_NULL_PTR_RETURN(_geniviLocalBluetoothName);
   FW_IF_NULL_PTR_RETURN(_geniviBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   FW_IF_NULL_PTR_RETURN(_geniviStartup);
   FW_IF_NULL_PTR_RETURN(_geniviSwitchBluetooth);

   // Alps Evolution Genivi SMs
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtDeviceIdServiceRecord);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBluetoothStackErrorHandling);

   // generic SMs
   _testMode->setInstance(_smExtension->getTestMode());
   _testMode->setControlIf(this);

   _rssi->setInstance(_geniviRssi);
   _rssi->setControlIf(this);
   _rssi->setDeviceManagerIf(_deviceManager);

   // nothing to do for _restrictedPairingConnecting

   _deviceManager->setInstance(_geniviDeviceManager);
   _deviceManager->setControlIf(this);
   _deviceManager->setTimerPoolIf(&TimerPool::getInstance());
   _deviceManager->registerObserver(_geniviLinkKey);
   _deviceManager->registerObserver(_serviceSearch->getDeviceObserver());
   _deviceManager->registerObserver(_protocolManagerNew->getDeviceObserver());

   _linkKey->setInstance(_geniviLinkKey);
   _linkKey->setDeviceManager(_deviceManager);

   _inquiry->setInstance(_geniviInquiry);
   _inquiry->setControlIf(this);
   _inquiry->setDeviceManagerIf(_deviceManager);
   _inquiry->setTimerPoolIf(&TimerPool::getInstance());

   _pairing->setInstance(_geniviPairing);
   _pairing->setControlIf(this);
   _pairing->setDeviceManager(_deviceManager);
   _pairing->setRestrictionIf(_restrictedPairingConnecting);
   _pairing->setConfigurationIf(_configuration);
   _pairing->setInquiryIf(_inquiry);

   _serviceSearch->setInstance(_geniviServiceSearch);
   _serviceSearch->setControlIf(this);
   _serviceSearch->setTimerPoolIf(&TimerPool::getInstance());
   _serviceSearch->setConfigurationIf(_configuration);
   _serviceSearch->setDeviceManagerIf(_deviceManager);
   _serviceSearch->setProtocolManagerIf(_protocolManagerNew);
   _serviceSearch->registerObserver(_protocolManagerNew->getServiceSearchObserver());

   // nothing to do for _configuration

   // NOTE: order of registration defines the order of configuration steps
   // before switching BT on:
   configurationMaster->registerConfigurationClient(_basicConfiguration->getConfigurationClientHandler(), true);
   configurationMaster->registerConfigurationClient(_switchBluetooth->getConfigurationClientHandler(), true); // last step
   // after switching BT on:
   configurationMaster->registerConfigurationClient(_localBluetoothName->getConfigurationClientHandler(), false);
   configurationMaster->registerConfigurationClient(_deviceIdServiceRecord->getConfigurationClientHandler(), false);
   configurationMaster->registerConfigurationClient(_sppPool->getConfigurationClientHandler(), false);
   configurationMaster->registerConfigurationClient(_localAdapterModes->getConfigurationClientHandler(), false);
   configurationMaster->registerConfigurationClient(_switchBluetooth->getConfigurationClientHandler(), false); // last step

   _basicConfiguration->setInstance(_geniviBasicConfiguration->getRequestIf(), BTS_CONFIG_PARAM_LOCAL_SERVICES, true);
   _basicConfiguration->setInstance(_alpsEvolutionExtBasicConfiguration->getRequestIf(), BTS_CONFIG_PARAM_PAGE_TIMEOUT, false, false);
   _basicConfiguration->setInstance(_alpsEvolutionExtBasicConfiguration->getRequestIf(), BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND, false, false);
   // NOTE: all configuration request interfaces have to be set before this line
   _basicConfiguration->setControlIf(this);
   _basicConfiguration->setTimerPoolIf(&TimerPool::getInstance());
   _basicConfiguration->setConfigurationIf(_configuration);

   _localAdapterModes->setInstance(_geniviLocalAdapterModes);
   _localAdapterModes->setControlIf(this);
   _localAdapterModes->setTimerPoolIf(&TimerPool::getInstance());
   _localAdapterModes->setSwitchBluetoothIf(_switchBluetooth);
   _localAdapterModes->setConfigurationIf(_configuration);

   _sppPool->setInstance(_geniviSppPool->getRequestIf());
   _sppPool->setControlIf(this);
   _sppPool->setTimerPoolIf(&TimerPool::getInstance());
   _sppPool->setConfigurationIf(_configuration);
   _sppPool->registerObserver(_protocolManagerNew->getSppPoolObserver());

   _protocolManagerNew->setInstance(_geniviHfpAvpConnectDisconnect->getRequestIf(), BTS_PROTO_HFP);
   _protocolManagerNew->setInstance(_geniviHfpAvpConnectDisconnect->getRequestIf(), BTS_PROTO_AVP);
   _protocolManagerNew->setInstance(_geniviObexConnectDisconnect->getRequestIf(), BTS_PROTO_PIM);
   _protocolManagerNew->setInstance(_geniviObexConnectDisconnect->getRequestIf(), BTS_PROTO_MSG);
   _protocolManagerNew->setInstance(_geniviPanConnectDisconnect->getRequestIf(), BTS_PROTO_PAN);
   _protocolManagerNew->setInstance(_alpsEvolutionExtSppConnectDisconnect->getRequestIf(), BTS_PROTO_DUN);
   _protocolManagerNew->setInstance(_alpsEvolutionExtSppConnectDisconnect->getRequestIf(), BTS_PROTO_PAN, false);
   _protocolManagerNew->setInstance(_geniviHfpAvpConnectDisconnect->getRequestIf(), BTS_PROTO_SPP);
   _protocolManagerNew->setControlIf(this);
   _protocolManagerNew->setTimerPoolIf(&TimerPool::getInstance());
   _protocolManagerNew->setDeviceManagerIf(_deviceManager);
   _protocolManagerNew->setServiceSearchIf(_serviceSearch);
   _protocolManagerNew->setSppPoolIf(_sppPool);
   _protocolManagerNew->setConfigurationIf(_configuration);
   _protocolManagerNew->setBasicConfigurationIf(_basicConfiguration);
   _protocolManagerNew->setRestrictionIf(_restrictedPairingConnecting);

   _connectionPriority->setControlIf(this);
   _connectionPriority->setDeviceManagerIf(_deviceManager);
   _connectionPriority->setProtocolManagerIf(_protocolManagerNew);
   _connectionPriority->setInquiryIf(_inquiry);
   _connectionPriority->setPairingIf(_pairing);
   _connectionPriority->setServiceSearchIf(_serviceSearch);

   _localBluetoothName->setInstance(_geniviLocalBluetoothName->getRequestIf());
   _localBluetoothName->setControlIf(this);
   _localBluetoothName->setTimerPoolIf(&TimerPool::getInstance());
   _localBluetoothName->setSwitchBluetoothIf(_switchBluetooth);
   _localBluetoothName->setConfigurationIf(_configuration);

   _deviceIdServiceRecord->setInstance(_alpsEvolutionExtDeviceIdServiceRecord->getRequestIf());
   _deviceIdServiceRecord->setControlIf(this);
   _deviceIdServiceRecord->setSwitchBluetoothIf(_switchBluetooth);
   _deviceIdServiceRecord->setConfigurationIf(_configuration);

   _hfpDeviceCapabilities->setInstance(_geniviHfpDeviceCapabilities->getRequestIf());
   _hfpDeviceCapabilities->setControlIf(this);

   _bluetoothStackErrorHandling->setInstance(_alpsEvolutionExtBluetoothStackErrorHandling->getRequestIf());
   _bluetoothStackErrorHandling->setControlIf(this);

   _startupNew->setInstance(_geniviStartup->getRequestIf());
   _startupNew->setControlIf(this);
   _startupNew->setConfigurationIf(_configuration);
   _startupNew->setSwitchBluetoothIf(_switchBluetooth);
   _startupNew->setLocalAdapterModesIf(_localAdapterModes);
   _startupNew->setTimerPoolIf(&TimerPool::getInstance());

   // NOTE: order of registration defines the order of status updates during BT stack restart handling
   _startupNew->registerSmEntry(_testMode->getSmEntryInterface());
   _startupNew->registerSmEntry(_rssi->getSmEntryInterface());
   _startupNew->registerSmEntry(_restrictedPairingConnecting->getSmEntryInterface());
   _startupNew->registerSmEntry(_linkKey->getSmEntryInterface());
   _startupNew->registerSmEntry(_inquiry->getSmEntryInterface());
   _startupNew->registerSmEntry(_pairing->getSmEntryInterface());
   _startupNew->registerSmEntry(_localAdapterModes->getSmEntryInterface());
   _startupNew->registerSmEntry(_sppPool->getSmEntryInterface());
   _startupNew->registerSmEntry(_localBluetoothName->getSmEntryInterface());
   _startupNew->registerSmEntry(_deviceIdServiceRecord->getSmEntryInterface());
   _startupNew->registerSmEntry(_hfpDeviceCapabilities->getSmEntryInterface());
   _startupNew->registerSmEntry(_protocolManagerNew->getSmEntryInterface());
   _startupNew->registerSmEntry(_serviceSearch->getSmEntryInterface());
   _startupNew->registerSmEntry(_deviceManager->getSmEntryInterface());
   _startupNew->registerSmEntry(_connectionPriority->getSmEntryInterface());
   _startupNew->registerSmEntry(_configuration->getSmEntryInterface());
   _startupNew->registerSmEntry(_basicConfiguration->getSmEntryInterface());
   _startupNew->registerSmEntry(_switchBluetooth->getSmEntryInterface());
   _startupNew->registerSmEntry(_bluetoothStackErrorHandling->getSmEntryInterface());
   _startupNew->registerSmEntry(_startupNew->getSmEntryInterface());

   _switchBluetooth->setInstance(_geniviSwitchBluetooth->getRequestIf());
   _switchBluetooth->setControlIf(this);
   _switchBluetooth->setDeviceManagerIf(_deviceManager);
   _switchBluetooth->setConfigurationIf(_configuration);
   _switchBluetooth->setTestModeIf(_testMode);
   _switchBluetooth->setLocalAdapterModesIf(_localAdapterModes);
   _switchBluetooth->setTimerPoolIf(&TimerPool::getInstance());

   // Genivi SMs
   _geniviRssi->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviDeviceManager->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviLinkKey->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviInquiry->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviPairing->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviServiceSearch->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   // nothing to do for _geniviLocalAdapterModes

   _geniviHfpAvpConnectDisconnect->setDeviceManagerIf(_deviceManager);
   _geniviHfpAvpConnectDisconnect->setServiceSearchIf(_serviceSearch);
   _geniviHfpAvpConnectDisconnect->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviObexConnectDisconnect->setSwitchBluetoothIf(_switchBluetooth);
   _geniviObexConnectDisconnect->setDeviceManagerIf(_deviceManager);
   _geniviObexConnectDisconnect->setServiceSearchIf(_serviceSearch);
   _geniviObexConnectDisconnect->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviPanConnectDisconnect->setConfigurationIf(_configuration);
   _geniviPanConnectDisconnect->setDeviceManagerIf(_deviceManager);
   _geniviPanConnectDisconnect->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   // nothing to do for _geniviSppPool

   // nothing to do for _geniviLocalBluetoothName

   // nothing to do for _geniviBasicConfiguration

   _geniviHfpDeviceCapabilities->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   _geniviStartup->setHfpDeviceCapabilitiesGeniviIf(_geniviHfpDeviceCapabilities);

   // nothing to do for _geniviSwitchBluetooth

   // Alps Evolution Genivi SMs
   // nothing to do for _alpsEvolutionExtBasicConfiguration

   _alpsEvolutionExtSppConnectDisconnect->setDeviceManagerIf(_deviceManager);
   _alpsEvolutionExtSppConnectDisconnect->setObjectPathManagerIf(&ObjectPathManager::getInstance());

   // nothing to do for _alpsEvolutionExtDeviceIdServiceRecord

   _alpsEvolutionExtBluetoothStackErrorHandling->setSwitchBluetoothIf(_switchBluetooth);

   // test command
   _testCommand.setControlIf(this);
   _testCommand.setServiceSearch(_serviceSearch);
   _testCommand.setProtocolManager(_protocolManagerNew);
   _testCommand.setBasicConfiguration(_basicConfiguration);
   _testCommand.setPairingIf(_pairing);
   _testCommand.setLocalAdapterModesIf(_localAdapterModes);
   _testCommand.setSwitchBluetoothIf(_switchBluetooth);
   _testCommand.setDeviceManagerIf(_deviceManager);
   _testCommand.setGeniviObexConnectDisconnectIf(_geniviObexConnectDisconnect);
   _testCommand.setGeniviPairingIf(_geniviPairing);
   _testCommand.setGeniviObjectPathManagerIf(&ObjectPathManager::getInstance());
}

void ConnectionControl::checkForAvailableBtBox(void)
{
   if(false == ::fw::doesFileExist("/dev/ttyLSIM_BTBOX_A"))
   {
      ETG_TRACE_FATAL((" checkForAvailableBtBox(): Bluetooth Box not detected"));
      DbusServiceInfo::setTimeout(250);
   }
}

void ConnectionControl::reorderApp2BtsQueue(INOUT MessageQueue< App2Bts_BaseMessage >& messageQueue, IN const bool reorderConnect) const
{
   // create a copy of waiting queue
   MessageQueue< App2Bts_BaseMessage > queueCopy;
   queueCopy.setWarningSize(messageQueue.getWarningSize());

   // list of "pending" items and "pending" flags
   ::std::vector< App2Bts_BaseMessage* > pendingItems;
   ::std::vector< bool > pendingFlags;
   pendingItems.reserve(messageQueue.getSize());
   pendingFlags.reserve(messageQueue.getSize());

   if(true == reorderConnect)
   {
      ETG_TRACE_USR4((" reorderApp2BtsQueue(): reorder connect"));
      reorderApp2BtsQueue4Connect(queueCopy, pendingItems, pendingFlags, messageQueue);
   }
   else
   {
      ETG_TRACE_USR4((" reorderApp2BtsQueue(): reorder search"));
      reorderApp2BtsQueue4Search(queueCopy, pendingItems, pendingFlags, messageQueue);
   }

   // add all remaining pending items
   for(size_t i = 0; (i < pendingItems.size()) && (i < pendingFlags.size()); i++)
   {
      if(true == pendingFlags[i])
      {
         // add to queue
         queueCopy.push(pendingItems[i], false);
         // reset pending flag
         pendingFlags[i] = false;
      }
   }

   // swap queues
   messageQueue.swap(queueCopy);

   // queue size check
   FW_NORMAL_ASSERT(messageQueue.getSize() == queueCopy.getSize());

   // compare queues and trace changes
   if(false == messageQueue.compare(queueCopy))
   {
      size_t counter(0);
      for(MessageQueue< App2Bts_BaseMessage >::It it = queueCopy.getBegin(); it != queueCopy.getEnd(); ++it)
      {
         BTSApp2BtsMessageCompareItem item;
         (*it)->getCompareItem(item);
         ETG_TRACE_USR4((" reorderApp2BtsQueue(): old: [%u]: 0x%08X protocol=%d address=%s", counter, (*it)->getMessageTraceId(), item.protocolId, item.deviceAddress.c_str()));
         ++counter;
      }
      counter = 0;
      for(MessageQueue< App2Bts_BaseMessage >::It it = messageQueue.getBegin(); it != messageQueue.getEnd(); ++it)
      {
         BTSApp2BtsMessageCompareItem item;
         (*it)->getCompareItem(item);
         ETG_TRACE_USR4((" reorderApp2BtsQueue(): new: [%u]: 0x%08X protocol=%d address=%s", counter, (*it)->getMessageTraceId(), item.protocolId, item.deviceAddress.c_str()));
         ++counter;
      }
   }
   else
   {
      ETG_TRACE_USR4((" reorderApp2BtsQueue(): no change"));
   }

   // clear queue to avoid freeing memory
   queueCopy.clear(false);
}

void ConnectionControl::reorderApp2BtsQueue4Connect(OUT MessageQueue< App2Bts_BaseMessage >& queueCopy, OUT ::std::vector< App2Bts_BaseMessage* >& pendingItems, OUT ::std::vector< bool >& pendingFlags, IN const MessageQueue< App2Bts_BaseMessage >& messageQueue) const
{
   // go through list
   for(MessageQueue< App2Bts_BaseMessage >::ConstIt it = messageQueue.getBegin(); it != messageQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

      if(((App2BtsOC_ConnectProtocol == item.opCode) || (App2BtsOC_DisconnectProtocol == item.opCode)) &&
         ((BTS_PROTO_PIM == item.protocolId) || (BTS_PROTO_MSG == item.protocolId)))
      {
         // add to pending list
         pendingItems.push_back((*it));
         pendingFlags.push_back(true);
      }
      else if(App2BtsOC_DisconnectDevice == item.opCode)
      {
         // add all pending items to queue having same address as current item
         for(size_t i = 0; (i < pendingItems.size()) && (i < pendingFlags.size()); i++)
         {
            if(true == pendingFlags[i])
            {
               BTSApp2BtsMessageCompareItem tmpItem;
               pendingItems[i]->getCompareItem(tmpItem);

               if(item.deviceAddress == tmpItem.deviceAddress)
               {
                  // add to queue
                  queueCopy.push(pendingItems[i], false);
                  // reset pending flag
                  pendingFlags[i] = false;
               }
            }
         }

         // add to queue
         queueCopy.push((*it), false);
      }
      else
      {
         // add to queue
         queueCopy.push((*it), false);
      }
   }
}

void ConnectionControl::reorderApp2BtsQueue4Search(OUT MessageQueue< App2Bts_BaseMessage >& queueCopy, OUT ::std::vector< App2Bts_BaseMessage* >& pendingItems, OUT ::std::vector< bool >& pendingFlags, IN const MessageQueue< App2Bts_BaseMessage >& messageQueue) const
{
   // go through list
   for(MessageQueue< App2Bts_BaseMessage >::ConstIt it = messageQueue.getBegin(); it != messageQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

      if(((App2BtsOC_StartRemoteServiceSearch == item.opCode) || (App2BtsOC_CancelRemoteServiceSearch == item.opCode)) &&
         (false == (*it)->getInternalMessageFlag()))
      {
         // add to pending list
         pendingItems.push_back((*it));
         pendingFlags.push_back(true);
      }
      else
      {
         // add to queue
         queueCopy.push((*it), false);
      }
   }
}
/*********** end here with specific class members + methods *******************/

void ConnectionControl::handleDoubledApp2BtsStartInternalCommunication(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartInternalCommunication* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do
}

void ConnectionControl::handleDoubledApp2BtsTriggerFbConnectionInitialized(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_TriggerFbConnectionInitialized* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do
}

void ConnectionControl::handleDoubledApp2BtsSwitchBluetoothOnOff(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SwitchBluetoothOnOff* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_switchBluetooth);
   ISwitchBluetooth& switchBluetooth = *_switchBluetooth;

   // send answer directly because doubled request - set all to success
   switchBluetooth.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsGetBluetoothOnOffStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetBluetoothOnOffStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_switchBluetooth);
   ISwitchBluetooth& switchBluetooth = *_switchBluetooth;

   // send answer directly because doubled request - set all to success
   switchBluetooth.sendStatus(bts2AppMsgList, *ptrMessage, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsSwitchBluetoothHardwareOnOff(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SwitchBluetoothHardwareOnOff* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // not implemented
}

void ConnectionControl::handleDoubledApp2BtsSetLocalBtName(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetLocalBtName* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localBluetoothName);
   ILocalBluetoothName& localBluetoothName = *_localBluetoothName;

   // send answer directly because doubled request - set all to success
   localBluetoothName.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsGetLocalBtName(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetLocalBtName* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localBluetoothName);
   ILocalBluetoothName& localBluetoothName = *_localBluetoothName;

   // send answer directly because doubled request - set all to success
   localBluetoothName.sendStatus(bts2AppMsgList, *ptrMessage, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsSetDeviceIdServiceRecord(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetDeviceIdServiceRecord* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // send answer directly because doubled request - set all to success
   // no Bts2App result/status message available
}

void ConnectionControl::handleDoubledApp2BtsSetDeviceForRestrictedPairingConnecting(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetDeviceForRestrictedPairingConnecting* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_restrictedPairingConnecting);
   IRestrictedPairingConnecting& restrictedPairingConnecting = *_restrictedPairingConnecting;

   // send answer directly because doubled request - set all to success
   restrictedPairingConnecting.setRestrictionAndSendStatus(bts2AppMsgList, *ptrMessage);
}

void ConnectionControl::handleDoubledApp2BtsSetLocalAdapterModes(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetLocalAdapterModes* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;

   // send answer directly because doubled request - set all to success
   localAdapterModes.storeNewModes(*ptrMessage);
   localAdapterModes.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsGetLocalAdapterModes(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetLocalAdapterModes* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;

   // send answer directly because doubled request - set all to success
   localAdapterModes.sendStatus(bts2AppMsgList, *ptrMessage, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsStartDiscovery(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartDiscovery* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_inquiry);
   IInquiry& inquiry = *_inquiry;

   // send answer directly because doubled request - set all to success
   inquiry.sendStatusAndStartResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsStopDiscovery(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StopDiscovery* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_inquiry);
   IInquiry& inquiry = *_inquiry;

   // send answer directly because doubled request - set all to success
   inquiry.sendStatusAndStopResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsGetDiscoveringStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetDiscoveringStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_inquiry);
   IInquiry& inquiry = *_inquiry;

   // send answer directly because doubled request - set all to success
   inquiry.sendStatus(bts2AppMsgList, *ptrMessage, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsStartPairing(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartPairing* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_pairing);
   IPairing& pairing = *_pairing;

   // send answer directly because doubled request - set all to success
   pairing.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsCancelPairing(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_CancelPairing* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do
}

void ConnectionControl::handleDoubledApp2BtsSetPinCode(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetPinCode* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do (doubled request for this interface cannot happen)
}

void ConnectionControl::handleDoubledApp2BtsConfirmSecureSimplePairing(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_ConfirmSecureSimplePairing* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do (doubled request for this interface cannot happen)
}

void ConnectionControl::handleDoubledApp2BtsSetLinkKey(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_SetLinkKey* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do (doubled request for this interface cannot happen)
}

void ConnectionControl::handleDoubledApp2BtsConnectProtocol(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_ConnectProtocol* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   IProtocolManager& protocolManager = *_protocolManagerNew;

   // send answer directly because doubled request - set all to success
   // all actions are done within handleApp2BtsConnectProtocol()
   protocolManager.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsAcceptRemoteProtocolConnect(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_AcceptRemoteProtocolConnect* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do (doubled request for this interface cannot happen)
}

void ConnectionControl::handleDoubledApp2BtsRejectRemoteProtocolConnect(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_RejectRemoteProtocolConnect* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do (doubled request for this interface cannot happen)
}

void ConnectionControl::handleDoubledApp2BtsDisconnectProtocol(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_DisconnectProtocol* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   IProtocolManager& protocolManager = *_protocolManagerNew;

   // send answer directly because doubled request - set all to success
   // all actions are done within handleApp2BtsDisconnectProtocol()
   protocolManager.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsDisconnectDevice(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_DisconnectDevice* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_deviceManager);
   IDeviceManager& deviceManager = *_deviceManager;

   // send answer directly because doubled request - set all to success
   deviceManager.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsGetProtocolConnectionStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetProtocolConnectionStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   IProtocolManager& protocolManager = *_protocolManagerNew;

   // send answer directly because doubled request - set all to success
   protocolManager.sendStatus(bts2AppMsgList, *ptrMessage, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsGetDeviceConnectionStatus(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_GetDeviceConnectionStatus* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(resultCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_deviceManager);
   IDeviceManager& deviceManager = *_deviceManager;

   // send answer directly because doubled request - set all to success
   deviceManager.sendStatus(bts2AppMsgList, *ptrMessage, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsStartRemoteServiceSearch(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartRemoteServiceSearch* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_serviceSearch);
   IServiceSearch& serviceSearch = *_serviceSearch;

   // send answer directly because doubled request - set all to success
   // all actions are done within handleApp2BtsStartRemoteServiceSearch()
   serviceSearch.sendStatusAndResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsCancelRemoteServiceSearch(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_CancelRemoteServiceSearch* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do
}

void ConnectionControl::handleDoubledApp2BtsQueryRemoteName(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_QueryRemoteName* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(resultCode);
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // send answer directly because doubled request - set all to success
   // TODO: [low]: implement if needed
}

void ConnectionControl::handleDoubledApp2BtsRequestLinkQuality(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_RequestLinkQuality* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_rssi);
   IRssi& rssi = *_rssi;

   // send answer directly because doubled request - set all to success
   rssi.sendResult(bts2AppMsgList, *ptrMessage, resultCode);
}

void ConnectionControl::handleDoubledApp2BtsStartTestMode(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StartTestMode* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_testMode);
   ITestMode& testMode = *_testMode;

   // send answer directly because doubled request - set all to success
   testMode.sendStatusAndStartResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsStopTestMode(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_StopTestMode* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_testMode);
   ITestMode& testMode = *_testMode;

   // send answer directly because doubled request - set all to success
   testMode.sendStatusAndStopResult(bts2AppMsgList, *ptrMessage, false, resultCode, statusCode);
}

void ConnectionControl::handleDoubledApp2BtsRequestTestModeLinkQuality(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN App2Bts_RequestTestModeLinkQuality* ptrMessage, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(statusCode);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_testMode);
   ITestMode& testMode = *_testMode;

   // send answer directly because doubled request - set all to success
   testMode.sendLinkQualityResult(bts2AppMsgList, *ptrMessage, resultCode);
}

void ConnectionControl::handleApp2BtsStartInternalCommunication(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartInternalCommunication* ptrMessage)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // nothing to do

   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsTriggerFbConnectionInitialized(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_TriggerFbConnectionInitialized* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_startupNew);
   IStartup& startup = *_startupNew;

   // forward request
   if(true == startup.handleFbConnectionInitialized(bts2IpcMsgList, bts2AppMsgList))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsSwitchBluetoothOnOff(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SwitchBluetoothOnOff* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_switchBluetooth);
   ISwitchBluetooth& switchBluetooth = *_switchBluetooth;

   // forward request
   if(true == switchBluetooth.switchBtOnOff(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsGetBluetoothOnOffStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetBluetoothOnOffStatus* ptrMessage)
{
   (void)(bts2IpcMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_switchBluetooth);
   ISwitchBluetooth& switchBluetooth = *_switchBluetooth;

   // send answer directly
   switchBluetooth.sendStatus(bts2AppMsgList, *ptrMessage, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsSwitchBluetoothHardwareOnOff(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SwitchBluetoothHardwareOnOff* ptrMessage)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // not implemented

   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsSetLocalBtName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetLocalBtName* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localBluetoothName);
   ILocalBluetoothName& localBluetoothName = *_localBluetoothName;

   // forward request
   if(true == localBluetoothName.setLocalBluetoothName(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsGetLocalBtName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetLocalBtName* ptrMessage)
{
   (void)(bts2IpcMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localBluetoothName);
   ILocalBluetoothName& localBluetoothName = *_localBluetoothName;

   // send answer directly
   localBluetoothName.sendStatus(bts2AppMsgList, *ptrMessage, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsSetDeviceIdServiceRecord(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetDeviceIdServiceRecord* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_deviceIdServiceRecord);
   IDeviceIdServiceRecord& deviceIdServiceRecord = *_deviceIdServiceRecord;

   // forward request
   if(true == deviceIdServiceRecord.setDeviceIdServiceRecord(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsSetDeviceForRestrictedPairingConnecting(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetDeviceForRestrictedPairingConnecting* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_restrictedPairingConnecting);
   IRestrictedPairingConnecting& restrictedPairingConnecting = *_restrictedPairingConnecting;

   // forward request
   if(true == restrictedPairingConnecting.setRestriction(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsSetLocalAdapterModes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetLocalAdapterModes* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;

   // forward request
   if(true == localAdapterModes.setModes(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsGetLocalAdapterModes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetLocalAdapterModes* ptrMessage)
{
   (void)(bts2IpcMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;

   // forward request
   localAdapterModes.sendStatus(bts2AppMsgList, *ptrMessage, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
   // request is answered => request can be deleted
   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsStartDiscovery(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartDiscovery* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.startDiscovery(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsStopDiscovery(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StopDiscovery* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.stopDiscovery(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsGetDiscoveringStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetDiscoveringStatus* ptrMessage)
{
   (void)(bts2IpcMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_inquiry);
   IInquiry& inquiry = *_inquiry;

   // forward request
   inquiry.sendStatus(bts2AppMsgList, *ptrMessage, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
   // request is answered => request can be deleted
   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsStartPairing(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartPairing* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   //===================================================================================================================
   // debug section start
   if((true == _testCommand.getSimulateStartPairingDuringIncomingPairing()) || (true == _testCommand.getSimulateStartPairingDuringIncomingPairing2()))
   {
      _testCommand.setSimulateStartPairingDuringIncomingPairingMsg(ptrMessage);
      return;
   }
   // debug section end
   //===================================================================================================================

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.startPairing(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsCancelPairing(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_CancelPairing* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.cancelPairing(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsSetPinCode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetPinCode* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_pairing);
   IPairing& pairing = *_pairing;

   // forward request
   if(true == pairing.setPinCode(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsConfirmSecureSimplePairing(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_ConfirmSecureSimplePairing* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_pairing);
   IPairing& pairing = *_pairing;

   // forward request
   if(true == pairing.confirmSecureSimplePairing(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsSetLinkKey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_SetLinkKey* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_linkKey);
   ILinkKey& linkKey = *_linkKey;

   // forward request
   if(true == linkKey.setLinkKey(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsConnectProtocol(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_ConnectProtocol* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.connectProtocol(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsAcceptRemoteProtocolConnect(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_AcceptRemoteProtocolConnect* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   IProtocolManager& protocolManager = *_protocolManagerNew;

   // this request has to be handled every time independent of any other requests in working or waiting queue or other ongoing connect/disconnect activities
   // only valid for HFP or AVP or SPP
   (void)protocolManager.acceptRemoteConnect(bts2IpcMsgList, bts2AppMsgList, *ptrMessage);

   // there is no callback message for this request
   deleteApp2BtsMessage = true;

   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getTriggerBlockAuthorizeServiceResponse())
   {
      if((0 < bts2IpcMsgList.size()) && (0 != bts2IpcMsgList[0]))
      {
         delete bts2IpcMsgList[0];
      }
      bts2IpcMsgList.clear();
   }
   // debug section end
   //===================================================================================================================
}

void ConnectionControl::handleApp2BtsRejectRemoteProtocolConnect(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_RejectRemoteProtocolConnect* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   IProtocolManager& protocolManager = *_protocolManagerNew;

   // this request has to be handled every time independent of any other requests in working or waiting queue or other ongoing connect/disconnect activities
   // only valid for HFP or AVP or SPP
   (void)protocolManager.rejectRemoteConnect(bts2IpcMsgList, bts2AppMsgList, *ptrMessage);

   // there is no callback message for this request
   deleteApp2BtsMessage = true;

   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getTriggerBlockAuthorizeServiceResponse())
   {
      if((0 < bts2IpcMsgList.size()) && (0 != bts2IpcMsgList[0]))
      {
         delete bts2IpcMsgList[0];
      }
      bts2IpcMsgList.clear();
   }
   // debug section end
   //===================================================================================================================
}

void ConnectionControl::handleApp2BtsDisconnectProtocol(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_DisconnectProtocol* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.disconnectProtocol(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsDisconnectDevice(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_DisconnectDevice* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.disconnectDevice(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsGetProtocolConnectionStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetProtocolConnectionStatus* ptrMessage)
{
   (void)(bts2IpcMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
   IProtocolManager& protocolManager = *_protocolManagerNew;

   // send answer directly
   protocolManager.sendStatus(bts2AppMsgList, *ptrMessage, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsGetDeviceConnectionStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_GetDeviceConnectionStatus* ptrMessage)
{
   (void)(bts2IpcMsgList);

   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_deviceManager);
   IDeviceManager& deviceManager = *_deviceManager;

   // forward request
   deviceManager.sendStatus(bts2AppMsgList, *ptrMessage, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
   // request is answered => request can be deleted
   deleteApp2BtsMessage = true;
}

void ConnectionControl::handleApp2BtsStartRemoteServiceSearch(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartRemoteServiceSearch* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.startSearch(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsCancelRemoteServiceSearch(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_CancelRemoteServiceSearch* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward request
   deleteApp2BtsMessage = checkRequestResult(connectionPriority.cancelSearch(bts2IpcMsgList, bts2AppMsgList, *ptrMessage), ptrMessage);
}

void ConnectionControl::handleApp2BtsQueryRemoteName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_QueryRemoteName* ptrMessage)
{
   (void)(bts2AppMsgList);
   (void)(deleteApp2BtsMessage);

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // TODO: [low]: implement if needed

   pushApp2BtsMsgToWorkingQueue(ptrMessage, false);

   bts2IpcMsgList.reserve(1);
}

void ConnectionControl::handleApp2BtsRequestLinkQuality(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_RequestLinkQuality* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_rssi);
   IRssi& rssi = *_rssi;

   // forward request
   if(true == rssi.getLinkQuality(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsStartTestMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StartTestMode* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_testMode);
   ITestMode& testMode = *_testMode;

   // forward request
   if(true == testMode.start(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsStopTestMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_StopTestMode* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_testMode);
   ITestMode& testMode = *_testMode;

   // forward request
   if(true == testMode.stop(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleApp2BtsRequestTestModeLinkQuality(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_RequestTestModeLinkQuality* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);
   FW_IF_NULL_PTR_RETURN(_testMode);
   ITestMode& testMode = *_testMode;

   // forward request
   if(true == testMode.getLinkQuality(bts2IpcMsgList, bts2AppMsgList, *ptrMessage))
   {
      pushApp2BtsMsgToWorkingQueue(ptrMessage, false);
   }
   else
   {
      deleteApp2BtsMessage = true;
   }
}

void ConnectionControl::handleIpc2BtsServiceAvailabilityConnection(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceAvailabilityConnection& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviStartup);
   IStartupGenivi& geniviStartup = *_geniviStartup;

   const BTSGenDbusServiceInterface interface(message.getInterface());
   const BTSDbusServiceAvailability availability(message.getAvailabilityEvent());

   if(BTS_GEN_DBUS_SERVICE_LAST <= interface)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_LAST <= availability)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   bool checkGroupAvailability(geniviStartup.updateDbusServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, interface, availability, message.getBusType(), message.getBusName(), message.getObjPath()));

   switch(interface)
   {
      case BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM:
         break;
      case BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SESSION:
         break;
      case BTS_GEN_DBUS_SERVICE_INTROSPECTABLE:
         break;
      case BTS_GEN_DBUS_SERVICE_PROPERTIES_GET_ALL:
         break;
      case BTS_GEN_DBUS_SERVICE_GEN_INITIALIZATION:
         break;
      case BTS_GEN_DBUS_SERVICE_HLI_INITIALIZATION:
         break;
      case BTS_GEN_DBUS_SERVICE_ADAPTER:
         break;
      case BTS_GEN_DBUS_SERVICE_AGENT:
         break;
      case BTS_GEN_DBUS_SERVICE_AGENT_MANAGER:
         break;
      case BTS_GEN_DBUS_SERVICE_DEVICE:
         FW_NORMAL_ASSERT(0 != _geniviDeviceManager);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviDeviceManager) && (0 != _connectionPriority))
         {
            IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to device manager SM: check done, messageItem is not modified because get requests will follow
            geniviDeviceManager.updateDbusServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_DM:
         break;
      case BTS_GEN_DBUS_SERVICE_SERVICE:
         FW_NORMAL_ASSERT(0 != _geniviHfpAvpConnectDisconnect);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviHfpAvpConnectDisconnect) && (0 != _connectionPriority))
         {
            IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to HFP/AVP connect SM: check done, messageItem is not modified because get requests will follow
            geniviHfpAvpConnectDisconnect.updateDbusServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_TRACE:
         break;
      case BTS_GEN_DBUS_SERVICE_GENIVI_TRACE:
         break;
      case BTS_GEN_DBUS_SERVICE_DID:
         break;
      case BTS_GEN_DBUS_SERVICE_SPP:
         break;
      case BTS_GEN_DBUS_SERVICE_OBEX_CLIENT:
         break;
      case BTS_GEN_DBUS_SERVICE_SERIAL:
         FW_NORMAL_ASSERT(0 != _geniviHfpAvpConnectDisconnect);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviHfpAvpConnectDisconnect) && (0 != _connectionPriority))
         {
            IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to HFP/AVP connect SM: check done, messageItem is not modified because get requests will follow
            geniviHfpAvpConnectDisconnect.updateDbusServiceSerialAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_CONNMAN_MANAGER:
         FW_NORMAL_ASSERT(0 != _geniviPanConnectDisconnect);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviPanConnectDisconnect) && (0 != _connectionPriority))
         {
            IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to PAN connect SM: check done, messageItem is not modified
            geniviPanConnectDisconnect.updateDbusServiceConnManAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_CONNMAN_TECHNOLOGY:
         // not needed
         break;
      case BTS_GEN_DBUS_SERVICE_CONNMAN_SERVICE:
         FW_NORMAL_ASSERT(0 != _geniviPanConnectDisconnect);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviPanConnectDisconnect) && (0 != _connectionPriority))
         {
            IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to PAN connect SM: check done, messageItem is not modified because get requests will follow
            geniviPanConnectDisconnect.updateDbusServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      case BTS_GEN_DBUS_SERVICE_HFP_HANDSFREE:
         FW_NORMAL_ASSERT(0 != _geniviHfpDeviceCapabilities);
         if(0 != _geniviHfpDeviceCapabilities)
         {
            IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;
            geniviHfpDeviceCapabilities.updateHandsfreeAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);
         }
         break;
      case BTS_GEN_DBUS_SERVICE_HFP_MANAGER:
         FW_NORMAL_ASSERT(0 != _geniviHfpDeviceCapabilities);
         if(0 != _geniviHfpDeviceCapabilities)
         {
            IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

#if 0
            //===================================================================================================================
            // debug section start
            if(BTS_DBUS_SERVICE_AVAILABLE == availability)
            {
               geniviHfpDeviceCapabilities.updateManagerAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_DBUS_SERVICE_WAITING);
            }
            else
            {
               geniviHfpDeviceCapabilities.updateManagerAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, availability);
            }
            // debug section end
            //===================================================================================================================
#else
            geniviHfpDeviceCapabilities.updateManagerAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, availability);
#endif

            if(true == geniviHfpDeviceCapabilities.getEnabled())
            {
               checkGroupAvailability = true;
            }
         }
         break;
      case BTS_GEN_DBUS_SERVICE_HFP_MODEM:
         FW_NORMAL_ASSERT(0 != _geniviHfpDeviceCapabilities);
         if(0 != _geniviHfpDeviceCapabilities)
         {
            IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;
            geniviHfpDeviceCapabilities.updateModemAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), availability);
         }
         break;
      case BTS_GEN_DBUS_SERVICE_DISABLED_BT_STACK_COMM:
         break;
      case BTS_GEN_DBUS_SERVICE_LAST:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      default:
      {
         // check extensions
         if(false == handleIpc2BtsExtensionMessage(bts2IpcMsgList, bts2AppMsgList, messageItem, &message))
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }
         break;
      }
   }

   geniviStartup.checkNewGroupAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, checkGroupAvailability);
}

void ConnectionControl::handleIpc2BtsGetAllPropertiesConnection(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetAllPropertiesConnection& message)
{
   const BTSDbusPropertyList& properties(message.getProperties());

   switch(message.getInterface())
   {
      case ::ccdbusif::evolution::IF_ADAPTER:
      {
         FW_IF_NULL_PTR_RETURN(_geniviStartup);
         FW_IF_NULL_PTR_RETURN(_geniviSwitchBluetooth);
         FW_IF_NULL_PTR_RETURN(_geniviLocalBluetoothName);
         FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
         IStartupGenivi& geniviStartup = *_geniviStartup;
         ISwitchBluetoothGenivi& geniviSwitchBluetooth = *_geniviSwitchBluetooth;
         ILocalBluetoothNameGenivi& geniviLocalBluetoothName = *_geniviLocalBluetoothName;
         LocalAdapterModesGenivi& geniviLocalAdapterModes = *_geniviLocalAdapterModes;

         BTSIpcCommonErrorCode errorCode(message.getIpcCommonErrorCode());

         //===================================================================================================================
         // debug section start
#if 0
         errorCode = BTS_IPC_ERROR;
#endif
         // debug section end
         //===================================================================================================================

         geniviStartup.handleAdapterProperties(bts2IpcMsgList, bts2AppMsgList, messageItem, properties, errorCode);
         geniviSwitchBluetooth.handleAdapterProperties(bts2IpcMsgList, bts2AppMsgList, messageItem, properties, errorCode);
         geniviLocalBluetoothName.handleAdapterProperties(bts2IpcMsgList, bts2AppMsgList, messageItem, properties, errorCode);
         geniviLocalAdapterModes.handleAdapterProperties(bts2IpcMsgList, bts2AppMsgList, messageItem, properties, errorCode);
         break;
      }
      case ::ccdbusif::evolution::IF_DEVICE:
      {
         FW_NORMAL_ASSERT(0 != _geniviDeviceManager);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviDeviceManager) && (0 != _connectionPriority))
         {
            IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to device manager SM: check done, messageItem could be modified
            geniviDeviceManager.deviceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, properties);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      }
      case ::ccdbusif::evolution::IF_SERVICE:
      {
         FW_NORMAL_ASSERT(0 != _geniviHfpAvpConnectDisconnect);
         FW_NORMAL_ASSERT(0 != _connectionPriority);
         if((0 != _geniviHfpAvpConnectDisconnect) && (0 != _connectionPriority))
         {
            IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
            IConnectionPriority& connectionPriority = *_connectionPriority;

            // forward to HFP/AVP connect SM: check done, messageItem could be modified
            geniviHfpAvpConnectDisconnect.protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, properties);

            // check for end of sequence
            connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
         }
         break;
      }
      case ::ccdbusif::evolution::IF_UNKNOWN:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
      default:
      {
         // check extensions
         if(false == handleIpc2BtsExtensionMessage(bts2IpcMsgList, bts2AppMsgList, messageItem, &message))
         {
            ETG_TRACE_ERR((" handleIpc2BtsGetAllPropertiesConnection(): check implementation"));
            // FW_NORMAL_ASSERT_ALWAYS();
         }
         break;
      }
   }
}

void ConnectionControl::handleIpc2BtsAddDeviceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddDeviceObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.deviceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getObjPath());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDelDeviceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelDeviceObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP connect SM: check done, messageItem is never modified
   geniviHfpAvpConnectDisconnect.removeConnectedService(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());
   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.deviceRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsAddProtocolObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddProtocolObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   switch(message.getProtocol())
   {
      case BTS_PROTO_HFP:
      case BTS_PROTO_AVP:
      case BTS_PROTO_SPP:
      {
         /* HFP, AVP, SPP:
          * service object path was created:
          * following data is given:
          * - protocol
          * - UUID
          * - device object path
          * - service object path
          * following data is needed:
          * - protocol
          * - UUID
          * - device address
          * - service object path
          */
         // forward to HFP/AVP connect SM
         geniviHfpAvpConnectDisconnect.protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getObjPath(), message.getProtocol(), message.getUuid());
         break;
      }
      case BTS_PROTO_PIM:
      case BTS_PROTO_MSG:
      {
         /* PBAP, MAP:
          * following data is given:
          * - protocol
          * - device address
          * - "protocol" object path
          * needed data is given
          */
         // forward to OBEX connect SM
         geniviObexConnectDisconnect.protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getObjPath(), message.getProtocol(), message.getInstanceId());
         break;
      }
      case BTS_PROTO_PAN:
      {
         /* PAN:
          * following data is given:
          * - protocol
          * - device address
          * - "protocol" object path
          * needed data is given
          */
         // forward to PAN connect SM
         geniviPanConnectDisconnect.protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getObjPath(), message.getProtocol());
         break;
      }
      case BTS_PROTO_DUN:
      case BTS_PROTO_LAST:
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDelProtocolObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelProtocolObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP disconnect SM
   geniviHfpAvpConnectDisconnect.protocolRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());
   // forward to OBEX disconnect SM
   geniviObexConnectDisconnect.protocolRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());
   // forward to PAN disconnect SM
   geniviPanConnectDisconnect.protocolRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsAddSerialObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddSerialObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP connect SM
   geniviHfpAvpConnectDisconnect.serialAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDelSerialObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelSerialObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP disconnect SM
   geniviHfpAvpConnectDisconnect.serialRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsRelease(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_Release& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);

   // not sure if this will happen
   FW_NORMAL_ASSERT_ALWAYS();
}

void ConnectionControl::handleIpc2BtsRequestPinCode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestPinCode& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   IPairingGenivi& geniviPairing = *_geniviPairing;

   // forward to pairing SM: check done, messageItem is never modified
   const BTSNumericValue numValue;
   geniviPairing.handlePairingRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), BTS_SSP_LAST, numValue, message.getDbusToken());

   // pairing sequence is ongoing => no check for end of sequence necessary
}

void ConnectionControl::handleIpc2BtsDisplayPinCode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DisplayPinCode& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   IPairingGenivi& geniviPairing = *_geniviPairing;

   // forward to pairing SM: check done, messageItem is never modified
   geniviPairing.handlePairingRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE, message.getPasskey(), message.getDbusToken());

   // pairing sequence is ongoing => no check for end of sequence necessary
}

void ConnectionControl::handleIpc2BtsRequestPasskey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestPasskey& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   IPairingGenivi& geniviPairing = *_geniviPairing;

   // forward to pairing SM: check done, messageItem is never modified
   const BTSNumericValue numValue;
   geniviPairing.handlePairingRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE, numValue, message.getDbusToken());

   // pairing sequence is ongoing => no check for end of sequence necessary
}

void ConnectionControl::handleIpc2BtsRequestConfirmation(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestConfirmation& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   IPairingGenivi& geniviPairing = *_geniviPairing;

   //===================================================================================================================
   // debug section start
   if(0 < _testCommand.getTriggerBlockRequestConfirmation())
   {
      _testCommand.setTriggerBlockRequestConfirmation(_testCommand.getTriggerBlockRequestConfirmation() - 1);
      if(1 == _testCommand.getTriggerBlockRequestConfirmation())
      {
         // pass
      }
      else if(0 == _testCommand.getTriggerBlockRequestConfirmation())
      {
         // block
         return;
      }
   }
   else if(0 < _testCommand.getAddressForDisconnectDuringPairing().size())
   {
      BTSObjectPath objPath;
      if(true == ObjectPathManager::getInstance().getObjectPath4Address(objPath, _testCommand.getAddressForDisconnectDuringPairing()))
      {
         Bts2Ipc_DeviceDisconnect* ptrMsg = ptrNew_Bts2Ipc_DeviceDisconnect();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(_testCommand.getAddressForDisconnectDuringPairing());
            ptrMsg->setDevice(objPath);
            sendInternalBts2IpcMessage(ptrMsg);
         }
      }
      _testCommand.getAddressForDisconnectDuringPairing().clear();
   }
   else if(true == _testCommand.getSimulateStartPairingDuringIncomingPairing())
   {
      _testCommand.setSimulateStartPairingDuringIncomingPairing(false);

      if(0 != _testCommand.getSimulateStartPairingDuringIncomingPairingMsg())
      {
         _testCommand.getSimulateStartPairingDuringIncomingPairingMsg()->doInputTrace();
         ::std::vector<Bts2Ipc_BaseMessage*> tmpBts2IpcMsgList;
         ::std::vector<Bts2App_BaseMessage*> tmpBts2AppMsgList;
         bool tmpDeleteApp2BtsMessage(false);
         App2Bts_StartPairing* tmpMsg(_testCommand.getSimulateStartPairingDuringIncomingPairingMsg());
         _testCommand.setSimulateStartPairingDuringIncomingPairingMsg(0);
         handleApp2BtsStartPairing(tmpBts2IpcMsgList, tmpBts2AppMsgList, tmpDeleteApp2BtsMessage, tmpMsg);
         if(0 < tmpBts2IpcMsgList.size())
         {
            sendInternalBts2IpcMessage(tmpBts2IpcMsgList[0]);
         }
         for(size_t i = 0; i < tmpBts2AppMsgList.size(); i++)
         {
            sendInternalBts2AppMessage(tmpBts2AppMsgList[i]);
         }
         if(true == tmpDeleteApp2BtsMessage)
         {
            delete tmpMsg;
         }
      }
   }
   // debug section end
   //===================================================================================================================

   // forward to pairing SM
   // use BTS_SSP_WITH_NUMERIC_COMPARISON also in case of empty passkey else we will have wrong mapping while sending response

   // forward to pairing SM: check done, messageItem is never modified
   geniviPairing.handlePairingRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), BTS_SSP_WITH_NUMERIC_COMPARISON, message.getPasskey(), message.getDbusToken());

   // pairing sequence is ongoing => no check for end of sequence necessary

   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getSimulateStartPairingDuringIncomingPairing2())
   {
      _testCommand.setSimulateStartPairingDuringIncomingPairing2(false);

      if(0 != _testCommand.getSimulateStartPairingDuringIncomingPairingMsg())
      {
         _testCommand.getSimulateStartPairingDuringIncomingPairingMsg()->doInputTrace();
         ::std::vector<Bts2Ipc_BaseMessage*> tmpBts2IpcMsgList;
         ::std::vector<Bts2App_BaseMessage*> tmpBts2AppMsgList;
         bool tmpDeleteApp2BtsMessage(false);
         App2Bts_StartPairing* tmpMsg(_testCommand.getSimulateStartPairingDuringIncomingPairingMsg());
         _testCommand.setSimulateStartPairingDuringIncomingPairingMsg(0);
         handleApp2BtsStartPairing(tmpBts2IpcMsgList, tmpBts2AppMsgList, tmpDeleteApp2BtsMessage, tmpMsg);
         if(0 < tmpBts2IpcMsgList.size())
         {
            sendInternalBts2IpcMessage(tmpBts2IpcMsgList[0]);
         }
         for(size_t i = 0; i < tmpBts2AppMsgList.size(); i++)
         {
            sendInternalBts2AppMessage(tmpBts2AppMsgList[i]);
         }
         if(true == tmpDeleteApp2BtsMessage)
         {
            delete tmpMsg;
         }
      }
   }
   // debug section end
   //===================================================================================================================
}

void ConnectionControl::handleIpc2BtsCancel(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_Cancel& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_geniviLinkKey);
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   IPairingGenivi& geniviPairing = *_geniviPairing;
   ILinkKeyGenivi& geniviLinkKey = *_geniviLinkKey;
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout RequestPinCode: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestPasskey: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestConfirmation: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout AutorizeService: 10s => will be handled by failed connect
    * - timeout RequestLinkkey: 25s => will be handled by failed connect
    */

   bool sent(false);

   // forward to pairing SM: check done, messageItem is never modified
   sent = geniviPairing.handleCancelRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, sent, message.getDbusToken());
   // forward to link key SM: check done, messageItem is never modified
   sent = geniviLinkKey.handleCancelRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, sent, message.getDbusToken());
   // forward to HFP/AVP connect SM: check done, messageItem is never modified
   sent = geniviHfpAvpConnectDisconnect.handleCancelRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, sent, message.getDbusToken());
   // forward to SPP connect SM: check done, messageItem is never modified
   sent = alpsEvolutionExtSppConnectDisconnect.handleCancelRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, sent, message.getDbusToken());

   FW_NORMAL_ASSERT(true == sent);
}

void ConnectionControl::handleIpc2BtsCancelRequest(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelRequest& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_geniviLinkKey);
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   IPairingGenivi& geniviPairing = *_geniviPairing;
   ILinkKeyGenivi& geniviLinkKey = *_geniviLinkKey;
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout RequestPinCode: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestPasskey: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestConfirmation: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout AutorizeService: 10s => will be handled by failed connect
    * - timeout RequestLinkkey: 25s => will be handled by failed connect
    */

   bool sent(false);

   // forward to pairing SM: check done, messageItem is never modified
   sent = geniviPairing.handleCancelRequestRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), sent, message.getDbusToken());
   // forward to link key SM: check done, messageItem is never modified
   sent = geniviLinkKey.handleCancelRequestRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), sent, message.getDbusToken());
   // forward to HFP/AVP connect SM: check done, messageItem is never modified
   sent = geniviHfpAvpConnectDisconnect.handleCancelRequestRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), sent, message.getDbusToken());
   // forward to SPP connect SM: check done, messageItem is never modified
   sent = alpsEvolutionExtSppConnectDisconnect.handleCancelRequestRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), sent, message.getDbusToken());

   FW_NORMAL_ASSERT(true == sent);
}

void ConnectionControl::handleIpc2BtsAuthorizeService(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AuthorizeService& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;

   bool handled(false);

   // forward to HFP/AVP connect SM: check done, messageItem is never modified
   if(true == geniviHfpAvpConnectDisconnect.handleRemoteConnect(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getProtocol(), message.getUuid(), message.getDbusToken()))
   {
      handled = true;
   }
   // forward to SPP connect SM: check done, messageItem is never modified
   else if(true == alpsEvolutionExtSppConnectDisconnect.handleRemoteConnect(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getProtocol(), message.getUuid(), message.getDbusToken()))
   {
      handled = true;
   }

   if(false == handled)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      geniviHfpAvpConnectDisconnect.createAuthorizeServiceResponse(bts2IpcMsgList, message.getDevice(), message.getDbusToken(), BTS_CONFIRM_REJECT, true);
   }
}

void ConnectionControl::handleIpc2BtsRequestAuthorization(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestAuthorization& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   IPairingGenivi& geniviPairing = *_geniviPairing;

   // forward to pairing SM: check done, messageItem is never modified
   const BTSNumericValue numValue;
   geniviPairing.handlePairingRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), BTS_SSP_WITH_JUST_WORK_MODE, numValue, message.getDbusToken());

   // pairing sequence is ongoing => no check for end of sequence necessary
}

void ConnectionControl::handleIpc2BtsRequestLinkkey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestLinkkey& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviLinkKey);
   ILinkKeyGenivi& geniviLinkKey = *_geniviLinkKey;

   // forward to link key SM
   geniviLinkKey.handleRequestLinkkey(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getDbusToken());
}

void ConnectionControl::handleIpc2BtsDisplayPasskey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DisplayPasskey& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);

   // TODO: [low]: implement if necessary
}

void ConnectionControl::handleIpc2BtsStartDiscoveryResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_StartDiscoveryResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviInquiry);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IInquiryGenivi& geniviInquiry = *_geniviInquiry;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to inquiry SM: check done, messageItem is only modified in case of finished existing sequence
   geniviInquiry.handleStartDiscoveryResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsStopDiscoveryResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_StopDiscoveryResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviInquiry);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IInquiryGenivi& geniviInquiry = *_geniviInquiry;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to inquiry SM: check done, messageItem is only modified in case of finished existing sequence
   geniviInquiry.handleStopDiscoveryResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsRemoveDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveDeviceResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.handleRemoveDeviceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsCreateDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateDeviceResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.handleCreateDeviceResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsSendHCICmdResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SendHCICmdResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterInitialization(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterInitialization& message)
{
   (void)(message);

   FW_IF_NULL_PTR_RETURN(_geniviStartup);
   IStartupGenivi& geniviStartup = *_geniviStartup;

   geniviStartup.handleAdapterInitialization(bts2IpcMsgList, bts2AppMsgList, messageItem);
}

void ConnectionControl::handleIpc2BtsDeviceCreated(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceCreated& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceRemoved& message)
{
   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getTriggerSimulateSDPProcessReport())
   {
      // simulate SDPProcessReport indication
      _testCommand.setTriggerSimulateSDPProcessReport(false);

      Ipc2Bts_SDPProcessReport sdpProcessReportMessage;

      sdpProcessReportMessage.setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      sdpProcessReportMessage.setDevice(message.getDevice());
      sdpProcessReportMessage.setActive(false);
      sdpProcessReportMessage.doInputTrace();

      handleIpc2BtsSDPProcessReport(bts2IpcMsgList, bts2AppMsgList, messageItem, sdpProcessReportMessage);
   }
   // debug section end
   //===================================================================================================================

   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtStackLog(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtStackLog& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsHCIEvent(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_HCIEvent& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceFound(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceFound& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviInquiry);
   IInquiryGenivi& geniviInquiry = *_geniviInquiry;

   // forward to geniviInquiry SM: check done, messageItem is never modified
   geniviInquiry.handleDeviceFound(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice());
}

void ConnectionControl::handleIpc2BtsAdapterAddressUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterAddressUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterClassUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterClassUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterAliasUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterAliasUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviLocalBluetoothName);
   FW_IF_NULL_PTR_RETURN(_configuration);
   ILocalBluetoothNameGenivi& geniviLocalBluetoothName = *_geniviLocalBluetoothName;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SM
   geniviLocalBluetoothName.handleLocalNameUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getAlias(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif
}

void ConnectionControl::handleIpc2BtsAdapterPoweredUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterPoweredUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviSwitchBluetooth);
   FW_IF_NULL_PTR_RETURN(_configuration);
   ISwitchBluetoothGenivi& geniviSwitchBluetooth = *_geniviSwitchBluetooth;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to switch BT SM
   geniviSwitchBluetooth.handlePoweredUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getPowered(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif
}

void ConnectionControl::handleIpc2BtsAdapterDiscoverableUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterDiscoverableUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
   FW_IF_NULL_PTR_RETURN(_configuration);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;
   ILocalAdapterModesGenivi& geniviLocalAdapterModes = *_geniviLocalAdapterModes;
   IConfiguration& configuration = *_configuration;

   // forward to set local adapter modes SM
   geniviLocalAdapterModes.handleDiscoverableUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDiscoverable(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for startup configuration
   if((true == localAdapterModes.getSettingLocalAdapterModesActive()) && (false == localAdapterModes.isSetLocalModesSequenceOngoing()) && (true == configuration.getOngoing()))
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
}

void ConnectionControl::handleIpc2BtsAdapterDiscoverabletimeoutUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterDiscoverabletimeoutUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
   ILocalAdapterModesGenivi& geniviLocalAdapterModes = *_geniviLocalAdapterModes;

   // forward to set local adapter modes SM
   geniviLocalAdapterModes.handleDiscoverableTimeoutUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDiscoverableTimeout(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());
}

void ConnectionControl::handleIpc2BtsAdapterPairableUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterPairableUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
   FW_IF_NULL_PTR_RETURN(_configuration);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;
   ILocalAdapterModesGenivi& geniviLocalAdapterModes = *_geniviLocalAdapterModes;
   IConfiguration& configuration = *_configuration;

   // forward to set local adapter modes SM
   geniviLocalAdapterModes.handlePairableUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getPairable(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for startup configuration
   if((true == localAdapterModes.getSettingLocalAdapterModesActive()) && (false == localAdapterModes.isSetLocalModesSequenceOngoing()) && (true == configuration.getOngoing()))
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
}

void ConnectionControl::handleIpc2BtsAdapterPairabletimeoutUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterPairabletimeoutUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
   ILocalAdapterModesGenivi& geniviLocalAdapterModes = *_geniviLocalAdapterModes;

   // forward to set local adapter modes SM
   geniviLocalAdapterModes.handlePairableTimeoutUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getPairableTimeout(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());
}

void ConnectionControl::handleIpc2BtsAdapterDiscoveringUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterDiscoveringUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviInquiry);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IInquiryGenivi& geniviInquiry = *_geniviInquiry;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to inquiry SM: check done, messageItem is only modified in case of finished existing sequence
   geniviInquiry.handleAdapterDiscoveringUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDiscovering(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsAdapterUuidsUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterUuidsUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_configuration);
   IBasicConfigurationGenivi& geniviBasicConfiguration = *_geniviBasicConfiguration;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to basic configuration SM
   geniviBasicConfiguration.handleAdapterUuidsUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getUuids(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif
}

void ConnectionControl::handleIpc2BtsAdapterWbsModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterWbsModeUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterConnectableUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterConnectableUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_localAdapterModes);
   FW_IF_NULL_PTR_RETURN(_geniviLocalAdapterModes);
   FW_IF_NULL_PTR_RETURN(_configuration);
   ILocalAdapterModes& localAdapterModes = *_localAdapterModes;
   ILocalAdapterModesGenivi& geniviLocalAdapterModes = *_geniviLocalAdapterModes;
   IConfiguration& configuration = *_configuration;

   // forward to set local adapter modes SM
   geniviLocalAdapterModes.handleConnectableUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getConnectable(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for startup configuration
   if((true == localAdapterModes.getSettingLocalAdapterModesActive()) && (false == localAdapterModes.isSetLocalModesSequenceOngoing()) && (true == configuration.getOngoing()))
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
}

void ConnectionControl::handleIpc2BtsAdapterHcimodeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterHcimodeUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_testMode);
   FW_IF_NULL_PTR_RETURN(_switchBluetooth);
   FW_IF_NULL_PTR_RETURN(_geniviSwitchBluetooth);
   ITestMode& testMode = *_testMode;
   ISwitchBluetooth& switchBluetooth = *_switchBluetooth;
   ISwitchBluetoothGenivi& geniviSwitchBluetooth = *_geniviSwitchBluetooth;

   // handle in all cases: update adapter mode
   geniviSwitchBluetooth.handleHciModeUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getHciMode(), message.getIpcCommonErrorCode());

   // inform test mode SM
   testMode.handleSetHciModeResult(bts2IpcMsgList, bts2AppMsgList, messageItem, (BTS_ADAPTER_MODE_HCI == switchBluetooth.getCurrentAdMode()), (BTS_IPC_SUCCESS == message.getIpcCommonErrorCode()));
}

void ConnectionControl::handleIpc2BtsAdapterVersionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterVersionUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterTraceLevelUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterTraceLevelUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterTraceCategoryUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterTraceCategoryUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAdapterAvailableSppServersUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AdapterAvailableSppServersUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviSppPool);
   FW_IF_NULL_PTR_RETURN(_configuration);
   ISppPoolGenivi& geniviSppPool = *_geniviSppPool;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SPP pool SM
   geniviSppPool.handleLocalSppUuidsUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getAvailableSppServers(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif
}

void ConnectionControl::handleIpc2BtsRegisterAgentResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RegisterAgentResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviStartup);
   IStartupGenivi& geniviStartup = *_geniviStartup;

   BTSIpcCommonErrorCode errorCode(message.getIpcCommonErrorCode());

   //===================================================================================================================
   // debug section start
#if 0
   errorCode = BTS_IPC_ERROR;
#endif
   // debug section end
   //===================================================================================================================

   geniviStartup.handleRegisterAgentResult(bts2IpcMsgList, bts2AppMsgList, messageItem, errorCode);
}

void ConnectionControl::handleIpc2BtsUnregisterAgentResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_UnregisterAgentResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsRequestDefaultAgentResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RequestDefaultAgentResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceConnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceConnectResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceDisconnectResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.handleDeviceDisconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsPairResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_PairResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPairingGenivi& geniviPairing = *_geniviPairing;
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to pairing SM: check done, messageItem is only modified in case of finished existing sequence
   geniviPairing.handlePairResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getIpcCommonErrorCode());

   // IncomingPairingComp (incoming pairing) < matches > PairResult (outgoing pairing)

   // handling in ALPS Evolution C6XX branch:
   // PairResult is still used to indicate a failed pairing
   // PairResult is no longer used to indicate the end of Evolution internal SDP search
   // instead signal SDPProcessReport shall be used to monitor Evolution internal SDP search

   // remember pairing status
   geniviServiceSearch.setPairingStatus(message.getDevice(), (BTS_IPC_SUCCESS == message.getIpcCommonErrorCode()));

   // start SDP search after successful pairing: check done, messageItem is never modified
   (void)geniviServiceSearch.startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice()); // bts2IpcMsgList, bts2AppMsgList, messageItem are not modified

   // check for end of sequence - in special cases the pairing sequence is not finished (parallel outgoing and incoming pairing requests)
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsCancelPairingResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelPairingResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPairingGenivi& geniviPairing = *_geniviPairing;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to pairing SM: check done, messageItem is never modified
   geniviPairing.handleCancelPairingResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDiscoverServicesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DiscoverServicesResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getTriggerCancelSearchDuringSearchResult())
   {
      _testCommand.setTriggerCancelSearchDuringSearchResult(false);

      if(0 != _serviceSearch)
      {
         BTSBDAddress address;
         if(true == ObjectPathManager::getInstance().getAddress4ObjectPath(address, message.getDevice()))
         {
            // trigger cancel search
            IServiceSearch& serviceSearchMain = *_serviceSearch;
            App2Bts_CancelRemoteServiceSearch request;
            request.setBDAddress(address);
            request.setSearchType(BTS_SEARCH_MAP);

            ::std::vector<Bts2Ipc_BaseMessage*> tmpBts2IpcMsgList;
            ::std::vector<Bts2App_BaseMessage*> tmpBts2AppMsgList;
            (void)serviceSearchMain.cancelSearch(tmpBts2IpcMsgList, tmpBts2AppMsgList, request);

            for(size_t i = 0; i < tmpBts2IpcMsgList.size(); i++)
            {
               sendInternalBts2IpcMessage(tmpBts2IpcMsgList[i]);
            }
            for(size_t i = 0; i < tmpBts2AppMsgList.size(); i++)
            {
               sendInternalBts2AppMessage(tmpBts2AppMsgList[i]);
            }
         }
      }
   }
   // debug section end
   //===================================================================================================================

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   bool firstConnectFailed(false);
   BTSBDAddress address;
   geniviServiceSearch.handleDiscoverServicesResult(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsCancelDiscoveryResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelDiscoveryResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.handleCancelDiscoveryResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsReportLinkkey(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportLinkkey& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPairingGenivi& geniviPairing = *_geniviPairing;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to pairing SM: check done, messageItem is only modified in case of finished existing sequence
   geniviPairing.handleReportLinkKey(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getLinkKey());

   // handling in ALPS Evolution C6XX branch:
   // Evolution internal SDP search is started after reception of IO Capability Request from phone (=> started before pairing is completed)
   // signal SDPProcessReport shall be used to monitor Evolution internal SDP search

   // remember pairing status => done by handling of IncomingPairingComp or PairResult

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
}

void ConnectionControl::handleIpc2BtsIncomingPairingComp(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_IncomingPairingComp& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPairing);
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPairingGenivi& geniviPairing = *_geniviPairing;
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to pairing SM: check done, messageItem is modified in case of finished existing sequence
   geniviPairing.handleIncomingPairingComp(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), convertPairingStatus2Result(message.getStatus()));

   // IncomingPairingComp (incoming pairing) < matches > PairResult (outgoing pairing)

   // handling in ALPS Evolution C6XX branch:
   // IncomingPairingComp is still used to indicate a failed pairing
   // IncomingPairingComp is no longer used to indicate the end of Evolution internal SDP search
   // instead signal SDPProcessReport shall be used to monitor Evolution internal SDP search

   // remember pairing status
   geniviServiceSearch.setPairingStatus(message.getDevice(), (BTS_STATUS_CODE_SUCCESS == message.getStatus()));

   // start SDP search after successful pairing: check done, messageItem is never modified
   (void)geniviServiceSearch.startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice()); // bts2IpcMsgList, bts2AppMsgList, messageItem are not modified

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsSDPProcessReport(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SDPProcessReport& message)
{
   //===================================================================================================================
   // debug section start
   if((false == message.getActive()) && (true == _testCommand.getTriggerIgnoreSDPProcessReport()))
   {
      // ignore
      _testCommand.setTriggerIgnoreSDPProcessReport(false);
      return;
   }
   // debug section end
   //===================================================================================================================

   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // handling in ALPS Evolution C6XX branch:
   // Evolution internal SDP search is started after reception of IO Capability Request from phone (=> started before pairing is completed)
   // signal SDPProcessReport shall be used to monitor Evolution internal SDP search
   geniviServiceSearch.setSdpSearchActiveAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getActive(), true);

   // start SDP search after successful pairing: check done, messageItem is never modified
   (void)geniviServiceSearch.startSdpSearchAfterPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice()); // bts2IpcMsgList, bts2AppMsgList, messageItem are not modified

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, ((true == messageItem.deleteMessage) && (false == message.getActive())));
}

void ConnectionControl::handleIpc2BtsDeviceAddressUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceAddressUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceClassUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceClassUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceIconUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceIconUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceUuidsUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceUuidsUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updateDeviceUuids(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getSupportedServices(), message.getUuidList());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDevicePairedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DevicePairedUpdate& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;

   // reset pairing status
   if(false == message.getPaired())
   {
      geniviServiceSearch.setPairingStatus(message.getDevice(), message.getPaired());
   }
}

void ConnectionControl::handleIpc2BtsDeviceConnectedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceConnectedUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getTriggerIgnoreDeviceConnectedUpdate())
   {
      _testCommand.setTriggerIgnoreDeviceConnectedUpdate(false);
      return;
   }
   // debug section end
   //===================================================================================================================

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.updateAclStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getConnected(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceTrustedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceTrustedUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceBlockedUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceBlockedUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceAliasUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceAliasUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.updateName(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getAlias(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceAdapterUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceAdapterUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceModaliasUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceModaliasUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDeviceRssiUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceRssiUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviRssi);
   IRssiGenivi& rssi = *_geniviRssi;

   // forward to RSSI SM
   rssi.handleDeviceRssiUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getRssi(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());
}

void ConnectionControl::handleIpc2BtsDeviceVendorIdUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceVendorIdUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updateDeviceVendorId(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getVendorId());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceVendorIdSourceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceVendorIdSourceUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updateDeviceVendorIdSource(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getVendorIdSource());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceProductIdUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceProductIdUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updateDeviceProductId(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getProductId());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceMapInfoUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceMapInfoUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updateMapInfo(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getMapInfoList(), message.getValid());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceConnectedServiceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceConnectedServiceUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;

   // forward to HFP/AVP connect SM: check done, messageItem is never modified
   geniviHfpAvpConnectDisconnect.updateConnectedService(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getConnectedService());
}

void ConnectionControl::handleIpc2BtsDeviceVersionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceVersionUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updateDeviceVersion(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getVersion());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDevicePbapInfoUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DevicePbapInfoUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM: check done, messageItem is only modified in case of finished existing sequence
   geniviServiceSearch.updatePbapInfo(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getDevice(), message.getPbapInfo(), message.getValid());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceModeUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceModeUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.updatePowerMode(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getMode(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceRoleUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceRoleUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.updateRole(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getMaster(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDeviceEnableAvpPauseUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DeviceEnableAvpPauseUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviDeviceManager);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IDeviceManagerGenivi& geniviDeviceManager = *_geniviDeviceManager;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to device manager SM: check done, messageItem could be modified
   geniviDeviceManager.updateEnableAvpPause(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getEnableAvpPause(), message.getIpcCommonErrorCode(), message.getResponseMessageFlag());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceConnectReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectReqResult& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to SPP connect SM: check done, messageItem could be modified
   const BTSProtocolId protocolId(convertServiceFunction2Protocol(message.getServiceFunction()));
   alpsEvolutionExtSppConnectDisconnect.handleConnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getInstance(), message.getOpId(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceDisconnectReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectReqResult& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to SPP connect SM: check done, messageItem could be modified
   const BTSProtocolId protocolId(convertServiceFunction2Protocol(message.getServiceFunction()));
   alpsEvolutionExtSppConnectDisconnect.handleDisconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getInstance(), message.getOpId(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplCancelConnectReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplCancelConnectReqResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtApplGetHwVersionReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplGetHwVersionReqResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtApplSetConfigurationReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSetConfigurationReqResult& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   FW_IF_NULL_PTR_RETURN(_configuration);
   alpsevolutionext::IBasicConfigurationAlpsEvolutionExt& alpsEvolutionExtBasicConfiguration = *_alpsEvolutionExtBasicConfiguration;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SM
   alpsEvolutionExtBasicConfiguration.handleSetConfigurationReqResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getInstance(), message.getConfigId());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplReadConfigurationReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplReadConfigurationReqResult& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   FW_IF_NULL_PTR_RETURN(_configuration);
   alpsevolutionext::IBasicConfigurationAlpsEvolutionExt& alpsEvolutionExtBasicConfiguration = *_alpsEvolutionExtBasicConfiguration;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SM
   alpsEvolutionExtBasicConfiguration.handleReadConfigurationReqResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getInstance(), message.getConfigId());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceConnectCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectCfm& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to SPP connect SM: check done, messageItem could be modified
   alpsEvolutionExtSppConnectDisconnect.handleConnectCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getOpId(), message.getStatus());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceConnectInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectInd& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   const BTSProtocolId protocolId(convertServiceFunction2Protocol(message.getServiceFunction()));

   // handle connect status
   // forward to HFP/AVP connect SM
   geniviHfpAvpConnectDisconnect.updateConnectStatus(message.getBDAddress(), protocolId, message.getStatus());
   // forward to OBEX connect SM
   geniviObexConnectDisconnect.updateConnectStatus(message.getBDAddress(), protocolId, message.getStatus());
   // forward to PAN connect SM
   geniviPanConnectDisconnect.updateConnectStatus(message.getBDAddress(), protocolId, message.getStatus());
   // forward to SPP connect SM
   alpsEvolutionExtSppConnectDisconnect.updateConnectStatus(message.getBDAddress(), protocolId, message.getInstance(), message.getStatus());

   // handle protocol connect
   // forward to HFP/AVP connect SM: check done, messageItem could be modified
   geniviHfpAvpConnectDisconnect.handleConnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getStatus());
   // forward to OBEX connect SM: check done, messageItem could be modified
   geniviObexConnectDisconnect.handleConnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getStatus());
   // forward to PAN connect SM: check done, messageItem could be modified
   geniviPanConnectDisconnect.handleConnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getStatus());
   // forward to SPP connect SM: check done, messageItem could be modified
   alpsEvolutionExtSppConnectDisconnect.handleConnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getInstance(), message.getStatus());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceDisconnectCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectCfm& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to SPP connect SM: check done, messageItem could be modified
   alpsEvolutionExtSppConnectDisconnect.handleDisconnectCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getOpId(), message.getStatus());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceDisconnectInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectInd& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   //===================================================================================================================
   // debug section start
   if((true == _testCommand.getTriggerSppDisconnectDuringRemoteDisconnect()) && (true == message.getServiceFunction().getBit(BTS_SRV_FUNC_SPP)))
   {
      ETG_TRACE_USR1((" SPP disconnect during remote disconnect"));

      _testCommand.setTriggerSppDisconnectDuringRemoteDisconnect(false);

      if(BTS_STATUS_CODE_SUCCESS == message.getStatus())
      {
         FW_IF_NULL_PTR_RETURN(_protocolManagerNew);
         IProtocolManager& protocolManager = *_protocolManagerNew;

         // trigger disconnect for SPP
         App2Bts_DisconnectProtocol* msg = ptrNew_App2Bts_DisconnectProtocol();
         if(0 != msg)
         {
            msg->setBDAddress(message.getBDAddress());
            msg->setProtocolId(BTS_PROTO_SPP);
            BTSUuid sppUuid;
            protocolManager.getFirstConnectedSppUuid(sppUuid, message.getBDAddress());
            msg->setUuid(sppUuid);

            ::std::vector<Bts2Ipc_BaseMessage*> tmpBts2IpcMsgList;
            ::std::vector<Bts2App_BaseMessage*> tmpBts2AppMsgList;
            bool tmpDeleteApp2BtsMessage(false);
            handleApp2BtsDisconnectProtocol(tmpBts2IpcMsgList, tmpBts2AppMsgList, tmpDeleteApp2BtsMessage, msg);
            for(size_t i = 0; i < tmpBts2IpcMsgList.size(); i++)
            {
               sendInternalBts2IpcMessage(tmpBts2IpcMsgList[i]);
            }
            for(size_t i = 0; i < tmpBts2AppMsgList.size(); i++)
            {
               sendInternalBts2AppMessage(tmpBts2AppMsgList[i]);
            }
            if(true == tmpDeleteApp2BtsMessage)
            {
               delete msg;
            }
         }
      }
   }
   // debug section end
   //===================================================================================================================

   const BTSProtocolId protocolId(convertServiceFunction2Protocol(message.getServiceFunction()));
   const BTSInternalDisconnectReason internalReason(convertDisconnectReason2Internal(message.getReason()));

   // handle disconnect reason only in success case
   if(BTS_STATUS_CODE_SUCCESS == message.getStatus())
   {
      // forward to HFP/AVP connect SM
      geniviHfpAvpConnectDisconnect.updateDisconnectReason(message.getBDAddress(), protocolId, internalReason);
      // forward to OBEX connect SM
      geniviObexConnectDisconnect.updateDisconnectReason(message.getBDAddress(), protocolId, internalReason);
      // forward to PAN connect SM
      geniviPanConnectDisconnect.updateDisconnectReason(message.getBDAddress(), protocolId, internalReason);
      // forward to SPP connect SM
      alpsEvolutionExtSppConnectDisconnect.updateDisconnectReason(message.getBDAddress(), protocolId, message.getInstance(), internalReason);
   }

   // handle protocol disconnect
   // forward to HFP/AVP connect SM: check done, messageItem could be modified
   geniviHfpAvpConnectDisconnect.handleDisconnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getStatus());
   // forward to OBEX connect SM: check done, messageItem could be modified
   geniviObexConnectDisconnect.handleDisconnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getStatus());
   // forward to PAN connect SM: check done, messageItem could be modified
   geniviPanConnectDisconnect.handleDisconnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getStatus());
   // forward to SPP connect SM: check done, messageItem could be modified
   alpsEvolutionExtSppConnectDisconnect.handleDisconnectInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), protocolId, message.getInstance(), message.getStatus());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceDisconnectCompInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceDisconnectCompInd& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to SPP connect SM: check done, messageItem could be modified
   alpsEvolutionExtSppConnectDisconnect.handleDisconnectCompInd(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getOpId());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplDeviceConnectCompInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDeviceConnectCompInd& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   bool firstConnectFailed(false);
   BTSBDAddress address;

   // forward to SPP connect SM: check done, messageItem could be modified
   alpsEvolutionExtSppConnectDisconnect.handleConnectCompInd(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getOpId());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsBtApplCancelConnectCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplCancelConnectCfm& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtApplStatusNotificationInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplStatusNotificationInd& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;

   // forward to SPP connect SM: check done, messageItem is never modified
   alpsEvolutionExtSppConnectDisconnect.handleDeviceNameInd(0, message.getDeviceName());
}

void ConnectionControl::handleIpc2BtsBtApplGetHwVersionCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplGetHwVersionCfm& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviSwitchBluetooth);
   ISwitchBluetoothGenivi& geniviSwitchBluetooth = *_geniviSwitchBluetooth;

   // check received data
   geniviSwitchBluetooth.handleGetHwVersionConfirmation(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getStatus(), message.getBtCoreVersion(), message.getManufacturer(), message.getHwSubVersion(), message.getExtInfo());
}

void ConnectionControl::handleIpc2BtsBtApplSetConfigurationCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSetConfigurationCfm& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   FW_IF_NULL_PTR_RETURN(_configuration);
   alpsevolutionext::IBasicConfigurationAlpsEvolutionExt& alpsEvolutionExtBasicConfiguration = *_alpsEvolutionExtBasicConfiguration;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SM
   alpsEvolutionExtBasicConfiguration.handleSetConfigurationCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getStatus(), message.getInstance(), message.getConfigId());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsBtApplReadConfigurationCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplReadConfigurationCfm& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBasicConfiguration);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   FW_IF_NULL_PTR_RETURN(_configuration);
   alpsevolutionext::IBasicConfigurationAlpsEvolutionExt& alpsEvolutionExtBasicConfiguration = *_alpsEvolutionExtBasicConfiguration;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SM
   alpsEvolutionExtBasicConfiguration.handleReadConfigurationCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getStatus(), message.getInstance(), message.getConfigId(), message.getData());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsConnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnectResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   bool firstConnectFailed(false);
   BTSBDAddress address;

   //===================================================================================================================
   // debug section start
   if(true == _testCommand.getSimulateRemotePairingDuringOngoingConnect())
   {
      _testCommand.setSimulateRemotePairingDuringOngoingConnect(false);

      FW_IF_NULL_PTR_RETURN(_geniviPairing);
      IPairingGenivi& geniviPairing = *_geniviPairing;

      // forward to pairing SM: check done, tmpMessageItem is never modified
      BTSHandleIpc2BtsMessageItem tmpMessageItem;
      geniviPairing.handlePairingRequest(bts2IpcMsgList, bts2AppMsgList, tmpMessageItem, "/org/bluez/hci1/dev_12_34_56_78_9A_BC", BTS_SSP_WITH_NUMERIC_COMPARISON, "456789", 0);
   }
   // debug section end
   //===================================================================================================================

   // forward to HFP/AVP connect SM: check done, messageItem could be modified
   geniviHfpAvpConnectDisconnect.handleConnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getService(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DisconnectResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP connect SM: check done, messageItem could be modified
   geniviHfpAvpConnectDisconnect.handleDisconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getService(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsServiceRemoteUuidUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceRemoteUuidUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsServiceStateUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceStateUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   bool firstConnectFailed(false);
   BTSBDAddress address;

   // forward to HFP/AVP connect SM: check done, messageItem could be modified
   geniviHfpAvpConnectDisconnect.handleServiceState(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getService(), message.getServiceState(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsServiceDeviceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceDeviceUpdate& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsServiceVersionUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ServiceVersionUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviServiceSearch);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IServiceSearchGenivi& geniviServiceSearch = *_geniviServiceSearch;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to service search SM
   geniviServiceSearch.updateServiceVersion(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getResponseMessageFlag(), message.getService(), message.getVersion());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsDebugTraceConfigResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DebugTraceConfigResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsProtocolTraceConfigResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ProtocolTraceConfigResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsReportDebugTrace(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportDebugTrace& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBluetoothStackErrorHandling);
   alpsevolutionext::IBluetoothStackErrorHandlingAlpsEvolutionExt& alpsEvolutionExtBluetoothStackErrorHandling = *_alpsEvolutionExtBluetoothStackErrorHandling;

   alpsEvolutionExtBluetoothStackErrorHandling.handleReportDebugTrace(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getTraceLocation(), message.getTraceLevel(), message.getTraceCategory(), message.getErrorCode(), message.getTraceData());
}

void ConnectionControl::handleIpc2BtsReportProtocolTrace(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportProtocolTrace& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsFatalError(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_FatalError& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBluetoothStackErrorHandling);
   alpsevolutionext::IBluetoothStackErrorHandlingAlpsEvolutionExt& alpsEvolutionExtBluetoothStackErrorHandling = *_alpsEvolutionExtBluetoothStackErrorHandling;

   alpsEvolutionExtBluetoothStackErrorHandling.handleFatalError(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getErrLocation(), message.getErrType());
}

void ConnectionControl::handleIpc2BtsGeneralError(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GeneralError& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBluetoothStackErrorHandling);
   alpsevolutionext::IBluetoothStackErrorHandlingAlpsEvolutionExt& alpsEvolutionExtBluetoothStackErrorHandling = *_alpsEvolutionExtBluetoothStackErrorHandling;

   alpsEvolutionExtBluetoothStackErrorHandling.handleGeneralError(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getStatus(), message.getBinaryStatus(), message.getOpcode());
}

void ConnectionControl::handleIpc2BtsReportGeniviDebugTrace(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ReportGeniviDebugTrace& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBluetoothStackErrorHandling);
   alpsevolutionext::IBluetoothStackErrorHandlingAlpsEvolutionExt& alpsEvolutionExtBluetoothStackErrorHandling = *_alpsEvolutionExtBluetoothStackErrorHandling;

   alpsEvolutionExtBluetoothStackErrorHandling.handleGeniviDebugTrace(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getErrFile(), message.getErrLine(), message.getLevel(), message.getCategory(), message.getErrorCode(), message.getData());
}

void ConnectionControl::handleIpc2BtsGeniviFatalError(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GeniviFatalError& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtBluetoothStackErrorHandling);
   alpsevolutionext::IBluetoothStackErrorHandlingAlpsEvolutionExt& alpsEvolutionExtBluetoothStackErrorHandling = *_alpsEvolutionExtBluetoothStackErrorHandling;

   alpsEvolutionExtBluetoothStackErrorHandling.handleGeniviFatalError(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode(), message.getErrFile(), message.getErrLine(), message.getErrCode(), message.getFmt(), message.getErrValue());
}

void ConnectionControl::handleIpc2BtsBtApplDidSetReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDidSetReqResult& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtDeviceIdServiceRecord);
   FW_IF_NULL_PTR_RETURN(_configuration);
   alpsevolutionext::IDeviceIdServiceRecordAlpsEvolutionExt& alpsEvolutionExtDeviceIdServiceRecord = *_alpsEvolutionExtDeviceIdServiceRecord;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   // in case of error is returned continue now else wait for CFM signal
   const bool doNextStep((true == configuration.getOngoing()) && (BTS_IPC_SUCCESS != message.getIpcCommonErrorCode()));
#endif

   // forward to SM
   alpsEvolutionExtDeviceIdServiceRecord.handleDidSetReqResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif
}

void ConnectionControl::handleIpc2BtsBtApplDidSetCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplDidSetCfm& message)
{
   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtDeviceIdServiceRecord);
   FW_IF_NULL_PTR_RETURN(_configuration);
   alpsevolutionext::IDeviceIdServiceRecordAlpsEvolutionExt& alpsEvolutionExtDeviceIdServiceRecord = *_alpsEvolutionExtDeviceIdServiceRecord;
   IConfiguration& configuration = *_configuration;

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   const bool doNextStep(true == configuration.getOngoing());
#endif

   // forward to SM
   alpsEvolutionExtDeviceIdServiceRecord.handleDidSetCfm(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getStatus());

#ifndef ENABLE_NEW_CONFIGURATION_HANDLING
   if(true == doNextStep)
   {
      doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
#endif
}

void ConnectionControl::handleIpc2BtsDunDeviceFilePathUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DunDeviceFilePathUpdate& message)
{
   // not needed at the moment
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtApplSppSetUuidReqResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSppSetUuidReqResult& message)
{
   // obsolete
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtApplSppSetUuidCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSppSetUuidCfm& message)
{
   // obsolete
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsBtApplSppVirtualDeviceInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BtApplSppVirtualDeviceInd& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_alpsEvolutionExtSppConnectDisconnect);
   alpsevolutionext::ISppConnectDisconnectAlpsEvolutionExt& alpsEvolutionExtSppConnectDisconnect = *_alpsEvolutionExtSppConnectDisconnect;

   // forward to SPP connect SM: check done, messageItem is never modified
   alpsEvolutionExtSppConnectDisconnect.handleDeviceNameInd(message.getInstance(), message.getDeviceName());
}

void ConnectionControl::handleIpc2BtsCreateSessionResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CreateSessionResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   bool firstConnectFailed(false);
   BTSBDAddress address;

   // forward to OBEX connect SM: check done, messageItem could be modified
   geniviObexConnectDisconnect.handleCreateSessionResult(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getBDAddress(), message.getProtocolId(), message.getInstanceId(), message.getSession(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsRemoveSessionResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_RemoveSessionResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to OBEX connect SM: check done, messageItem could be modified
   geniviObexConnectDisconnect.handleRemoveSessionResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getProtocolId(), message.getInstanceId(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsCancelSessionResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CancelSessionResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviObexConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to OBEX connect SM: check done, messageItem could be modified
   geniviObexConnectDisconnect.handleCancelSessionResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getProtocolId(), message.getInstanceId(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsSessionCreated(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SessionCreated& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsSessionRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SessionRemoved& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAddTechnologyObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddTechnologyObjectPath& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDelTechnologyObjectPath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelTechnologyObjectPath& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsAddNetworkObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddNetworkObjectPathMapping& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsDelNetworkObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelNetworkObjectPathMapping& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsGetConnManManagerPropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManManagerPropertiesResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsGetConnManTechnologiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManTechnologiesResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsGetConnManServicesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManServicesResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsConnManTechnologyAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManTechnologyAdded& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsConnManTechnologyRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManTechnologyRemoved& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsConnManServicesChanged(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServicesChanged& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsConnManServicesRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServicesRemoved& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsGetConnManServicePropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetConnManServicePropertiesResult& message)
{
   // ignore
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(message);
}

void ConnectionControl::handleIpc2BtsConnManServiceConnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceConnectResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   bool firstConnectFailed(false);
   BTSBDAddress address;

   // forward to PAN connect SM: check done, messageItem could be modified
   geniviPanConnectDisconnect.handleConnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsConnManServiceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceDisconnectResult& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to PAN connect SM: check done, messageItem could be modified
   geniviPanConnectDisconnect.handleDisconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDevice(), message.getIpcCommonErrorCode());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsConnManServiceStateUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceStateUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;
   bool firstConnectFailed(false);
   BTSBDAddress address;

   // forward to PAN connect SM: check done, messageItem could be modified
   geniviPanConnectDisconnect.handleServiceState(bts2IpcMsgList, bts2AppMsgList, messageItem, firstConnectFailed, address, message.getDevice(), message.getValue());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage), firstConnectFailed, address);
}

void ConnectionControl::handleIpc2BtsConnManServiceInterfaceUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ConnManServiceInterfaceUpdate& message)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_geniviPanConnectDisconnect);
   IPanConnectDisconnectGenivi& geniviPanConnectDisconnect = *_geniviPanConnectDisconnect;

   // forward to PAN connect SM: check done, messageItem is never modified
   geniviPanConnectDisconnect.handleInterfaceName(message.getDevice(), message.getInterface());
}

void ConnectionControl::handleIpc2BtsSerialDeviceFilePath(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_SerialDeviceFilePath& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP connect SM
   geniviHfpAvpConnectDisconnect.updateCharacterDeviceName(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getService(), message.getDeviceName());

   // check for end of sequence
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsCharacterDeviceAddedRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_CharacterDeviceAddedRemoved& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpAvpConnectDisconnect);
   FW_IF_NULL_PTR_RETURN(_connectionPriority);
   IHfpAvpConnectDisconnectGenivi& geniviHfpAvpConnectDisconnect = *_geniviHfpAvpConnectDisconnect;
   IConnectionPriority& connectionPriority = *_connectionPriority;

   // forward to HFP/AVP connect SM
   geniviHfpAvpConnectDisconnect.characterDeviceAddedRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getDeviceName(), message.getAdded());

   // check for end of sequence (no need to check for first connect failed because character device is only created in case of SPP connection was established successfully)
   connectionPriority.checkEndOfSequence(bts2IpcMsgList, bts2AppMsgList, messageItem, (true == messageItem.deleteMessage));
}

void ConnectionControl::handleIpc2BtsAddHfpInterfaceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_AddHfpInterfaceObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleAddedHfpInterface(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), message.getInterface());
}

void ConnectionControl::handleIpc2BtsDelHfpInterfaceObjectPathMapping(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_DelHfpInterfaceObjectPathMapping& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleRemovedHfpInterface(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getObjPath(), message.getInterface());
}

void ConnectionControl::handleIpc2BtsGetHandsfreeProperties(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetHandsfreeProperties& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleGetHandsfreePropertiesResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getModem(), message.getIpcCommonErrorCode());
}

void ConnectionControl::handleIpc2BtsHandsfreeFeaturesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_HandsfreeFeaturesUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleHandsfreeFeaturesUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getModem(), message.getFeatures());
}

void ConnectionControl::handleIpc2BtsHandsfreeInbandRingingUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_HandsfreeInbandRingingUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleHandsfreeInbandRingingUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getModem(), message.getInbandRingingEnabled());
}

void ConnectionControl::handleIpc2BtsGetModems(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetModems& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleGetModemsResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getIpcCommonErrorCode());
}

void ConnectionControl::handleIpc2BtsModemAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ModemAdded& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleModemAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getBDAddress(), message.getModem());
}

void ConnectionControl::handleIpc2BtsModemRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ModemRemoved& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleModemRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getModem());
}

void ConnectionControl::handleIpc2BtsGetModemProperties(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_GetModemProperties& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleGetModemPropertiesResult(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getModem(), message.getIpcCommonErrorCode());
}

void ConnectionControl::handleIpc2BtsModemInterfacesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_ModemInterfacesUpdate& message)
{
   FW_IF_NULL_PTR_RETURN(_geniviHfpDeviceCapabilities);
   IHfpDeviceCapabilitiesGenivi& geniviHfpDeviceCapabilities = *_geniviHfpDeviceCapabilities;

   geniviHfpDeviceCapabilities.handleModemInterfacesUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, message.getModem(), message.getInterfaces());
}

} //genivi
} //btstackif
