/**
 * @file ConnectionControlTestCommand.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Connection Control Test Command.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of connection control test command functionality.
 */

#include <fcntl.h>
#include <stddef.h>
#include <unistd.h>
#include <cerrno>
#include <climits>
#include <cstring>

#include "ConnectionControlTestCommand.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "SppTestApp.h"

// generic SMs
#include "IBasicControl.h"
#include "IServiceSearch.h"
#include "IProtocolManager.h"
#include "IBasicConfiguration.h"
#include "IPairing.h"
#include "ILocalAdapterModes.h"
#include "ISwitchBluetooth.h"
#include "IDeviceManager.h"

// Genivi SMs
#include "IObexConnectDisconnectGenivi.h"
#include "IPairingGenivi.h"
#include "IObjectPathManagerGenivi.h"

#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#include "FwUtils.h"
#include "FwBluetoothStringUtils.h"
#include "FwFormattedDataPrint.h"
#include "FwTestCommandData.h"
#include "Dispatcher.h"
#include "TraceClasses.h"
#include "FwErrmemPrint.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnectionControlTestCommand.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

ConnectionControlTestCommand::ConnectionControlTestCommand() :
_simulateDbusStartupError(false),
_simulateEvolutionStartupError(false),
_simulateStartPairingDuringIncomingPairing(false),
_simulateStartPairingDuringIncomingPairing2(false),
_simulateStartPairingDuringIncomingPairingMsg(0),
_triggerSppDisconnectDuringRemoteDisconnect(false),
_simulateRemotePairingDuringOngoingConnect(false),
_triggerCancelSearchDuringSearchResult(false),
_triggerLocalSppConnect(false),
_triggerIgnoreSDPProcessReport(false),
_triggerSimulateSDPProcessReport(false),
_triggerIgnoreDeviceConnectedUpdate(false),
_triggerSimulateStackResetDuringApp2BtsProcessing(false),
_triggerBlockRequestConfirmation(0),
_triggerBlockAuthorizeServiceResponse(false),
_addressForDisconnectDuringPairing(),
_controlIf(0),
_serviceSearch(0),
_protocolManager(0),
_basicConfiguration(0),
_pairingIf(0),
_localAdapterModesIf(0),
_switchBluetoothIf(0),
_deviceManagerIf(0),
_geniviObexConnectDisconnect(0),
_geniviPairingIf(0),
_geniviObjectPathManagerIf(0)
{
}

ConnectionControlTestCommand::~ConnectionControlTestCommand()
{
   _simulateStartPairingDuringIncomingPairingMsg = 0;
   _controlIf = 0;
   _serviceSearch = 0;
   _protocolManager = 0;
   _basicConfiguration = 0;
   _pairingIf = 0;
   _localAdapterModesIf = 0;
   _switchBluetoothIf = 0;
   _deviceManagerIf = 0;
   _geniviObexConnectDisconnect = 0;
   _geniviPairingIf = 0;
   _geniviObjectPathManagerIf = 0;
}

bool ConnectionControlTestCommand::handleTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   FW_IF_NULL_PTR_RETURN_FALSE(testCommand);

   bool handled(true);

   // no BTS_TTFIS_COMMAND

   if(0 == strcmp(BTS_SIMULATE_STARTPAIRING_DURING_INCOMINGPAIRING, testCommand))
   {
      if(1 == testData)
      {
         _simulateStartPairingDuringIncomingPairing = true;
      }
      else if(0 == testData)
      {
         _simulateStartPairingDuringIncomingPairing = false;
      }
   }
   else if(0 == strcmp(BTS_SIMULATE_STARTPAIRING_DURING_INCOMINGPAIRING_2, testCommand))
   {
      if(1 == testData)
      {
         _simulateStartPairingDuringIncomingPairing2 = true;
      }
      else if(0 == testData)
      {
         _simulateStartPairingDuringIncomingPairing2 = false;
      }
   }
   else if(0 == strcmp(BTS_SIMULATE_REMOTE_PAIRING_DURING_CONNECTING, testCommand))
   {
      if(1 == testData)
      {
         _simulateRemotePairingDuringOngoingConnect = true;
      }
      else if(0 == testData)
      {
         _simulateRemotePairingDuringOngoingConnect = false;
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_CANCEL_SEARCH_DURING_SEARCH_RESULT, testCommand))
   {
      if(1 == testData)
      {
         _triggerCancelSearchDuringSearchResult = true;
      }
      else if(0 == testData)
      {
         _triggerCancelSearchDuringSearchResult = false;
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_DISCONNECT_WHILE_DISCONNECTED, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerDisconnectWhileDisconnected((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_DISCONNECT_WHILE_DISCONNECTED_SET_STATE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerDisconnectWhileDisconnectedSetState((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_CANCEL_DURING_CONNECT_RESULT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTriggerCancelDuringConnectResult((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_DISABLE_SEARCHING, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestDisableSearching((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_DISABLE_SETTING, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestDisableSetting((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_BLOCK_SENDING_DISCONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestBlockSendingDisconnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_LOCAL_CONNECT_WHILE_REMOTE_CONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerLocalConnectWhileRemoteConnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_LOCAL_DISCONNECT_WHILE_REMOTE_CONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerLocalDisconnectWhileRemoteConnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_LOCAL_CONNECT_BEFORE_REMOTE_CONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerLocalConnectBeforeRemoteConnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_LOCAL_CONNECTED_BEFORE_REMOTE_CONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerLocalConnectedBeforeRemoteConnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_LOCAL_CONNECT_SAME_DEVICE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerLocalConnectSameDevice((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_PROTOCOL_MANAGER_TEST_ACTION, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
      IProtocolManager& protocolManager = *_protocolManager;

      protocolManager.executeTestAction(testData);
   }
   else if(0 == strcmp(BTS_SET_RETRY_MAX_COUNTER, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
      IProtocolManager& protocolManager = *_protocolManager;

      protocolManager.setTestMaxRetryNumber(testData);
   }
   else if(0 == strcmp(BTS_SET_RETRY_TIMEOUT, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
      IProtocolManager& protocolManager = *_protocolManager;

      protocolManager.setTestRetryTimeout(testData);
   }
   else if(0 == strcmp(BTS_TRIGGER_CANCEL_DURING_ACCEPT_REJECT_REMOTE_CONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTriggerCancelDuringAcceptRejectRemoteConnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_IGNORE_OBEX_PROTOCOL_ADDED, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_geniviObexConnectDisconnect);
         IObexConnectDisconnectGenivi& geniviObexConnectDisconnect = *_geniviObexConnectDisconnect;

         geniviObexConnectDisconnect.setTriggerIgnoreProtocolAdded((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_BASIC_CONF_TRIGGER_IGNORE_UPDATE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_basicConfiguration);
         IBasicConfiguration& basicConfiguration = *_basicConfiguration;

         basicConfiguration.setTestTriggerIgnoreUpdate((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_IGNORE_REPORTED_LINK_KEY, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_pairingIf);
         IPairing& pairingIf = *_pairingIf;

         pairingIf.setTestTriggerIgnoreReportedLinkKey((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_IGNORE_SDP_PROCESS_REPORT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         _triggerIgnoreSDPProcessReport = (bool)testData;
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_SIMULATE_SDP_PROCESS_REPORT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         _triggerSimulateSDPProcessReport = (bool)testData;
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_BLOCK_SET_DISCOVERABLE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_localAdapterModesIf);
         ILocalAdapterModes& localAdapterModesIf = *_localAdapterModesIf;

         localAdapterModesIf.setTestTriggerBlockSetDiscoverable((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_IGNORE_DEVICE_CONNECTED_UPDATE, testCommand))
   {
      if(1 == testData)
      {
         _triggerIgnoreDeviceConnectedUpdate = true;
      }
      else if(0 == testData)
      {
         _triggerIgnoreDeviceConnectedUpdate = false;
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_SIMULATE_STACK_RESET, testCommand))
   {
      if((0 == testData) || (1 == testData) || (2 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);

         Ipc2Bts_ServiceAvailabilityConnection* msg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();
         if(0 != msg)
         {
            ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setInterface(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
            msg->setAvailabilityEvent((BTSDbusServiceAvailability)testData);
            msg->setBusType((BTSCommonEnumClass)evoParser.getInterface2BusType(::ccdbusif::evolution::IF_ADAPTER));
            msg->setBusName(evoParser.getInterface2BusName(::ccdbusif::evolution::IF_ADAPTER));
            msg->setObjPath(evoParser.getRootObjectPath());

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_SIMULATE_STACK_RESET_DURING_PAIRING, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_geniviPairingIf);
         IPairingGenivi& geniviPairingIf = *_geniviPairingIf;

         geniviPairingIf.setTriggerSimulateStackReset((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_SIMULATE_STACK_RESET_DURING_APP2BTS, testCommand))
   {
      if(1 == testData)
      {
         _triggerSimulateStackResetDuringApp2BtsProcessing = true;
      }
      else if(0 == testData)
      {
         _triggerSimulateStackResetDuringApp2BtsProcessing = false;
      }
   }
   else if(0 == strcmp(BTS_SET_TEST_TRIGGER_CANCEL_SERVICE_SEARCH, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
         IServiceSearch& serviceSearch = *_serviceSearch;

         serviceSearch.setTestTriggerCancelServiceSearch((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_START_REMOTE_SERVICE_SEARCH, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IServiceSearch& serviceSearch = *_serviceSearch;
      IBasicControl& controlIf = *_controlIf;

      ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
      ::std::vector< Bts2App_BaseMessage* > bts2AppMsgList;
      App2Bts_StartRemoteServiceSearch msg;
      msg.setBDAddress("0009dd508fc4");
      msg.setSearchType(BTS_SEARCH_ALL);
      serviceSearch.startSearch(bts2IpcMsgList, bts2AppMsgList, msg);
      BTSApp2BtsMessageCompareItem compareItem;
      msg.getCompareItem(compareItem);
      controlIf.sendBts2IpcMessageList(bts2IpcMsgList, compareItem);
   }
   else if(0 == strcmp(BTS_GET_DEVICE_VENDOR_ID, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
      IServiceSearch& serviceSearch = *_serviceSearch;
      serviceSearch.handleTestCommandGetDeviceVendorId();
   }
   else if(0 == strcmp(BTS_GET_ALL_SEARCH_RESULTS, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
      IServiceSearch& serviceSearch = *_serviceSearch;
      serviceSearch.handleTestCommandGetAllSearchResults();
   }
   else if(0 == strcmp(BTS_TRIGGER_DISCOVER_SERVICES_BUSY, testCommand))
   {
      BTSIpcCommonErrorCode error = convertErrorCode2InternalValue("org.bluez.Error.Failed", "Input/output error");
      ETG_TRACE_USR1((" BTS_TRIGGER_DISCOVER_SERVICES_BUSY: error=%d", error));
   }
   else if(0 == strcmp(BTS_TRIGGER_CONNECT_RESULT_BUSY, testCommand))
   {
      BTSIpcCommonErrorCode error = convertErrorCode2InternalValue("org.bluez.Error.InProgress", "In Progress");
      ETG_TRACE_USR1((" BTS_TRIGGER_CONNECT_RESULT_BUSY: error=%d", error));
   }
   else if(0 == strcmp(BTS_START_DUN_CONNECT, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IProtocolManager& protocolManager = *_protocolManager;
      IBasicControl& controlIf = *_controlIf;

      ::std::vector< Bts2Ipc_BaseMessage* > bts2IpcMsgList;
      ::std::vector< Bts2App_BaseMessage* > bts2AppMsgList;
      App2Bts_ConnectProtocol msg;
      msg.setBDAddress("0009dd508fc4");
      msg.setProtocolId(BTS_PROTO_DUN);
      protocolManager.connectProtocol(bts2IpcMsgList, bts2AppMsgList, msg);
      BTSApp2BtsMessageCompareItem compareItem;
      msg.getCompareItem(compareItem);
      controlIf.sendBts2IpcMessageList(bts2IpcMsgList, compareItem);
   }
   else if(0 == strcmp(BTS_SET_PAN_CONNECT_MODE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setPanConnectionMode((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_SET_TEST_TRIGGER_DEVICE_CONNECTED_UPDATE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerDeviceConnectedUpdate((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_SET_TEST_TRIGGER_REMOTE_CONNECT_DURING_DEV_DISCONNECT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setTestTriggerRemoteConnectDuringDeviceDisconnect((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_SET_SIMULATE_FAILED_DISCONNECT_RESULT, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         FW_IF_NULL_PTR_RETURN_TRUE(_protocolManager);
         IProtocolManager& protocolManager = *_protocolManager;

         protocolManager.setSimulateFailedDisconnectResult((bool)testData);
      }
   }
   else if(0 == strcmp(BTS_BLOCK_REQUEST_CONFIRMATION, testCommand))
   {
      if(2 >= testData)
      {
         _triggerBlockRequestConfirmation = testData;
      }
   }
   else if(0 == strcmp(BTS_BLOCK_AUTHORIZE_SERVICE_RESPONSE, testCommand))
   {
      if((0 == testData) || (1 == testData))
      {
         _triggerBlockAuthorizeServiceResponse = (bool)testData;
      }
   }
   else if(0 == strcmp(BTS_SET_AGENT_CAPABILITY, testCommand))
   {
      if((const unsigned int)BTS_AGENT_CAPABILITY_LAST > testData)
      {
         // not implemented
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_INVALID_FW_DISPATCHER_CALL, testCommand))
   {
      Dispatcher::GetInstance().SendMessage(0);
      Dispatcher::GetInstance().SendMessage(0, 0);
      Dispatcher::GetInstance().SendMessageAnswer(0, 0);
      Dispatcher::GetInstance().SendMessageAnswer(0, 0, 0);
      Dispatcher::GetInstance().SendMessageAnswerLocal(0, 0);
   }
   else
   {
      handled = false;
   }

   return handled;
}

bool ConnectionControlTestCommand::handleTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   FW_IF_NULL_PTR_RETURN_FALSE(testCommand);
   FW_IF_NULL_PTR_RETURN_FALSE(testData);

   bool handled(true);

   if(0 == strcmp(BTS_TTFIS_COMMAND, testCommand))
   {
      // TTFis command received
      ETG_TRACE_USR1((" handleTestCommand(): TTFis command"));

      ::fw::TestCommandData cmd;

      // use 1 byte as command id
      const unsigned char opcode(cmd.getU8Value(testData));
      // data starts with testData[1]
      const unsigned char* data(&testData[1]);

      switch(opcode)
      {
         case 0x00:
         {
            // reserved
            break;
         }
         case 0x01:
         {
            BTSLocalMode mode = (BTSLocalMode)cmd.getU32Value(&testData[1]);
            if(BTS_MODE_LAST > mode)
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
               IBasicControl& controlIf = *_controlIf;

               App2Bts_SetLocalAdapterModes* ptrMsg = ptrNew_App2Bts_SetLocalAdapterModes();
               if(0 != ptrMsg)
               {
                  ptrMsg->setDiscoverableMode(mode);
                  ptrMsg->setDiscoverableTimeout(0);
                  ptrMsg->setConnectableMode(BTS_MODE_UNCHANGED);
                  ptrMsg->setConnectableTimeout(0);
                  ptrMsg->setInternalMessageFlag(true);

                  controlIf.sendInternalApp2BtsMessage(ptrMsg);
               }
            }
            break;
         }
         case 0x02:
         {
            BTSDeviceName name = cmd.getStringValue(&testData[1], 35);
            if(false == name.empty())
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
               IBasicControl& controlIf = *_controlIf;

               App2Bts_SetLocalBtName* ptrMsg = ptrNew_App2Bts_SetLocalBtName();
               if(0 != ptrMsg)
               {
                  ptrMsg->setDeviceName(name);
                  ptrMsg->setInternalMessageFlag(true);

                  controlIf.sendInternalApp2BtsMessage(ptrMsg);
               }
            }
            break;
         }
         case 0x03:
         {
            unsigned int tmpData = cmd.getU32Value(&testData[1]);
            if((0 == tmpData) || (1 == tmpData))
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
               IServiceSearch& serviceSearch = *_serviceSearch;

               serviceSearch.setSppSupportSimulation((bool)tmpData);
            }
            break;
         }
         case 0x04:
         {
            ::std::string filter(cmd.getStringValue(&testData[1], 35));
            listCharacterDevices(filter);
            break;
         }
         case 0x05:
         {
            ::std::string device(cmd.getStringValue(&testData[1], 35));
            unsigned int loops(cmd.getU32Value(&testData[36]));
            unsigned int delay(cmd.getU32Value(&testData[40]));
            SppTestApp::getInstance().openDevice(device, loops, delay);
            break;
         }
         case 0x06:
         {
            SppTestApp::getInstance().closeDevice();
            break;
         }
         case 0x07:
         {
            FW_IF_NULL_PTR_RETURN_TRUE(_switchBluetoothIf);
            ISwitchBluetooth& switchBluetoothIf = *_switchBluetoothIf;

            switchBluetoothIf.readRegistryData();
            break;
         }
         case 0x08:
         {
            unsigned int data = cmd.getU32Value(&testData[1]);
            if((0 == data) || (1 == data))
            {
               SppTestApp::getInstance().enableSending((bool)data);
            }
            break;
         }
         case 0x09:
         {
            BTSBDAddress address = cmd.getStringValue(&testData[1], 13);
            if(false == address.empty())
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
               IBasicControl& controlIf = *_controlIf;

               App2Bts_ConnectProtocol* ptrMsg = ptrNew_App2Bts_ConnectProtocol();
               if(0 != ptrMsg)
               {
                  ptrMsg->setBDAddress(address);
                  ptrMsg->setProtocolId(BTS_PROTO_PAN);
                  ptrMsg->setInternalMessageFlag(true);
                  controlIf.sendInternalApp2BtsMessage(ptrMsg);
               }
            }
            break;
         }
         case 0x0A:
         {
            BTSBDAddress address = cmd.getStringValue(&testData[1], 13);
            if(false == address.empty())
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
               IBasicControl& controlIf = *_controlIf;

               App2Bts_DisconnectProtocol* ptrMsg = ptrNew_App2Bts_DisconnectProtocol();
               if(0 != ptrMsg)
               {
                  ptrMsg->setBDAddress(address);
                  ptrMsg->setProtocolId(BTS_PROTO_PAN);
                  ptrMsg->setInternalMessageFlag(true);
                  controlIf.sendInternalApp2BtsMessage(ptrMsg);
               }
            }
            break;
         }
         case 0x0B:
         {
            unsigned int tmpData = cmd.getU32Value(&testData[1]);
            if((0 == tmpData) || (1 == tmpData))
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
               IServiceSearch& serviceSearch = *_serviceSearch;

               serviceSearch.setIgnoreServiceSearchData((bool)tmpData);
            }
            break;
         }
         case 0x0C:
         {
            const unsigned int enable(cmd.getU32Value(&data[0]));

            if((0 == enable) || (1 == enable))
            {
               _simulateDbusStartupError = (bool)enable;
            }
            break;
         }
         case 0x0D:
         {
            const unsigned int enable(cmd.getU32Value(&data[0]));

            if((0 == enable) || (1 == enable))
            {
               _simulateEvolutionStartupError = (bool)enable;
            }
            break;
         }
         case 0x0E:
         {
            const ::std::string interface(cmd.getStringValue(&data[0], 51));
            ::std::vector<unsigned char> macAddress;
            ::std::string macAddressString;

            const bool result = ::fw::getNetworkMacAddress(macAddress, macAddressString, interface);

            ETG_TRACE_USR1((" result=%d interface=%50s address=%s", result, interface.c_str(), macAddressString.c_str()));
            break;
         }
         case 0x0F:
         {
            ::fw::FormattedOutputU8 outU8dec1(0);
            ::fw::FormattedOutputU8 outU8hex1(0, true);
            ::fw::FormattedOutputU8 outU8dec2(UCHAR_MAX);
            ::fw::FormattedOutputU8 outU8hex2(UCHAR_MAX, true);

            ::fw::FormattedOutputS64 outS64dec1(LLONG_MIN);
            ::fw::FormattedOutputS64 outS64hex1(LLONG_MIN, true);
            ::fw::FormattedOutputS64 outS64dec2(0);
            ::fw::FormattedOutputS64 outS64hex2(0, true);
            ::fw::FormattedOutputS64 outS64dec3(LLONG_MAX);
            ::fw::FormattedOutputS64 outS64hex3(LLONG_MAX, true);

            ::fw::FormattedOutputFloat outFloatfix1(3.14159265358979323846f);
            ::fw::FormattedOutputFloat outFloatsce1(3.14159265358979323846f, true);
            ::fw::FormattedOutputDouble outDoublefix1(3.14159265358979323846);
            ::fw::FormattedOutputDouble outDoublesce1(3.14159265358979323846, true);
            ::fw::FormattedOutputFloat outFloatfix2(443.14159265358979323846f);
            ::fw::FormattedOutputFloat outFloatsce2(443.14159265358979323846f, true);
            ::fw::FormattedOutputDouble outDoublefix2(443.14159265358979323846);
            ::fw::FormattedOutputDouble outDoublesce2(443.14159265358979323846, true);
            ::fw::FormattedOutputDouble outDoublefix3(-4645.44666466);
            ::fw::FormattedOutputDouble outDoublesce3(-4645.44666466, true);

            ETG_TRACE_USR1((" outU8dec1      =%s", outU8dec1.c_str()));
            ETG_TRACE_USR1((" outU8hex1      =%s", outU8hex1.c_str()));
            ETG_TRACE_USR1((" outU8dec2      =%s", outU8dec2.c_str()));
            ETG_TRACE_USR1((" outU8hex2      =%s", outU8hex2.c_str()));

            ETG_TRACE_USR1((" outS64dec1     =%s", outS64dec1.c_str()));
            ETG_TRACE_USR1((" outS64hex1     =%s", outS64hex1.c_str()));
            ETG_TRACE_USR1((" outS64dec2     =%s", outS64dec2.c_str()));
            ETG_TRACE_USR1((" outS64hex2     =%s", outS64hex2.c_str()));
            ETG_TRACE_USR1((" outS64dec3     =%s", outS64dec3.c_str()));
            ETG_TRACE_USR1((" outS64hex3     =%s", outS64hex3.c_str()));

            ETG_TRACE_USR1((" outFloatfix1   =%s", outFloatfix1.c_str()));
            ETG_TRACE_USR1((" outFloatsce1   =%s", outFloatsce1.c_str()));
            ETG_TRACE_USR1((" outDoublefix1  =%s", outDoublefix1.c_str()));
            ETG_TRACE_USR1((" outDoublesce1  =%s", outDoublesce1.c_str()));
            ETG_TRACE_USR1((" outFloatfix2   =%s", outFloatfix2.c_str()));
            ETG_TRACE_USR1((" outFloatsce2   =%s", outFloatsce2.c_str()));
            ETG_TRACE_USR1((" outDoublefix2  =%s", outDoublefix2.c_str()));
            ETG_TRACE_USR1((" outDoublesce2  =%s", outDoublesce2.c_str()));
            ETG_TRACE_USR1((" outDoublefix3  =%s", outDoublefix3.c_str()));
            ETG_TRACE_USR1((" outDoublesce3  =%s", outDoublesce3.c_str()));
            break;
         }
         case 0x10:
         {
            const unsigned int enable(cmd.getU32Value(&data[0]));

            if((0 == enable) || (1 == enable))
            {
               _triggerSppDisconnectDuringRemoteDisconnect = (bool)enable;
            }
            break;
         }
         case 0x11:
         {
            unsigned int tmpData = cmd.getU32Value(&testData[1]);
            if((0 == tmpData) || (1 == tmpData))
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
               IServiceSearch& serviceSearch = *_serviceSearch;

               serviceSearch.setPanSupportSimulation((bool)tmpData);
            }
            break;
         }
         case 0x12:
         {
            const unsigned int enable(cmd.getU32Value(&data[0]));

            if((0 == enable) || (1 == enable))
            {
               _triggerLocalSppConnect = (bool)enable;
            }
            break;
         }
         case 0x13:
         {
            unsigned int a = 0;
            const char* b = 0;
            FW_ERRMEM_ASSERT(0 != a);
            FW_ERRMEM_ASSERT(0 != b);

            a = 1;
            b = "";
            FW_ERRMEM_ASSERT(0 != a);
            FW_ERRMEM_ASSERT(0 != b);
            break;
         }
         case 0x14:
         {
            // unused
            break;
         }
         case 0x15:
         {
            unsigned int tmpData = cmd.getU32Value(&testData[1]);
            if((0 == tmpData) || (1 == tmpData))
            {
               FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
               IServiceSearch& serviceSearch = *_serviceSearch;

               serviceSearch.setIgnoreSppUuids((bool)tmpData);
            }
            break;
         }
         case 0x16:
         {
            BTSBDAddress deviceAddress;
            getRandomBDAddress(deviceAddress);
            break;
         }
         case 0x17:
         {
            testFunctionIsValid128BitUuid();
            break;
         }
         case 0x18:
         {
            testFunctionIsValidBdAddress();
            break;
         }
         case 0x19:
         {
            const ::std::string tmpCommand(cmd.getStringValue(&data[0], 200));
            const unsigned int tmpData(cmd.getU32Value(&data[200]));
            handleTestCommand(tmpCommand.c_str(), tmpData);
            break;
         }
         case 0x1A:
         {
            testFunctionStackError(cmd.getU32Value(&data[0]));
            break;
         }
         //---------------------------------------------------------------------
         case 0xF0:
         {
            handleApp2BtsTestCommand(&testData[1]);
            break;
         }
         //---------------------------------------------------------------------
         default:
         {
            handled = false;
            break;
         }
      }
   }
   else if(0 == strcmp(BTS_TRIGGER_INTERNAL_SDP_SEARCH, testCommand))
   {
      ::fw::TestCommandData cmd;
      const BTSBDAddress address(cmd.getStringValue(&testData[0], 12));
      const BTSSearchType searchType((BTSSearchType)cmd.getU8Value(&testData[12]));

      FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
      IServiceSearch& serviceSearch = *_serviceSearch;

      serviceSearch.triggerSdpSearchRequest(address, searchType, false);
   }
   else if(0 == strcmp(BTS_TRIGGER_SDP_SEARCH_AFTER_PAIRING, testCommand))
   {
      ::fw::TestCommandData cmd;
      const BTSBDAddress address(cmd.getStringValue(&testData[0], 12));

      FW_IF_NULL_PTR_RETURN_TRUE(_serviceSearch);
      IServiceSearch& serviceSearch = *_serviceSearch;

      serviceSearch.startTestSdpSearchAfterPairing(address);
   }
   else if(0 == strcmp(BTS_SEND_DISCONNECT_DURING_PAIRING, testCommand))
   {
      ::fw::TestCommandData cmd;
      const BTSBDAddress address(cmd.getStringValue(&testData[0], 12));

      if(true == ::fw::isValidBdAddress(address))
      {
         _addressForDisconnectDuringPairing = address;
      }
   }
   else if(0 == strcmp(BTS_LIST_CHARACTER_DEVICES, testCommand))
   {
      const ::std::string filter((const char*)testData);
      listCharacterDevices(filter);
   }
   else if(0 == strcmp(BTS_OPEN_SPP_TEST_APP, testCommand))
   {
      const ::std::string device((const char*)testData);
      SppTestApp::getInstance().openDevice(device);
   }
   else if(0 == strcmp(BTS_CLOSE_SPP_TEST_APP, testCommand))
   {
      SppTestApp::getInstance().closeDevice();
   }
   else if(0 == strcmp(BTS_SET_SPP_RAW_MODE, testCommand))
   {
      const ::std::string data((const char*)testData);
      SppTestApp::getInstance().enableRawMode(('1' == data[0]));
   }
   else if(0 == strcmp(BTS_SIMULATE_OBEX_OBJECT_PUSH_AUTHORIZE, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IBasicControl& controlIf = *_controlIf;

      Ipc2Bts_AuthorizeService* ptrMsg = ptrNew_Ipc2Bts_AuthorizeService();
      if(0 != ptrMsg)
      {
         ptrMsg->setDevice("/org/bluez/hci1/dev_00_09_DD_50_8F_C4");
         ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;
         ptrMsg->setProtocol(convertUuid2Protocol(evoParser.getUuid2Enum("00001105-0000-1000-8000-00805f9b34fb")));
         ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
         ptrMsg->setDbusToken(1);
         controlIf.sendInternalIpc2BtsMessage(ptrMsg);
      }
   }
   else if(0 == strcmp(BTS_SIMULATE_HFP_AUTHORIZE, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IBasicControl& controlIf = *_controlIf;

      Ipc2Bts_AuthorizeService* ptrMsg = ptrNew_Ipc2Bts_AuthorizeService();
      if(0 != ptrMsg)
      {
         ptrMsg->setDevice("/org/bluez/hci1/dev_00_09_DD_50_8F_C4");
         ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;
         ptrMsg->setProtocol(convertUuid2Protocol(evoParser.getUuid2Enum("0000111e-0000-1000-8000-00805f9b34fb")));
         ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
         ptrMsg->setDbusToken(1);
         controlIf.sendInternalIpc2BtsMessage(ptrMsg);
      }
   }
   else if(0 == strcmp(BTS_SIMULATE_AVP_AUTHORIZE, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IBasicControl& controlIf = *_controlIf;

      Ipc2Bts_AuthorizeService* ptrMsg = ptrNew_Ipc2Bts_AuthorizeService();
      if(0 != ptrMsg)
      {
         ptrMsg->setDevice("/org/bluez/hci1/dev_00_09_DD_50_8F_C4");
         ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;
         ptrMsg->setProtocol(convertUuid2Protocol(evoParser.getUuid2Enum("0000110e-0000-1000-8000-00805f9b34fb")));
         ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
         ptrMsg->setDbusToken(1);
         controlIf.sendInternalIpc2BtsMessage(ptrMsg);
      }
   }
   else if(0 == strcmp(BTS_SIMULATE_SERVICE_STATE, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IBasicControl& controlIf = *_controlIf;

      Ipc2Bts_ServiceStateUpdate* ptrMsg = ptrNew_Ipc2Bts_ServiceStateUpdate();
      if(0 != ptrMsg)
      {
         BTSObjectPath service;
         BTSServiceState serviceState;
         const char* data = (const char*)testData;
         if('H' == data[0])
         {
            service = "/org/bluez/hci1/dev_00_09_DD_50_8F_C4/0000111f_0000_1000_8000_00805f9b34fb";
         }
         else
         {
            service = "/org/bluez/hci1/dev_00_09_DD_50_8F_C4/0000110a_0000_1000_8000_00805f9b34fb";
         }
         ++data;
         if(0 == strcmp("CONNECTING", data))
         {
            serviceState = BTS_SERVICE_STATE_CONNECTING;
         }
         else if(0 == strcmp("CONNECTED", data))
         {
            serviceState = BTS_SERVICE_STATE_CONNECTED;
         }
         else if(0 == strcmp("DISCONNECTING", data))
         {
            serviceState = BTS_SERVICE_STATE_DISCONNECTING;
         }
         else if(0 == strcmp("DISCONNECTED", data))
         {
            serviceState = BTS_SERVICE_STATE_DISCONNECTED;
         }
         else
         {
            serviceState = BTS_SERVICE_STATE_ERROR;
         }
         ptrMsg->setService(service);
         ptrMsg->setServiceState(serviceState);
         ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
         controlIf.sendInternalIpc2BtsMessage(ptrMsg);
      }
   }
   else if(0 == strcmp(BTS_CREATE_DEVICE, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      IBasicControl& controlIf = *_controlIf;

      const BTSBDAddress address((const char*)testData);
      Bts2Ipc_CreateDevice* ptrMsg = ptrNew_Bts2Ipc_CreateDevice();
      if(0 != ptrMsg)
      {
         ptrMsg->setBDAddress(address);
         controlIf.sendInternalBts2IpcMessage(ptrMsg);
      }
   }
   else if(0 == strcmp(BTS_START_CONNECT_PIM, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      FW_IF_NULL_PTR_RETURN_TRUE(_switchBluetoothIf);
      IBasicControl& controlIf = *_controlIf;
      ISwitchBluetooth& switchBluetoothIf = *_switchBluetoothIf;

      const BTSBDAddress address((const char*)testData);
      Bts2Ipc_CreateSession* ptrMsg = ptrNew_Bts2Ipc_CreateSession();
      if(0 != ptrMsg)
      {
         ptrMsg->setBDAddress(address);
         ptrMsg->setSourceAddress(switchBluetoothIf.getLocalAddress());
         ptrMsg->setProtocolId(BTS_PROTO_PIM);
         controlIf.sendInternalBts2IpcMessage(ptrMsg);
      }
   }
   else if(0 == strcmp(BTS_REMOVE_ALL_DEVICES, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      FW_IF_NULL_PTR_RETURN_TRUE(_deviceManagerIf);
      IBasicControl& controlIf = *_controlIf;
      IDeviceManager& deviceManagerIf = *_deviceManagerIf;

      ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
      ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
      deviceManagerIf.removeAllDevices(tmpBts2IpcMsgList, tmpBts2AppMsgList);
      FW_NORMAL_ASSERT(0 == tmpBts2AppMsgList.size());
      for(size_t i = 0; i < tmpBts2IpcMsgList.size(); i++)
      {
         controlIf.sendInternalBts2IpcMessage(tmpBts2IpcMsgList[i]);
      }
   }
   else if(0 == strcmp(BTS_START_REMOTE_SERVICE_SEARCH_2, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      FW_IF_NULL_PTR_RETURN_TRUE(_geniviObjectPathManagerIf);
      IBasicControl& controlIf = *_controlIf;
      IObjectPathManagerGenivi& geniviObjectPathManagerIf = *_geniviObjectPathManagerIf;

      const BTSBDAddress address((const char*)testData);
      BTSObjectPath objPath;
      if(true == geniviObjectPathManagerIf.getObjectPath4Address(objPath, address))
      {
         Bts2Ipc_DiscoverServices* ptrMsg = ptrNew_Bts2Ipc_DiscoverServices();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(address);
            ptrMsg->setSearchType(BTS_SEARCH_ALL);
            ptrMsg->setDevice(objPath);
            controlIf.sendInternalBts2IpcMessage(ptrMsg);
         }
      }
   }
   else if(0 == strcmp(BTS_START_CONNECT_AVP, testCommand))
   {
      FW_IF_NULL_PTR_RETURN_TRUE(_controlIf);
      FW_IF_NULL_PTR_RETURN_TRUE(_geniviObjectPathManagerIf);
      IBasicControl& controlIf = *_controlIf;
      IObjectPathManagerGenivi& geniviObjectPathManagerIf = *_geniviObjectPathManagerIf;

      const BTSBDAddress address((const char*)testData);
      BTSObjectPath objPath;
      if(true == geniviObjectPathManagerIf.getObjectPath4Protocol(objPath, address, BTS_PROTO_AVP))
      {
         Bts2Ipc_Connect* ptrMsg = ptrNew_Bts2Ipc_Connect();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(address);
            ptrMsg->setProtocol(BTS_PROTO_AVP);
            ptrMsg->setService(objPath);
            controlIf.sendInternalBts2IpcMessage(ptrMsg);
         }
      }
   }
   else
   {
      handled = false;
   }

   return handled;
}

void ConnectionControlTestCommand::getRandomBDAddress(OUT BTSBDAddress& deviceAddress) const
{
   const char lowerAscii[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
   const unsigned int bufferSize = 12; // size of Bluetooth address
   const char* randomDevice = "/dev/urandom"; // /dev/urandom will never block
   const int randomDeviceDescriptor = open(randomDevice, O_RDONLY);

   if(0 > randomDeviceDescriptor)
   {
      // open failed
      FW_ERRMEM_ASSERT_ALWAYS();

      const int errNumber(errno);
      ETG_TRACE_USR1((" getRandomBDAddress(): open failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
   }
   else
   {
      char randomData[bufferSize];
      const ssize_t result = read(randomDeviceDescriptor, randomData, sizeof(randomData));

      if(0 > result)
      {
         // read failed
         FW_ERRMEM_ASSERT_ALWAYS();

         const int errNumber(errno);
         ETG_TRACE_USR1((" getRandomBDAddress(): read failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      }
      else
      {
         char buffer[1 + bufferSize];

         for(size_t i = 0; i < (sizeof(buffer) - 1); i++)
         {
            const unsigned int index = (unsigned int)randomData[i];
            buffer[i] = lowerAscii[index % 16];
         }

         buffer[bufferSize] = '\0';

         deviceAddress = buffer;

         ETG_TRACE_USR1((" getRandomBDAddress(): deviceAddress=%s", deviceAddress.c_str()));
      }

      // close the device
      (void)close(randomDeviceDescriptor);
   }
}

void ConnectionControlTestCommand::testFunctionIsValid128BitUuid(void) const
{
   const BTSUuid uuid00("");
   const BTSUuid uuid01("00000001-0000-1000-8000-00805f9b34fb");
   const BTSUuid uuid02("00000001-0000-1000-8000-00805F9B34FB");
   const BTSUuid uuid03("0000000100001000800000805f9b34fb");
   const BTSUuid uuid04("0000000100001000800000805F9B34FB");

   const BTSUuid uuid05("00000001-0000-1000-8000-00805f9b34fb0");
   const BTSUuid uuid06("00000001-0000-1000-8000-00805F9B34FB0");
   const BTSUuid uuid07("0000000100001000800000805f9b34fb0");
   const BTSUuid uuid08("0000000100001000800000805F9B34FB0");

   const BTSUuid uuid09("00000001-0000-1000-8000-00805f9b34f");
   const BTSUuid uuid10("00000001-0000-1000-8000-00805F9B34F");
   const BTSUuid uuid11("0000000100001000800000805f9b34f");
   const BTSUuid uuid12("0000000100001000800000805F9B34F");

   const BTSUuid uuid13("00000001-0000-1000-8000-00805z9b34fb");
   const BTSUuid uuid14("00000001-0000-1000-8000-00805Z9B34FB");
   const BTSUuid uuid15("0000000100001000800000805z9b34fb");
   const BTSUuid uuid16("0000000100001000800000805Z9B34FB");

   const BTSUuid uuid17("0000000100000-1000-8000-00805f9b34fb");
   const BTSUuid uuid18("0000000100000-1000-8000-00805F9B34FB");

   const BTSUuid uuid19("-0000001-0000-1000-8000-00805f9b34fb");
   const BTSUuid uuid20("-0000001-0000-1000-8000-00805F9B34FB");

   const BTSUuid uuid21("0000-00100001000800000805f9b34fb");
   const BTSUuid uuid22("0000-00100001000800000805F9B34FB");

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid00.c_str(), ::fw::isValid128BitUuid(uuid00))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=1]", uuid00.c_str(), ::fw::isValid128BitUuid(uuid00, false, true))); // expected: 1

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid01.c_str(), ::fw::isValid128BitUuid(uuid01))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=1]", uuid01.c_str(), ::fw::isValid128BitUuid(uuid01, true))); // expected: 1

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid02.c_str(), ::fw::isValid128BitUuid(uuid02))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=1]", uuid02.c_str(), ::fw::isValid128BitUuid(uuid02, true))); // expected: 1

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=1]", uuid03.c_str(), ::fw::isValid128BitUuid(uuid03))); // expected: 1
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid03.c_str(), ::fw::isValid128BitUuid(uuid03, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=1]", uuid04.c_str(), ::fw::isValid128BitUuid(uuid04))); // expected: 1
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid04.c_str(), ::fw::isValid128BitUuid(uuid04, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid05.c_str(), ::fw::isValid128BitUuid(uuid05))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid05.c_str(), ::fw::isValid128BitUuid(uuid05, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid06.c_str(), ::fw::isValid128BitUuid(uuid06))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid06.c_str(), ::fw::isValid128BitUuid(uuid06, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid07.c_str(), ::fw::isValid128BitUuid(uuid07))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid07.c_str(), ::fw::isValid128BitUuid(uuid07, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid08.c_str(), ::fw::isValid128BitUuid(uuid08))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid08.c_str(), ::fw::isValid128BitUuid(uuid08, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid09.c_str(), ::fw::isValid128BitUuid(uuid09))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid09.c_str(), ::fw::isValid128BitUuid(uuid09, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid10.c_str(), ::fw::isValid128BitUuid(uuid10))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid10.c_str(), ::fw::isValid128BitUuid(uuid10, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid11.c_str(), ::fw::isValid128BitUuid(uuid11))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid11.c_str(), ::fw::isValid128BitUuid(uuid11, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid12.c_str(), ::fw::isValid128BitUuid(uuid12))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid12.c_str(), ::fw::isValid128BitUuid(uuid12, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid13.c_str(), ::fw::isValid128BitUuid(uuid13))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid13.c_str(), ::fw::isValid128BitUuid(uuid13, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid14.c_str(), ::fw::isValid128BitUuid(uuid14))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid14.c_str(), ::fw::isValid128BitUuid(uuid14, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid15.c_str(), ::fw::isValid128BitUuid(uuid15))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid15.c_str(), ::fw::isValid128BitUuid(uuid15, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid16.c_str(), ::fw::isValid128BitUuid(uuid16))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid16.c_str(), ::fw::isValid128BitUuid(uuid16, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid17.c_str(), ::fw::isValid128BitUuid(uuid17))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid17.c_str(), ::fw::isValid128BitUuid(uuid17, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid18.c_str(), ::fw::isValid128BitUuid(uuid18))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid18.c_str(), ::fw::isValid128BitUuid(uuid18, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid19.c_str(), ::fw::isValid128BitUuid(uuid19))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid19.c_str(), ::fw::isValid128BitUuid(uuid19, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid20.c_str(), ::fw::isValid128BitUuid(uuid20))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid20.c_str(), ::fw::isValid128BitUuid(uuid20, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid21.c_str(), ::fw::isValid128BitUuid(uuid21))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid21.c_str(), ::fw::isValid128BitUuid(uuid21, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid22.c_str(), ::fw::isValid128BitUuid(uuid22))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValid128BitUuid(): uuid=%100s valid=%d [expected=0]", uuid22.c_str(), ::fw::isValid128BitUuid(uuid22, true))); // expected: 0
}

void ConnectionControlTestCommand::testFunctionIsValidBdAddress(void) const
{
   const BTSUuid address00("");
   const BTSUuid address01("aa:bb:cc:dd:ee:ff");
   const BTSUuid address02("AA:BB:CC:DD:EE:FF");
   const BTSUuid address03("aabbccddeeff");
   const BTSUuid address04("AABBCCDDEEFF");

   const BTSUuid address05("aa:bb:cc:dd:ee:ff0");
   const BTSUuid address06("AA:BB:CC:DD:EE:FF0");
   const BTSUuid address07("aabbccddeeff0");
   const BTSUuid address08("AABBCCDDEEFF0");

   const BTSUuid address09("aa:bb:cc:dd:ee:f");
   const BTSUuid address10("AA:BB:CC:DD:EE:F");
   const BTSUuid address11("aabbccddeef");
   const BTSUuid address12("AABBCCDDEEF");

   const BTSUuid address13("aa:bb:cc:dd:ze:ff");
   const BTSUuid address14("AA:BB:CC:DD:ZE:FF");
   const BTSUuid address15("aabbccddzeff");
   const BTSUuid address16("AABBCCDDZEFF");

   const BTSUuid address17("aa:bb0cc:dd:ee:ff");
   const BTSUuid address18("AA:BB0CC:DD:EE:FF");

   const BTSUuid address19(":a:bb:cc:dd:ee:ff");
   const BTSUuid address20(":A:BB:CC:DD:EE:FF");

   const BTSUuid address21("aab:ccddeeff");
   const BTSUuid address22("AAB:CCDDEEFF");

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address00.c_str(), ::fw::isValidBdAddress(address00))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=1]", address00.c_str(), ::fw::isValidBdAddress(address00, false, true))); // expected: 1

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address01.c_str(), ::fw::isValidBdAddress(address01))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=1]", address01.c_str(), ::fw::isValidBdAddress(address01, true))); // expected: 1

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address02.c_str(), ::fw::isValidBdAddress(address02))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=1]", address02.c_str(), ::fw::isValidBdAddress(address02, true))); // expected: 1

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=1]", address03.c_str(), ::fw::isValidBdAddress(address03))); // expected: 1
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address03.c_str(), ::fw::isValidBdAddress(address03, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=1]", address04.c_str(), ::fw::isValidBdAddress(address04))); // expected: 1
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address04.c_str(), ::fw::isValidBdAddress(address04, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address05.c_str(), ::fw::isValidBdAddress(address05))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address05.c_str(), ::fw::isValidBdAddress(address05, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address06.c_str(), ::fw::isValidBdAddress(address06))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address06.c_str(), ::fw::isValidBdAddress(address06, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address07.c_str(), ::fw::isValidBdAddress(address07))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address07.c_str(), ::fw::isValidBdAddress(address07, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address08.c_str(), ::fw::isValidBdAddress(address08))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address08.c_str(), ::fw::isValidBdAddress(address08, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address09.c_str(), ::fw::isValidBdAddress(address09))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address09.c_str(), ::fw::isValidBdAddress(address09, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address10.c_str(), ::fw::isValidBdAddress(address10))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address10.c_str(), ::fw::isValidBdAddress(address10, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address11.c_str(), ::fw::isValidBdAddress(address11))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address11.c_str(), ::fw::isValidBdAddress(address11, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address12.c_str(), ::fw::isValidBdAddress(address12))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address12.c_str(), ::fw::isValidBdAddress(address12, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address13.c_str(), ::fw::isValidBdAddress(address13))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address13.c_str(), ::fw::isValidBdAddress(address13, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address14.c_str(), ::fw::isValidBdAddress(address14))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address14.c_str(), ::fw::isValidBdAddress(address14, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address15.c_str(), ::fw::isValidBdAddress(address15))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address15.c_str(), ::fw::isValidBdAddress(address15, true))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address16.c_str(), ::fw::isValidBdAddress(address16))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address16.c_str(), ::fw::isValidBdAddress(address16, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address17.c_str(), ::fw::isValidBdAddress(address17))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address17.c_str(), ::fw::isValidBdAddress(address17, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address18.c_str(), ::fw::isValidBdAddress(address18))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address18.c_str(), ::fw::isValidBdAddress(address18, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address19.c_str(), ::fw::isValidBdAddress(address19))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address19.c_str(), ::fw::isValidBdAddress(address19, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address20.c_str(), ::fw::isValidBdAddress(address20))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address20.c_str(), ::fw::isValidBdAddress(address20, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address21.c_str(), ::fw::isValidBdAddress(address21))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address21.c_str(), ::fw::isValidBdAddress(address21, true))); // expected: 0

   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address22.c_str(), ::fw::isValidBdAddress(address22))); // expected: 0
   ETG_TRACE_USR1((" testFunctionIsValidBdAddress(): address=%100s valid=%d [expected=0]", address22.c_str(), ::fw::isValidBdAddress(address22, true))); // expected: 0
}

void ConnectionControlTestCommand::testFunctionStackError(IN const unsigned int cmd) const
{
   FW_IF_NULL_PTR_RETURN(_controlIf);

   /*
    * Ipc2Bts_GeneralError (Trace::GeneralError)
    * Ipc2Bts_FatalError (Trace::FatalError)
    * Ipc2Bts_ReportDebugTrace (Trace::ReportDebugTrace)
    * Ipc2Bts_GeniviFatalError (GeniviTrace::FatalError)
    * Ipc2Bts_ReportGeniviDebugTrace (GeniviTrace::ReportDebugTrace)
    */

   switch(cmd)
   {
      case 0:
      {
         // Trace::GeneralError with opcode = 0xFFFE (stack error) + status != CONNECT_REJECT_SECURITY_REASONS => RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFE (stack error) + status != CONNECT_REJECT_SECURITY_REASONS => RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFE);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 1:
      {
         // Trace::GeneralError with opcode = 0xFFFE (stack error) + status = CONNECT_REJECT_SECURITY_REASONS => NO RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFE (stack error) + status = CONNECT_REJECT_SECURITY_REASONS => NO RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFE);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 2:
      {
         // Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status != CONNECT_REJECT_SECURITY_REASONS => RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status != CONNECT_REJECT_SECURITY_REASONS => RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFF);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 3:
      {
         // Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status = CONNECT_REJECT_SECURITY_REASONS => NO RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status = CONNECT_REJECT_SECURITY_REASONS => NO RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFF);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 4:
      {
         // Trace::GeneralError with opcode = 0xFFFC (invalid ACL packet) + status != CONNECT_REJECT_SECURITY_REASONS => NO RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFC (invalid ACL packet) + status != CONNECT_REJECT_SECURITY_REASONS => NO RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFC);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 5:
      {
         // Trace::GeneralError with opcode = 0xFFFC (invalid ACL packet) + status = CONNECT_REJECT_SECURITY_REASONS => NO RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFC (invalid ACL packet) + status = CONNECT_REJECT_SECURITY_REASONS => NO RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFC);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 6:
      {
         // Trace::GeneralError with opcode = 0xFFFE (stack error) + status != CONNECT_REJECT_SECURITY_REASONS + wrong common error code => NO RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFE (stack error) + status != CONNECT_REJECT_SECURITY_REASONS + wrong common error code => NO RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_ERROR);
            msg->setOpcode(0xFFFE);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_UNKNOWN_OPCODE);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 7:
      {
         // Trace::FatalError with err_location + err_type => RESET
         ETG_TRACE_USR1((" *** Trace::FatalError with err_location + err_type => RESET ***"));
         Ipc2Bts_FatalError* msg = ptrNew_Ipc2Bts_FatalError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setErrLocation(1);
            msg->setErrType(1);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 8:
      {
         // Trace::FatalError with err_location + err_type + wrong common error code => NO RESET
         ETG_TRACE_USR1((" *** Trace::FatalError with err_location + err_type + wrong common error code => NO RESET ***"));
         Ipc2Bts_FatalError* msg = ptrNew_Ipc2Bts_FatalError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_ERROR);
            msg->setErrLocation(1);
            msg->setErrType(1);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 9:
      {
         // Trace::ReportDebugTrace with TRACE_LEVEL_FATAL => WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** Trace::ReportDebugTrace with TRACE_LEVEL_FATAL => WRITE TO ERRMEM ***"));
         Ipc2Bts_ReportDebugTrace* msg = ptrNew_Ipc2Bts_ReportDebugTrace();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setTraceLocation(1234567890);
            msg->setTraceLevel(1);
            msg->setTraceCategory(::btstackif::genivi::BTS_DEBUG_TRACE_CAT_BTCOMMON);
            msg->setErrorCode(54321);
            ::std::vector< unsigned char > tmpData;
            tmpData.reserve(250);
            for(size_t i = 0; i < 250; i++)
            {
               tmpData.push_back((unsigned char)i);
            }
            msg->setTraceLength((::btstackif::genivi::BTSDebugTraceLength)tmpData.size());
            msg->setTraceData(tmpData);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 10:
      {
         // Trace::ReportDebugTrace with TRACE_LEVEL_ERROR => WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** Trace::ReportDebugTrace with TRACE_LEVEL_ERROR => WRITE TO ERRMEM ***"));
         Ipc2Bts_ReportDebugTrace* msg = ptrNew_Ipc2Bts_ReportDebugTrace();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setTraceLocation(1234567890);
            msg->setTraceLevel(2);
            msg->setTraceCategory(::btstackif::genivi::BTS_DEBUG_TRACE_CAT_BTCOMMON);
            msg->setErrorCode(54321);
            ::std::vector< unsigned char > tmpData;
            tmpData.reserve(500);
            for(size_t i = 0; i < 500; i++)
            {
               tmpData.push_back((unsigned char)i);
            }
            msg->setTraceLength((::btstackif::genivi::BTSDebugTraceLength)tmpData.size());
            msg->setTraceData(tmpData);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 11:
      {
         // Trace::ReportDebugTrace with TRACE_LEVEL_WARN => NO WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** Trace::ReportDebugTrace with TRACE_LEVEL_WARN => NO WRITE TO ERRMEM ***"));
         Ipc2Bts_ReportDebugTrace* msg = ptrNew_Ipc2Bts_ReportDebugTrace();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setTraceLocation(1234567890);
            msg->setTraceLevel(4);
            msg->setTraceCategory(::btstackif::genivi::BTS_DEBUG_TRACE_CAT_BTCOMMON);
            msg->setErrorCode(54321);
            ::std::vector< unsigned char > tmpData;
            tmpData.reserve(20);
            for(size_t i = 0; i < 20; i++)
            {
               tmpData.push_back((unsigned char)i);
            }
            msg->setTraceLength((::btstackif::genivi::BTSDebugTraceLength)tmpData.size());
            msg->setTraceData(tmpData);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 12:
      {
         // Trace::ReportDebugTrace with TRACE_LEVEL_WARN + wrong common error code => WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** Trace::ReportDebugTrace with TRACE_LEVEL_WARN + wrong common error code => WRITE TO ERRMEM ***"));
         Ipc2Bts_ReportDebugTrace* msg = ptrNew_Ipc2Bts_ReportDebugTrace();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_ERROR);
            msg->setTraceLocation(1234567890);
            msg->setTraceLevel(4);
            msg->setTraceCategory(::btstackif::genivi::BTS_DEBUG_TRACE_CAT_BTCOMMON);
            msg->setErrorCode(54321);
            ::std::vector< unsigned char > tmpData;
            tmpData.reserve(20);
            for(size_t i = 0; i < 20; i++)
            {
               tmpData.push_back((unsigned char)i);
            }
            msg->setTraceLength((::btstackif::genivi::BTSDebugTraceLength)tmpData.size());
            msg->setTraceData(tmpData);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 13:
      {
         // GeniviTrace::FatalError => NO RESET
         ETG_TRACE_USR1((" *** GeniviTrace::FatalError => NO RESET ***"));
         Ipc2Bts_GeniviFatalError* msg = ptrNew_Ipc2Bts_GeniviFatalError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setErrFile(54321);
            msg->setErrLine(45321);
            msg->setErrCode(21543);
            msg->setFmt("Test Error Code - GeniviTrace::FatalError");
            msg->setErrValue("Test Error Code - 1234567890ABCDEFGH");

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 14:
      {
         // GeniviTrace::FatalError + wrong common error code => NO RESET
         ETG_TRACE_USR1((" *** GeniviTrace::FatalError + wrong common error code => NO RESET ***"));
         Ipc2Bts_GeniviFatalError* msg = ptrNew_Ipc2Bts_GeniviFatalError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_ERROR);
            msg->setErrFile(54321);
            msg->setErrLine(45321);
            msg->setErrCode(21543);
            msg->setFmt("Test Error Code - GeniviTrace::FatalError");
            msg->setErrValue("Test Error Code - 1234567890ABCDEFGH");

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 15:
      {
         // GeniviTrace::ReportDebugTrace => WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** GeniviTrace::ReportDebugTrace => WRITE TO ERRMEM ***"));
         Ipc2Bts_ReportGeniviDebugTrace* msg = ptrNew_Ipc2Bts_ReportGeniviDebugTrace();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setErrFile(54321);
            msg->setErrLine(45321);
            msg->setLevel(21543);
            msg->setCategory(1234567890);
            msg->setErrorCode(31542);
            ::std::vector< unsigned char > tmpData;
            tmpData.reserve(200);
            for(size_t i = 0; i < 200; i++)
            {
               tmpData.push_back((unsigned char)i);
            }
            msg->setData(tmpData);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 16:
      {
         // GeniviTrace::ReportDebugTrace + wrong common error code => WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** GeniviTrace::ReportDebugTrace + wrong common error code => WRITE TO ERRMEM ***"));
         Ipc2Bts_ReportGeniviDebugTrace* msg = ptrNew_Ipc2Bts_ReportGeniviDebugTrace();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_ERROR);
            msg->setErrFile(54321);
            msg->setErrLine(45321);
            msg->setLevel(21543);
            msg->setCategory(1234567890);
            msg->setErrorCode(31542);
            ::std::vector< unsigned char > tmpData;
            tmpData.reserve(200);
            for(size_t i = 0; i < 200; i++)
            {
               tmpData.push_back((unsigned char)i);
            }
            msg->setData(tmpData);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 17:
      {
         // Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status == BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES => WRITE TO ERRMEM
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status == BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES => WRITE TO ERRMEM ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFF);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 18:
      {
         // Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status == <unknown status (0x60)> => RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status == <unknown status (0x60)> => RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFF);
            const BTSInternalOpcode binOpcode(0x60);
            msg->setStatus((BTSStatusCode)binOpcode);
            msg->setBinaryStatus(binOpcode);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      case 19:
      {
         // Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status == BTS_STATUS_CODE_DSP_LOAD_SEQUENCING_ERROR => RESET
         ETG_TRACE_USR1((" *** Trace::GeneralError with opcode = 0xFFFF (hardware failure) + status == BTS_STATUS_CODE_DSP_LOAD_SEQUENCING_ERROR => RESET ***"));
         Ipc2Bts_GeneralError* msg = ptrNew_Ipc2Bts_GeneralError();
         if(0 != msg)
         {
            msg->setIpcCommonErrorCode(::btstackif::BTS_IPC_SUCCESS);
            msg->setOpcode(0xFFFF);
            msg->setStatus(::btstackif::genivi::BTS_STATUS_CODE_DSP_LOAD_SEQUENCING_ERROR);
            msg->setBinaryStatus((BTSInternalOpcode)::btstackif::genivi::BTS_STATUS_CODE_DSP_LOAD_SEQUENCING_ERROR);

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
         break;
      }
      default:
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
      }
   }
}

void ConnectionControlTestCommand::handleApp2BtsTestCommand(IN const unsigned char* testData)
{
   if(0 == testData)
   {
      return;
   }

   ::fw::TestCommandData cmd;
   const unsigned short int opcode = cmd.getU16Value(testData);
   const unsigned char* data = &testData[2];

   switch(opcode)
   {
      case 0x0001:
      {
         BTSBDAddress address = cmd.getStringValue(&data[0], 13);
         BTSProtocolId protocol = (BTSProtocolId)cmd.getU8Value(&data[13]);
         BTSUuid uuid = cmd.getStringValue(&data[14], 33);
         FW_IF_NULL_PTR_RETURN(_controlIf);
         IBasicControl& controlIf = *_controlIf;
         App2Bts_ConnectProtocol* ptrMsg = ptrNew_App2Bts_ConnectProtocol();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(address);
            ptrMsg->setProtocolId(protocol);
            ptrMsg->setUuid(uuid);
            ptrMsg->doInputTrace();
            controlIf.sendInternalApp2BtsMessage(ptrMsg);
         }
         break;
      }
      case 0x0002:
      {
         BTSBDAddress address = cmd.getStringValue(&data[0], 13);
         BTSSearchType type = (BTSSearchType)cmd.getU8Value(&data[13]);
         FW_IF_NULL_PTR_RETURN(_controlIf);
         IBasicControl& controlIf = *_controlIf;
         App2Bts_StartRemoteServiceSearch* ptrMsg = ptrNew_App2Bts_StartRemoteServiceSearch();
         if(0 != ptrMsg)
         {
            ptrMsg->setBDAddress(address);
            ptrMsg->setSearchType(type);
            ptrMsg->doInputTrace();
            controlIf.sendInternalApp2BtsMessage(ptrMsg);
         }
         break;
      }
      default:
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
      }
   }
}

} //genivi
} //btstackif
