/**
 * @file ConnectionControlExtensions.cpp
 *
 * @par SW-Component
 * Connection Control
 *
 * @brief Implementation of connection control functionality for extensions.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of connection control functionality for extensions.
 */

#include "ConnectionControl_GEN.h"
#include "IControlHandler.h"
#include "BtsTimerPool.h"

namespace btstackif {
namespace genivi {

void ConnectionControl::initializeExtensions(IN const BTSLocalStackConfiguration& configuration, IN const BTSUserMode userMode, IN const BTSInterfaceType interface)
{
   (void)(configuration);
   (void)(userMode);
   (void)(interface);

   // add general extensions here if needed

   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         // set all interfaces before calling initialize
         _extensions[i]->setControlIf(this);
         _extensions[i]->setTimerPoolIf(&TimerPool::getInstance());
         _extensions[i]->initialize(); // TODO: pass configuration + userMode + interface to have better control to create extension or not
      }
   }
}

void ConnectionControl::destroyExtensions(void)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         _extensions[i]->destroy();
         delete _extensions[i];
      }
   }

   _extensions.clear();
}

void ConnectionControl::setExtensionStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         _extensions[i]->setStackConfiguration(component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
      }
   }
}

bool ConnectionControl::doApp2BtsExtensionMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->doApp2BtsMsgPrecheck(continueProcessing, rejectRequest, resultCode, statusCode, skipOpCodeCheck, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::getSimilarExtensionOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->getSimilarOpCodes(itemList, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::getMatchingExtensionOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->getMatchingOpCodes(itemList, highPrioItemList, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::handleDoubledApp2BtsExtensionMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleDoubledApp2BtsMessages(bts2AppMsgList, message, resultCode, statusCode))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::handleApp2BtsExtensionMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleApp2BtsMessage(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::handleIpc2BtsExtensionMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleIpc2BtsMessage(bts2IpcMsgList, bts2AppMsgList, messageItem, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::setSubControlExtensionTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSubControlTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::setSubControlExtensionTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSubControlTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::handleTTFisExtensionCommand(IN const unsigned char* testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleTTFisCommand(testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool ConnectionControl::handleApp2BtsExtensionTestCommand(IN const unsigned char* testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleApp2BtsTestCommand(testData))
         {
            return true;
         }
      }
   }

   return false;
}

} //genivi
} //btstackif
