/**
 * @file ConnectionControlTestCommand.h
 *
 * @par SW-Component
 * Main
 *
 * @brief Connection Control Test Command.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of connection control test command functionality.
 */

#ifndef _CONNECTION_CONTROL_TEST_COMMAND_H_
#define _CONNECTION_CONTROL_TEST_COMMAND_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class App2Bts_StartPairing;
class IBasicControl;
class IServiceSearch;
class IProtocolManager;
class IBasicConfiguration;
class IPairing;
class ILocalAdapterModes;
class ISwitchBluetooth;
class IDeviceManager;

namespace genivi {

// class forward declarations
class IObexConnectDisconnectGenivi;
class IPairingGenivi;
class IObjectPathManagerGenivi;

/**
 * Connection control test command class.
 */
class ConnectionControlTestCommand
{
public:
   /**
    * Default constructor.
    */
   ConnectionControlTestCommand();

   /**
    * Destructor.
    */
   virtual ~ConnectionControlTestCommand();

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control);

   /**
    * Set service search interface.
    *
    * @param[in] searchIf: service search interface
    */
   virtual void setServiceSearch(IN IServiceSearch* searchIf);

   /**
    * Set protocol manager interface.
    *
    * @param[in] protocolManagerIf: protocol manager interface
    */
   virtual void setProtocolManager(IN IProtocolManager* protocolManagerIf);

   /**
    * Set basic configuration interface.
    *
    * @param[in] basicConfigurationIf: basic configuration interface
    */
   virtual void setBasicConfiguration(IN IBasicConfiguration* basicConfigurationIf);

   /**
    * Set pairing interface.
    *
    * @param[in] pairingIf: pairing interface
    */
   virtual void setPairingIf(IN IPairing* pairingIf);

   /**
    * Set local adapter modes interface.
    *
    * @param[in] localAdapterModesIf: local adapter modes interface
    */
   virtual void setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModesIf);

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetoothIf: switch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf);

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   /**
    * Set Genivi OBEX connect/disconnect interface.
    *
    * @param[in] geniviObexConnectDisconnectIf: Genivi OBEX connect/disconnect interface
    */
   virtual void setGeniviObexConnectDisconnectIf(IN IObexConnectDisconnectGenivi* geniviObexConnectDisconnectIf);

   /**
    * Set Genivi pairing interface.
    *
    * @param[in] geniviPairingIf: Genivi pairing interface
    */
   virtual void setGeniviPairingIf(IN IPairingGenivi* geniviPairingIf);

   /**
    * Set Genivi object path manager interface.
    *
    * @param[in] geniviObjectPathManagerIf: Genivi object path manager interface
    */
   virtual void setGeniviObjectPathManagerIf(IN IObjectPathManagerGenivi* geniviObjectPathManagerIf);

   /**
    * Handle test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("handler is responsible for this request"),
    * @return = false: given message was not handled ("handler is not responsible for this request")
    */
   bool handleTestCommand(IN const char* testCommand, IN const unsigned int testData);

   /**
    * Handle test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("handler is responsible for this request"),
    * @return = false: given message was not handled ("handler is not responsible for this request")
    */
   bool handleTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Get random BT device address.
    *
    * @param[out] deviceAddress: BT device address
    */
   void getRandomBDAddress(OUT BTSBDAddress& deviceAddress) const;

   /**
    * Test function checking for valid 128 bit UUID.
    */
   void testFunctionIsValid128BitUuid(void) const;

   /**
    * Test function checking for valid Bluetooth device address.
    */
   void testFunctionIsValidBdAddress(void) const;

   /**
    * Test function checking for stack error.
    *
    * @param[in] cmd: test command
    */
   void testFunctionStackError(IN const unsigned int cmd) const;

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getSimulateDbusStartupError(void) const { return _simulateDbusStartupError; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getSimulateEvolutionStartupError(void) const { return _simulateEvolutionStartupError; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getSimulateStartPairingDuringIncomingPairing(void) const { return _simulateStartPairingDuringIncomingPairing; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setSimulateStartPairingDuringIncomingPairing(IN const bool enable) { _simulateStartPairingDuringIncomingPairing = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getSimulateStartPairingDuringIncomingPairing2(void) const { return _simulateStartPairingDuringIncomingPairing2; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setSimulateStartPairingDuringIncomingPairing2(IN const bool enable) { _simulateStartPairingDuringIncomingPairing2 = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline App2Bts_StartPairing* getSimulateStartPairingDuringIncomingPairingMsg(void) const { return _simulateStartPairingDuringIncomingPairingMsg; }

   /**
    * Set test flag.
    *
    * @param[in] msg: test message
    */
   inline void setSimulateStartPairingDuringIncomingPairingMsg(IN App2Bts_StartPairing* msg) { _simulateStartPairingDuringIncomingPairingMsg = msg; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerSppDisconnectDuringRemoteDisconnect(void) const { return _triggerSppDisconnectDuringRemoteDisconnect; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerSppDisconnectDuringRemoteDisconnect(IN const bool enable) { _triggerSppDisconnectDuringRemoteDisconnect = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getSimulateRemotePairingDuringOngoingConnect(void) const { return _simulateRemotePairingDuringOngoingConnect; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setSimulateRemotePairingDuringOngoingConnect(IN const bool enable) { _simulateRemotePairingDuringOngoingConnect = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerCancelSearchDuringSearchResult(void) const { return _triggerCancelSearchDuringSearchResult; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerCancelSearchDuringSearchResult(IN const bool enable) { _triggerCancelSearchDuringSearchResult = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerLocalSppConnect(void) const { return _triggerLocalSppConnect; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerIgnoreSDPProcessReport(void) const { return _triggerIgnoreSDPProcessReport; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerIgnoreSDPProcessReport(IN const bool enable) { _triggerIgnoreSDPProcessReport = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerSimulateSDPProcessReport(void) const { return _triggerSimulateSDPProcessReport; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerSimulateSDPProcessReport(IN const bool enable) { _triggerSimulateSDPProcessReport = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerIgnoreDeviceConnectedUpdate(void) const { return _triggerIgnoreDeviceConnectedUpdate; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerIgnoreDeviceConnectedUpdate(IN const bool enable) { _triggerIgnoreDeviceConnectedUpdate = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerSimulateStackResetDuringApp2BtsProcessing(void) const { return _triggerSimulateStackResetDuringApp2BtsProcessing; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerSimulateStackResetDuringApp2BtsProcessing(IN const bool enable) { _triggerSimulateStackResetDuringApp2BtsProcessing = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline unsigned int getTriggerBlockRequestConfirmation(void) const { return _triggerBlockRequestConfirmation; }

   /**
    * Set test flag.
    *
    * @param[in] value: value
    */
   inline void setTriggerBlockRequestConfirmation(IN const unsigned int value) { _triggerBlockRequestConfirmation = value; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline bool getTriggerBlockAuthorizeServiceResponse(void) const { return _triggerBlockAuthorizeServiceResponse; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setTriggerBlockAuthorizeServiceResponse(IN const bool enable) { _triggerBlockAuthorizeServiceResponse = enable; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline const BTSBDAddress& getAddressForDisconnectDuringPairing(void) const { return _addressForDisconnectDuringPairing; }

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   inline BTSBDAddress& getAddressForDisconnectDuringPairing(void) { return _addressForDisconnectDuringPairing; }

   /**
    * Set test flag.
    *
    * @param[in] enable: enable flag
    */
   inline void setAddressForDisconnectDuringPairing(IN const BTSBDAddress& address) { _addressForDisconnectDuringPairing = address; }

private:
   bool _simulateDbusStartupError; /**< flag for simulating DBUS startup error */
   bool _simulateEvolutionStartupError; /**< flag for simulating Evolution startup error */
   bool _simulateStartPairingDuringIncomingPairing; /**< test flag */
   bool _simulateStartPairingDuringIncomingPairing2; /**< test flag */
   App2Bts_StartPairing* _simulateStartPairingDuringIncomingPairingMsg; /**< test flag */
   bool _triggerSppDisconnectDuringRemoteDisconnect; /**< flag to trigger local SPP disconnect during remote disconnect */
   bool _simulateRemotePairingDuringOngoingConnect; /**< flag to trigger remote pairing request while ongoing connect */
   bool _triggerCancelSearchDuringSearchResult; /**< flag to trigger cancel search request during reception of search result */
   bool _triggerLocalSppConnect; /**< flag to trigger local SPP connect */
   bool _triggerIgnoreSDPProcessReport; /**< flag to ignore SDPProcessReport indication */
   bool _triggerSimulateSDPProcessReport; /**< flag to simulate SDPProcessReport indication */
   bool _triggerIgnoreDeviceConnectedUpdate; /**< flag to ignore device connected update */
   bool _triggerSimulateStackResetDuringApp2BtsProcessing; /**< flag to simulate stack reset */
   unsigned int _triggerBlockRequestConfirmation; /**< flag to block request confirmation request */
   bool _triggerBlockAuthorizeServiceResponse; /**< flag to block authorize service response */
   BTSBDAddress _addressForDisconnectDuringPairing; /**< address for disconnect during pairing */

   IBasicControl* _controlIf; /**< basic control interface */
   IServiceSearch* _serviceSearch; /**< generic SM for service search */
   IProtocolManager* _protocolManager; /**< generic SM for protocol manager */
   IBasicConfiguration* _basicConfiguration; /**< generic SM for basic configuration */
   IPairing* _pairingIf; /**< generic SM for pairing */
   ILocalAdapterModes* _localAdapterModesIf; /**< generic SM for local adapter modes */
   ISwitchBluetooth* _switchBluetoothIf; /**< generic SM for switch Bluetooth */
   IDeviceManager* _deviceManagerIf; /**< generic SM for device manager */

   IObexConnectDisconnectGenivi* _geniviObexConnectDisconnect; /**< Genivi SM for OBEX connect/disconnect */
   IPairingGenivi* _geniviPairingIf; /**< Genivi SM for pairing */
   IObjectPathManagerGenivi* _geniviObjectPathManagerIf; /**< Genivi SM for object path manager */

   void handleApp2BtsTestCommand(IN const unsigned char* testData);
};

/*
 * inline methods
 */

inline void ConnectionControlTestCommand::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;
}

inline void ConnectionControlTestCommand::setServiceSearch(IN IServiceSearch* searchIf)
{
   _serviceSearch = searchIf;
}

inline void ConnectionControlTestCommand::setProtocolManager(IN IProtocolManager* protocolManagerIf)
{
   _protocolManager = protocolManagerIf;
}

inline void ConnectionControlTestCommand::setBasicConfiguration(IN IBasicConfiguration* basicConfigurationIf)
{
   _basicConfiguration = basicConfigurationIf;
}

inline void ConnectionControlTestCommand::setPairingIf(IN IPairing* pairingIf)
{
   _pairingIf = pairingIf;
}

inline void ConnectionControlTestCommand::setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModesIf)
{
   _localAdapterModesIf = localAdapterModesIf;
}

inline void ConnectionControlTestCommand::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf)
{
   _switchBluetoothIf = switchBluetoothIf;
}

inline void ConnectionControlTestCommand::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;
}

inline void ConnectionControlTestCommand::setGeniviObexConnectDisconnectIf(IN IObexConnectDisconnectGenivi* geniviObexConnectDisconnectIf)
{
   _geniviObexConnectDisconnect = geniviObexConnectDisconnectIf;
}

inline void ConnectionControlTestCommand::setGeniviPairingIf(IN IPairingGenivi* geniviPairingIf)
{
   _geniviPairingIf = geniviPairingIf;
}

inline void ConnectionControlTestCommand::setGeniviObjectPathManagerIf(IN IObjectPathManagerGenivi* geniviObjectPathManagerIf)
{
   _geniviObjectPathManagerIf = geniviObjectPathManagerIf;
}

} //genivi
} //btstackif

#endif //_CONNECTION_CONTROL_TEST_COMMAND_H_
