/**
 * @file EvolutionGeniviUtils.cpp
 *
 * @par SW-Component
 * Utilities
 *
 * @brief Utilities for Evolution Genivi stack.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of utilities class.
 */

#include "EvolutionGeniviUtils.h"
#include "BtsUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviUtils.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

static const BTSBDAddress bdAddressEvolution = "AA:BB:CC:DD:EE:FF";
static const BTSBDAddress bdAddressInternal = "AABBCCDDEEFF";
static const unsigned char maxAsciiValue = 127;

void convertBdAddress2InternalValue(OUT ::std::string& stringOutput, IN const ::std::string& stringInput)
{
   if(bdAddressEvolution.size() == stringInput.size())
   {
      stringOutput.reserve(bdAddressInternal.size());

      for(size_t i = 0; i < stringInput.size(); i++)
      {
         if(':' != stringInput[i])
         {
            stringOutput.push_back(stringInput[i]);
         }
      }

      ::fw::convertString2LowerCase(stringOutput);
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();
      stringOutput.reserve(bdAddressInternal.size());
      for(size_t i = 0; i < bdAddressInternal.size(); i++)
      {
         stringOutput.push_back('0');
      }
   }
}

void convertBdAddress2ExternalValue(OUT ::std::string& stringOutput, IN const ::std::string& stringInput)
{
   if(bdAddressInternal.size() == stringInput.size())
   {
      stringOutput.reserve(bdAddressEvolution.size());

      for(size_t i = 0; i < stringInput.size(); i++)
      {
         if((0 != i) && (0 == (i % 2)))
         {
            stringOutput.push_back(':');
         }
         stringOutput.push_back(stringInput[i]);
      }

      ::fw::convertString2UpperCase(stringOutput);
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

BTSStatusCode convertStatusCode2InternalValue(IN const unsigned char code)
{
   BTSStatusCode returnCode = BTS_STATUS_CODE_FAILED;

   switch(code)
   {
      case 0x00:
         returnCode = BTS_STATUS_CODE_SUCCESS;
         break;
      case 0x01:
         returnCode = BTS_STATUS_CODE_UNKNOWN_OPCODE;
         break;
      case 0x02:
         returnCode = BTS_STATUS_CODE_ILLEGAL_FORMAT;
         break;
      case 0x03:
         returnCode = BTS_STATUS_CODE_ILLEGAL_PARAM;
         break;
      case 0x04:
         returnCode = BTS_STATUS_CODE_OPERATION_FAIL;
         break;
      case 0x05:
         returnCode = BTS_STATUS_CODE_SYSTEM_BUSY;
         break;
      case 0x06:
         returnCode = BTS_STATUS_CODE_ILLEGAL_OPERATION;
         break;
      case 0x07:
         returnCode = BTS_STATUS_CODE_ILLEGAL_STATE;
         break;
      case 0x08:
         returnCode = BTS_STATUS_CODE_PS_FAILURE;
         break;
      case 0x09:
         returnCode = BTS_STATUS_CODE_SDP_FAILURE;
         break;
      case 0x0A:
         returnCode = BTS_STATUS_CODE_ABNORMAL;
         break;
      case 0x0B:
         returnCode = BTS_STATUS_CODE_ABORTED;
         break;
      case 0x0C:
         returnCode = BTS_STATUS_CODE_EXTENDED_ERROR;
         break;
      case 0x0D:
         returnCode = BTS_STATUS_CODE_SDP_CONNECTION_ERROR;
         break;
      case 0x0E:
         returnCode = BTS_STATUS_CODE_TIMEOUT_ERROR;
         break;
      case 0x0F:
         returnCode = BTS_STATUS_CODE_RESTRICTED_CONNECTION;
         break;
      case 0x10:
         returnCode = BTS_STATUS_CODE_RAWDATA_ILLEGAL;
         break;
      case 0x11:
         returnCode = BTS_STATUS_CODE_SERVICE_NOTSUPPORTED;
         break;
      case 0x12:
         returnCode = BTS_STATUS_CODE_MOREDATA_EXPECTED;
         break;
      case 0x14:
         returnCode = BTS_STATUS_CODE_PAGE_TIMEOUT;
         break;
      case 0x16:
         returnCode = BTS_STATUS_CODE_LMP_TIMEOUT;
         break;
      case 0x17:
         returnCode = BTS_STATUS_CODE_SERVICE_CONNECT_TIMEOUT;
         break;
      case 0x19:
         returnCode = BTS_STATUS_CODE_AUTHENTICATION_FAILURE;
         break;
      case 0x1A:
         returnCode = BTS_STATUS_CODE_PIN_OR_KEY_MISSING;
         break;
      case 0x1B:
         returnCode = BTS_STATUS_CODE_ROLE_CHANGE_NOT_ALLOWED;
         break;
      case 0x1C:
         returnCode = BTS_STATUS_CODE_CONNECT_REJECT_BDADDR;
         break;
      case 0x1D:
         returnCode = BTS_STATUS_CODE_ACL_EXISTS;
         break;
      case 0x20:
         returnCode = BTS_STATUS_CODE_HW_FAILURE;
         break;
      case 0x21:
         returnCode = BTS_STATUS_CODE_CONNECT_REJECT_LIMITED_RESOURCES;
         break;
      case 0x22:
         returnCode = BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS;
         break;
      case 0x23:
         returnCode = BTS_STATUS_CODE_REMOTE_DISCONNECT_LOW_RESOURCES;
         break;
      case 0x24:
         returnCode = BTS_STATUS_CODE_REMOTE_DISCONNECT_POWEROFF;
         break;
      case 0x25:
         returnCode = BTS_STATUS_CODE_PAIRING_NOT_ALLOWED;
         break;
      case 0x26:
         returnCode = BTS_STATUS_CODE_CONNECTION_ACCEPT_TIMEOUT;
         break;
      case 0x27:
         returnCode = BTS_STATUS_CODE_LOCAL_REJECT;
         break;
      case 0x28:
         returnCode = BTS_STATUS_CODE_REMOTE_REJECT;
         break;
      case 0x29:
         returnCode = BTS_STATUS_CODE_INTERFACE_REMOVED;
         break;
      case 0x2C:
         returnCode = BTS_STATUS_CODE_REMOTE_DISCONNECT_ACL;
         break;
      case 0x30:
         returnCode = BTS_STATUS_CODE_ONLY_RSSI_SUCCESS;
         break;
      case 0x31:
         returnCode = BTS_STATUS_CODE_ONLY_LINK_QUALITY_SUCCESS;
         break;
      case 0x44:
         returnCode = BTS_STATUS_CODE_FAULT_LC_LMP_IN_INVALID_PACKET;
         break;
      case 0x58:
         returnCode = BTS_STATUS_CODE_FAULT_HOSTIO_FROZEN_HANDLES;
         break;
      case 0x76:
         returnCode = BTS_STATUS_CODE_DSP_LOAD_SEQUENCING_ERROR;
         break;
      case 0x80:
         returnCode = BTS_STATUS_CODE_SUCCESS; // original code: BTS_STATUS_CODE_SUCCESS_SLAVE
         break;
      case 0x81:
         returnCode = BTS_STATUS_CODE_SERVICE_INCOMING;
         break;
      case 0x82:
         returnCode = BTS_STATUS_CODE_INSTANCE_COLLISION;
         break;
      case 0x83:
         returnCode = BTS_STATUS_CODE_CONNECT_COLLISION;
         break;
      default:
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: convertStatusCode2InternalValue(): code=0x%02X", code));
         break;
   }

   ETG_TRACE_USR3((" convertStatusCode2InternalValue: code=0x%02X returnCode=%d", code, returnCode));

   return returnCode;
}

BTSStatusCode convertPairingStatusCode2InternalValue(IN const unsigned char code)
{
   BTSStatusCode returnCode = BTS_STATUS_CODE_UNKNOWN_REASON;

   switch(code)
   {
      case 0x00:
         returnCode = BTS_STATUS_CODE_SUCCESS;
         break;
      case 0x01:
         returnCode = BTS_STATUS_CODE_ALREADY_EXISTS;
         break;
      case 0x02:
         returnCode = BTS_STATUS_CODE_CONNECTION_LOST;
         break;
      case 0x03:
         returnCode = BTS_STATUS_CODE_AUTHENTICATION_ERROR;
         break;
      case 0x04:
         returnCode = BTS_STATUS_CODE_PAIRING_CANCELLED;
         break;
      case 0x05:
         returnCode = BTS_STATUS_CODE_PIN_OR_KEY_MISSING;
         break;
      case 0x06:
         returnCode = BTS_STATUS_CODE_UNKNOWN_REASON;
         break;
      case 0x07:
         returnCode = BTS_STATUS_CODE_RESTRICTED_CONNECTION;
         break;
      case 0x08:
         returnCode = BTS_STATUS_CODE_LMP_TIMEOUT;
         break;
      case 0x09:
         returnCode = BTS_STATUS_CODE_PAGE_TIMEOUT;
         break;
      case 0x0A:
         returnCode = BTS_STATUS_CODE_ROLE_CHANGE_NOT_ALLOWED;
         break;
      case 0x0B:
         returnCode = BTS_STATUS_CODE_CONNECT_REJECT_BDADDR;
         break;
      case 0x0C:
         returnCode = BTS_STATUS_CODE_ACL_EXISTS;
         break;
      case 0x0D:
         returnCode = BTS_STATUS_CODE_HW_FAILURE;
         break;
      case 0x0E:
         returnCode = BTS_STATUS_CODE_CONNECT_REJECT_LIMITED_RESOURCES;
         break;
      case 0x0F:
         returnCode = BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS;
         break;
      case 0x10:
         returnCode = BTS_STATUS_CODE_REMOTE_DISCONNECT_LOW_RESOURCES;
         break;
      case 0x11:
         returnCode = BTS_STATUS_CODE_REMOTE_DISCONNECT_POWEROFF;
         break;
      case 0x12:
         returnCode = BTS_STATUS_CODE_PAIRING_NOT_ALLOWED;
         break;
      case 0x13:
         returnCode = BTS_STATUS_CODE_CONNECTION_ACCEPT_TIMEOUT;
         break;
      case 0x14:
         returnCode = BTS_STATUS_CODE_LOCAL_REJECT;
         break;
      default:
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: convertPairingStatusCode2InternalValue(): code=0x%02X", code));
         break;
   }

   ETG_TRACE_USR3((" convertPairingStatusCode2InternalValue: code=0x%02X returnCode=%d", code, returnCode));

   return returnCode;
}

void convertAgentCapability2String(OUT ::std::string& stringOutput, IN const BTSAgentCapability capability)
{
   switch(capability)
   {
      case BTS_AGENT_CAPABILITY_DISPLAY_YES_NO:
         stringOutput = "DisplayYesNo";
         break;
      case BTS_AGENT_CAPABILITY_DISPLAY_ONLY:
         stringOutput = "DisplayOnly";
         break;
      case BTS_AGENT_CAPABILITY_KEYBOARD_ONLY:
         stringOutput = "KeyboardOnly";
         break;
      case BTS_AGENT_CAPABILITY_NO_INPUT_NO_OUTPUT:
         stringOutput = "NoInputNoOutput";
         break;
      case BTS_AGENT_CAPABILITY_KEYBOARD_DISPLAY:
         stringOutput = "KeyboardDisplay";
         break;
      case BTS_AGENT_CAPABILITY_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         stringOutput = "DisplayYesNo"; // default
         break;
   }
}

BTSServiceState convertServiceState2InternalValue(IN const ::ccdbusif::evolution::ServiceState value)
{
   BTSServiceState state = BTS_SERVICE_STATE_ERROR;

   switch(value)
   {
      case ::ccdbusif::evolution::SERVICE_STATE_DISCONNECTED:
         state = BTS_SERVICE_STATE_DISCONNECTED;
         break;
      case ::ccdbusif::evolution::SERVICE_STATE_CONNECTING:
         state = BTS_SERVICE_STATE_CONNECTING;
         break;
      case ::ccdbusif::evolution::SERVICE_STATE_CONNECTED:
         state = BTS_SERVICE_STATE_CONNECTED;
         break;
      case ::ccdbusif::evolution::SERVICE_STATE_DISCONNECTING:
         state = BTS_SERVICE_STATE_DISCONNECTING;
         break;
      case ::ccdbusif::evolution::SERVICE_STATE_ERROR:
         state = BTS_SERVICE_STATE_ERROR;
         break;
      case ::ccdbusif::evolution::SERVICE_STATE_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return state;
}

BTSProtocolId convertUuid2Protocol(IN const ::ccdbusif::evolution::UUID value)
{
   BTSProtocolId protocol = BTS_PROTO_LAST;

   switch(value)
   {
      case ::ccdbusif::evolution::UUID_HANDSFREE:
      case ::ccdbusif::evolution::UUID_HANDSFREEAUDIOGATEWAY:
         protocol = BTS_PROTO_HFP;
         break;
      case ::ccdbusif::evolution::UUID_AUDIOSOURCE:
      case ::ccdbusif::evolution::UUID_AUDIOSINK:
      case ::ccdbusif::evolution::UUID_A_V_REMOTECONTROLTARGET:
      case ::ccdbusif::evolution::UUID_ADVANCEDAUDIODISTRIBUTION:
      case ::ccdbusif::evolution::UUID_A_V_REMOTECONTROL:
      case ::ccdbusif::evolution::UUID_A_V_REMOTECONTROLCONTROLLER:
         protocol = BTS_PROTO_AVP;
         break;
      case ::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS_PCE:
      case ::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS_PSE:
      case ::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS:
         protocol = BTS_PROTO_PIM;
         break;
      case ::ccdbusif::evolution::UUID_MESSAGE_ACCESS_SERVER:
      case ::ccdbusif::evolution::UUID_MESSAGE_NOTIFICATION_SERVER:
      case ::ccdbusif::evolution::UUID_MESSAGE_ACCESS_PROFILE:
         protocol = BTS_PROTO_MSG;
         break;
      case ::ccdbusif::evolution::UUID_SERIALPORT:
         protocol = BTS_PROTO_SPP;
         break;
      case ::ccdbusif::evolution::UUID_PANU:
      case ::ccdbusif::evolution::UUID_NAP:
      case ::ccdbusif::evolution::UUID_GN:
         protocol = BTS_PROTO_PAN;
         break;
      case ::ccdbusif::evolution::UUID_DIALUPNETWORKING:
         protocol = BTS_PROTO_DUN;
         break;
      case ::ccdbusif::evolution::UUID_BASE:
      case ::ccdbusif::evolution::UUID_SDP:
      case ::ccdbusif::evolution::UUID_UDP:
      case ::ccdbusif::evolution::UUID_RFCOMM:
      case ::ccdbusif::evolution::UUID_TCP:
      case ::ccdbusif::evolution::UUID_TCS_BIN:
      case ::ccdbusif::evolution::UUID_TCS_AT:
      case ::ccdbusif::evolution::UUID_ATT:
      case ::ccdbusif::evolution::UUID_OBEX:
      case ::ccdbusif::evolution::UUID_IP:
      case ::ccdbusif::evolution::UUID_FTP:
      case ::ccdbusif::evolution::UUID_HTTP:
      case ::ccdbusif::evolution::UUID_WSP:
      case ::ccdbusif::evolution::UUID_BNEP:
      case ::ccdbusif::evolution::UUID_UPNP:
      case ::ccdbusif::evolution::UUID_HIDP:
      case ::ccdbusif::evolution::UUID_HARDCOPYCONTROLCHANNEL:
      case ::ccdbusif::evolution::UUID_HARDCOPYDATACHANNEL:
      case ::ccdbusif::evolution::UUID_HARDCOPYNOTIFICATION:
      case ::ccdbusif::evolution::UUID_AVCTP:
      case ::ccdbusif::evolution::UUID_AVDTP:
      case ::ccdbusif::evolution::UUID_CMTP:
      case ::ccdbusif::evolution::UUID_MCAPCONTROLCHANNEL:
      case ::ccdbusif::evolution::UUID_MCAPDATACHANNEL:
      case ::ccdbusif::evolution::UUID_L2CAP:
      case ::ccdbusif::evolution::UUID_SERVICEDISCOVERYSERVERSERVICECLASSID:
      case ::ccdbusif::evolution::UUID_BROWSEGROUPDESCRIPTORSERVICECLASSID:
      case ::ccdbusif::evolution::UUID_LANACCESSUSINGPPP:
      case ::ccdbusif::evolution::UUID_IRMCSYNC:
      case ::ccdbusif::evolution::UUID_OBEXOBJECTPUSH:
      case ::ccdbusif::evolution::UUID_OBEXFILETRANSFER:
      case ::ccdbusif::evolution::UUID_IRMCSYNCCOMMAND:
      case ::ccdbusif::evolution::UUID_HEADSET:
      case ::ccdbusif::evolution::UUID_CORDLESSTELEPHONY:
      case ::ccdbusif::evolution::UUID_INTERCOM:
      case ::ccdbusif::evolution::UUID_FAX:
      case ::ccdbusif::evolution::UUID_HEADSET_AUDIO_GATEWAY:
      case ::ccdbusif::evolution::UUID_WAP:
      case ::ccdbusif::evolution::UUID_WAP_CLIENT:
      case ::ccdbusif::evolution::UUID_DIRECTPRINTING:
      case ::ccdbusif::evolution::UUID_REFERENCEPRINTING:
      case ::ccdbusif::evolution::UUID_BASIC_IMAGING_PROFILE:
      case ::ccdbusif::evolution::UUID_IMAGINGRESPONDER:
      case ::ccdbusif::evolution::UUID_IMAGINGAUTOMATICARCHIVE:
      case ::ccdbusif::evolution::UUID_IMAGINGREFERENCEDOBJECTS:
      case ::ccdbusif::evolution::UUID_DIRECTPRINTINGREFERENCEOBJECTSSERVICE:
      case ::ccdbusif::evolution::UUID_REFLECTEDUI:
      case ::ccdbusif::evolution::UUID_BASICPRINTING:
      case ::ccdbusif::evolution::UUID_PRINTINGSTATUS:
      case ::ccdbusif::evolution::UUID_HUMANINTERFACEDEVICESERVICE:
      case ::ccdbusif::evolution::UUID_HARDCOPYCABLEREPLACEMENT:
      case ::ccdbusif::evolution::UUID_HCR_PRINT:
      case ::ccdbusif::evolution::UUID_HCR_SCAN:
      case ::ccdbusif::evolution::UUID_COMMON_ISDN_ACCESS:
      case ::ccdbusif::evolution::UUID_SIM_ACCESS:
      case ::ccdbusif::evolution::UUID_HEADSET_HS:
      case ::ccdbusif::evolution::UUID_GNSS:
      case ::ccdbusif::evolution::UUID_GNSS_SERVER:
      case ::ccdbusif::evolution::UUID_3D_DISPLAY:
      case ::ccdbusif::evolution::UUID_3D_GLASSES:
      case ::ccdbusif::evolution::UUID_3D_SYNCHRONIZATION:
      case ::ccdbusif::evolution::UUID_MPS_PROFILE:
      case ::ccdbusif::evolution::UUID_MPS_SC:
      case ::ccdbusif::evolution::UUID_CTN_ACCESS_SERVICE:
      case ::ccdbusif::evolution::UUID_CTN_NOTIFICATION_SERVICE:
      case ::ccdbusif::evolution::UUID_CTN_PROFILE:
      case ::ccdbusif::evolution::UUID_PNPINFORMATION:
      case ::ccdbusif::evolution::UUID_GENERICNETWORKING:
      case ::ccdbusif::evolution::UUID_GENERICFILETRANSFER:
      case ::ccdbusif::evolution::UUID_GENERICAUDIO:
      case ::ccdbusif::evolution::UUID_GENERICTELEPHONY:
      case ::ccdbusif::evolution::UUID_UPNP_SERVICE:
      case ::ccdbusif::evolution::UUID_UPNP_IP_SERVICE:
      case ::ccdbusif::evolution::UUID_ESDP_UPNP_IP_PAN:
      case ::ccdbusif::evolution::UUID_ESDP_UPNP_IP_LAP:
      case ::ccdbusif::evolution::UUID_ESDP_UPNP_L2CAP:
      case ::ccdbusif::evolution::UUID_VIDEOSOURCE:
      case ::ccdbusif::evolution::UUID_VIDEOSINK:
      case ::ccdbusif::evolution::UUID_VIDEODISTRIBUTION:
      case ::ccdbusif::evolution::UUID_HDP:
      case ::ccdbusif::evolution::UUID_HDP_SOURCE:
      case ::ccdbusif::evolution::UUID_HDP_SINK:
      case ::ccdbusif::evolution::UUID_SYNCML:
      case ::ccdbusif::evolution::UUID_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return protocol;
}

BTSSupportedServicesBit convertUuid2SupportedService(::std::string& convertedUuid, const ::std::string& receivedUuid)
{
   BTSSupportedServicesBit supportedService = BTS_SUPP_SRV_LAST;
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;

   if(false == parser.isValidUuid(receivedUuid))
   {
      // ignore
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      ::ccdbusif::evolution::UUID service;

      if(true == parser.isShortUuid(service, receivedUuid))
      {
         switch(service)
         {
            case ::ccdbusif::evolution::UUID_HANDSFREE:
            case ::ccdbusif::evolution::UUID_HANDSFREEAUDIOGATEWAY:
               supportedService = BTS_SUPP_SRV_HFP;
               break;
            case ::ccdbusif::evolution::UUID_AUDIOSOURCE:
            case ::ccdbusif::evolution::UUID_AUDIOSINK:
            case ::ccdbusif::evolution::UUID_ADVANCEDAUDIODISTRIBUTION:
               supportedService = BTS_SUPP_SRV_A2DP;
               break;
            case ::ccdbusif::evolution::UUID_A_V_REMOTECONTROLTARGET:
            case ::ccdbusif::evolution::UUID_A_V_REMOTECONTROL:
            case ::ccdbusif::evolution::UUID_A_V_REMOTECONTROLCONTROLLER:
               supportedService = BTS_SUPP_SRV_AVRCP;
               break;
            case ::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS_PCE:
            case ::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS_PSE:
            case ::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS:
               supportedService = BTS_SUPP_SRV_PBAP;
               break;
            case ::ccdbusif::evolution::UUID_MESSAGE_ACCESS_SERVER:
            case ::ccdbusif::evolution::UUID_MESSAGE_NOTIFICATION_SERVER:
            case ::ccdbusif::evolution::UUID_MESSAGE_ACCESS_PROFILE:
               supportedService = BTS_SUPP_SRV_MAP;
               break;
            case ::ccdbusif::evolution::UUID_SERIALPORT:
               supportedService = BTS_SUPP_SRV_SPP;
               // if SPP is supported in general we should use SPP UUID
               convertUuid2InternalValue(convertedUuid, parser.getUuid2String(::ccdbusif::evolution::UUID_SERIALPORT));
               break;
            case ::ccdbusif::evolution::UUID_PANU:
            case ::ccdbusif::evolution::UUID_NAP:
            case ::ccdbusif::evolution::UUID_GN:
               supportedService = BTS_SUPP_SRV_PAN;
               break;
            case ::ccdbusif::evolution::UUID_DIALUPNETWORKING:
               supportedService = BTS_SUPP_SRV_DUN;
               break;
            case ::ccdbusif::evolution::UUID_PNPINFORMATION:
               supportedService = BTS_SUPP_SRV_DID;
               break;
            case ::ccdbusif::evolution::UUID_BASE:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            case ::ccdbusif::evolution::UUID_SDP:
            case ::ccdbusif::evolution::UUID_UDP:
            case ::ccdbusif::evolution::UUID_RFCOMM:
            case ::ccdbusif::evolution::UUID_TCP:
            case ::ccdbusif::evolution::UUID_TCS_BIN:
            case ::ccdbusif::evolution::UUID_TCS_AT:
            case ::ccdbusif::evolution::UUID_ATT:
            case ::ccdbusif::evolution::UUID_OBEX:
            case ::ccdbusif::evolution::UUID_IP:
            case ::ccdbusif::evolution::UUID_FTP:
            case ::ccdbusif::evolution::UUID_HTTP:
            case ::ccdbusif::evolution::UUID_WSP:
            case ::ccdbusif::evolution::UUID_BNEP:
            case ::ccdbusif::evolution::UUID_UPNP:
            case ::ccdbusif::evolution::UUID_HIDP:
            case ::ccdbusif::evolution::UUID_HARDCOPYCONTROLCHANNEL:
            case ::ccdbusif::evolution::UUID_HARDCOPYDATACHANNEL:
            case ::ccdbusif::evolution::UUID_HARDCOPYNOTIFICATION:
            case ::ccdbusif::evolution::UUID_AVCTP:
            case ::ccdbusif::evolution::UUID_AVDTP:
            case ::ccdbusif::evolution::UUID_CMTP:
            case ::ccdbusif::evolution::UUID_MCAPCONTROLCHANNEL:
            case ::ccdbusif::evolution::UUID_MCAPDATACHANNEL:
            case ::ccdbusif::evolution::UUID_L2CAP:
            case ::ccdbusif::evolution::UUID_SERVICEDISCOVERYSERVERSERVICECLASSID:
            case ::ccdbusif::evolution::UUID_BROWSEGROUPDESCRIPTORSERVICECLASSID:
            case ::ccdbusif::evolution::UUID_LANACCESSUSINGPPP:
            case ::ccdbusif::evolution::UUID_IRMCSYNC:
            case ::ccdbusif::evolution::UUID_OBEXOBJECTPUSH:
            case ::ccdbusif::evolution::UUID_OBEXFILETRANSFER:
            case ::ccdbusif::evolution::UUID_IRMCSYNCCOMMAND:
            case ::ccdbusif::evolution::UUID_HEADSET:
            case ::ccdbusif::evolution::UUID_CORDLESSTELEPHONY:
            case ::ccdbusif::evolution::UUID_INTERCOM:
            case ::ccdbusif::evolution::UUID_FAX:
            case ::ccdbusif::evolution::UUID_HEADSET_AUDIO_GATEWAY:
            case ::ccdbusif::evolution::UUID_WAP:
            case ::ccdbusif::evolution::UUID_WAP_CLIENT:
            case ::ccdbusif::evolution::UUID_DIRECTPRINTING:
            case ::ccdbusif::evolution::UUID_REFERENCEPRINTING:
            case ::ccdbusif::evolution::UUID_BASIC_IMAGING_PROFILE:
            case ::ccdbusif::evolution::UUID_IMAGINGRESPONDER:
            case ::ccdbusif::evolution::UUID_IMAGINGAUTOMATICARCHIVE:
            case ::ccdbusif::evolution::UUID_IMAGINGREFERENCEDOBJECTS:
            case ::ccdbusif::evolution::UUID_DIRECTPRINTINGREFERENCEOBJECTSSERVICE:
            case ::ccdbusif::evolution::UUID_REFLECTEDUI:
            case ::ccdbusif::evolution::UUID_BASICPRINTING:
            case ::ccdbusif::evolution::UUID_PRINTINGSTATUS:
            case ::ccdbusif::evolution::UUID_HUMANINTERFACEDEVICESERVICE:
            case ::ccdbusif::evolution::UUID_HARDCOPYCABLEREPLACEMENT:
            case ::ccdbusif::evolution::UUID_HCR_PRINT:
            case ::ccdbusif::evolution::UUID_HCR_SCAN:
            case ::ccdbusif::evolution::UUID_COMMON_ISDN_ACCESS:
            case ::ccdbusif::evolution::UUID_SIM_ACCESS:
            case ::ccdbusif::evolution::UUID_HEADSET_HS:
            case ::ccdbusif::evolution::UUID_GNSS:
            case ::ccdbusif::evolution::UUID_GNSS_SERVER:
            case ::ccdbusif::evolution::UUID_3D_DISPLAY:
            case ::ccdbusif::evolution::UUID_3D_GLASSES:
            case ::ccdbusif::evolution::UUID_3D_SYNCHRONIZATION:
            case ::ccdbusif::evolution::UUID_MPS_PROFILE:
            case ::ccdbusif::evolution::UUID_MPS_SC:
            case ::ccdbusif::evolution::UUID_CTN_ACCESS_SERVICE:
            case ::ccdbusif::evolution::UUID_CTN_NOTIFICATION_SERVICE:
            case ::ccdbusif::evolution::UUID_CTN_PROFILE:
            case ::ccdbusif::evolution::UUID_GENERICNETWORKING:
            case ::ccdbusif::evolution::UUID_GENERICFILETRANSFER:
            case ::ccdbusif::evolution::UUID_GENERICAUDIO:
            case ::ccdbusif::evolution::UUID_GENERICTELEPHONY:
            case ::ccdbusif::evolution::UUID_UPNP_SERVICE:
            case ::ccdbusif::evolution::UUID_UPNP_IP_SERVICE:
            case ::ccdbusif::evolution::UUID_ESDP_UPNP_IP_PAN:
            case ::ccdbusif::evolution::UUID_ESDP_UPNP_IP_LAP:
            case ::ccdbusif::evolution::UUID_ESDP_UPNP_L2CAP:
            case ::ccdbusif::evolution::UUID_VIDEOSOURCE:
            case ::ccdbusif::evolution::UUID_VIDEOSINK:
            case ::ccdbusif::evolution::UUID_VIDEODISTRIBUTION:
            case ::ccdbusif::evolution::UUID_HDP:
            case ::ccdbusif::evolution::UUID_HDP_SOURCE:
            case ::ccdbusif::evolution::UUID_HDP_SINK:
            case ::ccdbusif::evolution::UUID_SYNCML:
               // ignore
               break;
            case ::ccdbusif::evolution::UUID_UNKNOWN:
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }
      }
      else
      {
         // assumption is that this is a UUID for a smartphone service
         // in this case we have to set the bit for SPP
         supportedService = BTS_SUPP_SRV_SPP;

         // convert UUID
         convertUuid2InternalValue(convertedUuid, receivedUuid);
      }
   }

   return supportedService;
}

void convertUuid2InternalValue(OUT ::std::string& stringOutput, IN const ::std::string& stringInput)
{
   // valid UUID must be passed

   stringOutput.reserve(stringInput.size());

   for(size_t i = 0; i < stringInput.size(); i++)
   {
      if('-' != stringInput[i])
      {
         stringOutput.push_back(stringInput[i]);
      }
   }

   ::fw::convertString2LowerCase(stringOutput);

   if(false == ::fw::isValid128BitUuid(stringOutput))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void convertUuid2ExternalValue(OUT ::std::string& stringOutput, IN const ::std::string& stringInput)
{
   // valid UUID must be passed

   if(false == ::fw::isValid128BitUuid(stringInput))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   const ::std::string& baseUuid(::fw::getBaseUuidWithHyphen());
   size_t counter(0);
   stringOutput.reserve(baseUuid.size());

   for(size_t i = 0; ((i < baseUuid.size()) && (counter < stringInput.size())); i++)
   {
      if('-' == baseUuid[i])
      {
         stringOutput.push_back('-');
      }
      else
      {
         stringOutput.push_back(stringInput[counter]);
         counter++;
      }
   }
}

BTSBtApplDisconnectReason convertBtApplDisconnectReason2InternalValue(IN const unsigned char reason)
{
   BTSBtApplDisconnectReason returnReason = BTS_BT_APPL_DISCONNECT_REASON_LOCAL_DEVICE;

   switch(reason)
   {
      case 0x01:
         returnReason = BTS_BT_APPL_DISCONNECT_REASON_LOCAL_DEVICE;
         break;
      case 0x02:
         returnReason = BTS_BT_APPL_DISCONNECT_REASON_REMOTE_DEVICE;
         break;
      case 0x03:
         returnReason = BTS_BT_APPL_DISCONNECT_REASON_EXCEPTION;
         break;
      case 0x04:
         returnReason = BTS_BT_APPL_DISCONNECT_REASON_LINK_LOSS;
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return returnReason;
}

void convertStringBinary2String(OUT ::std::string& stringOutput, IN const ::std::vector< unsigned char >& binaryInput)
{
   // given binary input is an array of characters (as unsigned char)
   stringOutput.reserve(binaryInput.size());
   for(size_t i = 0; i < binaryInput.size(); i++)
   {
      if(maxAsciiValue >= binaryInput[i])
      {
         // valid ASCII character
         stringOutput.push_back((char)binaryInput[i]);
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

BTSIpcCommonErrorCode convertErrorCode2InternalValue(IN const ::ccdbusif::evolution::ErrorCode error)
{
   BTSIpcCommonErrorCode returnCode = BTS_IPC_ERROR;

   switch(error)
   {
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_INVALIDARGUMENTS:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_INPROGRESS:
         returnCode = BTS_IPC_BUSY;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_ALREADYEXISTS:
         returnCode = BTS_IPC_ALREADY_CREATED;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_NOTSUPPORTED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_NOTCONNECTED:
         returnCode = BTS_IPC_NOT_CONNECTED;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_ALREADYCONNECTED:
         returnCode = BTS_IPC_ALREADY_CONNECTED;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_NOTAVAILABLE:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_NOTAUTHORIZED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_AGENTNOTAVAILABLE:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_NOTREADY:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_FAILED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_INVALIDARGUMENTS:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_INVALIDFORMAT:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_NOTIMPLEMENTED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_FAILED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_INPROGRESS:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_NOTFOUND:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_NOTACTIVE:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_NOTSUPPORTED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_NOTAVAILABLE:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_TIMEDOUT:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_OFONO_ERROR_INUSE:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_OBEX_ERROR_INVALIDARGUMENTS:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_OBEX_ERROR_FAILED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_OBEX_ERROR_FORBIDDEN:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_OBEX_NOTAUTHORIZED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_OBEX_INPROGRESS:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_CONNECTIONLOST:
         returnCode = BTS_IPC_PAIRING_CONN_LOST;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_AUTHENTICATIONERROR:
         returnCode = BTS_IPC_PAIRING_AUTHENTICATION_ERROR;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_PAIRINGCANCELLED:
         returnCode = BTS_IPC_PAIRING_PAIRING_CANCELLED;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_MISSINGPINORKEY:
         returnCode = BTS_IPC_PAIRING_MISSING_PIN;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_UNKNOWNREASON:
         returnCode = BTS_IPC_PAIRING_UNKNOWN;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_FREEDESKTOP_ERROR_UNKNOWNOBJECT:
         break;
      case ::ccdbusif::evolution::ERROR_UNKNOWN_OPCODE:
         break;
      case ::ccdbusif::evolution::ERROR_ILLEGAL_FORMAT:
         break;
      case ::ccdbusif::evolution::ERROR_ILLEGAL_PARAM:
         break;
      case ::ccdbusif::evolution::ERROR_OPERATION_FAIL:
         break;
      case ::ccdbusif::evolution::ERROR_SYSTEM_BUSY:
         returnCode = BTS_IPC_BUSY;
         break;
      case ::ccdbusif::evolution::ERROR_ILLEGAL_OPERATION:
         break;
      case ::ccdbusif::evolution::ERROR_ILLEGAL_STATE:
         break;
      case ::ccdbusif::evolution::ERROR_SDP_FAILURE:
         break;
      case ::ccdbusif::evolution::ERROR_ABNORMAL:
         break;
      case ::ccdbusif::evolution::ERROR_ABORTED:
         break;
      case ::ccdbusif::evolution::ERROR_EXTENDED_ERROR:
         break;
      case ::ccdbusif::evolution::ERROR_SDP_CONNECTION_ERROR:
         break;
      case ::ccdbusif::evolution::ERROR_TIMEOUT_ERROR:
         break;
      case ::ccdbusif::evolution::ERROR_RESTRICTED_CONNECTION:
         break;
      case ::ccdbusif::evolution::ERROR_RAW_DATA_ILLEGAL:
         break;
      case ::ccdbusif::evolution::ERROR_SERVICE_NOT_SUPPORTED:
         break;
      case ::ccdbusif::evolution::ERROR_MORE_DATA_EXPECTED:
         break;
      case ::ccdbusif::evolution::ERROR_SESSION_BUSY:
         returnCode = BTS_IPC_BUSY;
         break;
      case ::ccdbusif::evolution::ERROR_REMOTE_REJECT:
         break;
      case ::ccdbusif::evolution::ERROR_LMP_RESPONSE_TIMEOUT:
         break;
      case ::ccdbusif::evolution::ERROR_CONNECT_TIMEOUT:
         break;
      case ::ccdbusif::evolution::ERROR_ROLE_CHANGE_NOT_ALLOWED:
         break;
      case ::ccdbusif::evolution::ERROR_CONNECT_REJECT_BDADDR:
         break;
      case ::ccdbusif::evolution::ERROR_ACL_EXISTS:
         break;
      case ::ccdbusif::evolution::ERROR_HW_FAILURE:
         break;
      case ::ccdbusif::evolution::ERROR_CONNECT_REJECT_LIMITED_RESOURCES:
         break;
      case ::ccdbusif::evolution::ERROR_CONNECT_REJECT_SECURITY_REASONS:
         break;
      case ::ccdbusif::evolution::ERROR_REMOTE_DISCONNECT_LOW_RESOURCE:
         break;
      case ::ccdbusif::evolution::ERROR_REMOTE_DISCONNECT_POWER_OFF:
         break;
      case ::ccdbusif::evolution::ERROR_PAIRING_NOT_ALLOWED:
         break;
      case ::ccdbusif::evolution::ERROR_CONNECTION_ACCEPT_TIMEOUT:
         break;
      case ::ccdbusif::evolution::ERROR_LOCAL_REJECT:
         break;
      case ::ccdbusif::evolution::ERROR_INTERFACE_REMOVED:
         break;
      case ::ccdbusif::evolution::ERROR_PS_FAILURE:
         break;
      case ::ccdbusif::evolution::ERROR_EMPTY:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_FREEDESKTOP_ERROR_UNKNOWNMETHOD:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_DOESNOTEXIST:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_FREEDESKTOP_ERROR_NOSERVER:
         returnCode = BTS_IPC_DBUS_ERROR_NO_SERVER;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_DISCOVERSERVICESCANCELLED:
         returnCode = BTS_IPC_CANCELLED;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_FREEDESKTOP_ERROR_NOREPLY:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_FAILED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_INVALIDARGUMENTS:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_PERMISSIONDENIED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_PASSPHRASEREQUIRED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOTREGISTERED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOTUNIQUE:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOTSUPPORTED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOTIMPLEMENTED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOTFOUND:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOCARRIER:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_INPROGRESS:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_ALREADYEXISTS:
         returnCode = BTS_IPC_ALREADY_CREATED;
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_ALREADYENABLED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_ALREADYDISABLED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_ALREADYCONNECTED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_NOTCONNECTED:
         returnCode = BTS_IPC_NOT_CONNECTED;
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_OPERATIONABORTED:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_OPERATIONTIMEOUT:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_INVALIDSERVICE:
         break;
      case ::ccdbusif::evolution::ERROR_NET_CONNMAN_ERROR_INVALIDPROPERTY:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_OBEX_ALREADYCONNECTED:
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_SERVICEINCOMING:
         returnCode = BTS_IPC_CONNECTION_CONFLICT_SAME_DEVICE;
         break;
      case ::ccdbusif::evolution::ERROR_ORG_BLUEZ_ERROR_CONNECTIONCONFLICT:
         returnCode = BTS_IPC_CONNECTION_CONFLICT_SAME_DEVICE;
         break;
      case ::ccdbusif::evolution::ERROR_PAGE_TIMEOUT:
         break;
      case ::ccdbusif::evolution::ERROR_AUTHENTICATION_FAILURE:
         returnCode = BTS_IPC_PAIRING_AUTHENTICATION_ERROR;
         break;
      case ::ccdbusif::evolution::ERROR_REMOTE_DISCONNECT_ACL:
         break;
      case ::ccdbusif::evolution::ERROR_UNKNOWN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return returnCode;
}

BTSIpcCommonErrorCode convertErrorCode2InternalValue(IN const ::std::string& errorName, IN const ::std::string& errorMessage)
{
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   ::ccdbusif::evolution::ErrorCode error = parser.getErrorCode2Enum(errorName, errorMessage);

   ETG_TRACE_USR1((" convertErrorCode2InternalValue(): errorName=%50s errorMessage=%50s error=%d", errorName.c_str(), errorMessage.c_str(), error));

   return convertErrorCode2InternalValue(error);
}

BTSRequestResult convertPairingStatus2Result(IN const BTSStatusCode code)
{
   BTSRequestResult result = BTS_REQ_PAIRING_UNKNOWN;

   switch(code)
   {
      case BTS_STATUS_CODE_CONNECTION_LOST:
         result = BTS_REQ_PAIRING_CONN_LOST;
         break;
      case BTS_STATUS_CODE_AUTHENTICATION_ERROR:
         result = BTS_REQ_PAIRING_AUTHENTICATION_ERROR;
         break;
      case BTS_STATUS_CODE_PAIRING_CANCELLED:
         result = BTS_REQ_PAIRING_PAIRING_CANCELLED;
         break;
      case BTS_STATUS_CODE_PIN_OR_KEY_MISSING:
         result = BTS_REQ_PAIRING_MISSING_PIN;
         break;
      case BTS_STATUS_CODE_UNKNOWN_REASON:
      case BTS_STATUS_CODE_RESTRICTED_CONNECTION:
      case BTS_STATUS_CODE_LMP_TIMEOUT:
      case BTS_STATUS_CODE_PAGE_TIMEOUT:
      case BTS_STATUS_CODE_ROLE_CHANGE_NOT_ALLOWED:
      case BTS_STATUS_CODE_CONNECT_REJECT_BDADDR:
      case BTS_STATUS_CODE_ACL_EXISTS:
      case BTS_STATUS_CODE_HW_FAILURE:
      case BTS_STATUS_CODE_CONNECT_REJECT_LIMITED_RESOURCES:
      case BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS:
      case BTS_STATUS_CODE_REMOTE_DISCONNECT_LOW_RESOURCES:
      case BTS_STATUS_CODE_REMOTE_DISCONNECT_POWEROFF:
      case BTS_STATUS_CODE_PAIRING_NOT_ALLOWED:
      case BTS_STATUS_CODE_CONNECTION_ACCEPT_TIMEOUT:
      case BTS_STATUS_CODE_LOCAL_REJECT:
         result = BTS_REQ_PAIRING_UNKNOWN;
         break;
      case BTS_STATUS_CODE_ALREADY_EXISTS:
      case BTS_STATUS_CODE_SUCCESS:
         result = BTS_REQ_SUCCESS;
         break;
      default:
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: convertPairingStatus2Result(): code=%d", code));
         break;
   }

   ETG_TRACE_USR3((" convertPairingStatus2Result: code=%d result=%d", code, result));

   return result;
}

BTSRequestResult convertPairingCommonErrorCode2Result(IN const BTSIpcCommonErrorCode errorCode)
{
   BTSRequestResult result = BTS_REQ_PAIRING_UNKNOWN;

   switch(errorCode)
   {
      case BTS_IPC_PAIRING_CONN_LOST:
         result = BTS_REQ_PAIRING_CONN_LOST;
         break;
      case BTS_IPC_PAIRING_AUTHENTICATION_ERROR:
         result = BTS_REQ_PAIRING_AUTHENTICATION_ERROR;
         break;
      case BTS_IPC_PAIRING_PAIRING_CANCELLED:
         result = BTS_REQ_PAIRING_PAIRING_CANCELLED;
         break;
      case BTS_IPC_PAIRING_MISSING_PIN:
         result = BTS_REQ_PAIRING_MISSING_PIN;
         break;
      case BTS_IPC_PAIRING_UNKNOWN:
         result = BTS_REQ_PAIRING_UNKNOWN;
         break;
      case BTS_IPC_ALREADY_CREATED:
      case BTS_IPC_SUCCESS:
         result = BTS_REQ_SUCCESS;
         break;
      case BTS_IPC_BUSY:
         result = BTS_REQ_BUSY;
         break;
      case BTS_IPC_ERROR:
         result = BTS_REQ_PAIRING_UNKNOWN;
         break;
      default:
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: convertPairingCommonErrorCode2Result(): errorCode=%d", errorCode));
         break;
   }

   ETG_TRACE_USR3((" convertPairingCommonErrorCode2Result: errorCode=%d result=%d", errorCode, result));

   return result;
}

BTSRequestResult convertStatus2Result(IN const BTSStatusCode code)
{
   BTSRequestResult result = BTS_REQ_FAILED;

   switch(code)
   {
      case BTS_STATUS_CODE_SUCCESS:
         result = BTS_REQ_SUCCESS;
         break;
      case BTS_STATUS_CODE_UNKNOWN_OPCODE:
         break;
      case BTS_STATUS_CODE_ILLEGAL_FORMAT:
         result = BTS_REQ_INVALID_PARAM;
         break;
      case BTS_STATUS_CODE_ILLEGAL_PARAM:
         result = BTS_REQ_INVALID_PARAM;
         break;
      case BTS_STATUS_CODE_OPERATION_FAIL:
         break;
      case BTS_STATUS_CODE_SYSTEM_BUSY:
         result = BTS_REQ_BUSY;
         break;
      case BTS_STATUS_CODE_ILLEGAL_OPERATION:
         break;
      case BTS_STATUS_CODE_ILLEGAL_STATE:
         break;
      case BTS_STATUS_CODE_PS_FAILURE:
         break;
      case BTS_STATUS_CODE_SDP_FAILURE:
         break;
      case BTS_STATUS_CODE_ABNORMAL:
         break;
      case BTS_STATUS_CODE_ABORTED:
         break;
      case BTS_STATUS_CODE_EXTENDED_ERROR:
         break;
      case BTS_STATUS_CODE_SDP_CONNECTION_ERROR:
         break;
      case BTS_STATUS_CODE_TIMEOUT_ERROR:
         break;
      case BTS_STATUS_CODE_RESTRICTED_CONNECTION:
         break;
      case BTS_STATUS_CODE_RAWDATA_ILLEGAL:
         break;
      case BTS_STATUS_CODE_SERVICE_NOTSUPPORTED:
         result = BTS_REQ_PROTOCOL_NOT_SUPPORTED;
         break;
      case BTS_STATUS_CODE_MOREDATA_EXPECTED:
         break;
      case BTS_STATUS_CODE_PAGE_TIMEOUT:
         result = BTS_REQ_CONNECT_ACL_FAILED;
         break;
      case BTS_STATUS_CODE_LMP_TIMEOUT:
         break;
      case BTS_STATUS_CODE_SERVICE_CONNECT_TIMEOUT:
         break;
      case BTS_STATUS_CODE_AUTHENTICATION_FAILURE:
         break;
      case BTS_STATUS_CODE_PIN_OR_KEY_MISSING:
         break;
      case BTS_STATUS_CODE_ROLE_CHANGE_NOT_ALLOWED:
         break;
      case BTS_STATUS_CODE_CONNECT_REJECT_BDADDR:
         break;
      case BTS_STATUS_CODE_ACL_EXISTS:
         break;
      case BTS_STATUS_CODE_ACL_NOT_CONNECTED:
         result = BTS_REQ_NOT_CONNECTED;
         break;
      case BTS_STATUS_CODE_HW_FAILURE:
         break;
      case BTS_STATUS_CODE_CONNECT_REJECT_LIMITED_RESOURCES:
         break;
      case BTS_STATUS_CODE_CONNECT_REJECT_SECURITY_REASONS:
         break;
      case BTS_STATUS_CODE_REMOTE_DISCONNECT_LOW_RESOURCES:
         break;
      case BTS_STATUS_CODE_REMOTE_DISCONNECT_POWEROFF:
         break;
      case BTS_STATUS_CODE_PAIRING_NOT_ALLOWED:
         break;
      case BTS_STATUS_CODE_CONNECTION_ACCEPT_TIMEOUT:
         break;
      case BTS_STATUS_CODE_LOCAL_REJECT:
         break;
      case BTS_STATUS_CODE_REMOTE_REJECT:
         break;
      case BTS_STATUS_CODE_INTERFACE_REMOVED:
         break;
      case BTS_STATUS_CODE_REMOTE_DISCONNECT_ACL:
         break;
      case BTS_STATUS_CODE_ONLY_RSSI_SUCCESS:
         break;
      case BTS_STATUS_CODE_ONLY_LINK_QUALITY_SUCCESS:
         break;
      case BTS_STATUS_CODE_SUCCESS_SLAVE:
         result = BTS_REQ_SUCCESS;
         break;
      case BTS_STATUS_CODE_SERVICE_INCOMING:
         break;
      case BTS_STATUS_CODE_INSTANCE_COLLISION:
         break;
      case BTS_STATUS_CODE_INVALID_ARGUMENTS:
         result = BTS_REQ_INVALID_PARAM;
         break;
      case BTS_STATUS_CODE_IN_PROGRESS:
         break;
      case BTS_STATUS_CODE_ALREADY_EXISTS:
         break;
      case BTS_STATUS_CODE_NOT_SUPPORTED:
         break;
      case BTS_STATUS_CODE_NOT_CONNECTED:
         result = BTS_REQ_NOT_CONNECTED;
         break;
      case BTS_STATUS_CODE_ALREADY_CONNECTED:
         break;
      case BTS_STATUS_CODE_NOT_AVAILABLE:
         break;
      case BTS_STATUS_CODE_NOT_AUTHORIZED:
         break;
      case BTS_STATUS_CODE_AGENT_NOT_AVAILABLE:
         break;
      case BTS_STATUS_CODE_NOT_READY:
         break;
      case BTS_STATUS_CODE_FAILED:
         break;
      case BTS_STATUS_CODE_INVALID_FORMAT:
         result = BTS_REQ_INVALID_PARAM;
         break;
      case BTS_STATUS_CODE_NOT_IMPLEMENTED:
         break;
      case BTS_STATUS_CODE_NOT_FOUND:
         break;
      case BTS_STATUS_CODE_NOT_ACTIVE:
         break;
      case BTS_STATUS_CODE_TIMED_OUT:
         break;
      case BTS_STATUS_CODE_IN_USE:
         break;
      case BTS_STATUS_CODE_FORBIDDEN:
         break;
      case BTS_STATUS_CODE_CONNECT_FAILED:
         result = BTS_REQ_CONNECT_PROTOCOL_FAILED;
         break;
      case BTS_STATUS_CODE_DOES_NOT_EXIST:
         break;
      case BTS_STATUS_CODE_CONNECTION_LOST:
         break;
      case BTS_STATUS_CODE_AUTHENTICATION_ERROR:
         result = BTS_REQ_PAIRING_AUTHENTICATION_ERROR;
         break;
      case BTS_STATUS_CODE_PAIRING_CANCELLED:
         result = BTS_REQ_PAIRING_PAIRING_CANCELLED;
         break;
      case BTS_STATUS_CODE_UNKNOWN_REASON:
         break;
      case BTS_STATUS_CODE_RETRY_ABORTED:
         break;
      case BTS_STATUS_CODE_START_OTHER_CODES:
      case BTS_STATUS_CODE_LAST:
      default:
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: convertStatus2Result(): code=%d", code));
         break;
   }

   ETG_TRACE_USR3((" convertStatus2Result: code=%d result=%d", code, result));

   return result;
}

BTSProtocolId convertServiceFunction2Protocol(IN const BTSServiceFunction& serviceFunction)
{
   BTSProtocolId protocolId(BTS_PROTO_HFP);

   if(true == serviceFunction.getBit(BTS_SRV_FUNC_SPP))
   {
      protocolId = BTS_PROTO_SPP;
   }
   else if(true == serviceFunction.getBit(BTS_SRV_FUNC_HFP))
   {
      protocolId = BTS_PROTO_HFP;
   }
   else if(true == serviceFunction.getBit(BTS_SRV_FUNC_AVP))
   {
      protocolId = BTS_PROTO_AVP;
   }
   else if(true == serviceFunction.getBit(BTS_SRV_FUNC_PIM))
   {
      protocolId = BTS_PROTO_PIM;
   }
   else if(true == serviceFunction.getBit(BTS_SRV_FUNC_MSG))
   {
      protocolId = BTS_PROTO_MSG;
   }
   else if(true == serviceFunction.getBit(BTS_SRV_FUNC_PANU))
   {
      protocolId = BTS_PROTO_PAN;
   }
   else if(true == serviceFunction.getBit(BTS_SRV_FUNC_DUN))
   {
      protocolId = BTS_PROTO_DUN;
   }
   else
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return protocolId;
}

BTSInternalDisconnectReason convertDisconnectReason2Internal(IN const BTSBtApplDisconnectReason reason)
{
   BTSInternalDisconnectReason internalReason(BTS_INT_DISCONNECT_REASON_LOCAL_DEVICE);

   switch(reason)
   {
      case BTS_BT_APPL_DISCONNECT_REASON_LOCAL_DEVICE:
         internalReason = BTS_INT_DISCONNECT_REASON_LOCAL_DEVICE;
         break;
      case BTS_BT_APPL_DISCONNECT_REASON_REMOTE_DEVICE:
         internalReason = BTS_INT_DISCONNECT_REASON_REMOTE_DEVICE;
         break;
      case BTS_BT_APPL_DISCONNECT_REASON_EXCEPTION:
         internalReason = BTS_INT_DISCONNECT_REASON_EXCEPTION;
         break;
      case BTS_BT_APPL_DISCONNECT_REASON_LINK_LOSS:
         internalReason = BTS_INT_DISCONNECT_REASON_LINK_LOSS;
         break;
      case BTS_BT_APPL_DISCONNECT_REASON_LAST:
      default:
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return internalReason;
}

} //genivi
} //btstackif
