/**
 * @file TrcBts2Ipc_GEN.cpp
 *
 * @par SW-Component
 * Trace
 *
 * @brief Traces for Bts2Ipc messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of traces for Bts2Ipc messages.
 */

#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwTrace.h"
#include "BtsUtils.h"
#include "EvolutionGeniviStackTypes.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_BTS2IPC
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TrcBts2Ipc_GEN.cpp.trc.h"
#else
#include "BtStackIfTypesTrace.h"
#include "BtStackInternalTypesTrace.h"
#include "EvolutionGeniviStackTypesTrace.h"
#include "EvolutionGeniviStackOpcodesTrace.h"
#endif
#endif

namespace btstackif {
namespace genivi {

void Bts2Ipc_GetManagedObjectsConnection::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetAllPropertiesConnection::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): interface=%d objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            ETG_ENUM(TRC_CCDBUSIFInterface, _interface),
            _objPath.c_str() /*DONE: format*/));

#else

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): interface=%s objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            parser.getInterface2StringTrace(_interface).c_str(),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetIntrospectionConnection::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_ReleaseRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_RequestPinCodeRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X pinCode=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _pinCode.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X pinCode=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _pinCode.c_str()));

#endif
}

void Bts2Ipc_DisplayPinCodeRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2Ipc_RequestPasskeyRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X passkey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _passkey.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X passkey=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _passkey.c_str()));

#endif
}

void Bts2Ipc_RequestConfirmationRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X accept=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSConfirmationMode, _accept)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X accept=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getConfirmationMode2String(_accept)));

#endif
}

void Bts2Ipc_CancelRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_CancelRequestRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_AuthorizeServiceRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X accept=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSConfirmationMode, _accept)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X accept=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getConfirmationMode2String(_accept)));

#endif
}

void Bts2Ipc_RequestAuthorizationRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X accept=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSConfirmationMode, _accept)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X accept=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getConfirmationMode2String(_accept)));

#endif
}

void Bts2Ipc_RequestLinkkeyRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X linkKey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _linkKey.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X linkKey=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _linkKey.c_str()));

#endif
}

void Bts2Ipc_DisplayPasskeyRes::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2Ipc_StartDiscovery::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_StopDiscovery::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_RemoveDevice::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_CreateDevice::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2Ipc_SendHCICmd::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u length=%u data=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _length,
            "<TODO>" /*TODO: _data format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u length=%u data=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _length,
            "<TODO>" /*TODO: _data format*/));

#endif
}

void Bts2Ipc_GetAdapterAddress::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetAdapterClass::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetAdapterAlias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterAlias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u alias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _alias.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u alias=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _alias.c_str()));

#endif
}

void Bts2Ipc_GetAdapterPowered::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterPowered::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u powered=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _powered)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u powered=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_powered)));

#endif
}

void Bts2Ipc_GetAdapterDiscoverable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterDiscoverable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u discoverable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _discoverable)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u discoverable=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_discoverable)));

#endif
}

void Bts2Ipc_GetAdapterDiscoverabletimeout::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterDiscoverabletimeout::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u discoverableTimeout=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _discoverableTimeout));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u discoverableTimeout=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _discoverableTimeout));

#endif
}

void Bts2Ipc_GetAdapterPairable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterPairable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u pairable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _pairable)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u pairable=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_pairable)));

#endif
}

void Bts2Ipc_GetAdapterPairabletimeout::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterPairabletimeout::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u pairableTimeout=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _pairableTimeout));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u pairableTimeout=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _pairableTimeout));

#endif
}

void Bts2Ipc_GetAdapterDiscovering::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetAdapterUuids::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterUuids::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u uuids.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _uuids.size() /*DONE: format*/));
   for(size_t i = 0; i < _uuids.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): uuids[%u]=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
               i,
               _uuids[i].c_str() /*DONE: _uuids format*/));
   }

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u uuids.size()=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _uuids.size() /*DONE: format*/));
   for(size_t i = 0; i < _uuids.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): uuids[%u]=%s",
               getMessageTraceId(),
               getGenBts2IpcOpcode2String(opcode),
               i,
               _uuids[i].c_str() /*DONE: _uuids format*/));
   }

#endif
}

void Bts2Ipc_GetAdapterWbsMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterWbsMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u wbsMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _wbsMode)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u wbsMode=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_wbsMode)));

#endif
}

void Bts2Ipc_GetAdapterConnectable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterConnectable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u connectable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _connectable)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u connectable=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_connectable)));

#endif
}

void Bts2Ipc_GetAdapterHcimode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterHcimode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u hciMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _hciMode)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u hciMode=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_hciMode)));

#endif
}

void Bts2Ipc_GetAdapterVersion::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetAdapterTraceLevel::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterTraceLevel::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u traceLevel=0x%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _traceLevel.getData()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u traceLevel=0x%08X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _traceLevel.getData()));

#endif
}

void Bts2Ipc_GetAdapterTraceCategory::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterTraceCategory::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_traceCategory.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_traceCategory.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u traceCategory=0x%08X%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            u1, u2));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_traceCategory.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_traceCategory.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u traceCategory=0x%08X%08X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            u1, u2));

#endif
}

void Bts2Ipc_GetAdapterAvailableSppServers::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetAdapterAvailableSppServers::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u availableSppServers.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _availableSppServers.size() /*DONE: format*/));
   for(size_t i = 0; i < _availableSppServers.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): availableSppServers[%u]: uuid=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
               i,
               _availableSppServers[i].c_str() /*DONE: _availableSppServers format*/));
   }

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u availableSppServers.size()=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _availableSppServers.size() /*DONE: format*/));
   for(size_t i = 0; i < _availableSppServers.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): availableSppServers[%u]: uuid=%s",
               getMessageTraceId(),
               getGenBts2IpcOpcode2String(opcode),
               i,
               _availableSppServers[i].c_str() /*DONE: _availableSppServers format*/));
   }

#endif
}

void Bts2Ipc_RegisterAgent::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u capability=%d agent=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSAgentCapability, _capability),
            _agent.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u capability=%s agent=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getAgentCapability2String(_capability),
            _agent.c_str()));

#endif
}

void Bts2Ipc_UnregisterAgent::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u agent=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _agent.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u agent=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _agent.c_str()));

#endif
}

void Bts2Ipc_RequestDefaultAgent::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u agent=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _agent.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u agent=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _agent.c_str()));

#endif
}

void Bts2Ipc_DeviceConnect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_DeviceDisconnect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_Pair::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_CancelPairing::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_DiscoverServices::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X searchType=%d uuid=%40s device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSSearchType, _searchType),
            _uuid.c_str() /*DONE: format*/,
            _device.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X searchType=%s uuid=%40s device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getSearchType2String(_searchType),
            _uuid.c_str() /*DONE: format*/,
            _device.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_CancelDiscovery::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceAddress::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceClass::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceIcon::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceUuids::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDevicePaired::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceConnected::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceTrusted::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_SetDeviceTrusted::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X trusted=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBool, _trusted),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X trusted=%s device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getBool2String(_trusted),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceBlocked::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_SetDeviceBlocked::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X blocked=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBool, _blocked),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X blocked=%s device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getBool2String(_blocked),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceAlias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_SetDeviceAlias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X alias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _alias.c_str()));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X alias=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _alias.c_str()));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetDeviceAdapter::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceModalias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceRssi::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceVendorId::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceVendorIdSource::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceProductId::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceMapInfo::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceConnectedService::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceVersion::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDevicePbapInfo::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceRole::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetDeviceEnableAvpPause::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_SetDeviceEnableAvpPause::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X enableAvpPause=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBool, _enableAvpPause),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X enableAvpPause=%s device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getBool2String(_enableAvpPause),
            _device.c_str()));

#endif
}

void Bts2Ipc_BtApplDeviceConnectReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance));

#endif
}

void Bts2Ipc_BtApplDeviceDisconnectReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int u1 = (unsigned int)((_serviceFunction.getData() >> 32) & 0x00000000FFFFFFFF);
   unsigned int u2 = (unsigned int)((_serviceFunction.getData()      ) & 0x00000000FFFFFFFF);
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X serviceFunction=0x%08X%08X instance=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            u1, u2 /*DONE: format*/,
            _instance));

#endif
}

void Bts2Ipc_BtApplCancelConnectReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2Ipc_BtApplGetHwVersionReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_BtApplSetConfigurationReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int binSize = (unsigned int)_data.size();
   unsigned char* binData = &(_data[0]);
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u instance=%u configId=%d configLength=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _instance,
            ETG_ENUM(TRC_BTSEvoConfigIdEnum, _configId),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _data format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int binSize = (unsigned int)_data.size();
   unsigned char* binData = &(_data[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u instance=%u configId=%s configLength=%u data=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _instance,
            getEvoConfigIdEnum2String(_configId),
            binSize,
            tmpBuffer /*DONE: _data format*/));

#endif
}

void Bts2Ipc_BtApplReadConfigurationReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u instance=%u configId=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _instance,
            ETG_ENUM(TRC_BTSEvoConfigIdEnum, _configId)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u instance=%u configId=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _instance,
            getEvoConfigIdEnum2String(_configId)));

#endif
}

void Bts2Ipc_BtApplTestModeReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X enable=%d mode=%d role=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBool, _enable),
            ETG_ENUM(TRC_BTSTestMode, _mode),
            ETG_ENUM(TRC_BTSBtRole, _role)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X enable=%s mode=%s role=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getBool2String(_enable),
            getTestMode2String(_mode),
            getBtRole2String(_role)));

#endif
}

void Bts2Ipc_BtApplTestModeLinkQualityReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u mode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSTestMode, _mode)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u mode=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getTestMode2String(_mode)));

#endif
}

void Bts2Ipc_Connect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_Disconnect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetServiceRemoteUuid::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetServiceState::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetServiceDevice::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetServiceVersion::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocol=%d uuid=%40s service=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocol=%s uuid=%40s service=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocol),
            _uuid.c_str() /*DONE: format*/,
            _service.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_DebugTraceConfig::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u traceLevel=%u currentTime=%u traceMode=%d category1=0x%08X category2=0x%08X category3=0x%08X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _traceLevel,
            _currentTime,
            ETG_ENUM(TRC_BTSDebugTraceMode, _traceMode),
            _category1,
            _category2,
            _category3));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u traceLevel=%u currentTime=%u traceMode=%s category1=0x%08X category2=0x%08X category3=0x%08X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _traceLevel,
            _currentTime,
            getDebugTraceMode2String(_traceMode),
            _category1,
            _category2,
            _category3));

#endif
}

void Bts2Ipc_ProtocolTraceConfig::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u traceFilter=0x%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _traceFilter));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u traceFilter=0x%04X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _traceFilter));

#endif
}

void Bts2Ipc_BtApplDidSetReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u specificationID=0x%04X vendorID=0x%04X productID=0x%04X version=0x%04X primaryRecord=%d vendorIDSource=0x%04X clientExecutableURL=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _serviceRecord.specificationID,
            _serviceRecord.vendorID,
            _serviceRecord.productID,
            _serviceRecord.version,
            ETG_ENUM(TRC_BTSBool, _serviceRecord.primaryRecord),
            _serviceRecord.vendorIDSource,
            _serviceRecord.clientExecutableURL.c_str()));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): serviceDescription=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _serviceRecord.serviceDescription.c_str()));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): clientDocumentationURL=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _serviceRecord.clientDocumentationURL.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u specificationID=0x%04X vendorID=0x%04X productID=0x%04X version=0x%04X primaryRecord=%s vendorIDSource=0x%04X clientExecutableURL=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _serviceRecord.specificationID,
            _serviceRecord.vendorID,
            _serviceRecord.productID,
            _serviceRecord.version,
            getBool2String(_serviceRecord.primaryRecord),
            _serviceRecord.vendorIDSource,
            _serviceRecord.clientExecutableURL.c_str()));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): serviceDescription=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _serviceRecord.serviceDescription.c_str()));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): clientDocumentationURL=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _serviceRecord.clientDocumentationURL.c_str()));

#endif
}

void Bts2Ipc_GetDunDeviceFilePath::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_BtApplSppSetUuidReq::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u instance=%u outgoingUuid=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _instance,
            _outgoingUuid.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): incomingUuid=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _incomingUuid.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u instance=%u outgoingUuid=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _instance,
            _outgoingUuid.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): incomingUuid=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
             _incomingUuid.c_str()));

#endif
}

void Bts2Ipc_CreateSession::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   // DONE: second deviceAddress; --- not needed
   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocolId=%d channel=%u instanceId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _channel,
            _instanceId));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   // DONE: second deviceAddress; --- not needed
   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocolId=%s channel=%u instanceId=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _channel,
            _instanceId));

#endif
}

void Bts2Ipc_RemoveSession::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocolId=%d instanceId=%u session=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _instanceId,
            _session.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocolId=%s instanceId=%u session=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _instanceId,
            _session.c_str()));

#endif
}

void Bts2Ipc_CancelSession::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   // DONE: second deviceAddress; --- not needed
   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X protocolId=%d channel=%u instanceId=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _channel,
            _instanceId));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   // DONE: second deviceAddress; --- not needed
   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X protocolId=%s channel=%u instanceId=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _channel,
            _instanceId));

#endif
}

void Bts2Ipc_GetConnManManagerProperties::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetConnManTechnologies::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetConnManServices::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetConnManServiceProperties::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_ConnManServiceConnect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_ConnManServiceDisconnect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_CreateLeDevice::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2Ipc_RemoveLeDevice::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _device.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetLeAdapterAddressType::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetLeAdapterAddress::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetLeAdapterAlias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetLeAdapterAlias::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u alias=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _alias.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u alias=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _alias.c_str()));

#endif
}

void Bts2Ipc_GetLeAdapterAdvertisingTimeout::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetLeAdapterAdvertisingTimeout::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u advertisingTimeout=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _advertisingTimeout));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u advertisingTimeout=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _advertisingTimeout));

#endif
}

void Bts2Ipc_GetLeAdapterDiscoverableMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetLeAdapterDiscoverableMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u discoverableMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBLEDiscoverableMode, _discoverableMode)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u discoverableMode=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBLEDiscoverableMode2String(_discoverableMode)));

#endif
}

void Bts2Ipc_GetLeAdapterAdvertisingData::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetLeAdapterAdvertisingData::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int binSize = (unsigned int)_advertisingData.size();
   unsigned char* binData = &(_advertisingData[0]);
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u length=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _advertisingData format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int binSize = (unsigned int)_advertisingData.size();
   unsigned char* binData = &(_advertisingData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u length=%u data=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            binSize,
            tmpBuffer /*DONE: _advertisingData format*/));

#endif
}

void Bts2Ipc_GetLeAdapterAdvertiseEnable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_SetLeAdapterAdvertiseEnable::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u advertiseEnable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            ETG_ENUM(TRC_BTSBool, _advertiseEnable)));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u advertiseEnable=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            getBool2String(_advertiseEnable)));

#endif
}

void Bts2Ipc_LeDeviceDisconnect::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetLeDeviceAddressType::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetLeDeviceAddress::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetLeDeviceConnected::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_SendSeamlessPairingErrorCode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X errorCode=%20s device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _errorCode.c_str() /*DONE: _errorCode format*/,
            _device.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X errorCode=%20s device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _errorCode.c_str() /*DONE: _errorCode format*/,
            _device.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_SeamlessPairingAllowAccess::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X accept=%d device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBool, _accept),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X accept=%s device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            getBool2String(_accept),
            _device.c_str()));

#endif
}

void Bts2Ipc_GetSeamlessPairingPairingData::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _device.c_str()));

#endif
}

void Bts2Ipc_SetSeamlessPairingPairingData::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int binSize = (unsigned int)_pairingData.size();
   unsigned char* binData = &(_pairingData[0]);
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X length=%u data=%02X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            binSize,
            ETG_LIST(binSize, ETG_T8, binData) /*DONE: _pairingData format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): device=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _device.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   unsigned int binSize = (unsigned int)_pairingData.size();
   unsigned char* binData = &(_pairingData[0]);
   char tmpBuffer[1024] = { 0 };
   for(unsigned int i = 0; (i < binSize) && (i < ((sizeof(tmpBuffer) - 1) / 3)); i++)
   {
      sprintf(&tmpBuffer[3 * i], "%02X ", binData[i]);
   }
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X length=%u data=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            binSize,
            tmpBuffer /*DONE: _pairingData format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): device=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _device.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetManagedObjectsTelephony::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetAllPropertiesTelephony::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _interface.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): interface=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _interface.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetIntrospectionTelephony::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetHandsfreeProperties::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _modem.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X modem=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _modem.c_str()));

#endif
}

void Bts2Ipc_GetModems::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut()));

#endif
}

void Bts2Ipc_GetModemProperties::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u btAddr=%08X%04X modem=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _modem.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u btAddr=%08X%04X modem=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            traceAddr.getUpper(), traceAddr.getLower(),
            _modem.c_str()));

#endif
}

void Bts2Ipc_GetManagedObjectsPhonebook::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetAllPropertiesPhonebook::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _interface.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): interface=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _interface.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetIntrospectionPhonebook::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetPimSessionSource::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimSessionDestination::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimSessionChannel::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimSessionTarget::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_PimTransferCancel::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferSession::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferName::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferType::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferTime::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferSize::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferTransferred::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetPimTransferFilename::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetManagedObjectsMessaging::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetAllPropertiesMessaging::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _interface.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): interface=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _interface.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetIntrospectionMessaging::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetMsgSessionSource::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgSessionDestination::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgSessionChannel::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgSessionTarget::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_MsgTransferCancel::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferSession::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferName::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferType::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferTime::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferSize::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferTransferred::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetMsgTransferFilename::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _objPath.c_str()));

#endif
}

void Bts2Ipc_GetManagedObjectsMediaPlayer::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetAllPropertiesMediaPlayer::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): interface=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _interface.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): interface=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _interface.c_str() /*DONE: format*/));

#endif
}

void Bts2Ipc_GetIntrospectionMediaPlayer::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2Ipc: 0x%08X (%d): objPath=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSGenBts2IpcOpcode, opcode),
            _objPath.c_str() /*DONE: format*/));

#else

   BTSGenBts2IpcOpcode opcode = (BTSGenBts2IpcOpcode)_opCode;
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): TO=%u busType=%d busName=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            getMethodReturnTimeOut(),
            _busType,
            _busName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2Ipc: 0x%08X (%s): objPath=%s",
            getMessageTraceId(),
            getGenBts2IpcOpcode2String(opcode),
            _objPath.c_str() /*DONE: format*/));

#endif
}

} //genivi
} //btstackif
