/**
 * @file TestModeAlpsEvolutionExt.cpp
 *
 * @par SW-Component
 * State machine for test mode
 *
 * @brief Implementation of Alps Evolution Genivi test mode state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Alps Evolution Genivi test mode state machine.
 */

#include "TestModeAlpsEvolutionExt.h"
#include "ITestModeCallback.h"
#include "IBasicControl.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {
namespace alpsevolutionext {

TestModeAlpsEvolutionExt* TestModeAlpsEvolutionExt::_exemplar = 0;

TestModeAlpsEvolutionExt::TestModeAlpsEvolutionExt() :
_callback(0),
_controlIf(0),
_hci(false),
_enable(false),
_address(),
_timerHciMode(),
_timeoutHciMode(3000)
{
   _exemplar = this;
}

TestModeAlpsEvolutionExt::~TestModeAlpsEvolutionExt()
{
   _exemplar = 0;
   _callback = 0;
   _controlIf = 0;
}

void TestModeAlpsEvolutionExt::reset(void)
{
   _hci = false;
   _enable = false;
   _address.clear();
   _timerHciMode.stop();
}

void TestModeAlpsEvolutionExt::setCallback(IN ITestModeCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void TestModeAlpsEvolutionExt::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void TestModeAlpsEvolutionExt::start(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   // all necessary checks are done before

   /*
    * following steps have to be executed:
    * - switch to HCI mode if we are not in HCI mode (happens if we enter test mode during same power cycle)
    * - call BtApplTestModeReq
    * - wait for BtApplTestModeCfm
    * - wait for BtApplTestModeCompInd
    */
   _enable = true;
   _address = address;

   if(false == _hci)
   {
      // switch to HCI mode first
      // HINT: better way would be to trigger via _callback but HCI mode is currently only used for test mode
      createSetAdapterHciModeMsg(bts2IpcMsgList, true);
      _timerHciMode.start(_timeoutHciMode, &timeoutHciModeWrapper);
   }
   else
   {
      // already in HCI mode
      createBtApplTestModeReqMsg(bts2IpcMsgList, address, _enable, BTS_TEST_MODE_REMOTE_LOOPBACK, BTS_BT_MASTER);
   }
}

void TestModeAlpsEvolutionExt::stop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   // all necessary checks are done before

   /*
    * following steps have to be executed:
    * - call BtApplTestModeReq
    * - wait for BtApplTestModeCfm
    * - wait for BtApplTestModeCompInd
    *
    * after end of test mode we will not switch back to APP mode because this is not supported by ALPS Evolution Genivi stack
    * we will stay in HCI mode
    */
   _enable = false;

   createBtApplTestModeReqMsg(bts2IpcMsgList, address, _enable, BTS_TEST_MODE_REMOTE_LOOPBACK, BTS_BT_MASTER);
}

void TestModeAlpsEvolutionExt::getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // all necessary checks are done before

   /*
    * following steps have to be executed:
    * - call BtApplTestModeLinkQualityReq
    * - wait for BtApplTestModeLinkQualityCfm
    */
   createBtApplTestModeLinkQualityReqMsg(bts2IpcMsgList, BTS_TEST_MODE_REMOTE_LOOPBACK);
}

void TestModeAlpsEvolutionExt::handleSetHciModeStatus(IN const bool hciMode)
{
   _hci = hciMode;
}

void TestModeAlpsEvolutionExt::handleSetHciModeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const bool success)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   // intermediate message => get App2Bts message
   messageItem.message = _callback->getApp2BtsWorkingMessage();

   _timerHciMode.stop();

   if(true == success)
   {
      _hci = hciMode;
   }

   if(true == _hci)
   {
      // HCI mode entered => do next step
      createBtApplTestModeReqMsg(bts2IpcMsgList, _address, true, BTS_TEST_MODE_REMOTE_LOOPBACK, BTS_BT_MASTER);
   }
   else
   {
      // switch to HCI mode failed
      _callback->startResult(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
   }
}

void TestModeAlpsEvolutionExt::handleBtApplTestModeCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   // intermediate message => get App2Bts message
   messageItem.message = _callback->getApp2BtsWorkingMessage();

   if(BTS_STATUS_CODE_SUCCESS != status)
   {
      // action failed
      if(true == _enable)
      {
         _callback->startResult(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
      }
      else
      {
         _callback->stopResult(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
      }
   }
}

void TestModeAlpsEvolutionExt::handleBtApplTestModeCompInd(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const BTSStatusCode status)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_NORMAL_ASSERT(_enable == enabled);

   // final message
   bool success = false;

   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      success = true;
   }

   if(true == enabled)
   {
      _callback->startResult(bts2IpcMsgList, bts2AppMsgList, messageItem, success);
   }
   else
   {
      _callback->stopResult(bts2IpcMsgList, bts2AppMsgList, messageItem, success);
   }
}

void TestModeAlpsEvolutionExt::handleBtApplTestModeLinkQualityCfm(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSStatusCode status, IN const BTSRssi rssi, IN const BTSLinkQuality linkQuality)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   // final message
   bool success = false;
   bool rssiSupported = false;
   bool linkQualitySupported = false;

   if((BTS_STATUS_CODE_SUCCESS == status) || (BTS_STATUS_CODE_ONLY_RSSI_SUCCESS == status))
   {
      success = true;
      rssiSupported = true;
   }
   if((BTS_STATUS_CODE_SUCCESS == status) || (BTS_STATUS_CODE_ONLY_LINK_QUALITY_SUCCESS == status))
   {
      success = true;
      linkQualitySupported = true;
   }

   _callback->getLinkQualityResult(bts2IpcMsgList, bts2AppMsgList, messageItem, success, rssiSupported, rssi, linkQualitySupported, linkQuality);
}

void TestModeAlpsEvolutionExt::createSetAdapterHciModeMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const bool enable) const
{
   Bts2Ipc_SetAdapterHcimode* msg = ptrNew_Bts2Ipc_SetAdapterHcimode();
   if(0 != msg)
   {
      msg->setHciMode(enable);

      bts2IpcMsgList.push_back(msg);
   }
}

void TestModeAlpsEvolutionExt::createBtApplTestModeReqMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const bool enable, IN const BTSTestMode mode, IN const BTSBtRole role) const
{
   Bts2Ipc_BtApplTestModeReq* msg = ptrNew_Bts2Ipc_BtApplTestModeReq();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setEnable(enable);
      msg->setMode(mode);
      msg->setRole(role);

      bts2IpcMsgList.push_back(msg);
   }
}

void TestModeAlpsEvolutionExt::createBtApplTestModeLinkQualityReqMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSTestMode mode) const
{
   Bts2Ipc_BtApplTestModeLinkQualityReq* msg = ptrNew_Bts2Ipc_BtApplTestModeLinkQualityReq();
   if(0 != msg)
   {
      msg->setMode(mode);

      bts2IpcMsgList.push_back(msg);
   }
}

void TestModeAlpsEvolutionExt::timeoutHciMode(void)
{
   if(0 != _controlIf)
   {
      Ipc2Bts_AdapterHcimodeUpdate* msg = ptrNew_Ipc2Bts_AdapterHcimodeUpdate();
      if(0 != msg)
      {
         msg->setIpcCommonErrorCode(BTS_IPC_UPDATE_TIMEOUT);
      }
      _controlIf->sendInternalIpc2BtsMessage(msg);
   }
}

} //alpsevolutionext
} //genivi
} //btstackif
