/**
 * @file IBluetoothStackErrorHandlingAlpsEvolutionExt.h
 *
 * @par SW-Component
 * State machine for Bluetooth stack error handling
 *
 * @brief Interface definition for Alps Evolution Genivi Bluetooth stack error handling state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the Alps Evolution Genivi interface for Bluetooth stack error handling state machine.
 */

#ifndef _I_BLUETOOTH_STACK_ERROR_HANDLING_ALPS_EVOLUTION_EXT_H_
#define _I_BLUETOOTH_STACK_ERROR_HANDLING_ALPS_EVOLUTION_EXT_H_

#include "EvolutionGeniviStackTypes.h"

namespace btstackif {

// class forward declarations
class IBluetoothStackErrorHandlingRequest;
class ISwitchBluetooth;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace genivi {
namespace alpsevolutionext {

/**
 * Interface definition.
 */
class IBluetoothStackErrorHandlingAlpsEvolutionExt
{
public:
   /**
    * Destructor.
    */
   virtual ~IBluetoothStackErrorHandlingAlpsEvolutionExt() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IBluetoothStackErrorHandlingRequest* getRequestIf(void) = 0;

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetoothIf: switch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf) = 0;

   /**
    * Handle reported fatal error.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code
    * @param[in] location: location of error
    * @param[in] type: error type
    */
   virtual void handleFatalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSFatalErrorLocation location, IN const BTSFatalErrorType type) = 0;

   /**
    * Handle reported general error.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code
    * @param[in] status: status of operation
    * @param[in] binary status: status of operation (original value from Bluetooth stack)
    * @param[in] opcode: opcode
    */
   virtual void handleGeneralError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSStatusCode status, IN const BTSInternalOpcode binaryStatus, IN const BTSInternalOpcode opcode) = 0;

   /**
    * Handle reported genivi debug trace.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code
    * @param[in] errFile: error file
    * @param[in] errLine: error line
    * @param[in] level: debug level
    * @param[in] category: category
    * @param[in] errCode: error code
    * @param[in] data: trace data
    */
   virtual void handleGeniviDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSGeniviTraceNumber errFile, IN const BTSGeniviTraceNumber errLine, IN const BTSGeniviTraceNumber level, IN const BTSGeniviDebugTraceCategory category, IN const BTSGeniviTraceNumber errCode, IN const BTSGeniviTraceData& data) = 0;

   /**
    * Handle reported genivi fatal error.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code
    * @param[in] errFile: error file
    * @param[in] errLine: error line
    * @param[in] errCode: error code
    * @param[in] fmt: error description
    * @param[in] errValue: error value
    */
   virtual void handleGeniviFatalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSGeniviTraceNumber errFile, IN const BTSGeniviTraceNumber errLine, IN const BTSGeniviTraceNumber errCode, IN const BTSGeniviTraceError& fmt, IN const BTSGeniviTraceError& errValue) = 0;

   /**
    * Handle report debug trace.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] errorCode: error code
    * @param[in] location: trace location
    * @param[in] level: trace level
    * @param[in] category: category
    * @param[in] errCode: error code
    * @param[in] data: trace data
    */
   virtual void handleReportDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode, IN const BTSDebugTraceLocation location, IN const BTSDebugTraceLevel level, IN const BTSDebugTraceCategory category, IN const BTSDebugTraceErrorCode errCode, IN const BTSDebugTraceData& data) = 0;
};

} //alpsevolutionext
} //genivi
} //btstackif

#endif //_I_BLUETOOTH_STACK_ERROR_HANDLING_ALPS_EVOLUTION_EXT_H_
