/**
 * @file WblWiFiSetupStatus.cpp
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Implementation of generic WBL WiFi setup status state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic WBL WiFi setup status state machine.
 */

#include "WblWiFiSetupStatus.h"
#include "IWblWiFiSetupStatusRequest.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "IWblStartup.h"
#include "FwAssert.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblWiFiSetupStatus.cpp.trc.h"
#endif
#endif

namespace btstackif {

WblWiFiSetupStatus::WblWiFiSetupStatus() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_wblStartup(0),
_maxRetry(1),
_nmbRetry(0),
_timer(),
_timeoutRepeatGet(1000),
_getListPending(false),
_wifiSetupList(),
_lastUpdateWiFiSetupStatus(),
_initialWiFiSetupStatusUpdateDone(false),
_lastUpdateSupportedRestriction(),
_initialSupportedRestrictionUpdateDone(false),
_defaultHealthinessIndicatorValue(100),
_lastUpdateWiFiHealthinessIndicatorStatus(_defaultHealthinessIndicatorValue),
_initialWiFiHealthinessIndicatorStatusUpdateDone(false),
_initialGetDone(false)
{
}

WblWiFiSetupStatus::~WblWiFiSetupStatus()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _wblStartup = 0;
}

void WblWiFiSetupStatus::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _wblStartup
   _nmbRetry = 0;
   // stop and release timer
   releaseTimer(_timer);
   _getListPending = false;
   _wifiSetupList.clear();
   _lastUpdateWiFiSetupStatus.clear();
   _initialWiFiSetupStatusUpdateDone = false;
   _lastUpdateSupportedRestriction.clear();
   _initialSupportedRestrictionUpdateDone = false;
   _lastUpdateWiFiHealthinessIndicatorStatus = _defaultHealthinessIndicatorValue;
   _initialWiFiHealthinessIndicatorStatusUpdateDone = false;
   _initialGetDone = false;
}

void WblWiFiSetupStatus::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // update current access point list as empty
   _wifiSetupList.clear();
   createWblWiFiSetupStatusStatusMsg(bts2AppMsgList, 0, 0, true, _wifiSetupList);
   createWblSupportedRestrictionStatusMsg(bts2AppMsgList, 0, 0, true, _wifiSetupList);
   createWblWiFiHealthinessIndicatorStatusMsg(bts2AppMsgList, 0, 0, true, _wifiSetupList);

   // reset control data
   reset();
}

void WblWiFiSetupStatus::setInstance(IN IWblWiFiSetupStatusRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void WblWiFiSetupStatus::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void WblWiFiSetupStatus::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void WblWiFiSetupStatus::setWblStartupIf(IN IWblStartup* wblStartup)
{
   _wblStartup = wblStartup;

   FW_NORMAL_ASSERT(0 != _wblStartup);
}

void WblWiFiSetupStatus::sendWiFiSetupStatusStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status
   createWblWiFiSetupStatusStatusMsg(bts2AppMsgList, user, handle, false, _wifiSetupList);
}

void WblWiFiSetupStatus::sendWiFiSetupStatusStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblWiFiSetupStatus& request, IN const BTSCommonEnumClass statusCode) const
{
   sendWiFiSetupStatusStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

void WblWiFiSetupStatus::sendSupportedRestrictionsStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status
   createWblSupportedRestrictionStatusMsg(bts2AppMsgList, user, handle, false, _wifiSetupList);
}

void WblWiFiSetupStatus::sendSupportedRestrictionsStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblSupportedRestrictions& request, IN const BTSCommonEnumClass statusCode) const
{
   sendSupportedRestrictionsStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

void WblWiFiSetupStatus::sendWiFiHealthinessIndicatorStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status
   createWblWiFiHealthinessIndicatorStatusMsg(bts2AppMsgList, user, handle, false, _wifiSetupList);
}

void WblWiFiSetupStatus::sendWiFiHealthinessIndicatorStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblHealthinessIndicator& request, IN const BTSCommonEnumClass statusCode) const
{
   sendWiFiHealthinessIndicatorStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

bool WblWiFiSetupStatus::getWiFiSetupStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblWiFiSetupStatus& request)
{
   (void)(bts2IpcMsgList);

   if(true == _getListPending)
   {
      // update will be sent after get is finished
   }
   else
   {
      // send current status
      createWblWiFiSetupStatusStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _wifiSetupList);
   }

   return false;
}

bool WblWiFiSetupStatus::getSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblSupportedRestrictions& request)
{
   (void)(bts2IpcMsgList);

   if(true == _getListPending)
   {
      // update will be sent after get is finished
   }
   else
   {
      // send current status
      createWblSupportedRestrictionStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _wifiSetupList);
   }

   return false;
}

bool WblWiFiSetupStatus::getWiFiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblHealthinessIndicator& request)
{
   (void)(bts2IpcMsgList);

   if(true == _getListPending)
   {
      // update will be sent after get is finished
   }
   else
   {
      // send current status
      createWblWiFiHealthinessIndicatorStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _wifiSetupList);
   }

   return false;
}

IWblStartupObserver* WblWiFiSetupStatus::getStartupObserver(void)
{
   return this;
}

void WblWiFiSetupStatus::wifiSetupEntryAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   WblWiFiSetupStatusData& entry = checkList(id);

   entry.pending.setBit(WblWiFiSetupStatusData::GET_AVAILABILITY_PENDING);

   _getListPending = true;
}

void WblWiFiSetupStatus::wifiSetupEntryRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id)
{
   ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::iterator it = _wifiSetupList.find(id);
   if(_wifiSetupList.end() == it)
   {
      // no matching entry
      return;
   }

   _wifiSetupList.erase(it);

   ETG_TRACE_USR2((" wifiSetupEntryRemoved: _wifiSetupList.size()=%u", _wifiSetupList.size()));

   bool doUpdate(false);

   if(0 == _wifiSetupList.size())
   {
      doUpdate = true;
      _getListPending = false;
   }
   else
   {
      if(true == _getListPending)
      {
         if(true == isAnyRequestPending())
         {
            // wait for end of sequence
         }
         else
         {
            doUpdate = true;
            _getListPending = false;
         }
      }
      else
      {
         doUpdate = true;
      }
   }

   if(true == doUpdate)
   {
      validateValues(_wifiSetupList);
      if(true == _initialGetDone)
      {
         createWblWiFiSetupStatusStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         createWblSupportedRestrictionStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         createWblWiFiHealthinessIndicatorStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
      }
      else
      {
         // mark initial get as done
         handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
      }
   }
}

void WblWiFiSetupStatus::wifiSetupEntryAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id)
{
   (void)(messageItem);

   WblWiFiSetupStatusData& entry = checkList(id);

   entry.available = true;
   entry.pending.resetBit(WblWiFiSetupStatusData::GET_AVAILABILITY_PENDING);

   // request all parameters
   requestAll(bts2IpcMsgList, bts2AppMsgList, entry, id);

   _getListPending = true;
}

void WblWiFiSetupStatus::wifiSetupEntryUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id)
{
   ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::iterator it = _wifiSetupList.find(id);
   if(_wifiSetupList.end() == it)
   {
      // no matching entry
      return;
   }

   WblWiFiSetupStatusData& entry = it->second;

   clearEntry(entry); // reset all data for this entry

   bool doUpdate(false);

   if(true == _getListPending)
   {
      if(true == isAnyRequestPending())
      {
         // wait for end of sequence
      }
      else
      {
         doUpdate = true;
         _getListPending = false;
      }
   }
   else
   {
      doUpdate = true;
   }

   if(true == doUpdate)
   {
      validateValues(_wifiSetupList);
      if(true == _initialGetDone)
      {
         createWblWiFiSetupStatusStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         createWblSupportedRestrictionStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         createWblWiFiHealthinessIndicatorStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
      }
      else
      {
         // mark initial get as done
         handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
      }
   }
}

void WblWiFiSetupStatus::getWifiSetupListResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result)
{
   // listing is done; available entries were indicated before

   ETG_TRACE_USR2((" getWifiSetupListResult"));

   if(BTS_IPC_SUCCESS != result)
   {
      if(_maxRetry > _nmbRetry)
      {
         _nmbRetry++;

         // start timer
         startTimer(_timer, _timeoutRepeatGet);
      }
      else
      {
         // stop timer
         stopTimer(_timer);

         // no retries left, failed answer for get request

         _wifiSetupList.clear();
         _getListPending = false;
         if(true == _initialGetDone)
         {
            createWblWiFiSetupStatusStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
            createWblSupportedRestrictionStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
            createWblWiFiHealthinessIndicatorStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         }
         else
         {
            // mark initial get as done
            handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
         }
      }
   }
   else
   {
      // stop timer
      stopTimer(_timer);

      // successful answer for get list request

      // sequence is ongoing
   }
}

void WblWiFiSetupStatus::updateWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiMode mode)
{
   ETG_TRACE_USR2((" updateWifiMode: id=%u mode=%d", id, mode));

   WblWiFiSetupStatusData& entry = checkList(id);

   if(BTS_IPC_SUCCESS == result)
   {
      entry.status.mode = mode;
   }

   entry.pending.resetBit(WblWiFiSetupStatusData::GET_MODE_PENDING);

   checkForCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, UPDATE_WIFI_SETUP_STATUS);
}

void WblWiFiSetupStatus::updateWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiPowerState powerState)
{
   ETG_TRACE_USR2((" updateWifiPowerState: id=%u powerState=%d", id, powerState));

   WblWiFiSetupStatusData& entry = checkList(id);

   if(BTS_IPC_SUCCESS == result)
   {
      entry.status.powerState = powerState;
   }

   entry.pending.resetBit(WblWiFiSetupStatusData::GET_POWER_STATE_PENDING);

   checkForCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, UPDATE_WIFI_SETUP_STATUS);
}

void WblWiFiSetupStatus::updateWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiFrequency frequency)
{
   ETG_TRACE_USR2((" updateWifiFrequency: id=%u frequency=%d", id, frequency));

   WblWiFiSetupStatusData& entry = checkList(id);

   if(BTS_IPC_SUCCESS == result)
   {
      entry.status.frequency = frequency;
   }

   entry.pending.resetBit(WblWiFiSetupStatusData::GET_FREQUENCY_PENDING);

   checkForCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, UPDATE_WIFI_SETUP_STATUS);
}

void WblWiFiSetupStatus::updateAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblAccessPointConfigType type)
{
   ETG_TRACE_USR2((" updateAccessPointConfigType: id=%u type=%d", id, type));

   WblWiFiSetupStatusData& entry = checkList(id);

   if(BTS_IPC_SUCCESS == result)
   {
      entry.status.type = type;
   }

   entry.pending.resetBit(WblWiFiSetupStatusData::GET_AP_CONFIG_TYPE_PENDING);

   checkForCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, UPDATE_WIFI_SETUP_STATUS);
}

void WblWiFiSetupStatus::updateSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblSupportedRestrictionList& supportedRestrictions)
{
   ETG_TRACE_USR2((" updateSupportedRestrictions: id=%u supportedRestrictions.size()=%u", id, supportedRestrictions.size()));

   WblWiFiSetupStatusData& entry = checkList(id);

   if(BTS_IPC_SUCCESS == result)
   {
      entry.supportedRestrictions = supportedRestrictions;
   }

   entry.pending.resetBit(WblWiFiSetupStatusData::GET_SUPPORTED_RESTRICTIONS_PENDING);

   checkForCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, UPDATE_SUPPORTED_RESTRICTION_STATUS);
}

void WblWiFiSetupStatus::updateWifiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const bool connected, IN const BTSWblHealthinessIndicator healthIndicator)
{
   ETG_TRACE_USR2((" updateWifiHealthinessIndicator: id=%u connected=%d healthIndicator=%u", id, connected, healthIndicator));

   WblWiFiSetupStatusData& entry = checkList(id);

   if(BTS_IPC_SUCCESS == result)
   {
      entry.connected = connected;
      entry.healthinessIndicator = healthIndicator;
   }

   entry.pending.resetBit(WblWiFiSetupStatusData::GET_CONNECTION_STATUS_PENDING);

   checkForCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, UPDATE_WIFI_HEALTHINESS_INDICATOR_STATUS);
}

void WblWiFiSetupStatus::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   ETG_TRACE_USR2((" handleExtendedTimeout"));

   FW_IF_NULL_PTR_RETURN(_requestIf);

   // check timer id
   if(false == _timer.compare(timerId))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _requestIf->getWifiSetupList(bts2IpcMsgList, bts2AppMsgList);
}

void WblWiFiSetupStatus::serviceAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   if(BTS_DBUS_SERVICE_AVAILABLE == availability)
   {
      stopTimer(_timer);
      _nmbRetry = 0;

      // send get
      _requestIf->getWifiSetupList(bts2IpcMsgList, bts2AppMsgList);
      _getListPending = true;

      // mark initial get as not done
      handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, false);
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availability)
   {
      stopTimer(_timer);

      // reset status
      _wifiSetupList.clear();
      _getListPending = false;
      if(true == _initialGetDone)
      {
         createWblWiFiSetupStatusStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         createWblSupportedRestrictionStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
         createWblWiFiHealthinessIndicatorStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
      }
      else
      {
         // mark initial get as done
         handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
      }
   }
   else if(BTS_DBUS_SERVICE_WAITING == availability)
   {
      // ignore
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void WblWiFiSetupStatus::initialDataAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   if(BTS_DBUS_SERVICE_AVAILABLE == availability)
   {
      // all initial data is available => update now
      createWblWiFiSetupStatusStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
      createWblSupportedRestrictionStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
      createWblWiFiHealthinessIndicatorStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availability)
   {
      // ignore
   }
   else if(BTS_DBUS_SERVICE_WAITING == availability)
   {
      // ignore
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void WblWiFiSetupStatus::handleChangedInitialGetDone(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool newInitialGetDone)
{
   FW_IF_NULL_PTR_RETURN(_wblStartup);

   // handle only different status
   if(_initialGetDone == newInitialGetDone)
   {
      return;
   }

   // remember new status
   _initialGetDone = newInitialGetDone;

   // inform startup handler
   _wblStartup->informAboutInitialDataStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, WblStartupData::WIFI_SETUP_STATUS_PENDING, _initialGetDone);
}

WblWiFiSetupStatusData& WblWiFiSetupStatus::checkList(IN const BTSWblWiFiSetupId id)
{
   ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::iterator it = _wifiSetupList.find(id);
   if(_wifiSetupList.end() == it)
   {
      // add new entry
      ETG_TRACE_USR2((" checkList: _wifiSetupList.size()=%u", (1 + _wifiSetupList.size())));
      return _wifiSetupList[id];
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}

bool WblWiFiSetupStatus::isAnyRequestPending(void) const
{
   for(::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::const_iterator it = _wifiSetupList.begin(); it != _wifiSetupList.end(); ++it)
   {
      if(0 < it->second.pending.getData())
      {
         return true;
      }
   }

   return false;
}

void WblWiFiSetupStatus::clearEntry(INOUT WblWiFiSetupStatusData& entry)
{
   entry.reset();
}

void WblWiFiSetupStatus::requestAll(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, INOUT WblWiFiSetupStatusData& entry, IN const BTSWblWiFiSetupId id)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->getWifiMode(bts2IpcMsgList, bts2AppMsgList, id);
   _requestIf->getWifiPowerState(bts2IpcMsgList, bts2AppMsgList, id);
   _requestIf->getWifiFrequency(bts2IpcMsgList, bts2AppMsgList, id);
   _requestIf->getAccessPointConfigType(bts2IpcMsgList, bts2AppMsgList, id);
   _requestIf->getSupportedRestrictions(bts2IpcMsgList, bts2AppMsgList, id);
   _requestIf->getWifiHealthinessIndicator(bts2IpcMsgList, bts2AppMsgList, id);

   entry.pending.setBit(WblWiFiSetupStatusData::GET_MODE_PENDING);
   entry.pending.setBit(WblWiFiSetupStatusData::GET_POWER_STATE_PENDING);
   entry.pending.setBit(WblWiFiSetupStatusData::GET_FREQUENCY_PENDING);
   entry.pending.setBit(WblWiFiSetupStatusData::GET_AP_CONFIG_TYPE_PENDING);
   entry.pending.setBit(WblWiFiSetupStatusData::GET_SUPPORTED_RESTRICTIONS_PENDING);
   entry.pending.setBit(WblWiFiSetupStatusData::GET_CONNECTION_STATUS_PENDING);
}

void WblWiFiSetupStatus::validateValues(INOUT ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList)
{
   for(::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::iterator it = wifiSetupList.begin(); it != wifiSetupList.end(); ++it)
   {
      WblWiFiSetupStatusData& entry = it->second;

      // no check for available

      // check status
      if(BTS_WBL_WIFI_MODE_LAST <= entry.status.mode)
      {
         entry.status.mode = BTS_WBL_WIFI_MODE_STA;
      }
      if(BTS_WBL_WIFI_STATE_LAST <= entry.status.powerState)
      {
         entry.status.powerState = BTS_WBL_WIFI_STATE_DEVICE_UNAVAILABLE;
      }
      if(BTS_WBL_WIFI_FREQUENCY_LAST <= entry.status.frequency)
      {
         entry.status.frequency = BTS_WBL_WIFI_FREQUENCY_2_4_GHZ;
      }
      if(BTS_WBL_AP_CONFIG_LAST <= entry.status.type)
      {
         entry.status.type = BTS_WBL_AP_CONFIG_NORMAL;
      }

      // no check for supportedRestrictions

      // no check for pending
   }
}

void WblWiFiSetupStatus::checkForCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const MessageUpdateMode updateMode)
{
   // check if any pending flag is set
   if(true == isAnyRequestPending())
   {
      // wait for end of sequence
   }
   else
   {
      // sequence is finished
      _getListPending = false;

      // update
      validateValues(_wifiSetupList);
      if(true == _initialGetDone)
      {
         switch(updateMode)
         {
            case UPDATE_WIFI_SETUP_STATUS:
               createWblWiFiSetupStatusStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
               break;
            case UPDATE_SUPPORTED_RESTRICTION_STATUS:
               createWblSupportedRestrictionStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
               break;
            case UPDATE_WIFI_HEALTHINESS_INDICATOR_STATUS:
               createWblWiFiHealthinessIndicatorStatusMsgWrapper(bts2AppMsgList, 0, 0, true, _wifiSetupList);
               break;
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }
      }
      else
      {
         // mark initial get as done
         handleChangedInitialGetDone(bts2IpcMsgList, bts2AppMsgList, messageItem, true);
      }
   }
}

void WblWiFiSetupStatus::fillWiFiSetupStatusEntryList(OUT BTSWblWiFiSetupStatusEntryList& outList, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const
{
   outList.reserve(wifiSetupList.size());
   BTSWblWiFiSetupStatusEntry entry;

   for(::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::const_iterator it = wifiSetupList.begin(); it != wifiSetupList.end(); ++it)
   {
      if((true == it->second.available) && (BTS_WBL_WIFI_MODE_AP == it->second.status.mode))
      {
         entry.id = it->first;
         entry.status = it->second.status;
         outList.push_back(entry);
      }
   }
}

void WblWiFiSetupStatus::createWblWiFiSetupStatusStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList)
{
   BTSWblWiFiSetupStatusEntryList tmpList;
   fillWiFiSetupStatusEntryList(tmpList, wifiSetupList);

   if((true == _initialWiFiSetupStatusUpdateDone) && (tmpList == _lastUpdateWiFiSetupStatus))
   {
      return;
   }

   _initialWiFiSetupStatusUpdateDone = true;

   _lastUpdateWiFiSetupStatus = tmpList;

   createWblWiFiSetupStatusStatusMsg(bts2AppMsgList, user, handle, sendStatusToAll, wifiSetupList);
}

void WblWiFiSetupStatus::createWblWiFiSetupStatusStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const
{
   Bts2App_WblWiFiSetupStatus* msg = ptrNew_Bts2App_WblWiFiSetupStatus();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }

      BTSWblWiFiSetupStatusEntryList& entryList = msg->getWblWiFiSetupStatusEntryListMutable();
      fillWiFiSetupStatusEntryList(entryList, wifiSetupList);

      bts2AppMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatus::fillSupportedRestrictionEntryList(OUT BTSWblSupportedRestrictionEntryList& outList, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const
{
   outList.reserve(wifiSetupList.size());
   BTSWblSupportedRestrictionEntry entry;

   for(::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::const_iterator it = wifiSetupList.begin(); it != wifiSetupList.end(); ++it)
   {
      if((true == it->second.available) && (BTS_WBL_WIFI_MODE_AP == it->second.status.mode))
      {
         entry.id = it->first;
         entry.supportedRestrictions = it->second.supportedRestrictions;
         outList.push_back(entry);
      }
   }
}

void WblWiFiSetupStatus::createWblSupportedRestrictionStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList)
{
   BTSWblSupportedRestrictionEntryList tmpList;
   fillSupportedRestrictionEntryList(tmpList, wifiSetupList);

   if((true == _initialSupportedRestrictionUpdateDone) && (tmpList == _lastUpdateSupportedRestriction))
   {
      return;
   }

   _initialSupportedRestrictionUpdateDone = true;

   _lastUpdateSupportedRestriction = tmpList;

   createWblSupportedRestrictionStatusMsg(bts2AppMsgList, user, handle, sendStatusToAll, wifiSetupList);
}

void WblWiFiSetupStatus::createWblSupportedRestrictionStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const
{
   Bts2App_WblSupportedRestrictions* msg = ptrNew_Bts2App_WblSupportedRestrictions();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }

      BTSWblSupportedRestrictionEntryList& entryList = msg->getWblSupportedRestrictionEntryListMutable();
      fillSupportedRestrictionEntryList(entryList, wifiSetupList);

      bts2AppMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatus::fillWiFiHealthinessIndicatorStatusEntryValue(OUT BTSWblHealthinessIndicator& outValue, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const
{
   BTSWblHealthinessIndicator value(_defaultHealthinessIndicatorValue);

   for(::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >::const_iterator it = wifiSetupList.begin(); it != wifiSetupList.end(); ++it)
   {
      if(true == it->second.available)
      {
         if(true == it->second.connected)
         {
            // HU is connected to an AP => check healthiness indicator
            if(it->second.healthinessIndicator < value)
            {
               value = it->second.healthinessIndicator;
            }
         }
      }
   }

   outValue = value;
}

void WblWiFiSetupStatus::createWblWiFiHealthinessIndicatorStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList)
{
   BTSWblHealthinessIndicator tmpValue(_defaultHealthinessIndicatorValue);
   fillWiFiHealthinessIndicatorStatusEntryValue(tmpValue, wifiSetupList);

   if((true == _initialWiFiHealthinessIndicatorStatusUpdateDone) && (tmpValue == _lastUpdateWiFiHealthinessIndicatorStatus))
   {
      return;
   }

   _initialWiFiHealthinessIndicatorStatusUpdateDone = true;

   _lastUpdateWiFiHealthinessIndicatorStatus = tmpValue;

   createWblWiFiHealthinessIndicatorStatusMsg(bts2AppMsgList, user, handle, sendStatusToAll, wifiSetupList);
}

void WblWiFiSetupStatus::createWblWiFiHealthinessIndicatorStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const
{
   Bts2App_WblHealthinessIndicator* msg = ptrNew_Bts2App_WblHealthinessIndicator();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }

      BTSWblHealthinessIndicator tmpValue(_defaultHealthinessIndicatorValue);
      fillWiFiHealthinessIndicatorStatusEntryValue(tmpValue, wifiSetupList);
      msg->setWblHealthinessIndicator(tmpValue);

      bts2AppMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatus::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void WblWiFiSetupStatus::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void WblWiFiSetupStatus::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

} //btstackif
