/**
 * @file WblWiFiSetupStatusData.h
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Implementation of WBL WiFi setup status data.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of WBL WiFi setup status data.
 */

#ifndef _WBL_WIFI_SETUP_STATUS_DATA_H_
#define _WBL_WIFI_SETUP_STATUS_DATA_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 * WBL WiFi setup status data class.
 */
class WblWiFiSetupStatusData
{
public:
   /**
    * Type definition for device data information bits.
    * #GetDataPendingBit#ignore#
    */
   enum GetDataPendingBit
   {
      GET_MODE_PENDING = 0, /**< get for WiFi mode pending */
      GET_POWER_STATE_PENDING, /**< get for WiFi power state pending */
      GET_FREQUENCY_PENDING, /**< get for WiFi frequency pending */
      GET_AP_CONFIG_TYPE_PENDING, /**< get for access point config type pending */
      GET_SUPPORTED_RESTRICTIONS_PENDING, /**< get for supported restrictions pending */
      GET_CONNECTION_STATUS_PENDING, /**< get for WiFi connection status pending */
      GET_AVAILABILITY_PENDING, /**< availability pending */
      // -------------------------------------------------------------------------
      GET_DATA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for device data information.
    */
   typedef BTSBitfield<unsigned int, GetDataPendingBit, GET_DATA_LAST> GetDataPending;

   /**
    * Default constructor.
    */
   WblWiFiSetupStatusData();

   /**
    * Destructor.
    */
   virtual ~WblWiFiSetupStatusData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   WblWiFiSetupStatusData(const WblWiFiSetupStatusData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   WblWiFiSetupStatusData& operator=(const WblWiFiSetupStatusData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const WblWiFiSetupStatusData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const WblWiFiSetupStatusData& ref) const;

   /**
    * Reset all members.
    */
   void reset(void);

   // member with public access
   bool available; /**< available flag */
   BTSWblWiFiSetupStatus status; /**< WiFi setup status */
   BTSWblSupportedRestrictionList supportedRestrictions; /**< supported restrictions (supported frequencies) */
   GetDataPending pending; /**< get data pending information */
   bool connected; /**< connected flag */
   BTSWblHealthinessIndicator healthinessIndicator; /**< WiFi healthiness indicator */
};

} //btstackif

#endif //_WBL_WIFI_SETUP_STATUS_DATA_H_
