/**
 * @file WblWiFiSetupStatus.h
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Implementation of generic WBL WiFi setup status state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic WBL WiFi setup status state machine.
 */

#ifndef _WBL_WIFI_SETUP_STATUS_H_
#define _WBL_WIFI_SETUP_STATUS_H_

#include "StateMachine.h"
#include "IWblWiFiSetupStatus.h"
#include "IWblWiFiSetupStatusCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "IWblStartupObserver.h"
#include "ExtendedTimerEntry.h"
#include "WblWiFiSetupStatusData.h"

namespace btstackif {

/**
 * WblWiFiSetupStatus class.
 */
class WblWiFiSetupStatus : public StateMachine, public IWblWiFiSetupStatus, public IWblWiFiSetupStatusCallback, public IExtendedTimeoutHandler, public IWblStartupObserver
{
public:
   /**
    * Default constructor.
    */
   WblWiFiSetupStatus();

   /**
    * Destructor.
    */
   virtual ~WblWiFiSetupStatus();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IWblWiFiSetupStatusRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setWblStartupIf(IN IWblStartup* wblStartup);

   virtual void sendWiFiSetupStatusStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendWiFiSetupStatusStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblWiFiSetupStatus& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendSupportedRestrictionsStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendSupportedRestrictionsStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblSupportedRestrictions& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendWiFiHealthinessIndicatorStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendWiFiHealthinessIndicatorStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblHealthinessIndicator& request, IN const BTSCommonEnumClass statusCode) const;

   virtual bool getWiFiSetupStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblWiFiSetupStatus& request);

   virtual bool getSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblSupportedRestrictions& request);

   virtual bool getWiFiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblHealthinessIndicator& request);

   virtual IWblStartupObserver* getStartupObserver(void);

   virtual void wifiSetupEntryAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id);

   virtual void wifiSetupEntryRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id);

   virtual void wifiSetupEntryAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id);

   virtual void wifiSetupEntryUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblWiFiSetupId id);

   virtual void getWifiSetupListResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result);

   virtual void updateWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiMode mode);

   virtual void updateWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiPowerState powerState);

   virtual void updateWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblWifiFrequency frequency);

   virtual void updateAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblAccessPointConfigType type);

   virtual void updateSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const BTSWblSupportedRestrictionList& supportedRestrictions);

   virtual void updateWifiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblWiFiSetupId id, IN const bool connected, IN const BTSWblHealthinessIndicator healthIndicator);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void serviceAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability);

   virtual void initialDataAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability);

private:
   /**
    * Type definition for update messages.
    * #MessageUpdateMode#ignore#
    */
   enum MessageUpdateMode
   {
      UPDATE_WIFI_SETUP_STATUS = 0, /**< update WiFi setup status */
      UPDATE_SUPPORTED_RESTRICTION_STATUS, /**< update supported restrictions status */
      UPDATE_WIFI_HEALTHINESS_INDICATOR_STATUS /**< update WiFi healthiness indicator status */
   };

   IWblWiFiSetupStatusRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IWblStartup* _wblStartup; /**< WBL startup interface */
   const unsigned int _maxRetry; /**< max retry for requesting WiFi setup status */
   unsigned int _nmbRetry; /**< current number of retry for requesting WiFi setup status */
   ExtendedTimerEntry _timer; /**< timer */
   const BTSTimeValue _timeoutRepeatGet; /**< timeout value for requesting WiFi setup status again */
   bool _getListPending; /**< flag for pending get request */
   ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData > _wifiSetupList; /**< map containing WiFi setup status information */
   BTSWblWiFiSetupStatusEntryList _lastUpdateWiFiSetupStatus; /**< WiFi setup status list */
   bool _initialWiFiSetupStatusUpdateDone; /**< flag for initial update */
   BTSWblSupportedRestrictionEntryList _lastUpdateSupportedRestriction; /**< supported restriction list */
   bool _initialSupportedRestrictionUpdateDone; /**< flag for initial update */
   const BTSWblHealthinessIndicator _defaultHealthinessIndicatorValue; /**< default WiFi healthiness indicator value */
   BTSWblHealthinessIndicator _lastUpdateWiFiHealthinessIndicatorStatus; /**< WiFi healthiness indicator status */
   bool _initialWiFiHealthinessIndicatorStatusUpdateDone; /**< flag for initial update */
   bool _initialGetDone; /**< flag for finished initial get request */

   void handleChangedInitialGetDone(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool newInitialGetDone);

   WblWiFiSetupStatusData& checkList(IN const BTSWblWiFiSetupId id);

   bool isAnyRequestPending(void) const;

   void clearEntry(INOUT WblWiFiSetupStatusData& entry);

   void requestAll(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, INOUT WblWiFiSetupStatusData& entry, IN const BTSWblWiFiSetupId id);

   void validateValues(INOUT ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList);

   void checkForCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const MessageUpdateMode updateMode);

   void fillWiFiSetupStatusEntryList(OUT BTSWblWiFiSetupStatusEntryList& outList, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const;

   void createWblWiFiSetupStatusStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList);

   void createWblWiFiSetupStatusStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const;

   void fillSupportedRestrictionEntryList(OUT BTSWblSupportedRestrictionEntryList& outList, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const;

   void createWblSupportedRestrictionStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList);

   void createWblSupportedRestrictionStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const;

   void fillWiFiHealthinessIndicatorStatusEntryValue(OUT BTSWblHealthinessIndicator& outValue, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const;

   void createWblWiFiHealthinessIndicatorStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList);

   void createWblWiFiHealthinessIndicatorStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const ::std::map< BTSWblWiFiSetupId, WblWiFiSetupStatusData >& wifiSetupList) const;

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;
};

} //btstackif

#endif //_WBL_WIFI_SETUP_STATUS_H_
