/**
 * @file IWblWiFiSetupStatusRequest.h
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Request interface definition for WBL WiFi setup status state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific WBL WiFi setup status state machine.
 */

#ifndef _I_WBL_WIFI_SETUP_STATUS_REQUEST_H_
#define _I_WBL_WIFI_SETUP_STATUS_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IWblWiFiSetupStatusCallback;
class IBasicControl;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IWblWiFiSetupStatusRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblWiFiSetupStatusRequest() {}

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN IWblWiFiSetupStatusCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Get WiFi setup list.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void getWifiSetupList(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Get WiFi mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] id: identifier
    */
   virtual void getWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * Get WiFi power state
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] id: identifier
    */
   virtual void getWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * Get WiFi frequency.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] id: identifier
    */
   virtual void getWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * Get WiFi access point config type.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] id: identifier
    */
   virtual void getAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * Get supported restrictions (supported frequencies).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] id: identifier
    */
   virtual void getSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id) = 0;

   /**
    * Get WiFi healthiness indicator.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] id: identifier
    */
   virtual void getWifiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id) = 0;
};

} //btstackif

#endif //_I_WBL_WIFI_SETUP_STATUS_REQUEST_H_
