/**
 * @file WblStartupData.h
 *
 * @par SW-Component
 * State machine for WBL startup
 *
 * @brief Implementation of WBL startup data.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of WBL startup data.
 */

#ifndef _WBL_STARTUP_DATA_H_
#define _WBL_STARTUP_DATA_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 * WBL startup data class.
 */
class WblStartupData
{
public:
   /**
    * Type definition for device data information bits.
    * #InitialDataPendingBit#ignore#
    */
   enum InitialDataPendingBit
   {
      LAST_MODE_PENDING = 0, /**< get for last mode pending */
      WIFI_SETUP_STATUS_PENDING, /**< get for WiFi setup status pending */
      // -------------------------------------------------------------------------
      INITIAL_DATA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for device data information.
    */
   typedef BTSBitfield<unsigned int, InitialDataPendingBit, INITIAL_DATA_LAST> InitialDataPending;

   /**
    * Default constructor.
    */
   WblStartupData();

   /**
    * Destructor.
    */
   virtual ~WblStartupData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   WblStartupData(const WblStartupData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   WblStartupData& operator=(const WblStartupData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const WblStartupData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const WblStartupData& ref) const;

   /**
    * Reset all members.
    */
   void reset(void);

   // member with public access
   InitialDataPending pending; /**< initial data pending information */
};

} //btstackif

#endif //_WBL_STARTUP_DATA_H_
