/**
 * @file IWblStartup.h
 *
 * @par SW-Component
 * State machine for WBL startup
 *
 * @brief Interface definition for generic WBL startup state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for WBL startup state machine.
 */

#ifndef _I_WBL_STARTUP_H_
#define _I_WBL_STARTUP_H_

#include "BtStackInternalTypes.h"
#include "WblStartupData.h"

namespace btstackif {

// class forward declarations
class IWblStartupRequest;
class IBasicControl;
class ITimerPool;
class App2Bts_TriggerFbWblInitialized;
class App2Bts_GetWblServiceAvailability;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IWblStartupObserver;

/**
 * Interface definition.
 */
class IWblStartup
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblStartup() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IWblStartupRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] user: user
    * @param[in] handle: handle
    * @param[in] statusCode: status code
    */
   virtual void sendInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_TriggerFbWblInitialized& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] user: user
    * @param[in] handle: handle
    * @param[in] statusCode: status code
    */
   virtual void sendServiceAvailabilityStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendServiceAvailabilityStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblServiceAvailability& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Get service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool getServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblServiceAvailability& request) = 0;

   /**
    * Process trigger initialized callback request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool processTriggerInitializedCallback(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_TriggerFbWblInitialized& request) = 0;

   /**
    * Process trigger initialized callback request.
    * Note: This function might be called from another thread context.
    */
   virtual void processTriggerInitializedCallback(void) = 0;

   /**
    * Get WBL service availability.
    *
    * @return = true: WBL service is available,
    * @return = false: WBL service is not available
    */
   virtual bool isServiceAvailable(void) const = 0;

   /**
    * Get WBL startup flag.
    *
    * @return = true: WBL startup is ongoing,
    * @return = false: WBL startup is not ongoing (finished)
    */
   virtual bool isStartupOngoing(void) const = 0;

   /**
    * Register observer.
    *
    * @param[in] observer: observer
    */
   virtual void registerObserver(IN IWblStartupObserver* observer) = 0;

   /**
    * Inform about initial data status.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] pendingBit: pending data reference
    * @param[in] enable: enable flag
    */
   virtual void informAboutInitialDataStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const WblStartupData::InitialDataPendingBit pendingBit, IN const bool enable) = 0;
};

} //btstackif

#endif //_I_WBL_STARTUP_H_
