/**
 * @file WblLastMode.h
 *
 * @par SW-Component
 * State machine for WBL last mode
 *
 * @brief Implementation of generic WBL last mode state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic WBL last mode state machine.
 */

#ifndef _WBL_LAST_MODE_H_
#define _WBL_LAST_MODE_H_

#include "StateMachine.h"
#include "IWblLastMode.h"
#include "IWblLastModeCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "IWblStartupObserver.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {

/**
 * WblLastMode class.
 */
class WblLastMode : public StateMachine, public IWblLastMode, public IWblLastModeCallback, public IExtendedTimeoutHandler, public IWblStartupObserver
{
public:
   /**
    * Default constructor.
    */
   WblLastMode();

   /**
    * Destructor.
    */
   virtual ~WblLastMode();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IWblLastModeRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setWblStartupIf(IN IWblStartup* wblStartup);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblLastIntendedMode& request, IN const BTSCommonEnumClass statusCode) const;

   virtual bool getLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetWblLastIntendedMode& request);

   virtual IWblStartupObserver* getStartupObserver(void);

   virtual void updateLastMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSWblLastIntendedModeList& accessPointStatusList);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void serviceAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability);

   virtual void initialDataAvailabilityChanged(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability);

private:
   IWblLastModeRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IWblStartup* _wblStartup; /**< WBL startup interface */
   BTSWblLastIntendedModeList _accessPointList; /**< access point list */
   BTSWblLastIntendedModeList _lastUpdateList; /**< access point list */
   bool _initialUpdateDone; /**< flag for initial update */
   const unsigned int _maxRetry; /**< max retry for requesting last mode */
   unsigned int _nmbRetry; /**< current number of retry for requesting last mode */
   ExtendedTimerEntry _timer; /**< timer */
   const BTSTimeValue _timeoutRepeatGet; /**< timeout value for requesting last mode again */
   bool _getPending; /**< flag for pending get request */
   bool _initialGetDone; /**< flag for finished initial get request */

   void handleChangedInitialGetDone(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool newInitialGetDone);

   void createWblLastIntendedModeStatusMsgWrapper(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSWblLastIntendedModeList& accessPointStatusList);

   void createWblLastIntendedModeStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSWblLastIntendedModeList& accessPointStatusList) const;

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;
};

} //btstackif

#endif //_WBL_LAST_MODE_H_
