/**
 * @file TestMode.h
 *
 * @par SW-Component
 * State machine for test mode
 *
 * @brief Implementation of generic test mode state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic test mode state machine.
 */

#ifndef _TEST_MODE_H_
#define _TEST_MODE_H_

#include "StateMachine.h"
#include "ITestMode.h"
#include "ITestModeCallback.h"

namespace btstackif {

/**
 * Test mode class.
 */
class TestMode : public StateMachine, public ITestMode, public ITestModeCallback
{
public:
   /**
    * Default constructor.
    */
   TestMode();

   /**
    * Destructor.
    */
   virtual ~TestMode();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN ITestModeRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndStartResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartTestMode& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndStopResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopTestMode& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendLinkQualityResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestTestModeLinkQuality& request, IN const BTSCommonEnumClass resultCode) const;

   virtual bool isValidStartRequest(IN const App2Bts_StartTestMode& request) const;

   virtual bool isValidStopRequest(IN const App2Bts_StopTestMode& request) const;

   virtual bool isValidLinkQualityRequest(IN const App2Bts_RequestTestModeLinkQuality& request) const;

   virtual bool start(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartTestMode& request);

   virtual bool stop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopTestMode& request);

   virtual bool getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestTestModeLinkQuality& request);

   virtual bool isActive(void) const;

   virtual bool isActive(IN const BTSBDAddress& address) const;

   virtual const BTSBDAddress& getAddress(void) const;

   virtual void handleSetHciModeStatus(IN const bool hciMode);

   virtual void handleSetHciModeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode, IN const bool success);

   virtual void startResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool success);

   virtual void stopResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool success);

   virtual void status(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool active);

   virtual void getLinkQualityResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool success, IN const bool rssiSupported, IN const BTSRssi rssi, IN const bool linkQualitySupported, IN const BTSLinkQuality linkQuality);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(void);

private:
   ITestModeRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   BTSApp2BtsMessageRequestItem _requestItem; /**< request item */
   bool _active; /**< flag for active/inactive test mode */
   BTSBDAddress _address; /**< BT device address */

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void createTestModeStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const bool active) const;

   void createStartTestModeResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;

   void createStopTestModeResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;

   void createLinkQualityResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result, IN const BTSRssi rssi, IN const BTSLinkQuality linkQuality) const;
};

} //btstackif

#endif //_TEST_MODE_H_
