/**
 * @file SwitchBluetooth.h
 *
 * @par SW-Component
 * State machine for switch Bluetooth on/off
 *
 * @brief Implementation of generic switch Bluetooth on/off state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic switch Bluetooth on/off state machine.
 */

#ifndef _SWITCH_BLUETOOTH_H_
#define _SWITCH_BLUETOOTH_H_

#include "StateMachine.h"
#include "ISwitchBluetooth.h"
#include "ISwitchBluetoothCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "IConfigurationClient.h"
#include "ExtendedTimerEntry.h"
#include "BtsMessageQueue.h"

namespace btstackif {

/**
 * SwitchBluetooth class.
 */
class SwitchBluetooth : public StateMachine, public ISwitchBluetooth, public ISwitchBluetoothCallback, public IExtendedTimeoutHandler, public IConfigurationClient
{
public:
   /**
    * Default constructor.
    */
   SwitchBluetooth();

   /**
    * Destructor.
    */
   virtual ~SwitchBluetooth();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN ISwitchBluetoothRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   virtual void setConfigurationIf(IN IConfiguration* configurationIf);

   virtual IConfigurationClient* getConfigurationClientHandler(void);

   virtual void setTestModeIf(IN ITestMode* testModeIf);

   virtual void setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModes);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetBluetoothOnOffStatus& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SwitchBluetoothOnOff& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidGetRequest(IN const App2Bts_GetBluetoothOnOffStatus& request) const;

   virtual bool isValidSetRequest(IN const App2Bts_SwitchBluetoothOnOff& request) const;

   virtual bool switchBtOnOff(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SwitchBluetoothOnOff& request);

   virtual bool getAppBtMode(void) const;

   virtual bool getSwitchActive(void) const;

   virtual BTSAdapterMode getCurrentAdMode(void) const;

   virtual BTSBluetoothMode getNewBtMode(void) const;

   virtual const BTSBDAddress& getLocalAddress(void) const;

   virtual void updateCurrentStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) const;

   virtual void setCurrentBtMode(IN const BTSBluetoothMode mode);

   virtual void setBtOffReason(IN const BTSBluetoothOffReason reason);

   virtual void sendGetHwVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void readRegistryData(void);

   virtual void pushToSwitchBtQueue(IN App2Bts_BaseMessage* message, bool withLock = true);

   virtual BTSHciChipVendorId getHciChipVendorId(void) const;

   virtual BTSBtModuleId getBtModuleId(void) const;

   virtual void setStackLocalBDAddress(IN const BTSBDAddress& address);

   virtual void setStackPoweredMode(IN const bool enable);

   virtual void setStackHciMode(IN const bool enable);

   virtual void reportVersionInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem,
            IN const ::std::string& chipId, IN const ::std::string& buildId, IN const ::std::string& fwVersion, IN const BTSHciChipVendorId vendorId, IN const BTSBtModuleId moduleId);

   virtual void updateHciMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool hciMode);

   virtual void updatePowered(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool powered, IN const BTSIpcCommonErrorCode errorCode, IN const bool force = false);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void setConfigurationMasterIf(IN IConfigurationMaster* master);

   virtual void startGlobalConfiguration(void);

   virtual void stopGlobalConfiguration(void);

   virtual bool isSingleConfigurationOngoing(void) const;

   virtual bool setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred);

private:
   enum
   {
      TIMEOUT_UPDATE_ADAPTER_POWERED = 2000 /**< timeout for property update of Adapter::Powered after requesting a set */
   };

   ISwitchBluetoothRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IConfigurationMaster* _configurationMaster; /**< configuration master interface */
   ITestMode* _testModeIf; /**< test mode interface */
   ILocalAdapterModes* _localAdapterModesIf; /**< local adapter modes interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   BTSApp2BtsMessageRequestItem _requestItem; /**< request item */
   BTSBluetoothMode _newBtMode; /**< new BT mode */
   BTSBluetoothMode _currentBtMode; /**< current BT mode */
   BTSBluetoothOffReason _offReason; /**< Bluetooth off reason */
   BTSAdapterMode _newAdMode; /**< new adapter mode */
   BTSAdapterMode _currentAdMode; /**< current adapter mode */
   BTSBDAddress _localAddress; /**< local Bluetooth address */
   bool _switchActive; /**< flag indicating that switch BT on/off is active */
   bool _stackPowered; /**< current stack value */
   bool _stackHciMode; /**< current stack value */
   bool _setActive; /**< set request active flag */
   ExtendedTimerEntry _timerUpdate; /**< timer for status update */
   bool _globalConfigActive; /**< flag for global configuration */
   bool _configRequestOngoing; /**< flag for ongoing config request */
   bool _configRequestDone; /**< flag for finished config request */
   unsigned int _nmbRetry; /**< current number of retries */
   const unsigned int _maxRetry; /**< max number of retries */
   MessageQueue< App2Bts_BaseMessage > _app2BtsSwitchBtQueue; /**< switch BT queue */
   const ::std::string _regPath; /**< registry path */
   const ::std::string _chipIdKey; /**< registry key for chip id */
   const ::std::string _buildIdKey; /**< registry key for build id */
   const ::std::string _fwVersionKey; /**< registry key for FW version */
   BTSHciChipVendorId _vendorId; /**< vendor ID */
   BTSBtModuleId _moduleId; /**< module ID */

   void handleEndOfSwitchSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool switchOn);

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   App2Bts_BaseMessage* getApp2BtsWorkingMessage(void);

   void rejectStoredRequests(void);

   void writeRegistryData(IN const ::std::string& chipId, IN const ::std::string& buildId, IN const ::std::string& fwVersion);

   void sendPoweredMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool enable);

   void setRequestActive(IN const bool enable);

   bool getRequestActive(void) const;

   bool checkRetry(void);

   void resetRetry(void);

   BTSBluetoothMode getCurrentBtMode(void) const;

   void setCurrentAdMode(IN const BTSAdapterMode mode);

   void setNewBtMode(IN const BTSBluetoothMode mode);

   void setNewAdMode(IN const BTSAdapterMode mode);

   BTSAdapterMode getNewAdMode(void) const;

   void setSwitchActive(IN const bool enable);

   void forceLocalModesUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   void createStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll) const;

   void createResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;
};

} //btstackif

#endif //_SWITCH_BLUETOOTH_H_
