/**
 * @file Startup.cpp
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Implementation of generic startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic startup state machine.
 */

#include "Startup.h"
#include "IStartupRequest.h"
#include "IBasicControl.h"
#include "IConfiguration.h"
#include "ISwitchBluetooth.h"
#include "ILocalAdapterModes.h"
#include "ITimerPool.h"
#include "FwErrmemPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Startup.cpp.trc.h"
#endif
#endif

namespace btstackif {

Startup::Startup() :
_requestIf(0),
_controlIf(0),
_configurationIf(0),
_switchBluetoothIf(0),
_localAdapterModesIf(0),
_timerPoolIf(0),
_startupOngoing(false),
_serviceAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE),
_initStatus(BTS_REQ_FAILED),
_app2BtsStartupQueue(),
_smList(),
_startupTimer(),
_startupTimeout(15000),
_startupCounter(1)
{
   _app2BtsStartupQueue.setWarningSize(128);

   // reserve some entries
   _smList.reserve(30);
}

Startup::~Startup()
{
   _requestIf = 0;
   _controlIf = 0;
   _configurationIf = 0;
   _switchBluetoothIf = 0;
   _localAdapterModesIf = 0;
   _timerPoolIf = 0;
   _app2BtsStartupQueue.empty();
}

void Startup::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _configurationIf
   // keep _switchBluetoothIf
   // keep _localAdapterModesIf
   // keep _timerPoolIf
   _startupOngoing = false;
   _serviceAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;
   _initStatus = BTS_REQ_FAILED;
   rejectStoredRequests();
   // keep _smList
   stopTimer(_startupTimer);
   ++_startupCounter;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void Startup::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void Startup::setInstance(IN IStartupRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void Startup::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void Startup::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setAgentCapability(_configurationIf->getConfiguration().agentCapability);
}

void Startup::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth)
{
   _switchBluetoothIf = switchBluetooth;

   FW_ERRMEM_ASSERT(0 != _switchBluetoothIf);
}

void Startup::setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModes)
{
   _localAdapterModesIf = localAdapterModes;

   FW_ERRMEM_ASSERT(0 != _localAdapterModesIf);
}

void Startup::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setTimerPoolIf(_timerPoolIf);
}

IStateMachine* Startup::getSmEntryInterface(void)
{
   return this;
}

void Startup::handleTriggerInitializedCallback(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   if(true == _startupOngoing)
   {
      // wait for end of initialization; update happens automatically
   }
   else
   {
      _controlIf->sendInternalApp2BtsMessage(ptrNew_App2Bts_TriggerFbConnectionInitialized());
   }
}

bool Startup::handleFbConnectionInitialized(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2IpcMsgList);

   if(true == _startupOngoing)
   {
      // wait for end of initialization; update happens automatically
   }
   else
   {
      // update now
      updateInitializedStatus(bts2AppMsgList);
   }

   return false;
}

BTSDbusServiceAvailability Startup::getServiceAvailability(void) const
{
   // an active startup timer indicates a waiting state
   if(true == isTimerActive(_startupTimer))
   {
      return BTS_DBUS_SERVICE_WAITING;
   }
   else
   {
      return _serviceAvailability;
   }
}

bool Startup::getStartupOngoing(void) const
{
   return _startupOngoing;
}

void Startup::pushToStartupQueue(IN App2Bts_BaseMessage* message, bool withLock /*= true*/)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(message);

   _app2BtsStartupQueue.push(message, withLock);
}

void Startup::registerSmEntry(IN IStateMachine* entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(entry);

   // please ensure outside of this function that each entry is only registered once

   _smList.push_back(entry);
}

void Startup::startupSequenceStarted(IN const BTSDbusServiceAvailability availability)
{
   ETG_TRACE_USR2((" startupSequenceStarted: _serviceAvailability=%d availability=%d", _serviceAvailability, availability));

   _serviceAvailability = availability;

   _startupOngoing = true;

   ETG_TRACE_USR1((" ### startup started (count=%u) ###", _startupCounter));
}

void Startup::indicateAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability, IN const BTSBluetoothOffReason reason /*= BTS_BT_OFF_REASON_NOT_VALID*/)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_switchBluetoothIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_localAdapterModesIf);

   // handle only changed values
   if(availability == _serviceAvailability)
   {
      return;
   }

   ETG_TRACE_USR2((" indicateAvailability: _serviceAvailability=%d availability=%d", _serviceAvailability, availability));

   // store new value
   _serviceAvailability = availability;

   // handle new value
   if(BTS_DBUS_SERVICE_WAITING == _serviceAvailability)
   {
      // this is an intermediate state => wait for final state
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == _serviceAvailability)
   {
      // startup failed or Bluetooth stack crashed

      // stop timer
      stopTimer(_startupTimer);

      // set BT off reason
      _switchBluetoothIf->setBtOffReason(reason);

      // check available services to print collected data (for debug purpose)
      _requestIf->checkAvailableServices();

      // check current state
      if(true == _startupOngoing)
      {
         // reject all requests stored in startup queue
         rejectStoredRequests();

         // startup handling finished
         _startupOngoing = false;

         ETG_TRACE_USR1((" ### startup finished (failed) (count=%u) ###", _startupCounter));

         // BT off reason was set before, set current BT mode
         _switchBluetoothIf->setCurrentBtMode(BTS_BT_MODE_OFF);
         setInitializedStatus(bts2AppMsgList, BTS_REQ_DBUS_ERROR);
      }
      else
      {
         ETG_TRACE_USR1((" ### Bluetooth stack crashed !? (count=%u) ###", _startupCounter));

         // BT off reason was set before, set current BT mode
         _switchBluetoothIf->setCurrentBtMode(BTS_BT_MODE_OFF);
         setInitializedStatus(bts2AppMsgList, BTS_REQ_DBUS_ERROR);

         // reset all SMs (for each request in working queue an active SM (entry) shall exist; handle waiting queue entries after resetting SMs)
         handleStackReset(bts2IpcMsgList, bts2AppMsgList, messageItem);
      }
   }
   else if(BTS_DBUS_SERVICE_AVAILABLE == _serviceAvailability)
   {
      // stop timer
      stopTimer(_startupTimer);

      ::std::vector< Bts2Ipc_BaseMessage* > sendBts2IpcMsgList;
      sendBts2IpcMsgList.reserve(10);
      ::std::vector< Bts2App_BaseMessage* > sendBts2AppMsgList;
      sendBts2AppMsgList.reserve(10);

      // check for startup handling
      if(true == _startupOngoing)
      {
         // normal startup behavior
         _startupOngoing = false;

         ETG_TRACE_USR1((" ### startup finished (success) (count=%u) ###", _startupCounter));
      }
      else
      {
         // we have following situation: startup failed due to timeout; afterwards startup sequence was completed successfully
         ETG_TRACE_USR1((" ### startup finished (success after timeout) (count=%u) ###", _startupCounter));
      }

      // check for requesting introspection data
      _requestIf->getIntrospections(sendBts2IpcMsgList, sendBts2AppMsgList);

      // check for requesting version information TODO: [low]: add to configuration sequence
      _switchBluetoothIf->sendGetHwVersionRequest(sendBts2IpcMsgList, sendBts2AppMsgList);

      // quick hack for NCG3D-15977 TODO: [low]: add to configuration sequence
      _localAdapterModesIf->setDiscoverableTimeout(sendBts2IpcMsgList, 0);

      // send any Bts2App / Bts2Ipc message now else we have wrong order of messages because Bts2App / Bts2Ipc messages created during processStartupQueue() will be processed first
      BTSApp2BtsMessageCompareItem item;
      _controlIf->sendBts2IpcMessageList(sendBts2IpcMsgList, item);
      _controlIf->sendBts2AppMessageList(sendBts2AppMsgList);

      // set BT off reason
      _switchBluetoothIf->setBtOffReason(BTS_BT_OFF_REASON_STARTUP);
      setInitializedStatus(bts2AppMsgList, BTS_REQ_SUCCESS);

      // process startup queue
      processStartupQueue();
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

bool Startup::getStartupOngoingFlag(void) const
{
   return _startupOngoing;
}

void Startup::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   if(false == _startupTimer.compare(timerId))
   {
      return;
   }

   // timer was started after Bluetooth stack reset; in case of timeout the Bluetooth stack did not restart properly; assume that restart failed
   indicateAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_DBUS_SERVICE_WAITING);
   indicateAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_DBUS_SERVICE_NOT_AVAILABLE, BTS_BT_OFF_REASON_DBUS_ERROR);
}

void Startup::updateInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_switchBluetoothIf);

   _switchBluetoothIf->updateCurrentStatus(bts2AppMsgList);

   Bts2App_FbConnectionInitialized* msg = ptrNew_Bts2App_FbConnectionInitialized();
   if(0 != msg)
   {
      msg->setRequestResult(_initStatus);

      bts2AppMsgList.push_back(msg);
   }
}

void Startup::setInitializedStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSRequestResult status, IN const bool update /*= true*/)
{
   _initStatus = status;

   if(true == update)
   {
      updateInitializedStatus(bts2AppMsgList);
   }
}

void Startup::processStartupQueue(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsStartupQueue.getBegin(); it != _app2BtsStartupQueue.getEnd(); ++it)
   {
      _controlIf->pushApp2BtsMessage(*it);
   }

   _app2BtsStartupQueue.clear(false); // single worker thread
}

void Startup::rejectStoredRequests(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   // reject all requests stored in startup queue
   ::std::vector< App2Bts_BaseMessage* > msgList;
   msgList.reserve(_app2BtsStartupQueue.getSize());

   for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsStartupQueue.getBegin(); it != _app2BtsStartupQueue.getEnd(); ++it)
   {
      msgList.push_back(*it);
   }

   _controlIf->sendDirectAnswerForApp2BtsMessages(msgList, (BTSCommonEnumClass)BTS_REQ_DBUS_ERROR, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   _app2BtsStartupQueue.empty(false); // single worker thread
}

void Startup::handleStackReset(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   ETG_TRACE_USR1((" handleStackReset: start"));

   // reset all SMs (for each request in working queue an active SM (entry) shall exist; handle waiting queue entries after resetting SMs)

   // reset all SMs
   for(::std::vector< IStateMachine* >::iterator it = _smList.begin(); it != _smList.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->forceInitialState(bts2AppMsgList);
      }
   }

   // empty working queue
   _controlIf->emptyWorkingQueue(false); // single worker thread

   // answer requests in waiting queue
   ::std::vector<App2Bts_BaseMessage*> msgList;
   _controlIf->getWaitingQueueEntries(msgList, false); // single worker thread
   _controlIf->sendDirectAnswerForApp2BtsMessagesWrapper(bts2AppMsgList, msgList, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);

   // empty waiting queue
   _controlIf->emptyWaitingQueue(false); // single worker thread

   // trigger reset of DBus interface handler
   _controlIf->triggerResetOfDbusIfHandler();

   // start timer to monitor stack restart
   startTimer(_startupTimer, _startupTimeout);

   ETG_TRACE_USR1((" handleStackReset: end"));
}

void Startup::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void Startup::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void Startup::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool Startup::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

} //btstackif
