/**
 * @file IStartupRequest.h
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Request interface definition for startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific startup state machine.
 */

#ifndef _I_STARTUP_REQUEST_H_
#define _I_STARTUP_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IStartupCallback;
class IBasicControl;
class ITimerPool;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IStartupRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~IStartupRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN IStartupCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set agent capability.
    *
    * @param[in] capability: agent capability
    */
   virtual void setAgentCapability(IN const BTSAgentCapability capability) = 0;

   /**
    * Check available services.
    */
   virtual void checkAvailableServices(void) = 0;

   /**
    * Get introspections.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void getIntrospections(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;
};

} //btstackif

#endif //_I_STARTUP_REQUEST_H_
