/**
 * @file IStartup.h
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Interface definition for generic startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for startup state machine.
 */

#ifndef _I_STARTUP_H_
#define _I_STARTUP_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IStartupRequest;
class IBasicControl;
class IConfiguration;
class ISwitchBluetooth;
class ILocalAdapterModes;
class ITimerPool;
class IStateMachine;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;

/**
 * Interface definition.
 */
class IStartup
{
public:
   /**
    * Destructor.
    */
   virtual ~IStartup() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IStartupRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetooth: switch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth) = 0;

   /**
    * Set local adapter modes interface.
    *
    * @param[in] localAdapterModes: local adapter modes interface
    */
   virtual void setLocalAdapterModesIf(IN ILocalAdapterModes* localAdapterModes) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Handle trigger for initialized callback.
    */
   virtual void handleTriggerInitializedCallback(void) = 0;

   /**
    * Handle event for FbConnectionInitialized.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool handleFbConnectionInitialized(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Get service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    *
    * @return = service availability
    */
   virtual BTSDbusServiceAvailability getServiceAvailability(void) const = 0;

   /**
    * Get startup ongoing flag.
    *
    * @return = startup ongoing flag
    */
   virtual bool getStartupOngoing(void) const = 0;

   /**
    * Push message to startup queue.
    *
    * @param[in] message: message
    * @param[in] withLock: flag for locking
    */
   virtual void pushToStartupQueue(IN App2Bts_BaseMessage* message, bool withLock = true) = 0;

   /**
    * Register SM entry.
    *
    * @param[in] entry: SM entry
    */
   virtual void registerSmEntry(IN IStateMachine* entry) = 0;
};

} //btstackif

#endif //_I_STARTUP_H_
