/**
 * @file SppPool.cpp
 *
 * @par SW-Component
 * State machine for SPP pool
 *
 * @brief Implementation of generic SPP pool state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic SPP pool state machine.
 */

#include "SppPool.h"
#include "ISppPoolRequest.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "IConfiguration.h"
#include "IConfigurationMaster.h"
#include "ISppPoolObserver.h"
#include "FwAssert.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/SppPool.cpp.trc.h"
#endif
#endif

namespace btstackif {

SppPool::SppPool() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_configurationIf(0),
_configurationMaster(0),
_globalConfigActive(false),
_observerList(),
_configRequestOngoing(false),
_configRequestDone(false),
_timerLocalSppServices(),
_timeoutLocalSppServicesUpdate(3000)
{
}

SppPool::~SppPool()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _configurationIf = 0;
   _configurationMaster = 0;
}

void SppPool::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _configurationIf
   // keep _configurationMaster
   _globalConfigActive = false;
   // keep _observerList
   _configRequestOngoing = false;
   _configRequestDone = false;
   // stop all timer
   stopTimer(_timerLocalSppServices);

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void SppPool::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void SppPool::setInstance(IN ISppPoolRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void SppPool::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_IF_NULL_PTR_RETURN(_controlIf);
   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void SppPool::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setTimerPoolIf(_timerPoolIf);
}

void SppPool::setConfigurationIf(IN IConfiguration* configuration)
{
   _configurationIf = configuration;

   FW_NORMAL_ASSERT(0 != _configurationIf);

   initSppPool();
}

IConfigurationClient* SppPool::getConfigurationClientHandler(void)
{
   return this;
}

IStateMachine* SppPool::getSmEntryInterface(void)
{
   return this;
}

void SppPool::setMaxNumberSppConnections(IN const BTSSppInstanceId limit)
{
   (void)(limit);
   // ignore
   FW_NORMAL_ASSERT_ALWAYS();
}

BTSSppInstanceId SppPool::getMaxNumberSppConnections(void) const
{
   FW_IF_NULL_PTR_RETURN_NULL(_requestIf);

   return _requestIf->getMaxNumberSppConnections();
}

void SppPool::registerObserver(IN ISppPoolObserver* observer)
{
   FW_IF_NULL_PTR_RETURN(observer);

   _observerList.insert(observer);
}

void SppPool::updateLocalSppUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSUuidList& uuidList, IN const bool success)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(uuidList);
   (void)(success);

   FW_IF_NULL_PTR_RETURN(_configurationMaster);

   // check if request is ongoing (ignore other updates)
   if(true == _configRequestOngoing)
   {
      // local SPP services are only configured during stack configuration phase
      FW_NORMAL_ASSERT(true == _globalConfigActive);

      // stop timer
      stopTimer(_timerLocalSppServices);
      // mark as not ongoing
      _configRequestOngoing = false;
      // mark as finished
      _configRequestDone = true;
      // ignore result because there is no retry

#ifdef ENABLE_NEW_CONFIGURATION_HANDLING
      // inform configuration master to execute next step
      (void)_configurationMaster->doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
#endif
   }
}

void SppPool::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_requestIf);

   if(true == _timerLocalSppServices.compare(timerId))
   {
      if(true == _configRequestOngoing)
      {
         const BTSUuidList uuidList;
         _requestIf->sendVirtualLocalSppUuidsUpdate(bts2IpcMsgList, bts2AppMsgList, uuidList, BTS_IPC_UPDATE_TIMEOUT);
      }
   }
}

void SppPool::setConfigurationMasterIf(IN IConfigurationMaster* master)
{
   _configurationMaster = master;

   FW_NORMAL_ASSERT(0 != _configurationMaster);
}

void SppPool::startGlobalConfiguration(void)
{
   _globalConfigActive = true;
   _configRequestOngoing = false;
   // do not reset _configRequestDone because this configuration shall be done only once
}

void SppPool::stopGlobalConfiguration(void)
{
   _globalConfigActive = false;
   _configRequestOngoing = false;
}

bool SppPool::isSingleConfigurationOngoing(void) const
{
   if(false == _globalConfigActive)
   {
      return false;
   }

   // check if call to configure local SPP services is ongoing
   return _configRequestOngoing;
}

bool SppPool::setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred)
{
   (void)(messageItem);
   (void)(beforeBtOn);

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   if(false == _globalConfigActive)
   {
      return false;
   }

   // ETG_TRACE_USR3((" SppPool::setSingleConfiguration(): beforeBtOn=%d _configRequestOngoing=%d errorOccurred=%d _configRequestDone=%d", beforeBtOn, _configRequestOngoing, errorOccurred, _configRequestDone));

   if(true == _configRequestOngoing)
   {
      // configuration request is ongoing => wait for result
      return true;
   }

   if(true == errorOccurred)
   {
      // error occurred => do not continue
      return false;
   }

   if(false == _configRequestDone)
   {
      // configure local SPP services
      const BTSSppServiceInfoList& sppServiceInfoList(_configurationIf->getConfiguration().sppServiceInfoList);
      BTSUuidList uuidList;
      uuidList.reserve(sppServiceInfoList.size());

      // check for valid UUIDs was already done, only valid UUIDs are currently stored
      for(size_t i = 0; i < sppServiceInfoList.size(); i++)
      {
         if(false == sppServiceInfoList[i].localUuid.empty())
         {
            uuidList.push_back(sppServiceInfoList[i].localUuid);
         }
      }

      // do not send empty list because initial property value is an empty list
      if(0 == uuidList.size())
      {
         // mark as finished
         _configRequestDone = true;
         return false;
      }
      else
      {
         _requestIf->setLocalSppUuids(bts2IpcMsgList, bts2AppMsgList, uuidList);
         // start timer
         startTimer(_timerLocalSppServices, _timeoutLocalSppServicesUpdate);
         // mark as ongoing
         _configRequestOngoing = true;

         return true;
      }
   }

   return false;
}

void SppPool::initSppPool(void)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);
   FW_IF_NULL_PTR_RETURN(_configurationIf);

   const BTSSppServiceInfoList& sppServiceInfoList(_configurationIf->getConfiguration().sppServiceInfoList);
   size_t counter(0);

   for(::std::vector< BTSSppServiceInfo >::const_iterator it = sppServiceInfoList.begin(); it != sppServiceInfoList.end(); ++it)
   {
      if(false == it->localUuid.empty())
      {
         ++counter;
      }
   }

   if(counter > _requestIf->getMaxNumberLocalSppServices())
   {
      // more local UUIDs configured than available local SPP instances
      FW_NORMAL_ASSERT_ALWAYS();
   }

   // debug output
   printSppPool();
}

void SppPool::printSppPool(void) const
{
   FW_IF_NULL_PTR_RETURN(_requestIf);
   FW_IF_NULL_PTR_RETURN(_configurationIf);

   const BTSSppServiceInfoList& sppServiceInfoList(_configurationIf->getConfiguration().sppServiceInfoList);
   size_t counter(0);
   size_t i(0);

   for(::std::vector< BTSSppServiceInfo >::const_iterator it = sppServiceInfoList.begin(); it != sppServiceInfoList.end(); ++it, ++i)
   {
      const BTSSppServiceInfo& serviceInfo(*it);
      bool ignore(false);

      if(false == serviceInfo.localUuid.empty())
      {
         // check local SPP service limitation
         if(counter < _requestIf->getMaxNumberLocalSppServices())
         {
            // use
            ++counter;
         }
         else
         {
            // ignore
            ignore = true;
         }
      }

      if(false == ignore)
      {
         ETG_TRACE_USR1((" printSppPool: [%u]: localUuid=%40s remoteUuid=%40s serviceName=%s", i, serviceInfo.localUuid.c_str(), serviceInfo.remoteUuid.c_str(), serviceInfo.serviceName.c_str()));
      }
      else
      {
         ETG_TRACE_USR1((" printSppPool: [%u]: localUuid=%40s remoteUuid=%40s serviceName=%s (ignored)", i, serviceInfo.localUuid.c_str(), serviceInfo.remoteUuid.c_str(), serviceInfo.serviceName.c_str()));
      }
   }
}

void SppPool::informObserver(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSSppInstanceId instance, IN const BTSRequestResult result)
{
   for(::std::set< ISppPoolObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->settingUuidFinished(bts2IpcMsgList, bts2AppMsgList, messageItem, instance, result);
      }
   }
}

void SppPool::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void SppPool::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void SppPool::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool SppPool::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

} //btstackif
