/**
 * @file SppPool.h
 *
 * @par SW-Component
 * State machine for SPP pool
 *
 * @brief Implementation of generic SPP pool state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic SPP pool state machine.
 */

#ifndef _SPP_POOL_H_
#define _SPP_POOL_H_

#include "StateMachine.h"
#include "ISppPool.h"
#include "ISppPoolCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "IConfigurationClient.h"
#include "ExtendedTimerEntry.h"

#include <set>

namespace btstackif {

/**
 * SppPool class.
 */
class SppPool : public StateMachine, public ISppPool, public ISppPoolCallback, public IExtendedTimeoutHandler, public IConfigurationClient
{
public:
   /**
    * Default constructor.
    */
   SppPool();

   /**
    * Destructor.
    */
   virtual ~SppPool();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN ISppPoolRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setConfigurationIf(IN IConfiguration* configuration);

   virtual IConfigurationClient* getConfigurationClientHandler(void);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void setMaxNumberSppConnections(IN const BTSSppInstanceId limit);

   virtual BTSSppInstanceId getMaxNumberSppConnections(void) const;

   virtual void registerObserver(IN ISppPoolObserver* observer);

   virtual void updateLocalSppUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSUuidList& uuidList, IN const bool success);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void setConfigurationMasterIf(IN IConfigurationMaster* master);

   virtual void startGlobalConfiguration(void);

   virtual void stopGlobalConfiguration(void);

   virtual bool isSingleConfigurationOngoing(void) const;

   virtual bool setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred);

private:
   ISppPoolRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IConfigurationMaster* _configurationMaster; /**< configuration master interface */
   bool _globalConfigActive; /**< flag for global configuration */
   ::std::set< ISppPoolObserver* > _observerList; /**< observer list */
   bool _configRequestOngoing; /**< flag for ongoing config request */
   bool _configRequestDone; /**< flag for finished config request */
   ExtendedTimerEntry _timerLocalSppServices; /**< timer */
   const BTSTimeValue _timeoutLocalSppServicesUpdate; /**< timeout for update */

   void initSppPool(void);

   void printSppPool(void) const;

   void informObserver(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSSppInstanceId instance, IN const BTSRequestResult result);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;
};

} //btstackif

#endif //_SPP_POOL_H_
