/**
 * @file ISppPoolRequest.h
 *
 * @par SW-Component
 * State machine for SPP pool
 *
 * @brief Request interface definition for SPP pool state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific SPP pool state machine.
 */

#ifndef _I_SPP_POOL_REQUEST_H_
#define _I_SPP_POOL_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ISppPoolCallback;
class IBasicControl;
class ITimerPool;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ISppPoolRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~ISppPoolRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN ISppPoolCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Get max number of SPP connections.
    *
    * @return = max number of SPP connections
    */
   virtual BTSSppInstanceId getMaxNumberSppConnections(void) const = 0;

   /**
    * Get max number of local SPP services.
    *
    * @return = max number of local SPP services
    */
   virtual BTSSppInstanceId getMaxNumberLocalSppServices(void) const = 0;

   /**
    * Set SPP UUID.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] instance: SPP instance
    * @param[in] outgoingUuid: outgoing SPP UUID
    * @param[in] incomingUuid: incoming SPP UUID
    */
   virtual void setUuid(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSSppInstanceId instance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid) = 0;

   /**
    * Configure local SPP UUIDs.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] uuidList: UUID list
    */
   virtual void setLocalSppUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSUuidList& uuidList) = 0;

   /**
    * Send virtual update message for local SPP UUIDs (loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] uuidList: UUID list
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualLocalSppUuidsUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSUuidList& uuidList, IN const BTSIpcCommonErrorCode errorCode) = 0;
};

} //btstackif

#endif //_I_SPP_POOL_REQUEST_H_
