/**
 * @file ISppPool.h
 *
 * @par SW-Component
 * State machine for SPP pool
 *
 * @brief Interface definition for generic SPP pool state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for SPP pool state machine.
 */

#ifndef _I_SPP_POOL_H_
#define _I_SPP_POOL_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ISppPoolRequest;
class IBasicControl;
class ITimerPool;
class IConfiguration;
class IConfigurationClient;
class IStateMachine;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class ISppPoolObserver;

/**
 * Interface definition.
 */
class ISppPool
{
public:
   /**
    * Destructor.
    */
   virtual ~ISppPool() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN ISppPoolRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configuration: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configuration) = 0;

   /**
    * Get configuration client handler.
    *
    * @return = configuration client handler
    */
   virtual IConfigurationClient* getConfigurationClientHandler(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Set max number of SPP connections. To be used for testing purpose.
    *
    * @param[in] limit: max number of SPP connections
    */
   virtual void setMaxNumberSppConnections(IN const BTSSppInstanceId limit) = 0;

   /**
    * Get max number of SPP connections.
    *
    * @return = max number of SPP connections
    */
   virtual BTSSppInstanceId getMaxNumberSppConnections(void) const = 0;

   /**
    * Register observer.
    *
    * @param[in] observer: observer
    */
   virtual void registerObserver(IN ISppPoolObserver* observer) = 0;
};

} //btstackif

#endif //_I_SPP_POOL_H_
