/**
 * @file ServiceSearch.h
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Implementation of generic service search state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic service search state machine.
 */

#ifndef _SERVICE_SEARCH_H_
#define _SERVICE_SEARCH_H_

#include "StateMachine.h"
#include "IServiceSearch.h"
#include "IServiceSearchCallback.h"
#include "IDeviceObserver.h"
#include "IExtendedTimeoutHandler.h"
#include "ServiceSearchData.h"

#include <set>

namespace btstackif {

/**
 * Service search class.
 */
class ServiceSearch : public StateMachine, public IServiceSearch, public IServiceSearchCallback, public IDeviceObserver, public IExtendedTimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   ServiceSearch();

   /**
    * Destructor.
    */
   virtual ~ServiceSearch();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IServiceSearchRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setConfigurationIf(IN IConfiguration* configurationIf);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   virtual void setProtocolManagerIf(IN IProtocolManager* protocolManagerIf);

   virtual IDeviceObserver* getDeviceObserver(void);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode, IN const bool internalRequest = false) const;

   virtual bool isValidStartRequest(IN const App2Bts_StartRemoteServiceSearch& request) const;

   virtual bool isValidStartRequest(IN const BTSBDAddress& address, IN const BTSSearchType searchType) const;

   virtual bool isValidCancelRequest(IN const App2Bts_CancelRemoteServiceSearch& request) const;

   virtual bool startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request);

   virtual bool startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSTimeValue pageTimeout);

   virtual void triggerSdpSearchRequest(IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const bool highPrio = true);

   virtual bool cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelRemoteServiceSearch& request);

   virtual bool cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType);

   virtual void cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual bool isAnySearchOngoing(void) const;

   virtual bool isSearchOngoing(IN const BTSBDAddress& address) const;

   virtual bool isUuidAvailable(IN const BTSBDAddress& address, IN const BTSUuid& uuid) const;

   virtual bool isSdpSearchDone(IN const BTSBDAddress& address, IN const BTSSearchType searchType);

   virtual bool isSdpRecordAvailable(IN const BTSBDAddress& address, IN const BTSSearchType searchType);

   virtual bool isDeviceCreationOngoing(void) const;

   virtual bool getMasInstanceId(OUT BTSMasInstanceId& instanceId, const BTSBDAddress& address, IN const BTSMasInstanceName& instanceName) const;

   virtual bool getMasInstanceName(OUT BTSMasInstanceName& instanceName, const BTSBDAddress& address, IN const BTSMasInstanceId instanceId) const;

   virtual void requestServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol);

   virtual void setSppSupportSimulation(IN const bool enable);

   virtual void setPanSupportSimulation(IN const bool enable);

   virtual void setTestTriggerCancelServiceSearch(IN const bool enable);

   virtual void setIgnoreServiceSearchData(IN const bool enable);

   virtual void setIgnoreSppUuids(IN const bool enable);

   virtual void registerObserver(IN IServiceSearchObserver* observer);

   virtual void handleTestCommandGetDeviceVendorId(void);

   virtual void handleTestCommandGetAllSearchResults(void);

   virtual void setRetryMaxCounter(IN const unsigned int max);

   virtual void startTestSdpSearchAfterPairing(IN const BTSBDAddress& address);

   virtual void serviceSearchResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, IN const BTSBDAddress& address, IN const BTSIpcCommonErrorCode result);

   virtual void cancelServiceSearchResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSIpcCommonErrorCode result);

   virtual void setSdpSearchActiveAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool enable, IN const bool success);

   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool success);

   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void setPairingStatus(IN const BTSBDAddress& address, IN const bool success);

   virtual void updateDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSSupportedServices& supportedServices, IN const BTSUuidList& uuidList);

   virtual void updateServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSProfileVersion version);

   virtual void updateDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVendorId vendorId);

   virtual void updateDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVendorIdSource vendorIdSource);

   virtual void updateDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSProductId productId);

   virtual void updateDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSVersion version);

   virtual void updatePbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSPbapSdpRecord& record, IN const bool valid);

   virtual void updateMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSBDAddress& address, IN const BTSMapSdpRecordList& recordList, IN const bool valid);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address);

   virtual ServiceSearchData& getDataEntry(IN const BTSBDAddress& address);

   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected);

   virtual void deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   IServiceSearchRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   IProtocolManager* _protocolManagerIf; /**< protocol manager interface */
   ::std::map< BTSBDAddress, ServiceSearchData > _serviceSearchList; /**< map containing service search information */
   ::std::set< IServiceSearchObserver* > _observerList; /**< observer list */
   bool _simulateSppSupport; /**< flag for simulation of SPP support */
   bool _simulatePanSupport; /**< flag for simulation of PAN support */
   BTSUuid _simulateSppUuid; /**< UUID for simulation of SPP support */
   bool _testTriggerCancelServiceSearch; /**< flag for testing special sequence */
   bool _ignoreServiceSearchData; /**< flag for testing special sequence */
   bool _ignoreSppUuids; /**< flag for testing no SPP support */
   const BTSTimeValue _timeoutRetry; /**< timeout value for retry */
   unsigned int _maxRetry; /**< max value for retry */

   ServiceSearchData& checkServiceSearchList(IN const BTSBDAddress& address);

   ::std::map< BTSBDAddress, ServiceSearchData >::iterator getEntry(IN const BTSBDAddress& address);

   ::std::map< BTSBDAddress, ServiceSearchData >::const_iterator getEntry(IN const BTSBDAddress& address) const;

   ServiceSearchData::SdpRecordAvailableBit convertSearchType2SearchDoneBit(IN const BTSSearchType searchType) const;

   ServiceSearchData::SdpRecordAvailableBit convertSearchType2RecordAvailableBit(IN const BTSSearchType searchType) const;

   BTSProfileVersion getProfileVersion(IN const BTSServiceInfoList& infoList, IN const BTSSupportedServicesBit service);

   void setProfileVersion(IN BTSServiceInfoList& infoList, IN const BTSSupportedServicesBit service, IN const BTSProfileVersion version);

   void checkForSearchCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN ServiceSearchData& entry);

   void checkForOngoingSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   void checkDidServiceRecord(IN ServiceSearchData& entry);

   unsigned int getPendingGetDataBits(IN const BTSSearchType searchType, IN const bool sdpAfterPairing) const;

   void resetSearchDoneBits(IN ServiceSearchData::SdpRecordAvailable& recordAvailable) const;

   void triggerGetRequests(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const bool sdpAfterPairing, IN ServiceSearchData& entry);

   void createServiceSearchResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult resultServices, IN const BTSRequestResult resultSpp, IN const BTSRequestResult resultDid, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const;

   void createSupportedServicesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData& entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const;

   void createSupportedServicesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData* entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const;

   void createServiceRecordUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData& entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const;

   void createServiceRecordUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData* entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const;

   void createSppCapabilitiesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData& entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const;

   void createSppCapabilitiesUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const ServiceSearchData* entry, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) const;

   void startPageTimeoutTimer(IN ServiceSearchData& entry);

   void stopPageTimeoutTimer(IN ServiceSearchData& entry);

   bool pageTimeoutHappened(IN ServiceSearchData& entry) const;

   BTSRequestResult getSearchResult(IN const BTSRequestResult result, IN const bool pageTimeoutOccurred) const;

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   bool retryStarted(IN const BTSIpcCommonErrorCode errorCode, IN ServiceSearchData& entry, IN const BTSTimeValue timeout);

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSApp2BtsMessageCompareItem& item);

   App2Bts_BaseMessage* getWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item);

   void informObserver(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSRequestResult result);

   void handleEndOfFailedSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool* connectFailed, INOUT ServiceSearchData& entry, IN const BTSBDAddress& address);

   bool isUuidAvailableInSdpRecord(IN const ServiceSearchData& entry, IN const BTSUuid& uuid) const;

   bool isStartRequestPending(IN const BTSBDAddress& address, IN const BTSSearchType searchType) const;

   bool getAnyProfileConnected(IN const BTSBDAddress& address) const;

   bool isUuidInList(IN const BTSUuid& uuid, IN const BTSSppCapabilityList& capabilityList) const;
};

} //btstackif

#endif //_SERVICE_SEARCH_H_
