/**
 * @file IServiceSearch.h
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Interface definition for generic service search state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for service search state machine.
 */

#ifndef _I_SERVICE_SEARCH_H_
#define _I_SERVICE_SEARCH_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IServiceSearchRequest;
class IBasicControl;
class ITimerPool;
class IConfiguration;
class IDeviceManager;
class IProtocolManager;
class IDeviceObserver;
class IStateMachine;
class App2Bts_StartRemoteServiceSearch;
class App2Bts_CancelRemoteServiceSearch;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IServiceSearchObserver;

/**
 * Interface definition.
 */
class IServiceSearch
{
public:
   /**
    * Destructor.
    */
   virtual ~IServiceSearch() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IServiceSearchRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set protocol manager interface.
    *
    * @param[in] protocolManagerIf: protocol manager interface
    */
   virtual void setProtocolManagerIf(IN IProtocolManager* protocolManagerIf) = 0;

   /**
    * Returns device observer.
    *
    * @return = device observer
    */
   virtual IDeviceObserver* getDeviceObserver(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] searchType: search type
    * @param[in] user: user
    * @param[in] handle: session handle
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    * @param[in] internalRequest: flag for internal request
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode, IN const bool internalRequest = false) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidStartRequest(IN const App2Bts_StartRemoteServiceSearch& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] address: device address
    * @param[in] searchType: search type
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidStartRequest(IN const BTSBDAddress& address, IN const BTSSearchType searchType) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidCancelRequest(IN const App2Bts_CancelRemoteServiceSearch& request) const = 0;

   /**
    * Start service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request) = 0;

   /**
    * Start service search (internal).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] searchType: search type
    * @param[in] pageTimeout: page timeout
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSTimeValue pageTimeout) = 0;

   /**
    * Trigger internal SDP search request.
    *
    * @param[in] address: device address
    * @param[in] searchType: search type
    * @param[in] highPrio: high prio flag
    */
   virtual void triggerSdpSearchRequest(IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const bool highPrio = true) = 0;

   /**
    * Cancel service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelRemoteServiceSearch& request) = 0;

   /**
    * Cancel service search (internal).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] searchType: search type
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType) = 0;

   /**
    * Cancel service search (internal).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    */
   virtual void cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Check for any ongoing service search.
    *
    * @return = true: any service search ongoing,
    * @return = false: no service search ongoing
    */
   virtual bool isAnySearchOngoing(void) const = 0;

   /**
    * Check for ongoing service search activity. Only 1 service search can be active at the same time. Therefore search type is not needed.
    *
    * @param[in] address: device address
    *
    * @return = true: service search ongoing,
    * @return = false: no service search ongoing
    */
   virtual bool isSearchOngoing(IN const BTSBDAddress& address) const = 0;

   /**
    * Check for available UUID.
    *
    * @param[in] address: device address
    * @param[in] uuid: SPP uuid
    *
    * @return = true: UUID is offered by device,
    * @return = false: UUID is not offered by device
    */
   virtual bool isUuidAvailable(IN const BTSBDAddress& address, IN const BTSUuid& uuid) const = 0;

   /**
    * Checks if SDP search for given device / search type is finished.
    *
    * @param[in] address: device address
    * @param[in] searchType: search type
    *
    * @return = true: SDP search is finished,
    * @return = false: SDP search is not finished
    */
   virtual bool isSdpSearchDone(IN const BTSBDAddress& address, IN const BTSSearchType searchType) = 0;

   /**
    * Checks if SDP record for given device / search type is available.
    *
    * @param[in] address: device address
    * @param[in] searchType: search type
    *
    * @return = true: SDP record is available,
    * @return = false: SDP record is not available
    */
   virtual bool isSdpRecordAvailable(IN const BTSBDAddress& address, IN const BTSSearchType searchType) = 0;

   /**
    * Checks if any device creation is ongoing.
    *
    * @return = true: device creation is ongoing,
    * @return = false: no device creation is ongoing
    */
   virtual bool isDeviceCreationOngoing(void) const = 0;

   /**
    * Get MAS instance id.
    *
    * @param[out] instanceId: MAS instance id
    * @param[in] address: device address
    * @param[in] instanceName: MAS instance name
    *
    * @return = true: MAS instance found,
    * @return = false: MAS instance not found
    */
   virtual bool getMasInstanceId(OUT BTSMasInstanceId& instanceId, const BTSBDAddress& address, IN const BTSMasInstanceName& instanceName) const = 0;

   /**
    * Get MAS instance name.
    *
    * @param[out] instanceName: MAS instance name
    * @param[in] address: device address
    * @param[in] instanceId: MAS instance id
    *
    * @return = true: MAS instance found,
    * @return = false: MAS instance not found
    */
   virtual bool getMasInstanceName(OUT BTSMasInstanceName& instanceName, const BTSBDAddress& address, IN const BTSMasInstanceId instanceId) const = 0;

   /**
    * Request protocol version.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] deviceAddress: device address
    * @param[in] protocolId: protocol id
    */
   virtual void requestServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) = 0;

   /**
    * Activate/deactivate simulation of SPP support.
    *
    * @param[in] enable: enable flag
    */
   virtual void setSppSupportSimulation(IN const bool enable) = 0;

   /**
    * Activate/deactivate simulation of PAN support.
    *
    * @param[in] enable: enable flag
    */
   virtual void setPanSupportSimulation(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerCancelServiceSearch(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setIgnoreServiceSearchData(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setIgnoreSppUuids(IN const bool enable) = 0;

   /**
    * Register observer.
    *
    * @param[in] observer: observer
    */
   virtual void registerObserver(IN IServiceSearchObserver* observer) = 0;

   /**
    * Handle test command.
    */
   virtual void handleTestCommandGetDeviceVendorId(void) = 0;

   /**
    * Handle test command.
    */
   virtual void handleTestCommandGetAllSearchResults(void) = 0;

   /**
    * Set retry max counter. For testing purpose.
    *
    * @param[in] max: max counter
    */
   virtual void setRetryMaxCounter(IN const unsigned int max) = 0;

   /**
    * Send test trigger to start SDP search after pairing.
    *
    * @param[in] deviceAddress: device address
    */
   virtual void startTestSdpSearchAfterPairing(IN const BTSBDAddress& address) = 0;
};

} //btstackif

#endif //_I_SERVICE_SEARCH_H_
