/**
 * @file Rssi.cpp
 *
 * @par SW-Component
 * State machine for RSSI
 *
 * @brief Implementation of generic RSSI state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic RSSI + link quality state machine.
 */

#include "Rssi.h"
#include "IRssiRequest.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"

namespace btstackif {

Rssi::Rssi() :
_requestIf(0),
_controlIf(0),
_deviceManagerIf(0),
_requestItem(),
_address()
{
}

Rssi::~Rssi()
{
   _requestIf = 0;
   _controlIf = 0;
   _deviceManagerIf = 0;
}

void Rssi::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _deviceManagerIf
   _requestItem.reset();
   _address.clear();

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void Rssi::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current action
   switch(_requestItem.item.opCode)
   {
      case App2BtsOC_Last:
         // no request is pending => nothing to do
         break;
      case App2BtsOC_RequestLinkQuality:
         // getting link quality ongoing => update result as failed
         createLinkQualityResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, _address, BTS_REQ_FAILED, defaultRssiValue, defaultLinkQualityValue);
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   // reset control data
   reset();
}

void Rssi::setInstance(IN IRssiRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void Rssi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void Rssi::setDeviceManagerIf(IN IDeviceManager* deviceManager)
{
   _deviceManagerIf = deviceManager;

   FW_NORMAL_ASSERT(0 != _deviceManagerIf);
}

IStateMachine* Rssi::getSmEntryInterface(void)
{
   return this;
}

void Rssi::sendResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestLinkQuality& request, IN const BTSCommonEnumClass resultCode) const
{
   FW_IF_NULL_PTR_RETURN(_deviceManagerIf);

   // collect data for sending result
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;
   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   if(false == isValidRequest(request))
   {
      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else
   {
      if(false == _deviceManagerIf->isDeviceConnected(workingAddress))
      {
         sendResult = BTS_REQ_NOT_CONNECTED;
      }
      else
      {
         // in all other cases return failed
         sendResult = BTS_REQ_FAILED;
      }
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // send result for given control data
   createLinkQualityResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, sendResult, defaultRssiValue, defaultLinkQualityValue);
}

bool Rssi::isValidRequest(IN const App2Bts_RequestLinkQuality& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool Rssi::getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RequestLinkQuality& request)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   if(false == isValidRequest(request))
   {
      createLinkQualityResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, BTS_REQ_INVALID_PARAM, defaultRssiValue, defaultLinkQualityValue);

      return false;
   }

   if(false == _deviceManagerIf->isDeviceConnected(workingAddress))
   {
      createLinkQualityResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, BTS_REQ_NOT_CONNECTED, defaultRssiValue, defaultLinkQualityValue);

      return false;
   }

   // store data and process request
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   _address = workingAddress;
   _requestIf->getLinkQuality(bts2IpcMsgList, bts2AppMsgList, _address);

   return true;
}

void Rssi::getLinkQualityResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool success, IN const bool rssiSupported, IN const BTSRssi rssi, IN const bool linkQualitySupported, IN const BTSLinkQuality linkQuality)
{
   (void)(bts2IpcMsgList);

   if(_address != address)
   {
      // result for different address received => ignore
      return;
   }

   BTSRequestResult sendResult(BTS_REQ_FAILED);
   BTSRssi sendRssi(defaultRssiValue);
   BTSLinkQuality sendLinkQuality(defaultLinkQualityValue);

   if(true == success)
   {
      sendResult = BTS_REQ_SUCCESS;
      if(true == rssiSupported)
      {
         sendRssi = rssi;
      }
      if(true == linkQualitySupported)
      {
         sendLinkQuality = linkQuality;
      }
   }

   createLinkQualityResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, _address, sendResult, sendRssi, sendLinkQuality);

   // action is finished
   handleActionFinished(messageItem);

   // reset control data because action is finished
   _requestItem.reset();
   _address.clear();
}

App2Bts_BaseMessage* Rssi::getApp2BtsWorkingMessage(void)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

void Rssi::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   messageItem.deleteMessage = true;
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage();
   }
   FW_NORMAL_ASSERT(0 != messageItem.message);
}

void Rssi::createLinkQualityResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const BTSRssi rssi, IN const BTSLinkQuality linkQuality) const
{
   if(0 != user)
   {
      Bts2App_LinkQualityResult* msg = ptrNew_Bts2App_LinkQualityResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setBDAddress(address);
         msg->setRequestResult(result);
         msg->setRssi(rssi);
         msg->setLinkQuality(linkQuality);

         bts2AppMsgList.push_back(msg);
      }
   }
}

} //btstackif
