/**
 * @file IRssiCallback.h
 *
 * @par SW-Component
 * State machine for RSSI
 *
 * @brief Callback interface definition for RSSI state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific RSSI + link quality state machine.
 *          Callback interface must be provided by generic RSSI + link quality state machine.
 */

#ifndef _I_RSSI_CALLBACK_H_
#define _I_RSSI_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;

/**
 * Interface definition.
 */
class IRssiCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IRssiCallback() {}

   /**
    * Result for getting link quality.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] success: result
    * @param[in] rssiSupported: support flag for RSSI
    * @param[in] rssi: RSSI value
    * @param[in] linkQualitySupported: support flag for link quality
    * @param[in] linkQuality: link quality value
    */
   virtual void getLinkQualityResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool success, IN const bool rssiSupported, IN const BTSRssi rssi, IN const bool linkQualitySupported, IN const BTSLinkQuality linkQuality) = 0;

   /**
    * Get App2Bts message.
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(void) = 0;
};

} //btstackif

#endif //_I_RSSI_CALLBACK_H_
