/**
 * @file IRestrictedPairingConnecting.h
 *
 * @par SW-Component
 * State machine for restricted pairing and connecting
 *
 * @brief Interface definition for generic restricted pairing and connecting state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for restricted pairing and connecting state machine.
 */

#ifndef _I_RESTRICTED_PAIRING_CONNECTING_H_
#define _I_RESTRICTED_PAIRING_CONNECTING_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IStateMachine;
class App2Bts_SetDeviceForRestrictedPairingConnecting;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IRestrictedPairingConnecting
{
public:
   /**
    * Destructor.
    */
   virtual ~IRestrictedPairingConnecting() {}

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Set restriction and send status.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    */
   virtual void setRestrictionAndSendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetDeviceForRestrictedPairingConnecting& request) = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidRequest(IN const App2Bts_SetDeviceForRestrictedPairingConnecting& request) const = 0;

   /**
    * Set restricted pairing and connecting information.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool setRestriction(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetDeviceForRestrictedPairingConnecting& request) = 0;

   /**
    * Get restriction flag for pairing.
    *
    * @param[out] address: device address
    *
    * @return = true: pairing is restricted,
    * @return = false: pairing is not restricted
    */
   virtual bool isPairingRestricted(OUT BTSBDAddress& address) const = 0;

   /**
    * Get restriction flag for connecting.
    *
    * @param[out] address: device address
    *
    * @return = true: connecting is restricted,
    * @return = false: connecting is not restricted
    */
   virtual bool isConnectingRestricted(OUT BTSBDAddress& address) const = 0;
};

} //btstackif

#endif //_I_RESTRICTED_PAIRING_CONNECTING_H_
