/**
 * @file ProtocolSmPool.cpp
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of protocol state machine pool.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of protocol state machine pool.
 */

#include "ProtocolSmPool.h"
#include "ProtocolConnect.h"
#include "ProtocolDisconnect.h"
#include "ConfigurePageTimeout.h"
#include "FwAssert.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ProtocolSmPool.cpp.trc.h"
#endif
#endif

namespace btstackif {

ProtocolSmPool::ProtocolSmPool() :
_smPool()
{
}

ProtocolSmPool::~ProtocolSmPool()
{
}

ProtocolSmPool::ProtocolSmPool(const ProtocolSmPool& ref)
{
   _smPool = ref._smPool;
}

ProtocolSmPool& ProtocolSmPool::operator=(const ProtocolSmPool& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   _smPool = ref._smPool;

   return *this;
}

bool ProtocolSmPool::operator==(const ProtocolSmPool& ref) const
{
   bool result = true;

   result = (true == result) && (_smPool == ref._smPool);

   return result;
}

bool ProtocolSmPool::operator!=(const ProtocolSmPool& ref) const
{
   return !(operator==(ref));
}

ProtocolSmEntry& ProtocolSmPool::getSmEntry(IN const bool connect)
{
   if(true == connect)
   {
      return getSmEntryInternal(SM_TYPE_CONNECT);
   }
   else
   {
      return getSmEntryInternal(SM_TYPE_DISCONNECT);
   }
}

ProtocolSmEntry& ProtocolSmPool::getConfigureSmEntry(void)
{
   return getSmEntryInternal(SM_TYPE_CONFIGURE);
}

void ProtocolSmPool::freeSmContainer(IN ProtocolSmEntry& sm)
{
   if(sm.anySmAssigned())
   {
      for(size_t i = 0; i < _smPool.size(); i++)
      {
         if(sm == _smPool[i])
         {
            _smPool[i].setInUse(false);
            break;
         }
      }
   }
   sm.clearSm();

   ETG_TRACE_USR3((" freeSmContainer(): _smPool.size()=%u", _smPool.size()));
}

void ProtocolSmPool::freeAllSms(void)
{
   for(size_t i = 0; i < _smPool.size(); i++)
   {
      _smPool[i].destroySm();
   }
}

void ProtocolSmPool::reset(void)
{
   freeAllSms();
   _smPool.clear();
}

ProtocolSmEntry& ProtocolSmPool::getSmEntryInternal(IN const SmType smType)
{
   size_t index = 0;
   bool found = false;

   // check if unused SM exists
   for(size_t i = 0; ((i < _smPool.size()) && (false == found)); i++)
   {
      if(true == _smPool[i].getInUse())
      {
         // ignore
      }
      else
      {
         // check
         switch(smType)
         {
            case SM_TYPE_CONNECT:
               if(0 != _smPool[i].getConnectSm())
               {
                  index = i;
                  found = true;
               }
               break;
            case SM_TYPE_DISCONNECT:
               if(0 != _smPool[i].getDisconnectSm())
               {
                  index = i;
                  found = true;
               }
               break;
            case SM_TYPE_CONFIGURE:
               if(0 != _smPool[i].getConfigureSm())
               {
                  index = i;
                  found = true;
               }
               break;
            default:
               FW_NORMAL_ASSERT_ALWAYS();
               break;
         }
      }
   }

   // create new SM if needed
   if(false == found)
   {
      ProtocolSmEntry sm;
      switch(smType)
      {
         case SM_TYPE_CONNECT:
            sm.setConnectSm(new ProtocolConnect());
            FW_NORMAL_ASSERT(0 != sm.getConnectSm());
            break;
         case SM_TYPE_DISCONNECT:
            sm.setDisconnectSm(new ProtocolDisconnect());
            FW_NORMAL_ASSERT(0 != sm.getDisconnectSm());
            break;
         case SM_TYPE_CONFIGURE:
            sm.setConfigureSm(new ConfigurePageTimeout());
            FW_NORMAL_ASSERT(0 != sm.getConfigureSm());
            break;
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }

      index = _smPool.size();
      _smPool.push_back(sm);
   }

   // set SM to in use
   _smPool[index].setInUse(true);

   // reset SM
   _smPool[index].resetSm();

   ETG_TRACE_USR3((" getSmEntryInternal(): index=%u smType=%d _smPool.size()=%u", index, smType, _smPool.size()));

   return _smPool[index];
}

} //btstackif
