/**
 * @file ProtocolSmEntry.cpp
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of protocol state machine entry.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of protocol state machine entry.
 */

#include "ProtocolSmEntry.h"
#include "IProtocolConnect.h"
#include "IProtocolDisconnect.h"
#include "IConfigurePageTimeout.h"

namespace btstackif {

ProtocolSmEntry::ProtocolSmEntry() :
_inUse(false),
_smConnect(0),
_smDisconnect(0),
_smConfigure(0)
{
}

ProtocolSmEntry::~ProtocolSmEntry()
{
   // never destroy a SM within destructor
   _smConnect = 0;
   _smDisconnect = 0;
   _smConfigure = 0;
}

ProtocolSmEntry::ProtocolSmEntry(const ProtocolSmEntry& ref)
{
   _inUse = ref._inUse;
   _smConnect = 0;
   setConnectSm(ref._smConnect);
   _smDisconnect = 0;
   setDisconnectSm(ref._smDisconnect);
   _smConfigure = 0;
   setConfigureSm(ref._smConfigure);
}

ProtocolSmEntry& ProtocolSmEntry::operator=(const ProtocolSmEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   _inUse = ref._inUse;
   _smConnect = 0;
   setConnectSm(ref._smConnect);
   _smDisconnect = 0;
   setDisconnectSm(ref._smDisconnect);
   _smConfigure = 0;
   setConfigureSm(ref._smConfigure);

   return *this;
}

bool ProtocolSmEntry::operator==(const ProtocolSmEntry& ref) const
{
   bool result = true;

   result = (true == result) && (_inUse == ref._inUse);
   result = (true == result) && (_smConnect == ref._smConnect);
   result = (true == result) && (_smDisconnect == ref._smDisconnect);
   result = (true == result) && (_smConfigure == ref._smConfigure);

   return result;
}

bool ProtocolSmEntry::operator!=(const ProtocolSmEntry& ref) const
{
   return !(operator==(ref));
}

void ProtocolSmEntry::destroySm(void)
{
   // destroy must be explicitly triggered from outside
   if(0 != _smConnect)
   {
      delete _smConnect;
      _smConnect = 0;
   }
   if(0 != _smDisconnect)
   {
      delete _smDisconnect;
      _smDisconnect = 0;
   }
   if(0 != _smConfigure)
   {
      delete _smConfigure;
      _smConfigure = 0;
   }
}

void ProtocolSmEntry::clearSm(void)
{
   // reset in use flag
   _inUse = false;
   // set all SMs to NULL
   _smConnect = 0;
   _smDisconnect = 0;
   _smConfigure = 0;
}

bool ProtocolSmEntry::anySmAssigned(void) const
{
   // check all SMs
   if(0 != _smConnect)
   {
      return true;
   }
   else if(0 != _smDisconnect)
   {
      return true;
   }
   else if(0 != _smConfigure)
   {
      return true;
   }

   return false;
}

void ProtocolSmEntry::resetSm(void)
{
   // check all SMs
   if(0 != _smConnect)
   {
      _smConnect->resetStateMachine();
   }
   else if(0 != _smDisconnect)
   {
      _smDisconnect->resetStateMachine();
   }
   else if(0 != _smConfigure)
   {
      _smConfigure->resetStateMachine();
   }
}

bool ProtocolSmEntry::isSmAssigned(IN const bool connect) const
{
   if(true == connect)
   {
      return (0 != _smConnect);
   }
   else
   {
      return (0 != _smDisconnect);
   }
}

} //btstackif
