/**
 * @file ProtocolManager.cpp
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of generic protocol manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic protocol manager state machine.
 */

#include "ProtocolManager.h"
#include "IProtocolManagerRequest.h"
#include "IBasicControl.h"
#include "IProtocolObserver.h"
#include "ITimerPool.h"
#include "IDeviceManager.h"
#include "IServiceSearch.h"
#include "ISppPool.h"
#include "IConfiguration.h"
#include "IRestrictedPairingConnecting.h"
#include "IProtocolConnect.h"
#include "IProtocolDisconnect.h"
#include "IConfigurePageTimeout.h"
#include "FwErrmemPrint.h"
#include "FwUtils.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "FwFormattedDataPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ProtocolManager.cpp.trc.h"
#endif
#endif

namespace btstackif {

#define IF_INVALID_PROTOCOL_INSTANCES_RETURN if(ProtocolManagerData::PROTO_INST_LAST != _protocolInstances.size()) { FW_ERRMEM_ASSERT_ALWAYS(); return; }

ProtocolManager::ProtocolManager() :
_controlIf(0),
_timerPoolIf(0),
_deviceManagerIf(0),
_serviceSearchIf(0),
_sppPoolIf(0),
_configurationIf(0),
_basicConfigurationIf(0),
_restrictionIf(0),
_protocolList(),
_observerList(),
_protocolInstances(),
_smPool(),
_monitoredDevices(),
_waitForCharDevTimeout(3000),
_currentPageTimeout(0),
_pageTimeoutConfigured(false),
_defaultSppInstanceId(255),
_defaultMasInstanceId(0),
_emptyMasInstanceName(),
_retryTimeout(200),
_defaultRetryMax(2),
_connectTimeout(7000),
_disconnectTimeout(40000),
_panViaConnMan(true),
_testTriggerDisconnectWhileDisconnected(false),
_testTriggerDisconnectWhileDisconnectedSetState(false),
_testTriggerCancelDuringConnectResult(false),
_testTriggerCancelDuringAcceptRejectRemoteConnect(false),
_testDisableSearching(false),
_testDisableSetting(false),
_testBlockSendingDisconnect(false),
_testTriggerLocalConnectWhileRemoteConnect(false),
_testTriggerLocalDisconnectWhileRemoteConnect(false),
_testTriggerLocalConnectBeforeRemoteConnect(false),
_testTriggerLocalConnectedBeforeRemoteConnect(false),
_testTriggerLocalConnectSameDevice(false)
{
   _protocolInstances.reserve(ProtocolManagerData::PROTO_INST_LAST);
   _protocolInstances.resize(ProtocolManagerData::PROTO_INST_LAST);
}

ProtocolManager::~ProtocolManager()
{
   _smPool.freeAllSms();

   _controlIf = 0;
   _timerPoolIf = 0;
   _deviceManagerIf = 0;
   _serviceSearchIf = 0;
   _sppPoolIf = 0;
   _configurationIf = 0;
   _basicConfigurationIf = 0;
   _restrictionIf = 0;
}

void ProtocolManager::reset(void)
{
   StateMachine::reset();
   // keep _controlIf
   // keep _timerPoolIf
   // keep _deviceManagerIf
   // keep _serviceSearchIf
   // keep _sppPoolIf
   // keep _configurationIf
   // keep _basicConfigurationIf
   // keep _restrictionIf
   // stop and release all timer
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      releaseTimer(it->second.timer);
      releaseTimer(it->second.charDevTimer);
      releaseTimer(it->second.connectTimer);
      releaseTimer(it->second.disconnectTimer);
   }
   _protocolList.clear();
   // keep _observerList
   // keep _protocolInstances
   _smPool.reset();
   _monitoredDevices.clear();
   _currentPageTimeout = 0;
   _pageTimeoutConfigured = false;
   _testTriggerDisconnectWhileDisconnected = false;
   _testTriggerDisconnectWhileDisconnectedSetState = false;
   _testTriggerCancelDuringConnectResult = false;
   _testTriggerCancelDuringAcceptRejectRemoteConnect = false;
   _testDisableSearching = false;
   _testDisableSetting = false;
   _testBlockSendingDisconnect = false;
   _testTriggerLocalConnectWhileRemoteConnect = false;
   _testTriggerLocalDisconnectWhileRemoteConnect = false;
   _testTriggerLocalConnectBeforeRemoteConnect = false;
   _testTriggerLocalConnectedBeforeRemoteConnect = false;
   _testTriggerLocalConnectSameDevice = false;

   for(size_t i = 0; i < _protocolInstances.size(); i++)
   {
      if(0 != _protocolInstances[i])
      {
         _protocolInstances[i]->reset();
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }
   }

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);
   _currentPageTimeout = _configurationIf->getConfiguration().defaultPageTimeout;
}

void ProtocolManager::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current state/action
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& protocol = it->first;
      const ProtocolManagerData& entry = it->second;

      // check for ongoing connect/disconnect
      if(true == entry.sm.isSmAssigned(true))
      {
         // send connect result as failed
         ProtocolManagerData entryConnectFailed(entry);
         entryConnectFailed.connected = false;
         entryConnectFailed.reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED;
         entryConnectFailed.rfCommDevice.clear();

         createProtocolConnectionStatusUpdate(bts2AppMsgList, protocol.deviceAddress, protocol.protocolId, protocol.sppUuid, protocol.masInstanceId, entryConnectFailed, 0, 0);
         createConnectDisconnectProtocolResult(bts2AppMsgList, protocol.deviceAddress, protocol.protocolId, protocol.sppUuid, protocol.masInstanceId, BTS_REQ_CONNECT_ABORTED, true, entry.requestItem.user, entry.requestItem.handle);
      }
      else if(true == entry.sm.isSmAssigned(false))
      {
         // send disconnect result as success
         ProtocolManagerData entryDisconnectSuccess(entry);
         entryDisconnectSuccess.connected = false;
         entryDisconnectSuccess.reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL;
         entryDisconnectSuccess.rfCommDevice.clear();

         createProtocolConnectionStatusUpdate(bts2AppMsgList, protocol.deviceAddress, protocol.protocolId, protocol.sppUuid, protocol.masInstanceId, entryDisconnectSuccess, 0, 0);
         createConnectDisconnectProtocolResult(bts2AppMsgList, protocol.deviceAddress, protocol.protocolId, protocol.sppUuid, protocol.masInstanceId, BTS_REQ_SUCCESS, false, entry.requestItem.user, entry.requestItem.handle);
      }
      else
      {
         // check for connected
         if(true == entry.connected)
         {
            // update status as disconnected
            ProtocolManagerData entryConnectionLoss(entry);
            entryConnectionLoss.connected = false;
            entryConnectionLoss.reason = BTS_DISCONNECT_REASON_ABNORMAL_LOSS;
            entryConnectionLoss.rfCommDevice.clear();

            createProtocolConnectionStatusUpdate(bts2AppMsgList, protocol.deviceAddress, protocol.protocolId, protocol.sppUuid, protocol.masInstanceId, entryConnectionLoss, 0, 0);
         }
      }
   }

   // reset control data
   reset();
}

void ProtocolManager::setInstance(IN IProtocolManagerRequest* instance, IN const BTSProtocolId protocol, IN const bool panViaConnMan /*= true*/)
{
   IF_INVALID_PROTOCOL_INSTANCES_RETURN

   _protocolInstances[convertProtocol2Instance(protocol, panViaConnMan)] = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_protocolInstances[convertProtocol2Instance(protocol, panViaConnMan)]);

   _protocolInstances[convertProtocol2Instance(protocol, panViaConnMan)]->setCallback(this);
}

void ProtocolManager::setControlIf(IN IBasicControl* control)
{
   IF_INVALID_PROTOCOL_INSTANCES_RETURN

   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);

   for(size_t i = 0; i < _protocolInstances.size(); i++)
   {
      if(0 != _protocolInstances[i])
      {
         _protocolInstances[i]->setControlIf(_controlIf);
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }
   }
}

void ProtocolManager::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);

   for(size_t i = 0; i < _protocolInstances.size(); i++)
   {
      if(0 != _protocolInstances[i])
      {
         _protocolInstances[i]->setTimerPoolIf(_timerPoolIf);
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }
   }
}

void ProtocolManager::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void ProtocolManager::setServiceSearchIf(IN IServiceSearch* serviceSearchIf)
{
   _serviceSearchIf = serviceSearchIf;

   FW_ERRMEM_ASSERT(0 != _serviceSearchIf);
}

void ProtocolManager::setSppPoolIf(IN ISppPool* sppPool)
{
   _sppPoolIf = sppPool;

   FW_ERRMEM_ASSERT(0 != _sppPoolIf);
}

void ProtocolManager::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   _currentPageTimeout = _configurationIf->getConfiguration().defaultPageTimeout;
}

void ProtocolManager::setBasicConfigurationIf(IN IBasicConfiguration* basicConfigurationIf)
{
   _basicConfigurationIf = basicConfigurationIf;

   FW_ERRMEM_ASSERT(0 != _basicConfigurationIf);
}

void ProtocolManager::setRestrictionIf(IN IRestrictedPairingConnecting* restrictionIf)
{
   _restrictionIf = restrictionIf;

   FW_ERRMEM_ASSERT(0 != _restrictionIf);
}

IDeviceObserver* ProtocolManager::getDeviceObserver(void)
{
   return this;
}

IServiceSearchObserver* ProtocolManager::getServiceSearchObserver(void)
{
   return this;
}

ISppPoolObserver* ProtocolManager::getSppPoolObserver(void)
{
   return this;
}

IStateMachine* ProtocolManager::getSmEntryInterface(void)
{
   return this;
}

void ProtocolManager::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetProtocolConnectionStatus& request, IN const BTSCommonEnumClass statusCode) const
{
   sendStatus(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), request.getUser(), request.getSessionHandle(), statusCode);
}

void ProtocolManager::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   if(false == isValidGetRequest(address, protocol, sppUuid, masInstance))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);

   // check if entry is available
   if(_protocolList.end() == it)
   {
      const ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, masInstance, dummyEntry, user, handle);
   }
   else
   {
      const ProtocolManagerData& updateEntry(getUpdateEntry(it->second, getLinkageEntry(workingAddress, protocol, workingUuid, it->second)));
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, instanceId, updateEntry, user, handle);
   }
}

void ProtocolManager::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(sendStatusToAll);
   (void)(statusCode);

   if(false == isValidConnectRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // use given address
      ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), BTS_REQ_INVALID_PARAM, true, request.getUser(), request.getSessionHandle());
      return;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);
   const BTSProtocolId protocol(request.getProtocolId());
   BTSUuid workingUuid(request.getUuid());
   ::fw::convertString2LowerCase(workingUuid);

   if(false == isProtocolEnabled(protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // send status + result
      ProtocolManagerData dummyEntry;
      dummyEntry.reason = BTS_DISCONNECT_REASON_PROTO_NOT_SUPPORTED;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_PROTOCOL_NOT_SUPPORTED, true, request.getUser(), request.getSessionHandle());
      return;
   }

   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, request.getMasInstanceName()));
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);

   // check if entry is available
   if(_protocolList.end() == it)
   {
      sendResult = BTS_REQ_CONNECT_PROTOCOL_FAILED;

      // check given result
      if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult))
      {
         // use given result
         sendResult = inputResult;
      }

      const ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), sendResult, true, request.getUser(), request.getSessionHandle());
   }
   else
   {
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, it->second);

      if((true == it->second.connected) || (true == isLinkageEntryConnected(linkageIt)))
      {
         sendResult = BTS_REQ_SUCCESS;
      }
      else
      {
         sendResult = BTS_REQ_CONNECT_PROTOCOL_FAILED;
      }

      // check given result
      if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
      {
         // use given result
         sendResult = inputResult;
      }

      const ProtocolManagerData& updateEntry(getUpdateEntry(it->second, linkageIt));
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, instanceId, updateEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), sendResult, true, request.getUser(), request.getSessionHandle());
   }
}

void ProtocolManager::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(sendStatusToAll);
   (void)(statusCode);

   if(false == isValidDisconnectRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // use given address
      ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), BTS_REQ_INVALID_PARAM, false, request.getUser(), request.getSessionHandle());
      return;
   }

   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;
   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);
   const BTSProtocolId protocol(request.getProtocolId());
   BTSUuid workingUuid(request.getUuid());
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, request.getMasInstanceName()));
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);

   // check if entry is available
   if(_protocolList.end() == it)
   {
      sendResult = BTS_REQ_SUCCESS;

      const ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), sendResult, false, request.getUser(), request.getSessionHandle());
   }
   else
   {
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, it->second);

      if((true == it->second.connected) || (true == isLinkageEntryConnected(linkageIt)))
      {
         sendResult = BTS_REQ_FAILED;
      }
      else
      {
         sendResult = BTS_REQ_SUCCESS;
      }

      // check given result
      if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
      {
         // use given result
         sendResult = inputResult;
      }

      const ProtocolManagerData& updateEntry(getUpdateEntry(it->second, linkageIt));
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, instanceId, updateEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), sendResult, false, request.getUser(), request.getSessionHandle());
   }
}

bool ProtocolManager::isValidGetRequest(IN const App2Bts_GetProtocolConnectionStatus& request) const
{
   return isValidGetRequest(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName());
}

bool ProtocolManager::isValidGetRequest(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const
{
   (void)(masInstance);

   if(false == ::fw::isValidBdAddress(address))
   {
      return false;
   }

   if(BTS_PROTO_LAST <= protocol)
   {
      return false;
   }

   // check only for empty UUID; if "invalid" UUID no matching protocol entry will be found
   if((BTS_PROTO_SPP == protocol) && (true == sppUuid.empty()))
   {
      return false;
   }

   // no check for masInstance because empty MAS instance name means default instance

   return true;
}

bool ProtocolManager::isValidConnectRequest(IN const App2Bts_ConnectProtocol& request) const
{
   if(false == ::fw::isValidBdAddress(request.getBDAddress()))
   {
      return false;
   }

   if(BTS_PROTO_LAST <= request.getProtocolId())
   {
      return false;
   }

   // check only for empty UUID; if "invalid" UUID no matching entry in service search list will be found
   if((BTS_PROTO_SPP == request.getProtocolId()) && (true == request.getUuid().empty()))
   {
      return false;
   }

   // no check for masInstance because empty MAS instance name means default instance

   return true;
}

bool ProtocolManager::isValidAcceptRequest(IN const App2Bts_AcceptRemoteProtocolConnect& request) const
{
   if(false == ::fw::isValidBdAddress(request.getBDAddress()))
   {
      return false;
   }

   // currently only remote connect for HFP + AVP + SPP supported
   if((BTS_PROTO_HFP != request.getProtocolId()) && (BTS_PROTO_AVP != request.getProtocolId()) && (BTS_PROTO_SPP != request.getProtocolId()))
   {
      return false;
   }

   // in case of SPP a valid UUID shall be given
   if((BTS_PROTO_SPP == request.getProtocolId()) && (true == request.getUuid().empty()))
   {
      return false;
   }

   return true;
}

bool ProtocolManager::isValidRejectRequest(IN const App2Bts_RejectRemoteProtocolConnect& request) const
{
   if(false == ::fw::isValidBdAddress(request.getBDAddress()))
   {
      return false;
   }

   // currently only remote connect for HFP + AVP + SPP supported
   if((BTS_PROTO_HFP != request.getProtocolId()) && (BTS_PROTO_AVP != request.getProtocolId()) && (BTS_PROTO_SPP != request.getProtocolId()))
   {
      return false;
   }

   // in case of SPP a valid UUID shall be given
   if((BTS_PROTO_SPP == request.getProtocolId()) && (true == request.getUuid().empty()))
   {
      return false;
   }

   return true;
}

bool ProtocolManager::isValidDisconnectRequest(IN const App2Bts_DisconnectProtocol& request) const
{
   if(false == ::fw::isValidBdAddress(request.getBDAddress()))
   {
      return false;
   }

   if(BTS_PROTO_LAST <= request.getProtocolId())
   {
      return false;
   }

   // check only for empty UUID; if "invalid" UUID no matching protocol entry will be found
   if((BTS_PROTO_SPP == request.getProtocolId()) && (true == request.getUuid().empty()))
   {
      return false;
   }

   // no check for masInstance because empty MAS instance name means default instance

   return true;
}

bool ProtocolManager::isProtocolEnabled(IN const BTSProtocolId protocol) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   bool enabled(false);

   switch(protocol)
   {
      case BTS_PROTO_HFP:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_HFP);
         break;
      case BTS_PROTO_AVP:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_A2DP) || _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_AVRCP);
         break;
      case BTS_PROTO_PIM:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_PBAP);
         break;
      case BTS_PROTO_MSG:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_MAP);
         break;
      case BTS_PROTO_DUN:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_DUN);
         break;
      case BTS_PROTO_PAN:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_PAN);
         break;
      case BTS_PROTO_SPP:
         enabled = _configurationIf->getConfiguration().supportedServices.getBit(BTS_SUPP_SRV_SPP);
         break;
      case BTS_PROTO_LAST:
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return enabled;
}

bool ProtocolManager::connectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request)
{
   ETG_TRACE_USR2((" connectProtocol: address=%s", request.getBDAddress().c_str()));

   if(false == isValidConnectRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // use given address
      ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), BTS_REQ_INVALID_PARAM, true, request.getUser(), request.getSessionHandle());
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);
   const BTSProtocolId protocol(request.getProtocolId());
   BTSUuid workingUuid(request.getUuid());
   ::fw::convertString2LowerCase(workingUuid);

   if(false == isProtocolEnabled(protocol))
   {
      // send status + result
      ProtocolManagerData dummyEntry;
      dummyEntry.reason = BTS_DISCONNECT_REASON_PROTO_NOT_SUPPORTED;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_PROTOCOL_NOT_SUPPORTED, true, request.getUser(), request.getSessionHandle());
      return false;
   }

   const BTSMasInstanceId masInstanceId(getMasInstanceId(workingAddress, protocol, request.getMasInstanceName()));
   ProtocolManagerData& entry = checkProtocolList(workingAddress, protocol, workingUuid, masInstanceId);
   entry.sdpRecordPending = (false == isSdpRecordAvailable(workingAddress, protocol));
   entry.masName = getVerifiedMasInstanceName(protocol, request.getMasInstanceName());
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, entry);

   if((true == entry.connected) || (true == isLinkageEntryConnected(linkageIt)))
   {
      // protocol already connected => send direct answer
      const ProtocolManagerData& updateEntry(getUpdateEntry(entry, linkageIt));
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), updateEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_SUCCESS, true, request.getUser(), request.getSessionHandle());
      return false;
   }

   // check request instance
   IProtocolManagerRequest* requestIf(getRequestIf(protocol));
   if(0 == requestIf)
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // internal error => send direct answer
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), entry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_FAILED, true, request.getUser(), request.getSessionHandle());
      return false;
   }

   // check for free SPP slot
   if((BTS_PROTO_SPP == protocol) && (false == isSppSlotAvailable(workingUuid)))
   {
      // send answer directly because SPP connection is not possible
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), entry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_SPP_LIMIT_REACHED, true, request.getUser(), request.getSessionHandle());
      return false;
   }

   // SPP: do not check available SPP UUID => improve auto/re connect handling for iAP

   // protocol entry was created, protocol is not connected, request interface is available, free SPP slot is available for SPP

   // check for assigned SM but continue
   FW_ERRMEM_ASSERT(false == entry.sm.anySmAssigned());

   // get SM
   getSM(entry.sm, true, requestIf);
   if((false == entry.sm.isSmAssigned(true)) || (0 == entry.sm.getConnectSm()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // internal error => send direct answer
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), entry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_FAILED, true, request.getUser(), request.getSessionHandle());
      return false;
   }

   // SPP: get SPP instance
   BTSSppInstanceId sppInstanceId(_defaultSppInstanceId);
   if(BTS_PROTO_SPP == protocol)
   {
      (void)getSppSlot(sppInstanceId, workingUuid);
   }
   // DUN: use instance 0
   else if(BTS_PROTO_DUN == protocol)
   {
      sppInstanceId = 0;
   }
   // PAN (proprietary): use instance 0
   else if((BTS_PROTO_PAN == protocol) && (false == _panViaConnMan))
   {
      sppInstanceId = 0;
   }

   // store data and process request
   if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.connectionOriginator)
   {
      entry.connectionOriginator = BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE;
   }
   // entry.connected
   // entry.stackReason
   // entry.reason
   // entry.rfCommDevice
   // entry.remotePending
   // entry.localPending
   // entry.acceptPending
   // entry.info
   // entry.secondaryInfo
   entry.operationId = 0;
   entry.sppInstance = sppInstanceId;
   // entry.sm
   request.getCompareItem(entry.requestItem.item);
   entry.requestItem.user = request.getUser();
   entry.requestItem.handle = request.getSessionHandle();

   // stop any retry
   stopTimer(entry.timer);
   stopTimer(entry.charDevTimer);
   stopTimer(entry.connectTimer);
   stopTimer(entry.disconnectTimer);

   // reset retry data
   entry.retryMax = _defaultRetryMax;
   entry.retryNmb = 0;

   // entry.sdpRecordPending was set before
   // entry.masName was set before

   // store working data
   const BTSUuid incomingUuid;
   entry.sm.getConnectSm()->setConnectionData(workingAddress, protocol, entry.sppInstance, workingUuid, incomingUuid, request.getMasInstanceName(), masInstanceId, isDeviceConnected(workingAddress));
   entry.sm.getConnectSm()->setPageTimeout(request.getTimeValue());

   //===================================================================================================================
   // debug section start
   if(true == _testDisableSearching)
   {
      _testDisableSearching = false;

      entry.sm.getConnectSm()->setTestDisableSearching(true);
   }
   // debug section end
   //===================================================================================================================

   //===================================================================================================================
   // debug section start
   if(true == _testDisableSetting)
   {
      _testDisableSetting = false;

      entry.sm.getConnectSm()->setTestDisableSetting(true);
   }
   // debug section end
   //===================================================================================================================

   // send start event
   entry.sm.getConnectSm()->sendStartEvent();

   // send connect event
   entry.sm.getConnectSm()->sendConnectEvent();

   // do SM processing
   BTSHandleIpc2BtsMessageItem messageItem;
   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, entry);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);

   return true;
}

bool ProtocolManager::acceptRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_AcceptRemoteProtocolConnect& request)
{
   ETG_TRACE_USR2((" acceptRemoteConnect: address=%s", request.getBDAddress().c_str()));

   if(false == isValidAcceptRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // wait for end of sequence
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);
   const BTSProtocolId protocol(request.getProtocolId());
   BTSUuid workingUuid(request.getUuid());
   ::fw::convertString2LowerCase(workingUuid);

   if(false == isProtocolEnabled(protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // wait for end of sequence
      return false;
   }

   const BTSMasInstanceId instanceId(_defaultMasInstanceId);
   bool sendStatus(true);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);

   // check if entry is available
   if(_protocolList.end() == it)
   {
      // device / protocol is not in list => nothing to do
   }
   else
   {
      ProtocolManagerData& tmpEntry = it->second;
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, tmpEntry);

      // check if connected
      if((true == tmpEntry.connected) || (true == isLinkageEntryConnected(linkageIt)))
      {
         // accept for an already connected protocol => what to do? => ignore
      }
      else
      {
         // accept remote connect
         // note: in case of parallel connect (local + remote) situation together with UUID linkage:
         //       - remote connect will be automatically rejected to avoid a situation with 2 connections for "same" UUID
         //       => distinguish between described situation and normal one
         if((0 != tmpEntry.sm.getConnectSm()) && ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.sm.getConnectSm())))
         {
            // parallel connect (local + remote) situation together with UUID linkage
            // ignore
            // do not send any status now => any protocol connection status will follow
            sendStatus = false;
         }
         else
         {
            // following scenarios are possible here:
            // - UUID linkage available and connect for linkage UUID + no main connect SM (single connect for linkage UUID) => only possible for remote connect
            // - UUID linkage available but no connect for linkage UUID + main connect SM or no main connect SM
            // - no UUID linkage available + main connect SM or no main connect SM
            ProtocolManagerData& entry(getRemoteConnectEntry(tmpEntry, linkageIt));

            if((0 != entry.sm.getConnectSm()) && (BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE == entry.connectionOriginator) /*&& (true == entry.acceptPending)*/)
            {
               // related entry found
               // remote connection request is answered by application
               entry.acceptPending = false;

               entry.sm.getConnectSm()->sendAcceptEvent();

               // do SM processing
               BTSHandleIpc2BtsMessageItem messageItem;
               doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, instanceId, entry);
               FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
               FW_ERRMEM_ASSERT(0 == messageItem.message);

               // any protocol connection status will follow
               sendStatus = false;

               //===================================================================================================================
               // debug section start
               if(true == _testTriggerCancelDuringAcceptRejectRemoteConnect)
               {
                  _testTriggerCancelDuringAcceptRejectRemoteConnect = false;

                  handleTestTriggerCancelDuringAcceptRejectRemoteConnect(workingAddress, protocol, workingUuid, _emptyMasInstanceName);
               }
               // debug section end
               //===================================================================================================================
            }
         }
      }
   }

   if(true == sendStatus)
   {
      const BTSMasInstanceName masName;

      if(_protocolList.end() == it)
      {
         const ProtocolManagerData dummyEntry;
         createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, masName, dummyEntry, request.getUser(), request.getSessionHandle());
      }
      else
      {
         const ProtocolManagerData& updateEntry(getUpdateEntry(it->second, getLinkageEntry(workingAddress, protocol, workingUuid, it->second)));
         createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, masName, updateEntry, request.getUser(), request.getSessionHandle());
      }
   }

   return false;
}

bool ProtocolManager::rejectRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RejectRemoteProtocolConnect& request)
{
   ETG_TRACE_USR2((" rejectRemoteConnect: address=%s", request.getBDAddress().c_str()));

   if(false == isValidRejectRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // wait for end of sequence
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);
   const BTSProtocolId protocol(request.getProtocolId());
   BTSUuid workingUuid(request.getUuid());
   ::fw::convertString2LowerCase(workingUuid);

   if(false == isProtocolEnabled(protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // wait for end of sequence
      return false;
   }

   const BTSMasInstanceId instanceId(_defaultMasInstanceId);
   bool sendStatus(true);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);

   // check if entry is available
   if(_protocolList.end() == it)
   {
      // device / protocol is not in list => nothing to do
   }
   else
   {
      ProtocolManagerData& tmpEntry = it->second;
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, tmpEntry);

      // check if connected
      if((true == tmpEntry.connected) || (true == isLinkageEntryConnected(linkageIt)))
      {
         // reject for an already connected protocol => what to do? => ignore
      }
      else
      {
         // reject remote connect
         // note: in case of parallel connect (local + remote) situation together with UUID linkage:
         //       - remote connect will be automatically rejected to avoid a situation with 2 connections for "same" UUID
         //       => distinguish between described situation and normal one
         if((0 != tmpEntry.sm.getConnectSm()) && ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.sm.getConnectSm())))
         {
            // parallel connect (local + remote) situation together with UUID linkage
            // ignore
            // do not send any status now => any protocol connection status will follow
            sendStatus = false;
         }
         else
         {
            // following scenarios are possible here:
            // - UUID linkage available and connect for linkage UUID + no main connect SM (single connect for linkage UUID) => only possible for remote connect
            // - UUID linkage available but no connect for linkage UUID + main connect SM or no main connect SM
            // - no UUID linkage available + main connect SM or no main connect SM
            ProtocolManagerData& entry(getRemoteConnectEntry(tmpEntry, linkageIt));

            if((0 != entry.sm.getConnectSm()) && (BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE == entry.connectionOriginator) /*&& (true == entry.acceptPending)*/)
            {
               // related entry found
               // remote connection request is answered by application
               entry.acceptPending = false;

               entry.sm.getConnectSm()->sendRejectEvent();

               // do SM processing
               BTSHandleIpc2BtsMessageItem messageItem;
               doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, instanceId, entry);
               FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
               FW_ERRMEM_ASSERT(0 == messageItem.message);

               // any protocol connection status will follow
               sendStatus = false;

               //===================================================================================================================
               // debug section start
               if(true == _testTriggerCancelDuringAcceptRejectRemoteConnect)
               {
                  _testTriggerCancelDuringAcceptRejectRemoteConnect = false;

                  handleTestTriggerCancelDuringAcceptRejectRemoteConnect(workingAddress, protocol, workingUuid, _emptyMasInstanceName);
               }
               // debug section end
               //===================================================================================================================
            }
         }
      }
   }

   if(true == sendStatus)
   {
      const BTSMasInstanceName masName;

      if(_protocolList.end() == it)
      {
         const ProtocolManagerData dummyEntry;
         createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, masName, dummyEntry, request.getUser(), request.getSessionHandle());
      }
      else
      {
         const ProtocolManagerData& updateEntry(getUpdateEntry(it->second, getLinkageEntry(workingAddress, protocol, workingUuid, it->second)));
         createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, masName, updateEntry, request.getUser(), request.getSessionHandle());
      }
   }

   return false;
}

bool ProtocolManager::disconnectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request)
{
   ETG_TRACE_USR2((" disconnectProtocol: address=%s", request.getBDAddress().c_str()));

   if(false == isValidDisconnectRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // use given address
      ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), BTS_REQ_INVALID_PARAM, false, request.getUser(), request.getSessionHandle());
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);
   const BTSProtocolId protocol(request.getProtocolId());
   BTSUuid workingUuid(request.getUuid());
   ::fw::convertString2LowerCase(workingUuid);

   if(false == isProtocolEnabled(protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // send status + result
      ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_SUCCESS, false, request.getUser(), request.getSessionHandle());
      return false;
   }

   const BTSMasInstanceId masInstanceId(getMasInstanceId(workingAddress, protocol, request.getMasInstanceName()));
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, masInstanceId);

   if(_protocolList.end() == it)
   {
      // device / protocol not in list => send direct answer
      ProtocolManagerData dummyEntry;
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), dummyEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_SUCCESS, false, request.getUser(), request.getSessionHandle());
      return false;
   }

   ProtocolManagerData& tmpEntry = it->second;
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, tmpEntry);

   if((true == tmpEntry.connected) || (true == isLinkageEntryConnected(linkageIt)))
   {
      // continue
   }
   else
   {
      // protocol already disconnected => send direct answer
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), tmpEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_SUCCESS, false, request.getUser(), request.getSessionHandle());
      return false;
   }

   // check request instance
   IProtocolManagerRequest* requestIf(getRequestIf(protocol));
   if(0 == requestIf)
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // internal error => send direct answer
      const ProtocolManagerData& updateEntry(getUpdateEntry(tmpEntry, linkageIt));
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), updateEntry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_FAILED, false, request.getUser(), request.getSessionHandle());
      return false;
   }

   // protocol entry was created, protocol is connected, request interface is available

   // continue with correct entry
   ProtocolManagerData& entry(getLocalDisconnectEntry(tmpEntry, linkageIt));
   workingUuid = getLocalDisconnectUuid(workingUuid, tmpEntry, linkageIt);

   // check for assigned SM but continue
   FW_ERRMEM_ASSERT(false == entry.sm.anySmAssigned());

   // get SM
   getSM(entry.sm, false, requestIf);
   if((false == entry.sm.isSmAssigned(false)) || (0 == entry.sm.getDisconnectSm()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // internal error => send direct answer
      createProtocolConnectionStatusUpdate(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), entry, request.getUser(), request.getSessionHandle());
      createConnectDisconnectProtocolResult(bts2AppMsgList, workingAddress, protocol, workingUuid, request.getMasInstanceName(), BTS_REQ_FAILED, false, request.getUser(), request.getSessionHandle());
      return false;
   }

   // store data and process request
   if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.connectionOriginator)
   {
      entry.connectionOriginator = BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE;
   }
   // entry.connected
   // entry.stackReason
   // entry.reason
   // entry.rfCommDevice
   // entry.remotePending
   // entry.localPending
   // entry.acceptPending
   // entry.info
   // entry.secondaryInfo
   entry.operationId = 0;
   // entry.sppInstance
   // entry.sm
   request.getCompareItem(entry.requestItem.item);
   entry.requestItem.user = request.getUser();
   entry.requestItem.handle = request.getSessionHandle();

   // stop any retry
   stopTimer(entry.timer);
   stopTimer(entry.charDevTimer);
   stopTimer(entry.connectTimer);
   stopTimer(entry.disconnectTimer);

   // reset retry data
   entry.retryMax = _defaultRetryMax;
   entry.retryNmb = 0;

   entry.sdpRecordPending = false;
   entry.masName.clear();

   // store working data
   const BTSUuid incomingUuid;
   entry.sm.getDisconnectSm()->setConnectionData(workingAddress, protocol, entry.sppInstance, workingUuid, request.getMasInstanceName(), masInstanceId, request.getPauseBtStreaming());

   //===================================================================================================================
   // debug section start
   if(true == _testBlockSendingDisconnect)
   {
      _testBlockSendingDisconnect = false;

      entry.sm.getDisconnectSm()->setTestBlockSendingDisconnect(true);
   }
   // debug section end
   //===================================================================================================================

   // send connect event
   entry.sm.getDisconnectSm()->sendDisconnectEvent();

   // do SM processing
   BTSHandleIpc2BtsMessageItem messageItem;
   doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, entry);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);

   return true;
}

void ProtocolManager::setRequestedConnectionState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool connectRequested, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool remote /*= false*/)
{
   (void)(user);
   (void)(handle);
   (void)(remote);

   ETG_TRACE_USR2((" setRequestedProtocolConnectionState: address=%s", address.c_str()));

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);

   const BTSMasInstanceId masInstanceId(getMasInstanceId(workingAddress, protocol, masInstance));
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, masInstanceId);

   // check requested state
   if(true == connectRequested)
   {
      // we want to connect => do not create new entry
      if(_protocolList.end() == it)
      {
         // device / protocol is not in list => nothing to do
      }
      else
      {
         // in case UUID linkage available: this function is always called with master UUID
         ProtocolManagerData& tmpEntry = it->second;
         ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, tmpEntry);

         // check if connected
         if((true == tmpEntry.connected) || (true == isLinkageEntryConnected(linkageIt)))
         {
            ProtocolManagerData& entry(getLocalDisconnectEntry(tmpEntry, linkageIt));

            // try to abort disconnect
            if(0 != entry.sm.getDisconnectSm())
            {
               // protocol connected, disconnect ongoing, connect requested => abort disconnect (if possible)

               // send cancel event
               entry.sm.getDisconnectSm()->sendCancelEvent();

               // do SM processing
               BTSHandleIpc2BtsMessageItem messageItem;
               doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, entry);
               FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
               FW_ERRMEM_ASSERT(0 == messageItem.message);
            }
         }
         else
         {
            // accept remote connect
            // note: in case of parallel connect (local + remote) situation together with UUID linkage:
            //       - remote connect will be automatically rejected to avoid a situation with 2 connections for "same" UUID
            //       => distinguish between described situation and normal one
            if((0 != tmpEntry.sm.getConnectSm()) && ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.sm.getConnectSm())))
            {
               // parallel connect (local + remote) situation together with UUID linkage
               // ignore
            }
            else
            {
               // following scenarios are possible here:
               // - UUID linkage available and connect for linkage UUID + no main connect SM (single connect for linkage UUID) => only possible for remote connect
               // - UUID linkage available but no connect for linkage UUID + main connect SM or no main connect SM
               // - no UUID linkage available + main connect SM or no main connect SM
               ProtocolManagerData& entry(getRemoteConnectEntry(tmpEntry, linkageIt));

               if((0 != entry.sm.getConnectSm()) && (BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE == entry.connectionOriginator) /*&& (true == entry.acceptPending)*/)
               {
                  // related entry found
                  // remote connection request is answered by application
                  entry.acceptPending = false;

                  entry.sm.getConnectSm()->sendAcceptEvent();

                  // do SM processing
                  BTSHandleIpc2BtsMessageItem messageItem;
                  doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, entry);
                  FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
                  FW_ERRMEM_ASSERT(0 == messageItem.message);
               }
            }
         }
      }
   }
   else
   {
      // we want to disconnect => do not create new entry; do not remove entry
      if(_protocolList.end() == it)
      {
         // device / protocol is not in list => nothing to do
      }
      else
      {
         // in case UUID linkage available: this function is always called with master UUID
         ProtocolManagerData& tmpEntry = it->second;
         ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, tmpEntry);

         // check if connected
         if((true == tmpEntry.connected) || (true == isLinkageEntryConnected(linkageIt)))
         {
            // connected, request for disconnect will follow => nothing to do
         }
         else
         {
            // reject remote connect / abort connect
            // note: in case of parallel connect (local + remote) situation together with UUID linkage:
            //       - remote connect will be automatically rejected to avoid a situation with 2 connections for "same" UUID
            //       => distinguish between described situation and normal one
            if((0 != tmpEntry.sm.getConnectSm()) && ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.sm.getConnectSm())))
            {
               // parallel connect (local + remote) situation together with UUID linkage
               // remote connect is automatically rejected
               // cancel local connect
               if(0 != tmpEntry.sm.getConnectSm())
               {
                  // send cancel event
                  tmpEntry.sm.getConnectSm()->sendCancelEvent();

                  // do SM processing
                  BTSHandleIpc2BtsMessageItem messageItem;
                  doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, tmpEntry);
                  FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
                  FW_ERRMEM_ASSERT(0 == messageItem.message);
               }
            }
            else
            {
               // following scenarios are possible here:
               // - UUID linkage available and connect for linkage UUID + no main connect SM (single connect for linkage UUID) => only possible for remote connect
               // - UUID linkage available but no connect for linkage UUID + main connect SM or no main connect SM
               // - no UUID linkage available + main connect SM or no main connect SM
               ProtocolManagerData& entry(getRemoteConnectEntry(tmpEntry, linkageIt));

               if((0 != entry.sm.getConnectSm()) && (BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE == entry.connectionOriginator) /*&& (true == entry.acceptPending)*/)
               {
                  // related entry found
                  // remote connection request is answered by application
                  entry.acceptPending = false;

                  entry.sm.getConnectSm()->sendRejectEvent();

                  // do SM processing
                  BTSHandleIpc2BtsMessageItem messageItem;
                  doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, entry);
                  FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
                  FW_ERRMEM_ASSERT(0 == messageItem.message);
               }
               else if((0 != entry.sm.getConnectSm()))
               {
                  // send cancel event
                  entry.sm.getConnectSm()->sendCancelEvent();

                  // do SM processing
                  BTSHandleIpc2BtsMessageItem messageItem;
                  doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, protocol, workingUuid, masInstanceId, entry);
                  FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
                  FW_ERRMEM_ASSERT(0 == messageItem.message);
               }
            }
         }
      }
   }
}

void ProtocolManager::setRequestedConnectionState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool connectRequested)
{
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      BTSBDAddress workingAddress(address);
      ::fw::convertString2LowerCase(workingAddress);

      if(workingAddress == it->first.deviceAddress)
      {
         BTSMasInstanceName instanceName;
         getMasInstanceName(instanceName, workingAddress, it->first.protocolId, it->first.masInstanceId);

         setRequestedConnectionState(bts2IpcMsgList, bts2AppMsgList, workingAddress, it->first.protocolId, it->first.sppUuid, instanceName, connectRequested, 0, 0);
      }
   }
}

void ProtocolManager::getConnectedProtocols(OUT ::std::vector< BTSProtocolId >& connectedProtocols, IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      if(workingAddress == it->first.deviceAddress)
      {
         if(true == it->second.connected)
         {
            connectedProtocols.push_back(it->first.protocolId);
         }
      }
   }
}

bool ProtocolManager::isProtocolConnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);
   if(_protocolList.end() != it)
   {
      const ProtocolManagerData& updateEntry(getUpdateEntry(it->second, getLinkageEntry(address, protocol, workingUuid, it->second)));

      return updateEntry.connected;
   }

   return false;
}

bool ProtocolManager::isAnyProtocolConnectDisconnectOngoing(void) const
{
   // check list: if there is any SM assigned a connect or disconnect or configuration is ongoing
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      if((true == it->second.sm.anySmAssigned()) || (true == it->second.configureSm.anySmAssigned()))
      {
         return true;
      }
   }

   return false;
}

bool ProtocolManager::isProtocolConnectDisconnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);
   if(_protocolList.end() == it)
   {
      // can happen if device / protocol is not in list
      return false;
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, it->second);

   const bool check1((true == it->second.sm.anySmAssigned()) || ((_protocolList.end() != linkageIt) && (true == linkageIt->second.sm.anySmAssigned())));
   const bool check2((true == it->second.configureSm.anySmAssigned()) || ((_protocolList.end() != linkageIt) && (true == linkageIt->second.configureSm.anySmAssigned())));

   return ((true == check1) || (true == check2));
}

bool ProtocolManager::isProtocolConnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);
   if(_protocolList.end() == it)
   {
      // can happen if device / protocol is not in list
      return false;
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, it->second);

   const bool check1((0 != it->second.sm.getConnectSm()) || ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.sm.getConnectSm())));
   const bool check2((0 != it->second.configureSm.getConfigureSm()) || ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.configureSm.getConfigureSm())));

   return ((true == check1) || (true == check2));
}

bool ProtocolManager::isProtocolDisconnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);
   if(_protocolList.end() == it)
   {
      // can happen if device / protocol is not in list
      return false;
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(workingAddress, protocol, workingUuid, it->second);

   return ((0 != it->second.sm.getDisconnectSm()) || ((_protocolList.end() != linkageIt) && (0 != linkageIt->second.sm.getDisconnectSm())));
}

bool ProtocolManager::isProtocolAdded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);
   if(_protocolList.end() != it)
   {
      return it->second.info.getBit(ProtocolManagerData::PROTOCOL_ADDED);
   }

   return false;
}

bool ProtocolManager::isProtocolAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2 /*= true*/) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(sppUuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masInstance));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(workingAddress, protocol, workingUuid, instanceId);
   if(_protocolList.end() != it)
   {
      if(true == level2)
      {
         return it->second.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2);
      }
      else
      {
         return it->second.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE);
      }
   }

   return false;
}

bool ProtocolManager::isSppSlotAvailable(IN const BTSUuid& uuid) const
{
   (void)(uuid);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_sppPoolIf);

   // check number of connected SPP instances
   size_t counter(0);
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      if((BTS_PROTO_SPP == it->first.protocolId) && ((true == it->second.connected) || (0 != it->second.sm.getConnectSm())))
      {
         ++counter;
      }
   }

   return (counter < _sppPoolIf->getMaxNumberSppConnections());
}

void ProtocolManager::setMaxSppConnections(IN const unsigned int limit)
{
   (void)(limit);
}

void ProtocolManager::setPanConnectionMode(IN const bool mode)
{
   (void)(mode);
}

void ProtocolManager::setTestTriggerDeviceConnectedUpdate(IN const bool enable)
{
   (void)(enable);
}

bool ProtocolManager::getTestTriggerDeviceConnectedUpdate(void) const
{
   return false;
}

void ProtocolManager::setTestTriggerRemoteConnectDuringDeviceDisconnect(IN const bool enable)
{
   (void)(enable);
}

bool ProtocolManager::getTestTriggerRemoteConnectDuringDeviceDisconnect(void) const
{
   return false;
}

void ProtocolManager::setSimulateFailedDisconnectResult(IN const bool enable)
{
   (void)(enable);
}

void ProtocolManager::registerObserver(IN IProtocolObserver* observer)
{
   (void)(observer);
}

void ProtocolManager::setDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason)
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   BTSUuid workingUuid(uuid);
   ::fw::convertString2LowerCase(workingUuid);
   const BTSMasInstanceId instanceId(getMasInstanceId(workingAddress, protocol, masName));

   // ensure that an entry to protocol list is added to have disconnect reason stored
   ProtocolManagerData& entry = checkProtocolList(workingAddress, protocol, workingUuid, instanceId);

   // set disconnect reason independent of connection status
   entry.stackReason = reason;
   entry.reason = reason;

   setDisconnectReasonForLinkageEntry(workingAddress, protocol, workingUuid, entry);
}

void ProtocolManager::setTestTriggerDisconnectWhileDisconnected(IN const bool enable)
{
   _testTriggerDisconnectWhileDisconnected = enable;
}

void ProtocolManager::setTestTriggerDisconnectWhileDisconnectedSetState(IN const bool enable)
{
   _testTriggerDisconnectWhileDisconnectedSetState = enable;
}

void ProtocolManager::setTriggerCancelDuringConnectResult(IN const bool enable)
{
   _testTriggerCancelDuringConnectResult = enable;
}

void ProtocolManager::setTriggerCancelDuringAcceptRejectRemoteConnect(IN const bool enable)
{
   _testTriggerCancelDuringAcceptRejectRemoteConnect = enable;
}

void ProtocolManager::setTestDisableSearching(IN const bool enable)
{
   _testDisableSearching = enable;
}

void ProtocolManager::setTestDisableSetting(IN const bool enable)
{
   _testDisableSetting = enable;
}

void ProtocolManager::setTestBlockSendingDisconnect(IN const bool enable)
{
   // check if any disconnect SM is ongoing
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      ProtocolManagerData& entry(it->second);

      if(0 != entry.sm.getDisconnectSm())
      {
         entry.sm.getDisconnectSm()->setTestBlockSendingDisconnect(enable);
         return;
      }
   }

   _testBlockSendingDisconnect = enable;
}

void ProtocolManager::setTestTriggerLocalConnectWhileRemoteConnect(IN const bool enable)
{
   _testTriggerLocalConnectWhileRemoteConnect = enable;
}

void ProtocolManager::setTestTriggerLocalDisconnectWhileRemoteConnect(IN const bool enable)
{
   _testTriggerLocalDisconnectWhileRemoteConnect = enable;
}

void ProtocolManager::setTestTriggerLocalConnectBeforeRemoteConnect(IN const bool enable)
{
   _testTriggerLocalConnectBeforeRemoteConnect = enable;
}

void ProtocolManager::setTestTriggerLocalConnectedBeforeRemoteConnect(IN const bool enable)
{
   _testTriggerLocalConnectedBeforeRemoteConnect = enable;
}

void ProtocolManager::setTestTriggerLocalConnectSameDevice(IN const bool enable)
{
   _testTriggerLocalConnectSameDevice = enable;
}

void ProtocolManager::executeTestAction(IN const unsigned int action)
{
   internalExecuteTestAction(action);
}

void ProtocolManager::setTestMaxRetryNumber(IN const unsigned int number)
{
   _defaultRetryMax = number;
}

void ProtocolManager::setTestRetryTimeout(IN const unsigned int timeout)
{
   _retryTimeout = (BTSTimeValue)timeout;
}

bool ProtocolManager::getFirstConnectedSppUuid(OUT BTSUuid& uuid, IN const BTSBDAddress& address) const
{
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;

      if((address == keyData.deviceAddress) && (BTS_PROTO_SPP == keyData.protocolId))
      {
         const ProtocolManagerData& entry = it->second;

         if(true == entry.connected)
         {
            uuid = keyData.sppUuid;
            return true;
         }
      }
   }

   return false;
}

bool ProtocolManager::isPanViaConnMan(void) const
{
   return _panViaConnMan;
}

void ProtocolManager::waitingForServiceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getConnectSm());

   if(BTS_REQ_SUCCESS == result)
   {
      entry.sm.getConnectSm()->sendWaitingForServiceSuccessEvent();
   }
   else
   {
      entry.sm.getConnectSm()->sendWaitingForServiceFailedEvent();
   }

   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
}

void ProtocolManager::waitingForProtocolResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getConnectSm());

   if(BTS_REQ_SUCCESS == result)
   {
      entry.sm.getConnectSm()->sendWaitingForProtocolSuccessEvent();
   }
   else
   {
      entry.sm.getConnectSm()->sendWaitingForProtocolFailedEvent();
   }

   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
}

void ProtocolManager::connectIndication(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result)
{
   // not needed at the moment
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(result);
}

bool ProtocolManager::connectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, false);

   ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(entry.sm.getConnectSm());

   //===================================================================================================================
   // debug section start
   if(true == _testTriggerCancelDuringConnectResult)
   {
      _testTriggerCancelDuringConnectResult = false;

      handleTestTriggerCancelDuringConnectResult(address, protocol, uuid, masInstance);
   }
   // debug section end
   //===================================================================================================================

   if(BTS_REQ_BUSY == result)
   {
      entry.sm.getConnectSm()->sendBusyEvent();
   }
   else
   {
      entry.sm.getConnectSm()->sendConnectResultEvent(result);
   }

   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);

   // check connect failed due to page timeout
   if(0 == entry.sm.getConnectSm())
   {
      // sequence finished
      if((BTS_DISCONNECT_REASON_OUT_OF_RANGE == entry.reason) || (BTS_DISCONNECT_REASON_ACL_CONNECT_FAILED == entry.reason))
      {
         return true;
      }
   }

   return false;
}

void ProtocolManager::connecting(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getConnectSm());

   entry.sm.getConnectSm()->sendConnectingEvent();

   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
}

void ProtocolManager::disconnectIndication(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result)
{
   // not needed at the moment
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(result);
}

void ProtocolManager::disconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   // consider that disconnect result is received during cancel connect sequence
   // note: connect and disconnect SM never run in parallel
   if(0 != entry.sm.getConnectSm())
   {
      entry.sm.getConnectSm()->sendCancelResultEvent(result);

      doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   }
   else if(0 != entry.sm.getDisconnectSm())
   {
      if(BTS_REQ_BUSY == result)
      {
         entry.sm.getDisconnectSm()->sendBusyEvent();
      }
      else
      {
         entry.sm.getDisconnectSm()->sendDisconnectResultEvent(result);
      }

      doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void ProtocolManager::disconnecting(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   // consider that disconnecting is indicated during cancel connect sequence
   // note: connect and disconnect SM never run in parallel
   if(0 != entry.sm.getConnectSm())
   {
      entry.sm.getConnectSm()->sendDisconnectingEvent();

      doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   }
   else if(0 != entry.sm.getDisconnectSm())
   {
      entry.sm.getDisconnectSm()->sendDisconnectingEvent();

      doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

bool ProtocolManager::updateStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason)
{
   (void)(sppInstance);
   (void)(masName);
   (void)(reason);

   ETG_TRACE_USR3((" updateStatus: address=%s", address.c_str()));

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, false);

   ProtocolManagerData& entry = it->second;
   bool connectOngoing(false);

   // either connect or disconnect
   if(0 != entry.sm.getConnectSm())
   {
      // local or remote protocol connect ongoing
      connectOngoing = true;

      const BTSRequestResult result((BTS_CONN_CONNECTED == status) ? BTS_REQ_SUCCESS : BTS_REQ_CONNECT_PROTOCOL_FAILED);

      bool localConnect(false);
      if(true == entry.sm.getConnectSm()->isConnectingState(localConnect))
      {
         // SM is in connecting state => check local/remote connect
         if(false == localConnect)
         {
            entry.sm.getConnectSm()->sendConnectResultEvent(result);
         }
      }
      else if(true == entry.sm.getConnectSm()->isDisconnectedState())
      {
         // SM is in disconnected state
         entry.sm.getConnectSm()->sendConnectResultEvent(result);
      }

      entry.sm.getConnectSm()->sendFinalStateEvent(result);

      doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   }
   else if(0 != entry.sm.getDisconnectSm())
   {
      // local protocol disconnect ongoing
      const BTSRequestResult result((BTS_CONN_DISCONNECTED == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

      entry.sm.getDisconnectSm()->sendFinalStateEvent(result);

      doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   }
   else
   {
      // local device disconnect ongoing, remote disconnect or connection loss
      bool doUpdate(true);

      if(BTS_CONN_DISCONNECTED == status)
      {
         //===================================================================================================================
         // debug section start
         if(true == _testTriggerDisconnectWhileDisconnected)
         {
            handleTestTriggerDisconnectWhileDisconnected(address, protocol, uuid, masInstance);
         }
         // debug section end
         //===================================================================================================================

         // disconnect request is successful
         entry.connected = false;
         // entry.reason was set before
         // entry.stackReason was set before

         // check for ongoing device disconnect to correct disconnect reason
         if(true == isDeviceDisconnectOngoing(address))
         {
            entry.reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL;
            entry.stackReason = entry.reason;

            setDisconnectReasonForLinkageEntry(address, protocol, uuid, entry);
         }
      }
      else
      {
         if(true == entry.connected)
         {
            // protocol is already connected => nothing to do
            doUpdate = false;
         }
         else
         {
            // this is normally handled by if-blocks before
            FW_ERRMEM_ASSERT_ALWAYS();

            // connect request is successful
            entry.connected = true;
            entry.reason = convertConnectResult2DisconnectReason(BTS_REQ_SUCCESS);
            entry.stackReason = entry.reason;
         }
      }

      if(true == doUpdate)
      {
         // update connection status
         const ProtocolManagerData& updateEntry(getUpdateEntry(entry, getLinkageEntry(address, protocol, uuid, entry)));
         createProtocolConnectionStatusUpdate(bts2AppMsgList, address, protocol, uuid, masInstance, updateEntry, 0, 0);
         handleChangedProtocolConnectionStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, updateEntry.reason);

         // reset data
         entry.connectionOriginator = BTS_PAIR_CONNECT_INITIATED_LAST;
         entry.requestItem.reset();

         //===================================================================================================================
         // debug section start
         if(true == _testTriggerDisconnectWhileDisconnected)
         {
            _testTriggerDisconnectWhileDisconnected = false;

            handleTestTriggerDisconnectWhileDisconnected(entry.requestItem.item, address, protocol, uuid, masInstance);
         }
         // debug section end
         //===================================================================================================================

         if(BTS_CONN_DISCONNECTED == status)
         {
            // free SPP instance
            freeSppSlot(protocol, entry.sppInstance);
            entry.sppInstance = _defaultSppInstanceId;

            // reset MAS instance name because disconnect is only triggered if profile was connected
            entry.masName.clear();
         }

         // stop any retry
         stopTimer(entry.timer);
         stopTimer(entry.charDevTimer);
         stopTimer(entry.connectTimer);
         stopTimer(entry.disconnectTimer);

         // reset retry data
         entry.retryMax = _defaultRetryMax;
         entry.retryNmb = 0;
      }
   }

   // check connect failed due to page timeout
   if((true == connectOngoing) && (0 == entry.sm.getConnectSm()))
   {
      // sequence finished
      if((BTS_DISCONNECT_REASON_OUT_OF_RANGE == entry.reason) || (BTS_DISCONNECT_REASON_ACL_CONNECT_FAILED == entry.reason))
      {
         return true;
      }
   }

   return false;
}

void ProtocolManager::updateFailedConnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   // PBAP/MAP:
   // messages on session bus can be faster than messages on system bus
   // therefore it can happen that during this moment PBAP/MAP is already connected
   // => failed connect reason is received after reception of connected message
   // => therefore ignore failed connect reason and set reason during last step of protocol connect SM

   if(false == entry.connected)
   {
      // check for ongoing protocol connect
      if(0 != entry.sm.getConnectSm())
      {
         entry.stackReason = reason;
         entry.reason = reason;

         setDisconnectReasonForLinkageEntry(address, protocol, uuid, entry);
      }
      else
      {
         // protocol connect sequence already finished
      }
   }
   else
   {
      // entry is already marked as connected
   }
}

void ProtocolManager::updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   // PBAP/MAP:
   // messages on session bus can be faster than messages on system bus
   // therefore it can happen that during this moment PBAP/MAP is already disconnected
   // => disconnect reason is received after disconnected message
   // => therefore ignore disconnect reason and set reason during last step of protocol disconnect SM

   if(true == entry.connected)
   {
      // check for correction of disconnect reason: device disconnect (=> NORMAL_LOSS_LOCAL)
      if(true == isDeviceDisconnectOngoing(address))
      {
         entry.stackReason = reason;
         entry.reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL;
      }
      // check for correction of disconnect reason: protocol disconnect (=> NORMAL_LOSS_LOCAL)
      else if(0 != entry.sm.getDisconnectSm())
      {
         entry.stackReason = reason;
         entry.reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL;
      }
      // remote disconnect or connection loss
      else
      {
         entry.stackReason = reason;
         entry.reason = reason;
      }

      setDisconnectReasonForLinkageEntry(address, protocol, uuid, entry);
   }
   else
   {
      // entry is already marked as disconnected
   }
}

void ProtocolManager::updateVirtualDeviceName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDeviceName& name)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getConnectSm());

   entry.rfCommDevice = name;
}

void ProtocolManager::characterDeviceAddedRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& name, IN const bool added)
{
   if(0 == name.size())
   {
      return;
   }

   if(true == added)
   {
      (void)_monitoredDevices.insert(name);
   }
   else
   {
      (void)_monitoredDevices.erase(name);
   }

   if(true == added)
   {
      // loop over all active SPP connect SMs and inform about reported character device
      for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
      {
         const BTSProtocolBaseEntry& keyData = it->first;

         if(BTS_PROTO_SPP == keyData.protocolId)
         {
            ProtocolManagerData& entry(it->second);

            // send event only in case of character device was already reported by Bluetooth stack
            if((0 != entry.sm.getConnectSm()) && (name == entry.rfCommDevice))
            {
               entry.sm.getConnectSm()->sendCharacterDeviceAvailableEvent();

               // do SM processing
               doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);
            }
         }
      }
   }
}

void ProtocolManager::remoteConnectRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_restrictionIf);

   ProtocolManagerData& entry = checkProtocolList(address, protocol, uuid, masInstance);

   //===================================================================================================================
   // debug section start
   if((true == _testTriggerLocalConnectBeforeRemoteConnect) && ((BTS_PROTO_SPP == protocol) || (BTS_PROTO_AVP == protocol)))
   {
      _testTriggerLocalConnectBeforeRemoteConnect = false;

      handleTestTriggerLocalConnectBeforeRemoteConnect(address, protocol, uuid, entry, _testTriggerLocalConnectSameDevice);
   }
   // debug section end
   //===================================================================================================================

   //===================================================================================================================
   // debug section start
   if((true == _testTriggerLocalConnectedBeforeRemoteConnect) && ((BTS_PROTO_SPP == protocol) || (BTS_PROTO_AVP == protocol)))
   {
      _testTriggerLocalConnectedBeforeRemoteConnect = false;

      handleTestTriggerLocalConnectedBeforeRemoteConnect(address, protocol, uuid, entry);
   }
   // debug section end
   //===================================================================================================================

   // reset remotePending flag
   entry.remotePending = false;

   // check for ongoing disconnect
   FW_ERRMEM_IF_FALSE_RETURN(0 == entry.sm.getDisconnectSm());

   // check request instance
   IProtocolManagerRequest* requestIf(getRequestIf(protocol));
   FW_ERRMEM_IF_NULL_PTR_RETURN(requestIf);

   const bool connectOngoing(0 != entry.sm.getConnectSm());

   // get SM if needed
   getSM(entry.sm, true, requestIf);
   if((false == entry.sm.isSmAssigned(true)) || (0 == entry.sm.getConnectSm()))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // update originator
   if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.connectionOriginator)
   {
      entry.connectionOriginator = BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE;
   }

   if(false == connectOngoing)
   {
      // stop any retry
      stopTimer(entry.timer);
      stopTimer(entry.charDevTimer);
      stopTimer(entry.connectTimer);
      stopTimer(entry.disconnectTimer);

      // reset retry data
      entry.retryMax = _defaultRetryMax;
      entry.retryNmb = 0;

      entry.sdpRecordPending = false;
      entry.masName.clear();
   }

   // remote connection request is sent to application; wait for answer
   // not necessary --- entry.acceptPending = true;

   // store working data
   const BTSUuid incomingUuid;
   entry.sm.getConnectSm()->setConnectionData(address, protocol, entry.sppInstance, uuid, incomingUuid, masName, masInstance, isDeviceConnected(address));

   // check for restricted connecting
   BTSBDAddress restrictedAddress;
   if(true == _restrictionIf->isConnectingRestricted(restrictedAddress))
   {
      if(address != restrictedAddress)
      {
         // we have to reject given connect request
         entry.sm.getConnectSm()->setRestrictedConnecting();
      }
   }

   // check for UUID linkage and parallel connect (in case of such situation reject automatically remote connection request to avoid a situation with 2 connections for "same" UUID)
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(address, protocol, uuid, entry);
   if((_protocolList.end() != linkageIt) && ((0 != linkageIt->second.sm.getConnectSm()) || (true == linkageIt->second.connected)))
   {
      // reuse restricted connecting
      entry.sm.getConnectSm()->setRestrictedConnecting();
   }

   // send start event
   entry.sm.getConnectSm()->sendStartEvent();

   // send connect event
   entry.sm.getConnectSm()->sendRemoteConnectEvent();

   // do SM processing
   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);

   //===================================================================================================================
   // debug section start
   if((true == _testTriggerLocalConnectWhileRemoteConnect) && ((BTS_PROTO_SPP == protocol) || (BTS_PROTO_AVP == protocol)))
   {
      _testTriggerLocalConnectWhileRemoteConnect = false;

      handleTestTriggerLocalConnectWhileRemoteConnect(address, protocol, uuid, entry);
   }
   // debug section end
   //===================================================================================================================

   //===================================================================================================================
   // debug section start
   if((true == _testTriggerLocalDisconnectWhileRemoteConnect) && ((BTS_PROTO_SPP == protocol) || (BTS_PROTO_AVP == protocol)))
   {
      _testTriggerLocalDisconnectWhileRemoteConnect = false;

      handleTestTriggerLocalDisconnectWhileRemoteConnect(address, protocol, uuid, entry);
   }
   // debug section end
   //===================================================================================================================
}

void ProtocolManager::remoteConnectCanceled(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);
   (void)(masName);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getConnectSm());

   entry.sm.getConnectSm()->sendCanceledEvent();

   doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
}

void ProtocolManager::protocolAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);

   ProtocolManagerData& entry = checkProtocolList(address, protocol, uuid, masInstance);

   if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_ADDED))
   {
      ETG_TRACE_USR2((" protocolAdded: protocol=%d address=%s", protocol, address.c_str()));

      entry.info.setBit(ProtocolManagerData::PROTOCOL_ADDED);

      // check observer list
      for(::std::set< IProtocolObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masName);
         }
      }
   }
}

void ProtocolManager::protocolRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);
   if(_protocolList.end() != it)
   {
      // stop and release timer before removing from list
      releaseTimer(it->second.timer);

      _protocolList.erase(it);

      ETG_TRACE_USR2((" protocolRemoved: _protocolList.size()=%u protocol=%d address=%s", _protocolList.size(), protocol, address.c_str()));

      // check observer list
      for(::std::set< IProtocolObserver* >::const_iterator itObs = _observerList.begin(); itObs != _observerList.end(); ++itObs)
      {
         if(0 != *itObs)
         {
            (*itObs)->protocolRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masName);
         }
      }
   }
}

void ProtocolManager::protocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ProtocolManagerData::ProtocolAvailableInfo& protocolAvailableInfo)
{
   (void)(sppInstance);

   ProtocolManagerData& entry = checkProtocolList(address, protocol, uuid, masInstance);
   bool sendEventAndInformObserverL1(false);
   bool sendEventAndInformObserverL2(false);

   // first step: store data
   if(true == protocolAvailableInfo.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1))
   {
      if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE))
      {
         ETG_TRACE_USR2((" protocolAvailable: protocol=%d address=%s", protocol, address.c_str()));

         entry.info.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE);

         sendEventAndInformObserverL1 = true;
      }
   }
   if(true == protocolAvailableInfo.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2))
   {
      if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2))
      {
         ETG_TRACE_USR2((" protocolAvailable: protocol=%d address=%s (level2)", protocol, address.c_str()));

         entry.info.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2);

         sendEventAndInformObserverL2 = true;
      }
   }

   // second step: send SM event
   if(true == sendEventAndInformObserverL1)
   {
      // check for pending local connect
      if(0 != entry.sm.getConnectSm())
      {
         // send create success event
         entry.sm.getConnectSm()->sendCreateSuccessEvent();

         // do SM processing
         doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masInstance, entry);
      }
   }
   if(true == sendEventAndInformObserverL2)
   {
      // no event
   }

   // third step: inform observer
   if(true == sendEventAndInformObserverL1)
   {
      // check observer list
      for(::std::set< IProtocolObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masName, false);
         }
      }
   }
   if(true == sendEventAndInformObserverL2)
   {
      // check observer list
      for(::std::set< IProtocolObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masName, true);
         }
      }
   }
}

void ProtocolManager::protocolUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ProtocolManagerData::ProtocolAvailableInfo& protocolAvailableInfo)
{
   (void)(sppInstance);

   ProtocolManagerData& entry = checkProtocolList(address, protocol, uuid, masInstance);
   bool sendEventAndInformObserverL1(false);
   bool sendEventAndInformObserverL2(false);

   // first step: store data
   if(true == protocolAvailableInfo.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1))
   {
      if(true == entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE))
      {
         ETG_TRACE_USR2((" protocolUnavailable: protocol=%d address=%s", protocol, address.c_str()));

         entry.info.resetBit(ProtocolManagerData::PROTOCOL_AVAILABLE);

         sendEventAndInformObserverL1 = true;
      }
   }
   if(true == protocolAvailableInfo.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2))
   {
      if(true == entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2))
      {
         ETG_TRACE_USR2((" protocolUnavailable: protocol=%d address=%s (level2)", protocol, address.c_str()));

         entry.info.resetBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2);

         sendEventAndInformObserverL2 = true;
      }
   }

   // second step: send SM event
   if(true == sendEventAndInformObserverL1)
   {
      // no event
   }
   if(true == sendEventAndInformObserverL2)
   {
      // no event
   }

   // third step: inform observer
   if(true == sendEventAndInformObserverL1)
   {
      // check observer list
      for(::std::set< IProtocolObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masName, false);
         }
      }
   }
   if(true == sendEventAndInformObserverL2)
   {
      // check observer list
      for(::std::set< IProtocolObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, masName, true);
         }
      }
   }
}

App2Bts_BaseMessage* ProtocolManager::getApp2BtsWorkingMessage(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   (void)(sppInstance);
   (void)(masName);

   ProtocolManagerData& entry = checkProtocolList(address, protocol, uuid, masInstance);

   return _controlIf->findApp2BtsWorkingMessageWrapper(entry.requestItem.item);
}

ProtocolManagerData& ProtocolManager::getEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(sppInstance);
   (void)(masName);

   return checkProtocolList(address, protocol, uuid, masInstance);
}

const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& ProtocolManager::getProtocolList(void) const
{
   return _protocolList;
}

bool ProtocolManager::isProtocolAvailableInternal(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2 /*= true*/) const
{
   return isProtocolAvailable(address, protocol, sppUuid, masInstance, level2);
}

void ProtocolManager::getMasInstanceNameInternal(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const
{
   getMasInstanceName(instanceName, address, protocol, instanceId);
}

void ProtocolManager::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void ProtocolManager::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void ProtocolManager::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // find matching connect SMs
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      if(address == it->first.deviceAddress)
      {
         const ProtocolManagerData& entry = it->second;

         if(0 != entry.sm.getConnectSm())
         {
            IProtocolManagerRequest* requestIf(getRequestIf(it->first.protocolId));
            if(0 != requestIf)
            {
               BTSMasInstanceName instanceName;

               // it might be possible that SDP record for MAP is not available during this moment
               if(true == isSdpRecordAvailable(address, it->first.protocolId))
               {
                  getMasInstanceName(instanceName, address, it->first.protocolId, it->first.masInstanceId);
               }

               requestIf->setDeviceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, address, it->first.protocolId, entry.sppInstance, it->first.sppUuid, it->first.masInstanceId, instanceName, true);
            }
         }
      }
   }
}

void ProtocolManager::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void ProtocolManager::deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(aclTransition);
   (void)(anyProfileTransition);

   // ETG_TRACE_USR3((" deviceConnectionStatus: address=%s", address.c_str()));

   // find matching connect SMs
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      if(address == it->first.deviceAddress)
      {
         ProtocolManagerData& entry = it->second;

         if(0 != entry.sm.getConnectSm())
         {
            entry.sm.getConnectSm()->setDeviceConnectionStatus(((true == aclConnected) || (true == anyProfileConnected)));
         }
      }
   }
}

void ProtocolManager::deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);

   // ignore in case of success, wait for protocol available
   if(BTS_REQ_SUCCESS == result)
   {
      return;
   }

   // TODO: [low]: timer is needed for device available
}

void ProtocolManager::deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void ProtocolManager::serviceSearchFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSRequestResult result)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_serviceSearchIf);

   const BTSProtocolId protocol(convertSearchType2Protocol(searchType));

   // check for pending SDP record (needed for MAP (MAS instances))
   if(BTS_SEARCH_MAP == searchType)
   {
      // find matching connect SMs
      for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
      {
         const BTSProtocolBaseEntry& keyData = it->first;

         if((address == keyData.deviceAddress) && (protocol == keyData.protocolId))
         {
            ProtocolManagerData& entry = it->second;

            if(0 != entry.sm.getConnectSm())
            {
               // check flag
               if(true == entry.sdpRecordPending)
               {
                  entry.sdpRecordPending = false;

                  // check/correction of MAS instance id is needed
                  if(true == _serviceSearchIf->isSdpRecordAvailable(keyData.deviceAddress, searchType))
                  {
                     // SDP record is now available => check MAS instance id
                     const BTSMasInstanceId oldMasInstanceId(keyData.masInstanceId);
                     const BTSMasInstanceId newMasInstanceId(getMasInstanceId(keyData.deviceAddress, keyData.protocolId, entry.masName));

                     if(newMasInstanceId != oldMasInstanceId)
                     {
                        BTSProtocolBaseEntry tmpKey;
                        tmpKey.deviceAddress = keyData.deviceAddress;
                        tmpKey.protocolId = keyData.protocolId;
                        tmpKey.sppUuid = keyData.sppUuid;
                        tmpKey.masInstanceId = newMasInstanceId;
                        const ProtocolManagerData tmpData(it->second);

                        if(_protocolList.find(tmpKey) != _protocolList.end())
                        {
                           // entry already exists => something went wrong!?
                           FW_ERRMEM_ASSERT_ALWAYS();
                        }
                        else
                        {
                           _protocolList.erase(it);
                           _protocolList[tmpKey] = tmpData;

                           // update MAS instance data in connect SM
                           entry.sm.getConnectSm()->setMasInstanceNameAndId(tmpData.masName, tmpKey.masInstanceId);
                        }
                     }

                     ETG_TRACE_USR2((" serviceSearchFinished: MAS INSTANCE SWITCHED from %u to %u for address=%s", oldMasInstanceId, newMasInstanceId, address.c_str()));
                  }

                  // only 1 MAP connect to the same device at the same time is allowed => end loop now
                  break;
               }
            }
         }
      }
   }

   // find matching connect SMs
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;

      if((address == keyData.deviceAddress) && (protocol == keyData.protocolId))
      {
         ProtocolManagerData& entry = it->second;

         if(0 != entry.sm.getConnectSm())
         {
            // check stored SDP record data
            if(BTS_SEARCH_LAST <= searchType)
            {
               FW_ERRMEM_ASSERT_ALWAYS();
            }
            else
            {
               if(true == _serviceSearchIf->isSdpRecordAvailable(keyData.deviceAddress, searchType))
               {
                  entry.sm.getConnectSm()->sendSearchSuccessEvent();
               }
               else
               {
                  if(BTS_REQ_SUCCESS == result)
                  {
                     entry.sm.getConnectSm()->sendSearchFailedEvent(BTS_REQ_PROTOCOL_NOT_SUPPORTED);
                  }
                  else
                  {
                     entry.sm.getConnectSm()->sendSearchFailedEvent(result);
                  }
               }

               // do SM processing
               doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);
            }
         }
      }
   }
}

void ProtocolManager::settingUuidFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSSppInstanceId instance, IN const BTSRequestResult result)
{
   // find matching connect SMs
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;
      ProtocolManagerData& entry = it->second;

      if((BTS_PROTO_SPP == keyData.protocolId) && (0 != entry.sm.getConnectSm()) && (instance == entry.sppInstance))
      {
         // matching entry found
         if(BTS_REQ_SUCCESS == result)
         {
            entry.sm.getConnectSm()->sendSettingSuccessEvent();
         }
         else
         {
            entry.sm.getConnectSm()->sendSettingFailedEvent();
         }

         // do SM processing
         doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);

         // only 1 matching entry is possible
         break;
      }
   }
}

void ProtocolManager::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   // timeout after retry = continue connect/disconnect

   // find matching entry via timer id
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it;
   for(it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const ProtocolManagerData& entry = it->second;

      if((true == entry.timer.compare(timerId)) || (true == entry.charDevTimer.compare(timerId)) || (true == entry.connectTimer.compare(timerId)) || (true == entry.disconnectTimer.compare(timerId)))
      {
         // matching timer found
         break;
      }
   }

   if(_protocolList.end() == it)
   {
      // no match
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSProtocolBaseEntry& keyData = it->first;
   ProtocolManagerData& entry = it->second;

   ETG_TRACE_USR2((" handleExtendedTimeout: address=%s", keyData.deviceAddress.c_str()));

   if(true == entry.timer.compare(timerId))
   {
      if(0 != entry.sm.getConnectSm())
      {
         entry.sm.getConnectSm()->sendConnectEvent();

         // do SM processing
         doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);
         FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
         FW_ERRMEM_ASSERT(0 == messageItem.message);
      }
      else if(0 != entry.sm.getDisconnectSm())
      {
         entry.sm.getDisconnectSm()->sendDisconnectEvent();

         // do SM processing
         doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);
         FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
         FW_ERRMEM_ASSERT(0 == messageItem.message);
      }
      else
      {
         // no match
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(true == entry.charDevTimer.compare(timerId))
   {
      if(0 != entry.sm.getConnectSm())
      {
         entry.sm.getConnectSm()->sendCharacterDeviceAvailableEvent();

         // do SM processing
         doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);
         FW_ERRMEM_ASSERT(true == messageItem.deleteMessage);
      }
   }
   else if(true == entry.connectTimer.compare(timerId))
   {
      // nothing to do
   }
   else if(true == entry.disconnectTimer.compare(timerId))
   {
      IProtocolManagerRequest* requestIf(getRequestIf(keyData.protocolId));

      FW_ERRMEM_IF_NULL_PTR_RETURN(requestIf);
      FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getDisconnectSm());

      // send virtual profile status update message (to follow normal sequence)
      BTSMasInstanceName instanceName;
      getMasInstanceName(instanceName, keyData.deviceAddress, keyData.protocolId, keyData.masInstanceId);
      requestIf->sendVirtualConnectedUpdate(bts2IpcMsgList, bts2AppMsgList, keyData.deviceAddress, keyData.protocolId, entry.sppInstance, keyData.sppUuid, keyData.masInstanceId, instanceName, false, BTS_IPC_SUCCESS);

      // write error memory entry
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: disconnect for device=%12s profile=%d uuid=%32s instanceId=%u failed", keyData.deviceAddress.c_str(), keyData.protocolId, keyData.sppUuid.c_str(), keyData.masInstanceId));

      /*
       * background information:
       * This is a workaround for failed profile disconnect (assumption: caused by ALPS Evolution stack).
       * Action 1: Collect device and profile information for later analysis.
       * Action 2: Change profile connection status to disconnected and continue.
       * Normally BM core will trigger a device disconnect after all profiles are disconnected.
       * This gives Bluetooth stack a second chance to disconnect the profile.
       * There is no guaranty that this approach will work but we will avoid blocking situation and hence a possible system reset.
       */
   }
}

void ProtocolManager::createRemoteConnectRequest(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   (void)(masInstance);

   Bts2App_RemoteProtocolConnectRequest* msg = ptrNew_Bts2App_RemoteProtocolConnectRequest();
   if(NULL != msg)
   {
      msg->setUser(NULL);
      msg->setSessionHandle(0);
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setUuid(getMasterUuid(address, protocol, uuid));

      bts2AppMsgList.push_back(msg);
   }
}

BTSDisconnectReason ProtocolManager::convertConnectResult2DisconnectReason(IN const BTSRequestResult result) const
{
   BTSDisconnectReason reason = BTS_DISCONNECT_REASON_NOT_VALID;

   switch(result)
   {
      case BTS_REQ_SUCCESS:
         reason = BTS_DISCONNECT_REASON_NOT_VALID;
         break;
      case BTS_REQ_CONNECT_PROTOCOL_FAILED:
         reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED;
         break;
      case BTS_REQ_CONNECT_ACL_FAILED:
         reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE;
         break;
      case BTS_REQ_CONNECT_ABORTED:
         reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED;
         break;
      case BTS_REQ_PROTOCOL_NOT_SUPPORTED:
         reason = BTS_DISCONNECT_REASON_PROTO_NOT_SUPPORTED;
         break;
      case BTS_REQ_BT_INIT_FAILED:
         reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED; // if BT initialization fails no connect request shall be triggered
         break;
      case BTS_REQ_SPP_UUID_NOT_FOUND:
         reason = BTS_DISCONNECT_REASON_SPP_UUID_NOT_FOUND;
         break;
      case BTS_REQ_SPP_LIMIT_REACHED:
         reason = BTS_DISCONNECT_REASON_SPP_LIMIT_REACHED;
         break;
      case BTS_REQ_CONNECT_TIMEOUT:
         reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_TIMEOUT;
         break;
      case BTS_REQ_CONNECT_REJECTED:
         reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_REJECTED;
         break;

      // not valid for connect
      case BTS_REQ_FAILED:
      case BTS_REQ_DISCONNECT_ABORTED:
      case BTS_REQ_DISCOVERY_ALREADY_STARTED:
      case BTS_REQ_PAIRING_ABORTED:
      case BTS_REQ_PAIRING_CONN_LOST:
      case BTS_REQ_PAIRING_AUTHENTICATION_ERROR:
      case BTS_REQ_PAIRING_PAIRING_CANCELLED:
      case BTS_REQ_PAIRING_MISSING_PIN:
      case BTS_REQ_PAIRING_UNKNOWN:
      case BTS_REQ_SEARCH_NOT_STARTED:
      case BTS_REQ_DBUS_ERROR:
      case BTS_REQ_LAST:
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return reason;
}

BTSDisconnectReason ProtocolManager::getDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, BTS_DISCONNECT_REASON_LAST);

   const ProtocolManagerData& entry = it->second;

   return entry.reason;
}

void ProtocolManager::setConnectionStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connected, IN const BTSDisconnectReason reason)
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   entry.connected = connected;

   // do not set reason if given value is BTS_DISCONNECT_REASON_LAST (valid used case)
   if(BTS_DISCONNECT_REASON_LAST > reason)
   {
      entry.reason = reason;

      if(false == entry.connected)
      {
         setDisconnectReasonForLinkageEntry(address, protocol, uuid, entry, false);
      }
   }
}

void ProtocolManager::updateConnectionStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool sendStatusToAll) const
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   const ProtocolManagerData& entry = it->second;

   if(true == sendStatusToAll)
   {
      createProtocolConnectionStatusUpdate(bts2AppMsgList, address, protocol, uuid, masInstance, entry, 0, 0);
   }
   else
   {
      createProtocolConnectionStatusUpdate(bts2AppMsgList, address, protocol, uuid, masInstance, entry, entry.requestItem.user, entry.requestItem.handle);
   }
}

void ProtocolManager::updateConnectionResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool isConnectRequest, IN const BTSRequestResult result) const
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   const ProtocolManagerData& entry = it->second;

   createConnectDisconnectProtocolResult(bts2AppMsgList, address, protocol, uuid, masInstance, result, isConnectRequest, entry.requestItem.user, entry.requestItem.handle);
}

void ProtocolManager::handleChangedProtocolConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDisconnectReason stackReason /*= BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL*/)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_deviceManagerIf);

   // note: connection status was updated before

   bool anyProfileConnected(false);
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_NOT_VALID);

   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& entry = it->first;
      const ProtocolManagerData& info = it->second;

      if(entry.deviceAddress == address)
      {
         if(true == info.connected)
         {
            anyProfileConnected = true;
            break;
         }
      }
   }

   // in case of last protocol disconnect we have to pass a valid disconnect reason (last protocol disconnect reason is used for device disconnect reason)
   if(false == anyProfileConnected)
   {
      reason = stackReason;
   }

   // inform device manager
   _deviceManagerIf->updateAnyProfileConnected(bts2IpcMsgList, bts2AppMsgList, messageItem, address, anyProfileConnected, reason);
}

bool ProtocolManager::isRetryAllowed(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, false);

   const ProtocolManagerData& entry = it->second;

   return (entry.retryMax > entry.retryNmb);
}

void ProtocolManager::startRetryTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   startTimer(entry.timer, _retryTimeout);

   entry.retryNmb++;
}

void ProtocolManager::stopRetryTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   // stop any retry
   stopTimer(entry.timer);
}

void ProtocolManager::increaseRetryCounter(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   entry.retryNmb++;
}

void ProtocolManager::setSppUuid(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSppInstanceId instance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(instance);
   (void)(outgoingUuid);
   (void)(incomingUuid);
   FW_ERRMEM_ASSERT_ALWAYS();
}

bool ProtocolManager::isAnyProtocolConnected(IN const BTSBDAddress& address) const
{
   // check list: if there is any protocol connected
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      if(address == it->first.deviceAddress)
      {
         if(true == it->second.connected)
         {
            return true;
         }
      }
   }

   return false;
}

bool ProtocolManager::wasCharacterDeviceReported(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const
{
   // check only for SPP
   if(BTS_PROTO_SPP != protocol)
   {
      return true;
   }

   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, true);

   const ProtocolManagerData& entry = it->second;

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(0 < entry.rfCommDevice.size(), true);

   // search for RFCOMM device in reported list
   return (_monitoredDevices.end() != _monitoredDevices.find(entry.rfCommDevice));
}

void ProtocolManager::startWaitForCharacterDeviceTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   // check only for SPP
   if(BTS_PROTO_SPP != protocol)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   startTimer(entry.charDevTimer, _waitForCharDevTimeout);
}

void ProtocolManager::stopWaitForCharacterDeviceTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   // check only for SPP
   if(BTS_PROTO_SPP != protocol)
   {
      return;
   }

   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   stopTimer(entry.charDevTimer);
}

void ProtocolManager::checkAccessRightsForCharacterDevice(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const
{
   (void)(masInstance);

   // check only for SPP
   if(BTS_PROTO_SPP != protocol)
   {
      return;
   }

   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, 0);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   const ProtocolManagerData& entry = it->second;

   // check access rights only for connected state
   if(false == entry.connected)
   {
      return;
   }

   // check access rights only for available RFCOMM device name
   if(0 == entry.rfCommDevice.size())
   {
      return;
   }

   ETG_TRACE_USR2((" checkAccessRightsForCharacterDevice: address=%20s uuid=%40s device=%100s (start)", address.c_str(), uuid.c_str(), entry.rfCommDevice.c_str()));

   __gid_t groupId(0);
   FW_ERRMEM_IF_FALSE_RETURN(true == ::fw::getGroupId4GroupName(groupId, "dialout"));

   /*
    * to minimize the delay we will do following
    * - repeat check for maximum 20 loops a 10ms
    * - => maximum delay = 200ms
    *
    * expected group id is 20 (dialout) => fixed value
    * expected read flag for group is enabled
    * expected write flag for group is enabled
    */

   const unsigned int maxRetry(20);
   const unsigned int retryDelayInMS(10);
   unsigned int retry(0);

   while(retry < maxRetry)
   {
      ++retry;

      struct stat statBuffer;
      if(0 != stat(entry.rfCommDevice.c_str(), &statBuffer))
      {
         // failed
         ETG_TRACE_ERR(("checkAccessRightsForCharacterDevice(): stat character device failed: %d", errno));
         // force end of loop
         retry = maxRetry;
      }
      else
      {
         // success
         char tmpBuffer[10];
         if(statBuffer.st_mode & S_IRUSR) { tmpBuffer[0] = 'r'; } else { tmpBuffer[0] = '-'; }
         if(statBuffer.st_mode & S_IWUSR) { tmpBuffer[1] = 'w'; } else { tmpBuffer[1] = '-'; }
         if(statBuffer.st_mode & S_IXUSR) { tmpBuffer[2] = 'x'; } else { tmpBuffer[2] = '-'; }
         if(statBuffer.st_mode & S_IRGRP) { tmpBuffer[3] = 'r'; } else { tmpBuffer[3] = '-'; }
         if(statBuffer.st_mode & S_IWGRP) { tmpBuffer[4] = 'w'; } else { tmpBuffer[4] = '-'; }
         if(statBuffer.st_mode & S_IXGRP) { tmpBuffer[5] = 'x'; } else { tmpBuffer[5] = '-'; }
         if(statBuffer.st_mode & S_IROTH) { tmpBuffer[6] = 'r'; } else { tmpBuffer[6] = '-'; }
         if(statBuffer.st_mode & S_IWOTH) { tmpBuffer[7] = 'w'; } else { tmpBuffer[7] = '-'; }
         if(statBuffer.st_mode & S_IXOTH) { tmpBuffer[8] = 'x'; } else { tmpBuffer[8] = '-'; }
         tmpBuffer[9] = '\0';

         if((groupId == statBuffer.st_gid) && (statBuffer.st_mode & S_IRGRP) && (statBuffer.st_mode & S_IWGRP))
         {
            ETG_TRACE_USR2((" checkAccessRightsForCharacterDevice(): try=%u/%u: group id=%u ugo=%s (success)", retry, maxRetry, statBuffer.st_gid, tmpBuffer));
            // force end of loop
            retry = maxRetry;
         }
         else
         {
            ETG_TRACE_ERR((" checkAccessRightsForCharacterDevice(): try=%u/%u: group id=%u ugo=%s (failed)", retry, maxRetry, statBuffer.st_gid, tmpBuffer));
            // wait and try again
            (void)usleep(retryDelayInMS * 1000L);
         }
      }
   }

   ETG_TRACE_USR2((" checkAccessRightsForCharacterDevice: address=%20s uuid=%40s device=%100s (end)", address.c_str(), uuid.c_str(), entry.rfCommDevice.c_str()));
}

bool ProtocolManager::configurePageTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSTimeValue pageTimeout)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   // reset marker
   _pageTimeoutConfigured = false;

   // check given page timeout
   if(0 == pageTimeout)
   {
      return false;
   }

   // check device connection status
   if(true == isDeviceConnected(address))
   {
      return false;
   }

   // compare page timeout values
   if(_currentPageTimeout == pageTimeout)
   {
      return false;
   }

   // find protocol entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, false);

   ProtocolManagerData& entry = it->second;

   getConfigureSM(entry.configureSm);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(entry.configureSm.getConfigureSm());

   // start configuration
   if(true == entry.configureSm.getConfigureSm()->configure(bts2IpcMsgList, bts2AppMsgList, pageTimeout, *_configurationIf))
   {
      // configuration started => remember that page timeout configuration was started => reset page timeout after end of connect
      _pageTimeoutConfigured = true;
      entry.setPageTimeoutActive = true;
      return true;
   }
   else
   {
      // free configure SM
      freeSmEntry(entry.configureSm);
      return false;
   }
}

void ProtocolManager::handleEndOfPageTimeoutConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result, IN const BTSTimeValue pageTimeout)
{
   if(BTS_REQ_SUCCESS == result)
   {
      _currentPageTimeout = pageTimeout;
   }

   // note: handling below works only if there are no parallel connects for first protocol (ACL)
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      ProtocolManagerData& entry(it->second);

      if(0 != entry.configureSm.getConfigureSm())
      {
         if(true == entry.setPageTimeoutActive)
         {
            // set page timeout sequence
            if(0 != entry.sm.getConnectSm())
            {
               // free configure SM
               freeSmEntry(entry.configureSm);

               // ignore result => just continue
               const BTSProtocolBaseEntry& keyData = it->first;

               entry.sm.getConnectSm()->sendSetPageTimeoutSuccessEvent();

               // do SM processing
               doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, keyData.masInstanceId, entry);
            }
            else
            {
               FW_ERRMEM_ASSERT_ALWAYS();
            }
         }
         else
         {
            // reset page timeout sequence
            // free configure SM
            freeSmEntry(entry.configureSm);

            const BTSProtocolBaseEntry& keyData = it->first;

            BTSMasInstanceName instanceName;
            getMasInstanceName(instanceName, keyData.deviceAddress, keyData.protocolId, keyData.masInstanceId);

            // set message item
            messageItem.item.opCode = App2BtsOC_ConnectProtocol;
            messageItem.item.deviceAddress = keyData.deviceAddress;
            messageItem.item.protocolId = keyData.protocolId;
            messageItem.item.sppUuid = getMasterUuid(keyData.deviceAddress, keyData.protocolId, keyData.sppUuid, entry);
            messageItem.item.masInstance = instanceName;
            messageItem.deleteMessage = true;
         }

         break;
      }
   }
}

void ProtocolManager::startConnectTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction)
{
   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   if(true == connectAction)
   {
      startTimer(entry.connectTimer, _connectTimeout);
   }
   else
   {
      startTimer(entry.disconnectTimer, _disconnectTimeout);
   }
}

void ProtocolManager::stopConnectTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction)
{
   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN(_protocolList.end() != it);

   ProtocolManagerData& entry = it->second;

   if(true == connectAction)
   {
      stopTimer(entry.connectTimer);
   }
   else
   {
      stopTimer(entry.disconnectTimer);
   }
}

bool ProtocolManager::isConnectTimerActive(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction) const
{
   // find matching entry
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, masInstance);

   FW_ERRMEM_IF_FALSE_RETURN_ERROR(_protocolList.end() != it, false);

   const ProtocolManagerData& entry = it->second;

   if(true == connectAction)
   {
      return isTimerActive(entry.connectTimer);
   }
   else
   {
      return isTimerActive(entry.disconnectTimer);
   }
}

void ProtocolManager::doConnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getConnectSm());

   if(true == entry.sm.getConnectSm()->doConnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, entry.requestItem.user, entry.requestItem.handle))
   {
      // SM is in finished state
      // => connect sequence is finished
      // status and result updates are done by SM
      // keep protocol entry in every case

      // get working message
      if(BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.connectionOriginator)
      {
         if(0 == messageItem.message)
         {
            messageItem.message = getWorkingMessage(entry.requestItem.item);
         }
         FW_ERRMEM_ASSERT(0 != messageItem.message);
      }

      BTSMasInstanceName instanceName;
      getMasInstanceName(instanceName, address, protocol, instance);

      // set message item (in case of remote connect we have no user request)
      messageItem.item.opCode = App2BtsOC_ConnectProtocol;
      messageItem.item.deviceAddress = address;
      messageItem.item.protocolId = protocol;
      messageItem.item.sppUuid = getMasterUuid(address, protocol, uuid, entry);
      messageItem.item.masInstance = instanceName;
      messageItem.deleteMessage = true;

      // free SPP instance (if not connected)
      if(false == entry.connected)
      {
         freeSppSlot(protocol, entry.sppInstance);
         entry.sppInstance = _defaultSppInstanceId;
      }

      // reset MAS instance name (if connected) because SDP record is available
      if(true == entry.connected)
      {
         entry.masName.clear();
      }

      // reset control data because action is finished
      resetProtocolEntryControlData(entry);

      // free SM
      freeSmEntry(entry.sm);

      // check for resetting page timeout
      checkResettingPageTimeout(bts2IpcMsgList, bts2AppMsgList, entry);
   }
}

void ProtocolManager::doDisconnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.sm.getDisconnectSm());

   if(true == entry.sm.getDisconnectSm()->doDisconnectSmProcessing(bts2IpcMsgList, bts2AppMsgList, messageItem, entry.requestItem.user, entry.requestItem.handle))
   {
      // SM is in finished state
      // => disconnect sequence is finished
      // status and result updates are done by SM
      // keep protocol entry in every case

      // get working message
      if(0 == messageItem.message)
      {
         messageItem.message = getWorkingMessage(entry.requestItem.item);
      }
      FW_ERRMEM_ASSERT(0 != messageItem.message);

#if 0
      ETG_TRACE_USR2((" doDisconnectSmProcessing: opCode=%d deviceAddress=%20s protocolId=%d", entry.requestItem.item.opCode, entry.requestItem.item.deviceAddress.c_str(), entry.requestItem.item.protocolId));
#endif

      BTSMasInstanceName instanceName;
      getMasInstanceName(instanceName, address, protocol, instance);

      // set message item
      messageItem.item.opCode = App2BtsOC_DisconnectProtocol;
      messageItem.item.deviceAddress = address;
      messageItem.item.protocolId = protocol;
      messageItem.item.sppUuid = getMasterUuid(address, protocol, uuid, entry);
      messageItem.item.masInstance = instanceName;
      messageItem.deleteMessage = true;

      // free SPP instance (if disconnected)
      if(false == entry.connected)
      {
         freeSppSlot(protocol, entry.sppInstance);
         entry.sppInstance = _defaultSppInstanceId;
      }

      // reset MAS instance name because disconnect is only triggered if profile was connected
      entry.masName.clear();

      // reset control data because action is finished
      resetProtocolEntryControlData(entry);

      // free SM
      freeSmEntry(entry.sm);
   }
}

void ProtocolManager::freeSmEntry(IN ProtocolSmEntry& sm)
{
   _smPool.freeSmContainer(sm);
}

bool ProtocolManager::getSppSlot(OUT BTSSppInstanceId& instance, IN const BTSUuid& uuid)
{
   (void)(instance);
   (void)(uuid);
   return true;
}

void ProtocolManager::freeSppSlot(IN const BTSProtocolId protocolId, IN const BTSSppInstanceId instance)
{
   if(BTS_PROTO_SPP != protocolId)
   {
      return;
   }

   (void)(instance);
}

void ProtocolManager::resetProtocolEntryControlData(IN ProtocolManagerData& entry)
{
   entry.connectionOriginator = BTS_PAIR_CONNECT_INITIATED_LAST;
   // no change --- entry.connected
   // no change --- entry.reason
   // no change --- entry.rfCommDevice
   entry.remotePending = false;
   entry.localPending = false;
   entry.acceptPending = false;
   // no change --- entry.info
   // no change --- entry.secondaryInfo
   entry.operationId = 0;
   // no change --- entry.sppInstance
   // no change --- entry.sm
   entry.requestItem.reset();

   // stop any retry
   stopTimer(entry.timer);
   stopTimer(entry.charDevTimer);
   stopTimer(entry.connectTimer);
   stopTimer(entry.disconnectTimer);

   // reset retry data
   entry.retryMax = _defaultRetryMax;
   entry.retryNmb = 0;

   entry.sdpRecordPending = false;
   // no change --- entry.masName
}

ProtocolManagerData& ProtocolManager::checkProtocolList(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance)
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;

   // check protocol
   if(BTS_PROTO_SPP == protocol)
   {
      // uuid needed
      entry.sppUuid = uuid;
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      // instance needed
      entry.masInstanceId = instance;
   }
   else
   {
      // uuid + instance not needed => ignore
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData> ::iterator it = _protocolList.find(entry);

   if(_protocolList.end() == it)
   {
      // add new entry
      ProtocolManagerData& protocolEntry(_protocolList[entry]);
      ETG_TRACE_USR3((" checkProtocolList: _protocolList.size()=%u", _protocolList.size()));
      addLinkageEntry(address, protocol, uuid, protocolEntry);
      return protocolEntry;
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}

::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator ProtocolManager::getProtocolEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance)
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;

   // check protocol
   if(BTS_PROTO_SPP == protocol)
   {
      // uuid needed
      entry.sppUuid = uuid;
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      // instance needed
      entry.masInstanceId = instance;
   }
   else
   {
      // uuid + instance not needed => ignore
   }

   return _protocolList.find(entry);
}

::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator ProtocolManager::getProtocolEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance) const
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;

   // check protocol
   if(BTS_PROTO_SPP == protocol)
   {
      // uuid needed
      entry.sppUuid = uuid;
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      // instance needed
      entry.masInstanceId = instance;
   }
   else
   {
      // uuid + instance not needed => ignore
   }

   return _protocolList.find(entry);
}

void ProtocolManager::getSM(OUT ProtocolSmEntry& smEntry, IN const bool connect, IN IProtocolManagerRequest* requestIf)
{
   if(true == smEntry.isSmAssigned(connect))
   {
      return;
   }

   smEntry = _smPool.getSmEntry(connect);

   if(0 != smEntry.getConnectSm())
   {
      smEntry.getConnectSm()->setProtocolManager(this);
      smEntry.getConnectSm()->setDeviceManager(_deviceManagerIf);
      smEntry.getConnectSm()->setServiceSearch(_serviceSearchIf);
      smEntry.getConnectSm()->setProtocolManagerRequest(requestIf);
      smEntry.getConnectSm()->setProtocolSmHelper(this);
   }
   else if(0 != smEntry.getDisconnectSm())
   {
      smEntry.getDisconnectSm()->setProtocolManagerRequest(requestIf);
      smEntry.getDisconnectSm()->setProtocolSmHelper(this);
   }
}

void ProtocolManager::getConfigureSM(OUT ProtocolSmEntry& smEntry)
{
   if(0 != smEntry.getConfigureSm())
   {
      return;
   }

   smEntry = _smPool.getConfigureSmEntry();

   if(0 != smEntry.getConfigureSm())
   {
      smEntry.getConfigureSm()->setBasicConfigurationIf(_basicConfigurationIf);
      smEntry.getConfigureSm()->setProtocolSmHelper(this);
   }
}

IProtocolManagerRequest* ProtocolManager::getRequestIf(IN const BTSProtocolId protocol) const
{
   if(BTS_PROTO_LAST <= protocol)
   {
      return 0;
   }

   const ProtocolManagerData::ProtocolInstance instance(convertProtocol2Instance(protocol, _panViaConnMan));

   if(_protocolInstances.size() <= instance)
   {
      return 0;
   }

   return _protocolInstances[instance];
}

ProtocolManagerData::ProtocolInstance ProtocolManager::convertProtocol2Instance(IN const BTSProtocolId protocol, IN const bool panViaConnMan /*= true*/) const
{
   ProtocolManagerData::ProtocolInstance instance(ProtocolManagerData::PROTO_INST_HFP);

   switch(protocol)
   {
      case BTS_PROTO_HFP:
         instance = ProtocolManagerData::PROTO_INST_HFP;
         break;
      case BTS_PROTO_AVP:
         instance = ProtocolManagerData::PROTO_INST_AVP;
         break;
      case BTS_PROTO_PIM:
         instance = ProtocolManagerData::PROTO_INST_PIM;
         break;
      case BTS_PROTO_MSG:
         instance = ProtocolManagerData::PROTO_INST_MSG;
         break;
      case BTS_PROTO_DUN:
         instance = ProtocolManagerData::PROTO_INST_DUN;
         break;
      case BTS_PROTO_PAN:
         if(true == panViaConnMan)
         {
            instance = ProtocolManagerData::PROTO_INST_PAN;
         }
         else
         {
            instance = ProtocolManagerData::PROTO_INST_PAN_PROPRIETARY;
         }
         break;
      case BTS_PROTO_SPP:
         instance = ProtocolManagerData::PROTO_INST_SPP;
         break;
      case BTS_PROTO_LAST:
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return instance;
}

void ProtocolManager::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void ProtocolManager::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void ProtocolManager::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool ProtocolManager::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

bool ProtocolManager::isDeviceDisconnectOngoing(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDisconnectOngoing(address);
}

bool ProtocolManager::isDeviceConnected(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDeviceConnected(address, true);
}

const BTSMasInstanceName& ProtocolManager::getVerifiedMasInstanceName(IN const BTSProtocolId protocol, IN const BTSMasInstanceName& masName) const
{
   if(BTS_PROTO_MSG != protocol)
   {
      // return default MAS instance name
      return _emptyMasInstanceName;
   }

   return masName;
}

bool ProtocolManager::isSdpRecordAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const
{
   if(BTS_PROTO_MSG != protocol)
   {
      // return true because SDP record is only needed for MAS instances (MAP)
      return true;
   }

   FW_ERRMEM_IF_NULL_PTR_RETURN_TRUE(_serviceSearchIf);

   if(true == _serviceSearchIf->isSdpRecordAvailable(address, BTS_SEARCH_MAP))
   {
      return true;
   }

   ETG_TRACE_USR2((" isSdpRecordAvailable: NOT AVAILABLE for address=%s", address.c_str()));

   return false;
}

BTSMasInstanceId ProtocolManager::getMasInstanceId(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceName& instanceName) const
{
   if(BTS_PROTO_MSG != protocol)
   {
      // return 0 because MAS instances exist only for MAP (BTS_PROTO_MSG)
      return _defaultMasInstanceId;
   }

   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_serviceSearchIf);

   BTSMasInstanceId instanceId(_defaultMasInstanceId); // default value
   if(false == _serviceSearchIf->getMasInstanceId(instanceId, address, instanceName))
   {
      // no matching MAS instance found
      if(true == isAnyProtocolInstanceConnected(address, protocol))
      {
         // but at least one MAS instance of same device is already connected
         // => should never happen
         // continue with default value
         FW_ERRMEM_ASSERT_ALWAYS();
      }
      // else: no MAS instance of same device is connected; it can happen that related SDP record is not available; continue with default value
   }

   return instanceId;
}

void ProtocolManager::getMasInstanceName(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const
{
   if(BTS_PROTO_MSG != protocol)
   {
      // return empty name because MAS instances exist only for MAP (BTS_PROTO_MSG)
      instanceName = _emptyMasInstanceName;
      return;
   }

   if(0 == _serviceSearchIf)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      instanceName = _emptyMasInstanceName;
      return;
   }

   if(false == _serviceSearchIf->getMasInstanceName(instanceName, address, instanceId))
   {
      // no matching MAS instance found
      // but function is only used while connect or disconnect is ongoing; during this moment the MAS instance information must be available
      // but function is also called in case of failed SDP search => during this moment the MAS instance information is not available
      instanceName = _emptyMasInstanceName;

      // fallback solution for failed SDP search => use data provided via connect call
      BTSUuid uuid;
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, instanceId);
      if(_protocolList.end() != it)
      {
         const ProtocolManagerData& entry = it->second;

         if(false == entry.masName.empty())
         {
            instanceName = entry.masName;

            ETG_TRACE_USR2((" getMasInstanceName: MAS INSTANCE NAME fallback for instanceName=%s", instanceName.c_str()));
         }
      }
   }
}

bool ProtocolManager::isAnyProtocolInstanceConnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const
{
   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& entry = it->first;
      const ProtocolManagerData& info = it->second;

      if((entry.deviceAddress == address) && (entry.protocolId == protocol))
      {
         if(true == info.connected)
         {
            return true;
         }
      }
   }

   return false;
}

void ProtocolManager::createProtocolConnectionStatusUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN const ProtocolManagerData& entry, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const
{
   BTSMasInstanceName instanceName;
   getMasInstanceName(instanceName, address, protocol, instance);
   createProtocolConnectionStatusUpdate(bts2AppMsgList, address, protocol, uuid, instanceName, entry, user, handle);
}

void ProtocolManager::createProtocolConnectionStatusUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& instance, IN const ProtocolManagerData& entry, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const
{
   Bts2App_ProtocolConnectionStatus* msg = ptrNew_Bts2App_ProtocolConnectionStatus();
   if(0 != msg)
   {
      if(0 == user)
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }

      msg->setBDAddress(address);
      if(true == entry.connected)
      {
         msg->setConnectionStatus(BTS_CONN_CONNECTED);
      }
      else
      {
         msg->setConnectionStatus(BTS_CONN_DISCONNECTED);
      }
      msg->setDisconnectReason(entry.reason);
      msg->setProtocolId(protocol);
      msg->setUuid(getMasterUuid(address, protocol, uuid, entry));
      msg->setMasInstanceName(instance);
      if(true == entry.connected)
      {
         msg->setDeviceName(entry.rfCommDevice);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void ProtocolManager::createConnectDisconnectProtocolResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN const BTSRequestResult result, IN const bool isConnect, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const
{
   BTSMasInstanceName instanceName;
   getMasInstanceName(instanceName, address, protocol, instance);
   createConnectDisconnectProtocolResult(bts2AppMsgList, address, protocol, uuid, instanceName, result, isConnect, user, handle);
}

void ProtocolManager::createConnectDisconnectProtocolResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& instance, IN const BTSRequestResult result, IN const bool isConnect, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const
{
   if(0 != user)
   {
      if(true == isConnect)
      {
         Bts2App_ConnectProtocolResult* msg = ptrNew_Bts2App_ConnectProtocolResult();
         if(0 != msg)
         {
            msg->setUser(user);
            msg->setSessionHandle(handle);
            msg->setRequestResult(result);
            msg->setBDAddress(address);
            msg->setProtocolId(protocol);
            msg->setUuid(getMasterUuid(address, protocol, uuid));
            msg->setMasInstanceName(instance);

            bts2AppMsgList.push_back(msg);
         }
      }
      else
      {
         Bts2App_DisconnectProtocolResult* msg = ptrNew_Bts2App_DisconnectProtocolResult();
         if(0 != msg)
         {
            msg->setUser(user);
            msg->setSessionHandle(handle);
            msg->setRequestResult(result);
            msg->setBDAddress(address);
            msg->setProtocolId(protocol);
            msg->setUuid(getMasterUuid(address, protocol, uuid));
            msg->setMasInstanceName(instance);

            bts2AppMsgList.push_back(msg);
         }
      }
   }
}

App2Bts_BaseMessage* ProtocolManager::getWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(item);
}

void ProtocolManager::handleTestTriggerDisconnectWhileDisconnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   App2Bts_DisconnectProtocol* msg = ptrNew_App2Bts_DisconnectProtocol();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setUuid(uuid);
      getMasInstanceName(msg->getMasInstanceNameMutable(), address, protocol, instance);

      ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
      ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
      if(true == disconnectProtocol(tmpBts2IpcMsgList, tmpBts2AppMsgList, *msg))
      {
         _controlIf->pushApp2BtsMsgToWorkingQueue(msg, false);

         for(size_t i = 0; i < tmpBts2IpcMsgList.size(); i++)
         {
            if(0 != tmpBts2IpcMsgList[i])
            {
               _controlIf->sendInternalBts2IpcMessage(tmpBts2IpcMsgList[i]);
            }
         }
      }
      else
      {
         delete msg;
      }

      if(true == _testTriggerDisconnectWhileDisconnectedSetState)
      {
         _testTriggerDisconnectWhileDisconnectedSetState = false;

         ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator it = getProtocolEntry(address, protocol, uuid, instance);

         if(_protocolList.end() != it)
         {
            ProtocolManagerData& entry = it->second;

            if(0 != entry.sm.getDisconnectSm())
            {
               entry.sm.getDisconnectSm()->sendFinalStateEvent(BTS_REQ_SUCCESS);

               ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
               ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
               BTSHandleIpc2BtsMessageItem tmpMessageItem;
               doDisconnectSmProcessing(tmpBts2IpcMsgList, tmpBts2AppMsgList, tmpMessageItem, address, protocol, uuid, instance, entry);
               FW_ERRMEM_ASSERT(0 == tmpBts2IpcMsgList.size());
               FW_ERRMEM_ASSERT(0 == tmpBts2AppMsgList.size());
               FW_ERRMEM_ASSERT(false == tmpMessageItem.deleteMessage);
               FW_ERRMEM_ASSERT(0 == tmpMessageItem.message);
            }
         }
      }
   }
}

void ProtocolManager::handleTestTriggerDisconnectWhileDisconnected(OUT BTSApp2BtsMessageCompareItem& compareItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance)
{
   App2Bts_DisconnectProtocol msg;
   msg.setBDAddress(address);
   msg.setProtocolId(protocol);
   msg.setUuid(uuid);
   getMasInstanceName(msg.getMasInstanceNameMutable(), address, protocol, instance);
   msg.getCompareItem(compareItem);
}

void ProtocolManager::handleTestTriggerCancelDuringConnectResult(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   App2Bts_DisconnectProtocol* msg = ptrNew_App2Bts_DisconnectProtocol();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setUuid(uuid);
      getMasInstanceName(msg->getMasInstanceNameMutable(), address, protocol, instance);

      _controlIf->pushApp2BtsMessage(msg);
   }
}

void ProtocolManager::handleTestTriggerCancelDuringAcceptRejectRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& instance)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   App2Bts_DisconnectProtocol* msg = ptrNew_App2Bts_DisconnectProtocol();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setUuid(uuid);
      msg->setMasInstanceName(instance);

      _controlIf->pushApp2BtsMessage(msg);
   }
}

void ProtocolManager::handleTestTriggerLocalConnectWhileRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSUuid workingUuid(uuid);

   // check for UUID linkage
   if(BTS_PROTO_SPP == protocol)
   {
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(address, protocol, uuid, entry);

      if(_protocolList.end() != linkageIt)
      {
         if(false == entry.linkageMaster)
         {
            workingUuid = entry.linkageUuid;
         }
      }
   }

   App2Bts_ConnectProtocol* msg = ptrNew_App2Bts_ConnectProtocol();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setUuid(workingUuid);

      _controlIf->pushApp2BtsMessage(msg);
   }
}

void ProtocolManager::handleTestTriggerLocalDisconnectWhileRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSUuid workingUuid(uuid);

   // check for UUID linkage
   if(BTS_PROTO_SPP == protocol)
   {
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(address, protocol, uuid, entry);

      if(_protocolList.end() != linkageIt)
      {
         if(false == entry.linkageMaster)
         {
            workingUuid = entry.linkageUuid;
         }
      }
   }

   App2Bts_DisconnectProtocol* msg = ptrNew_App2Bts_DisconnectProtocol();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setUuid(workingUuid);

      _controlIf->pushApp2BtsMessage(msg);
   }
}

void ProtocolManager::handleTestTriggerLocalConnectBeforeRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry, IN const bool sameDevice)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSUuid workingUuid(uuid);

   // check for UUID linkage
   if(BTS_PROTO_SPP == protocol)
   {
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(address, protocol, uuid, entry);

      if(_protocolList.end() != linkageIt)
      {
         if(false == entry.linkageMaster)
         {
            workingUuid = entry.linkageUuid;
         }
      }
   }

   App2Bts_ConnectProtocol* msg = ptrNew_App2Bts_ConnectProtocol();
   if(0 != msg)
   {
      if(true == sameDevice)
      {
         msg->setBDAddress(address);
      }
      else
      {
         const BTSBDAddress dummyAddress("01234567890a");
         msg->setBDAddress(dummyAddress);
      }
      msg->setProtocolId(protocol);
      msg->setUuid(workingUuid);

      ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
      ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
      if(true == connectProtocol(tmpBts2IpcMsgList, tmpBts2AppMsgList, *msg))
      {
         _controlIf->pushApp2BtsMsgToWorkingQueue(msg, false);

         for(size_t i = 0; i < tmpBts2IpcMsgList.size(); i++)
         {
            if(0 != tmpBts2IpcMsgList[i])
            {
               _controlIf->sendInternalBts2IpcMessage(tmpBts2IpcMsgList[i]);
            }
         }
      }
      else
      {
         delete msg;
      }
   }
}

void ProtocolManager::handleTestTriggerLocalConnectedBeforeRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   BTSUuid workingUuid(uuid);

   // check for UUID linkage
   if(BTS_PROTO_SPP == protocol)
   {
      ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getLinkageEntry(address, protocol, uuid, entry);

      if(_protocolList.end() != linkageIt)
      {
         if(false == entry.linkageMaster)
         {
            workingUuid = entry.linkageUuid;
         }
      }
   }

   ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
   ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;
   BTSHandleIpc2BtsMessageItem messageItem;
   const BTSMasInstanceName masName;
   (void)updateStatus(tmpBts2IpcMsgList, tmpBts2AppMsgList, messageItem, address, protocol, 0, workingUuid, 0, masName, BTS_CONN_CONNECTED, BTS_DISCONNECT_REASON_NOT_VALID);

   for(size_t i = 0; i < tmpBts2AppMsgList.size(); i++)
   {
      if(0 != tmpBts2AppMsgList[i])
      {
         _controlIf->sendInternalBts2AppMessage(tmpBts2AppMsgList[i]);
      }
   }
}

void ProtocolManager::internalExecuteTestAction(IN const unsigned int action)
{
   switch(action)
   {
      case 1:
      case 2:
      {
         for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = _protocolList.begin(); it != _protocolList.end(); ++it)
         {
            const BTSProtocolBaseEntry& key = it->first;
            const ProtocolManagerData& entry = it->second;
            bool useEntry(true);

            if(BTS_PROTO_SPP == key.protocolId)
            {
               if(false == entry.linkageUuid.empty())
               {
                  if(false == entry.linkageMaster)
                  {
                     useEntry = false;
                  }
               }
            }

            if(true == useEntry)
            {
               bool result(false);
               App2Bts_BaseMessage* workingMsg(0);
               ::std::vector< Bts2Ipc_BaseMessage* > tmpBts2IpcMsgList;
               ::std::vector< Bts2App_BaseMessage* > tmpBts2AppMsgList;

               if(1 == action)
               {
                  App2Bts_ConnectProtocol* msg = ptrNew_App2Bts_ConnectProtocol();
                  if(0 != msg)
                  {
                     msg->setBDAddress(key.deviceAddress);
                     msg->setProtocolId(key.protocolId);
                     msg->setUuid(key.sppUuid);

                     result = connectProtocol(tmpBts2IpcMsgList, tmpBts2AppMsgList, *msg);

                     workingMsg = msg;
                  }
               }
               else
               {
                  App2Bts_DisconnectProtocol* msg = ptrNew_App2Bts_DisconnectProtocol();
                  if(0 != msg)
                  {
                     msg->setBDAddress(key.deviceAddress);
                     msg->setProtocolId(key.protocolId);
                     msg->setUuid(key.sppUuid);

                     result = disconnectProtocol(tmpBts2IpcMsgList, tmpBts2AppMsgList, *msg);

                     workingMsg = msg;
                  }
               }

               if(0 != workingMsg)
               {
                  if(true == result)
                  {
                     _controlIf->pushApp2BtsMsgToWorkingQueue(workingMsg, false);

                     for(size_t i = 0; i < tmpBts2IpcMsgList.size(); i++)
                     {
                        if(0 != tmpBts2IpcMsgList[i])
                        {
                           _controlIf->sendInternalBts2IpcMessage(tmpBts2IpcMsgList[i]);
                        }
                     }
                  }
                  else
                  {
                     delete workingMsg;

                     for(size_t i = 0; i < tmpBts2AppMsgList.size(); i++)
                     {
                        if(0 != tmpBts2AppMsgList[i])
                        {
                           _controlIf->sendInternalBts2AppMessage(tmpBts2AppMsgList[i]);
                        }
                     }
                  }
               }

               break;
            }
         }
         break;
      }
      default:
      {
         break;
      }
   }
}

BTSProtocolId ProtocolManager::convertSearchType2Protocol(IN const BTSSearchType searchType) const
{
   BTSProtocolId protocol(BTS_PROTO_LAST);

   switch(searchType)
   {
      case BTS_SEARCH_ALL:
         break;
      case BTS_SEARCH_SPP:
         protocol = BTS_PROTO_SPP;
         break;
      case BTS_SEARCH_DID:
         break;
      case BTS_SEARCH_PBAP:
         protocol = BTS_PROTO_PIM;
         break;
      case BTS_SEARCH_MAP:
         protocol = BTS_PROTO_MSG;
         break;
      case BTS_SEARCH_PAN:
         protocol = BTS_PROTO_PAN;
         break;
      case BTS_SEARCH_LAST:
      default:
         break;
   }

   return protocol;
}

BTSSearchType ProtocolManager::convertProtocol2SearchType(IN const BTSProtocolId protocol) const
{
   BTSSearchType searchType(BTS_SEARCH_LAST);

   switch(protocol)
   {
      case BTS_PROTO_HFP:
         break;
      case BTS_PROTO_AVP:
         break;
      case BTS_PROTO_PIM:
         searchType = BTS_SEARCH_PBAP;
         break;
      case BTS_PROTO_MSG:
         searchType = BTS_SEARCH_MAP;
         break;
      case BTS_PROTO_DUN:
         break;
      case BTS_PROTO_PAN:
         searchType = BTS_SEARCH_PAN;
         break;
      case BTS_PROTO_SPP:
         searchType = BTS_SEARCH_SPP;
         break;
      case BTS_PROTO_LAST:
      default:
         break;
   }

   return searchType;
}

::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator ProtocolManager::getLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry)
{
   (void)(uuid);

   if(BTS_PROTO_SPP != protocol)
   {
      return _protocolList.end();
   }

   if(true == entry.linkageUuid.empty())
   {
      return _protocolList.end();
   }

   return getProtocolEntry(address, protocol, entry.linkageUuid, 0);
}

::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator ProtocolManager::getLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry) const
{
   (void)(uuid);

   if(BTS_PROTO_SPP != protocol)
   {
      return _protocolList.end();
   }

   if(true == entry.linkageUuid.empty())
   {
      return _protocolList.end();
   }

   return getProtocolEntry(address, protocol, entry.linkageUuid, 0);
}

const ProtocolManagerData& ProtocolManager::getUpdateEntry(IN const ProtocolManagerData& mainEntry, IN const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator& linkageIt) const
{
   if(true == isLinkageEntryConnected(linkageIt))
   {
      return linkageIt->second;
   }
   else
   {
      return mainEntry;
   }
}

ProtocolManagerData& ProtocolManager::getRemoteConnectEntry(IN ProtocolManagerData& mainEntry, IN ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator& linkageIt)
{
   if(_protocolList.end() == linkageIt)
   {
      return mainEntry;
   }

   if(0 == linkageIt->second.sm.getConnectSm())
   {
      return mainEntry;
   }

   return linkageIt->second;
}

ProtocolManagerData& ProtocolManager::getLocalDisconnectEntry(IN ProtocolManagerData& mainEntry, IN ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator& linkageIt)
{
   if(_protocolList.end() == linkageIt)
   {
      return mainEntry;
   }

   if(false == linkageIt->second.connected)
   {
      return mainEntry;
   }

   return linkageIt->second;
}

const BTSUuid& ProtocolManager::getLocalDisconnectUuid(IN const BTSUuid& mainUuid, IN const ProtocolManagerData& mainEntry, IN const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator& linkageIt) const
{
   if(_protocolList.end() == linkageIt)
   {
      return mainUuid;
   }

   if(false == linkageIt->second.connected)
   {
      return mainUuid;
   }

   return mainEntry.linkageUuid;
}

const BTSUuid& ProtocolManager::getMasterUuid(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) const
{
   if(BTS_PROTO_SPP != protocol)
   {
      return uuid;
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = getProtocolEntry(address, protocol, uuid, 0);

   if(_protocolList.end() == it)
   {
      return uuid;
   }

   return getMasterUuid(address, protocol, uuid, it->second);
}

const BTSUuid& ProtocolManager::getMasterUuid(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry) const
{
   if(BTS_PROTO_SPP != protocol)
   {
      return uuid;
   }

   if(true == entry.linkageUuid.empty())
   {
      return uuid;
   }

   if(true == entry.linkageMaster)
   {
      return uuid;
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator linkageIt = getProtocolEntry(address, protocol, entry.linkageUuid, 0);

   if(_protocolList.end() == linkageIt)
   {
      // entry shall exist
      FW_ERRMEM_ASSERT_ALWAYS();
      return uuid;
   }

   if(true == linkageIt->second.linkageUuid.empty())
   {
      // UUID shall be not empty
      FW_ERRMEM_ASSERT_ALWAYS();
      return uuid;
   }

   if(false == linkageIt->second.linkageMaster)
   {
      // entry shall be linkage master
      FW_ERRMEM_ASSERT_ALWAYS();
      return uuid;
   }

   return entry.linkageUuid;
}

void ProtocolManager::setDisconnectReasonForLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry, IN const bool setStackReason /*= true*/)
{
   (void)(uuid);

   if(BTS_PROTO_SPP != protocol)
   {
      return;
   }

   if(true == entry.linkageUuid.empty())
   {
      return;
   }

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator linkageIt = getProtocolEntry(address, protocol, entry.linkageUuid, 0);

   if(_protocolList.end() == linkageIt)
   {
      // entry shall exist
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // do not set disconnect reason if linkage entry is connected or connect is ongoing for it
   if(true == linkageIt->second.connected)
   {
      return;
   }

   if(0 != linkageIt->second.sm.getConnectSm())
   {
      return;
   }

   ETG_TRACE_USR2((" setDisconnectReasonForLinkageEntry: address=%s", address.c_str()));

   linkageIt->second.reason = entry.reason;
   if(true == setStackReason)
   {
      linkageIt->second.stackReason = entry.stackReason;
   }
}

void ProtocolManager::addLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   if(BTS_PROTO_SPP != protocol)
   {
      return;
   }

   const BTSSppServiceInfoList& sppServiceInfoList(_configurationIf->getConfiguration().sppServiceInfoList);

   for(::std::vector< BTSSppServiceInfo >::const_iterator it = sppServiceInfoList.begin(); it != sppServiceInfoList.end(); ++it)
   {
      const BTSSppServiceInfo& serviceInfo(*it);

      if((false == serviceInfo.localUuid.empty()) && (false == serviceInfo.remoteUuid.empty()) && (serviceInfo.localUuid != serviceInfo.remoteUuid))
      {
         if(uuid == serviceInfo.localUuid)
         {
            // slave UUID
            entry.linkageUuid = serviceInfo.remoteUuid;
            entry.linkageMaster = false;
            break;
         }
         else if(uuid == serviceInfo.remoteUuid)
         {
            // master UUID
            entry.linkageUuid = serviceInfo.localUuid;
            entry.linkageMaster = true;
            break;
         }
      }
   }

   // check if something was found, configure linkage information
   if(false == entry.linkageUuid.empty())
   {
      BTSProtocolBaseEntry key;
      key.deviceAddress = address;
      key.protocolId = protocol;
      key.sppUuid = entry.linkageUuid;

      ProtocolManagerData& linkageEntry(_protocolList[key]);
      ETG_TRACE_USR3((" addLinkageEntry: _protocolList.size()=%u", _protocolList.size()));
      linkageEntry.linkageUuid = uuid;
      linkageEntry.linkageMaster = (false == entry.linkageMaster);
      if(true == entry.linkageMaster)
      {
         ETG_TRACE_USR3((" addLinkageEntry: localUuid=%32s remoteUuid=%32s", entry.linkageUuid.c_str(), linkageEntry.linkageUuid.c_str()));
      }
      else
      {
         ETG_TRACE_USR3((" addLinkageEntry: localUuid=%32s remoteUuid=%32s", linkageEntry.linkageUuid.c_str(), entry.linkageUuid.c_str()));
      }
   }
}

void ProtocolManager::checkResettingPageTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN ProtocolManagerData& entry)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   // check whether page timeout was configured during connect sequence
   if(false == _pageTimeoutConfigured)
   {
      return;
   }

   getConfigureSM(entry.configureSm);

   FW_ERRMEM_IF_NULL_PTR_RETURN(entry.configureSm.getConfigureSm());

   // start configuration
   if(true == entry.configureSm.getConfigureSm()->configure(bts2IpcMsgList, bts2AppMsgList, _configurationIf->getConfiguration().defaultPageTimeout, *_configurationIf))
   {
      entry.setPageTimeoutActive = false;
   }
   else
   {
      // free configure SM
      freeSmEntry(entry.configureSm);
   }
}

} //btstackif
