/**
 * @file ConfigurePageTimeout.cpp
 *
 * @par SW-Component
 * State machine for configure page timeout
 *
 * @brief Implementation of configure page timeout state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of configure page timeout state machine.
 */

#include "ConfigurePageTimeout.h"
#include "IBasicConfiguration.h"
#include "IProtocolSmHelper.h"
#include "IConfiguration.h"
#include "FwErrmemPrint.h"

namespace btstackif {

ConfigurePageTimeout::ConfigurePageTimeout() :
_basicConfigurationIf(0),
_protocolSmHelper(0),
_requestedPageTimeout(0),
_calculatedPageTimeout(0),
_ongoingConfiguration(BTS_CONFIG_PARAM_PAGE_TIMEOUT)
{
}

ConfigurePageTimeout::~ConfigurePageTimeout()
{
   _basicConfigurationIf = 0;
   _protocolSmHelper = 0;
}

void ConfigurePageTimeout::setBasicConfigurationIf(IN IBasicConfiguration* basicConfigurationIf)
{
   _basicConfigurationIf = basicConfigurationIf;

   FW_ERRMEM_ASSERT(0 != _basicConfigurationIf);
}

void ConfigurePageTimeout::setProtocolSmHelper(IN IProtocolSmHelper* helper)
{
   _protocolSmHelper = helper;

   FW_ERRMEM_ASSERT(0 != _protocolSmHelper);
}

void ConfigurePageTimeout::resetStateMachine(void)
{
   _ongoingConfiguration = BTS_CONFIG_PARAM_PAGE_TIMEOUT;
}

bool ConfigurePageTimeout::configure(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSTimeValue pageTimeout, IN const IConfiguration& configurationIf)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_basicConfigurationIf);

   // check given page timeout
   if(0 == pageTimeout)
   {
      return false;
   }

   // reset control data
   resetStateMachine();

   // remember requested page timeout
   _requestedPageTimeout = pageTimeout;

   // calculate timeout value
   _calculatedPageTimeout = calculateTimeout(pageTimeout, configurationIf);

   if(false == _basicConfigurationIf->isStartRuntimeConfigurationPossible())
   {
      return false;
   }

   // start configuration
   _ongoingConfiguration = BTS_CONFIG_PARAM_PAGE_TIMEOUT;
   _basicConfigurationIf->setRuntimeConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, BTS_CONFIG_PARAM_PAGE_TIMEOUT, this, _calculatedPageTimeout);

   return true;
}

void ConfigurePageTimeout::handleBasicResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_basicConfigurationIf);

   if(BTS_CONFIG_PARAM_PAGE_TIMEOUT == _ongoingConfiguration)
   {
      // if failed then abort configuration sequence
      if(BTS_REQ_SUCCESS != result)
      {
         handleEndOfConfiguration(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
      }
      else
      {
         _ongoingConfiguration = BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND;
         _basicConfigurationIf->setRuntimeConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND, this, _calculatedPageTimeout);
      }
   }
   else if(BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND == _ongoingConfiguration)
   {
      handleEndOfConfiguration(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
   }
   else
   {
      // error
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

BTSTimeValue ConfigurePageTimeout::calculateTimeout(IN const BTSTimeValue pageTimeout, IN const IConfiguration& configurationIf) const
{
   // internal requirement: allowed maximum value is given via configuration

   BTSTimeValue timeout(pageTimeout);

   if(configurationIf.getConfiguration().defaultPageTimeout < timeout)
   {
      timeout = configurationIf.getConfiguration().defaultPageTimeout;
   }

   return timeout;
}

void ConfigurePageTimeout::handleEndOfConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_protocolSmHelper);

   _protocolSmHelper->handleEndOfPageTimeoutConfiguration(bts2IpcMsgList, bts2AppMsgList, messageItem, result, _requestedPageTimeout);
}

} //btstackif
