/**
 * @file ProtocolSmPool.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of protocol state machine pool.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of protocol state machine pool.
 */

#ifndef _PROTOCOL_SM_POOL_H_
#define _PROTOCOL_SM_POOL_H_

#include "BtStackInternalTypes.h"
#include "ProtocolSmEntry.h"

namespace btstackif {

/**
 * Protocol state machine pool class.
 */
class ProtocolSmPool
{
public:
   /**
    * Default constructor.
    */
   ProtocolSmPool();

   /**
    * Destructor.
    */
   virtual ~ProtocolSmPool();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   ProtocolSmPool(const ProtocolSmPool& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   ProtocolSmPool& operator=(const ProtocolSmPool& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const ProtocolSmPool& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const ProtocolSmPool& ref) const;

   /**
    * Get SM. Create new SM if necessary.
    *
    * @param[in] connect: connect/disconnect flag
    *
    * @return = SM
    */
   ProtocolSmEntry& getSmEntry(IN const bool connect);

   /**
    * Get configure SM. Create new SM if necessary.
    *
    * @return = SM
    */
   ProtocolSmEntry& getConfigureSmEntry(void);

   /**
    * Free SM (mark as free in SM pool).
    */
   void freeSmContainer(IN ProtocolSmEntry& sm);

   /**
    * Free all SMs.
    */
   void freeAllSms(void);

   /**
    * Reset SM pool.
    */
   void reset(void);

private:
   /**
    * Type definition for different state machine types.
    * #SmType#ignore#
    */
   enum SmType
   {
      SM_TYPE_CONNECT = 0, /**< connect SM */
      SM_TYPE_DISCONNECT, /**< disconnect SM */
      SM_TYPE_CONFIGURE /**< configure SM */
   };

   ::std::vector< ProtocolSmEntry > _smPool; /**< pool with SMs for connect/disconnect */


   ProtocolSmEntry& getSmEntryInternal(IN const SmType smType);
};

} //btstackif

#endif //_PROTOCOL_SM_POOL_H_
