/**
 * @file ProtocolManagerData.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of protocol manager data.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of protocol manager data.
 */

#ifndef _PROTOCOL_MANAGER_DATA_H_
#define _PROTOCOL_MANAGER_DATA_H_

#include "BtStackInternalTypes.h"
#include "ProtocolSmEntry.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {

/**
 * Protocol manager data class.
 */
class ProtocolManagerData
{
public:
   /**
    * Type definition for different connect/disconnect instances.
    * #ProtocolInstance#ignore#
    */
   enum ProtocolInstance
   {
      PROTO_INST_HFP = 0, /**< HFP */
      PROTO_INST_AVP, /**< AVP */
      PROTO_INST_PIM, /**< PIM */
      PROTO_INST_MSG, /**< MSG */
      PROTO_INST_DUN, /**< DUN */
      PROTO_INST_PAN, /**< PAN via ConnMan */
      PROTO_INST_PAN_PROPRIETARY, /**< PAN not via ConnMan */
      PROTO_INST_SPP, /**< SPP */
      // -------------------------------------------------------------------------
      PROTO_INST_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for protocol data available.
    * #ProtocolDataInfoBit#ignore#
    */
   enum ProtocolDataInfoBit
   {
      PROTOCOL_ADDED = 0, /**< protocol entry was added */
      PROTOCOL_AVAILABLE, /**< protocol is available (level 1) */
      PROTOCOL_AVAILABLE_L2, /**< protocol is available (level 2) */
      STATUS_UPDATED, /**< status flag was updated */
      // -------------------------------------------------------------------------
      PROTOCOL_DATA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for protocol available bits.
    * #ProtocolAvailableInfoBit#ignore#
    */
   enum ProtocolAvailableInfoBit
   {
      PROTOCOL_AVAILABLE_LEVEL_1 = 0, /**< protocol available (level 1) */
      PROTOCOL_AVAILABLE_LEVEL_2, /**< protocol available (level 2) */
      // -------------------------------------------------------------------------
      PROTOCOL_AVAILABLE_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for protocol data available.
    */
   typedef BTSBitfield<unsigned char, ProtocolDataInfoBit, PROTOCOL_DATA_LAST> ProtocolDataInfo;

   /**
    * Type definition for protocol available info.
    */
   typedef BTSBitfield<unsigned char, ProtocolAvailableInfoBit, PROTOCOL_AVAILABLE_LAST> ProtocolAvailableInfo;

   /**
    * Default constructor.
    */
   ProtocolManagerData();

   /**
    * Destructor.
    */
   virtual ~ProtocolManagerData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   ProtocolManagerData(const ProtocolManagerData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   ProtocolManagerData& operator=(const ProtocolManagerData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const ProtocolManagerData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const ProtocolManagerData& ref) const;

   // member with public access
   BTSPairConnectOriginator connectionOriginator; /**< set connection originator type (needed for HFP/AVP) */
   bool connected; /**< connected flag */
   BTSDisconnectReason stackReason; /**< internal disconnect reason */
   BTSDisconnectReason reason; /**< disconnect reason */
   BTSDeviceName rfCommDevice; /**< RF comm device name */
   bool remotePending; /**< remote pending flag */
   bool localPending; /**< local pending flag */
   bool acceptPending; /**< accept pending flag */
   ProtocolDataInfo info; /**< protocol data information */
   unsigned int secondaryInfo; /**< secondary protocol data information */
   unsigned int operationId; /**< secondary protocol data information */
   BTSSppInstanceId sppInstance; /**< SPP instance */
   ProtocolSmEntry sm; /**< related SM */
   ProtocolSmEntry configureSm; /**< related SM */
   bool setPageTimeoutActive; /**< marker for set/reset page timeout */
   BTSApp2BtsMessageRequestItem requestItem; /**< request item */
   ExtendedTimerEntry timer; /**< timer */
   unsigned int retryMax; /**< maximum number of retries */
   unsigned int retryNmb; /**< current retry number */
   bool sdpRecordPending; /**< flag indicating that SDP record is pending, needed for MAS instances (MAP) */
   BTSMasInstanceName masName; /**< MAS instance name (MAP) */
   bool linkageMaster; /**< SPP: flag indicating linkage master */
   BTSUuid linkageUuid; /**< SPP: linkage UUID */
   ExtendedTimerEntry charDevTimer; /**< timer to wait for character device */
   ExtendedTimerEntry connectTimer; /**< timer to measure time needed for connect */
   ExtendedTimerEntry disconnectTimer; /**< timer to monitor profile disconnect */
};

} //btstackif

#endif //_PROTOCOL_MANAGER_DATA_H_
