/**
 * @file ProtocolManager.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of generic protocol manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic protocol manager state machine.
 */

#ifndef _PROTOCOL_MANAGER_H_
#define _PROTOCOL_MANAGER_H_

#include "StateMachine.h"
#include "IProtocolManager.h"
#include "IProtocolManagerCallback.h"
#include "IDeviceObserver.h"
#include "IServiceSearchObserver.h"
#include "ISppPoolObserver.h"
#include "IExtendedTimeoutHandler.h"
#include "IProtocolSmHelper.h"
#include "ProtocolManagerData.h"
#include "ProtocolSmPool.h"

#include <set>

namespace btstackif {

/**
 * Protocol manager class.
 */
class ProtocolManager : public StateMachine, public IProtocolManager, public IProtocolManagerCallback, public IDeviceObserver, public IServiceSearchObserver, public ISppPoolObserver, public IExtendedTimeoutHandler, public IProtocolSmHelper
{
public:
   /**
    * Default constructor.
    */
   ProtocolManager();

   /**
    * Destructor.
    */
   virtual ~ProtocolManager();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IProtocolManagerRequest* instance, IN const BTSProtocolId protocol, IN const bool panViaConnMan = true);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf);

   virtual void setSppPoolIf(IN ISppPool* sppPool);

   virtual void setConfigurationIf(IN IConfiguration* configurationIf);

   virtual void setBasicConfigurationIf(IN IBasicConfiguration* basicConfigurationIf);

   virtual void setRestrictionIf(IN IRestrictedPairingConnecting* restrictionIf);

   virtual IDeviceObserver* getDeviceObserver(void);

   virtual IServiceSearchObserver* getServiceSearchObserver(void);

   virtual ISppPoolObserver* getSppPoolObserver(void);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetProtocolConnectionStatus& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidGetRequest(IN const App2Bts_GetProtocolConnectionStatus& request) const;

   virtual bool isValidGetRequest(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const;

   virtual bool isValidConnectRequest(IN const App2Bts_ConnectProtocol& request) const;

   virtual bool isValidAcceptRequest(IN const App2Bts_AcceptRemoteProtocolConnect& request) const;

   virtual bool isValidRejectRequest(IN const App2Bts_RejectRemoteProtocolConnect& request) const;

   virtual bool isValidDisconnectRequest(IN const App2Bts_DisconnectProtocol& request) const;

   virtual bool isProtocolEnabled(IN const BTSProtocolId protocol) const;

   virtual bool connectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request);

   virtual bool acceptRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_AcceptRemoteProtocolConnect& request);

   virtual bool rejectRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RejectRemoteProtocolConnect& request);

   virtual bool disconnectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request);

   virtual void setRequestedConnectionState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool connectRequested, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool remote = false);

   virtual void setRequestedConnectionState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool connectRequested);

   virtual void getConnectedProtocols(OUT ::std::vector< BTSProtocolId >& connectedProtocols, IN const BTSBDAddress& address) const;

   virtual bool isProtocolConnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const;

   virtual bool isAnyProtocolConnectDisconnectOngoing(void) const;

   virtual bool isProtocolConnectDisconnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const;

   virtual bool isProtocolConnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const;

   virtual bool isProtocolDisconnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const;

   virtual bool isProtocolAdded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const;

   virtual bool isProtocolAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2 = true) const;

   virtual bool isSppSlotAvailable(IN const BTSUuid& uuid) const;

   virtual void setMaxSppConnections(IN const unsigned int limit);

   virtual void setPanConnectionMode(IN const bool mode);

   virtual void setTestTriggerDeviceConnectedUpdate(IN const bool enable);

   virtual bool getTestTriggerDeviceConnectedUpdate(void) const;

   virtual void setTestTriggerRemoteConnectDuringDeviceDisconnect(IN const bool enable);

   virtual bool getTestTriggerRemoteConnectDuringDeviceDisconnect(void) const;

   virtual void setSimulateFailedDisconnectResult(IN const bool enable);

   virtual void registerObserver(IN IProtocolObserver* observer);

   virtual void setDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason);

   virtual void setTestTriggerDisconnectWhileDisconnected(IN const bool enable);

   virtual void setTestTriggerDisconnectWhileDisconnectedSetState(IN const bool enable);

   virtual void setTriggerCancelDuringConnectResult(IN const bool enable);

   virtual void setTriggerCancelDuringAcceptRejectRemoteConnect(IN const bool enable);

   virtual void setTestDisableSearching(IN const bool enable);

   virtual void setTestDisableSetting(IN const bool enable);

   virtual void setTestBlockSendingDisconnect(IN const bool enable);

   virtual void setTestTriggerLocalConnectWhileRemoteConnect(IN const bool enable);

   virtual void setTestTriggerLocalDisconnectWhileRemoteConnect(IN const bool enable);

   virtual void setTestTriggerLocalConnectBeforeRemoteConnect(IN const bool enable);

   virtual void setTestTriggerLocalConnectedBeforeRemoteConnect(IN const bool enable);

   virtual void setTestTriggerLocalConnectSameDevice(IN const bool enable);

   virtual void executeTestAction(IN const unsigned int action);

   virtual void setTestMaxRetryNumber(IN const unsigned int number);

   virtual void setTestRetryTimeout(IN const unsigned int timeout);

   virtual bool getFirstConnectedSppUuid(OUT BTSUuid& uuid, IN const BTSBDAddress& address) const;

   virtual bool isPanViaConnMan(void) const;

   virtual void waitingForServiceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result);

   virtual void waitingForProtocolResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result);

   virtual void connectIndication(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result);

   virtual bool connectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result);

   virtual void connecting(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void disconnectIndication(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result);

   virtual void disconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result);

   virtual void disconnecting(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual bool updateStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason);

   virtual void updateFailedConnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason);

   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason);

   virtual void updateVirtualDeviceName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDeviceName& name);

   virtual void characterDeviceAddedRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& name, IN const bool added);

   virtual void remoteConnectRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void remoteConnectCanceled(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void protocolAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void protocolRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual void protocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ProtocolManagerData::ProtocolAvailableInfo& protocolAvailableInfo);

   virtual void protocolUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ProtocolManagerData::ProtocolAvailableInfo& protocolAvailableInfo);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual ProtocolManagerData& getEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName);

   virtual const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& getProtocolList(void) const;

   virtual bool isProtocolAvailableInternal(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2 = true) const;

   virtual void getMasInstanceNameInternal(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const;

   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected);

   virtual void deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void serviceSearchFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSRequestResult result);

   virtual void settingUuidFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSSppInstanceId instance, IN const BTSRequestResult result);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void createRemoteConnectRequest(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   virtual BTSDisconnectReason convertConnectResult2DisconnectReason(IN const BTSRequestResult result) const;

   virtual BTSDisconnectReason getDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const;

   virtual void setConnectionStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connected, IN const BTSDisconnectReason reason);

   virtual void updateConnectionStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool sendStatusToAll) const;

   virtual void updateConnectionResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool isConnectRequest, IN const BTSRequestResult result) const;

   virtual void handleChangedProtocolConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDisconnectReason stackReason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL);

   virtual bool isRetryAllowed(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const;

   virtual void startRetryTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   virtual void stopRetryTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   virtual void increaseRetryCounter(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   virtual void setSppUuid(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSppInstanceId instance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid);

   virtual bool isAnyProtocolConnected(IN const BTSBDAddress& address) const;

   virtual bool wasCharacterDeviceReported(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const;

   virtual void startWaitForCharacterDeviceTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   virtual void stopWaitForCharacterDeviceTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance);

   virtual void checkAccessRightsForCharacterDevice(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const;

   virtual bool configurePageTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSTimeValue pageTimeout);

   virtual void handleEndOfPageTimeoutConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result, IN const BTSTimeValue pageTimeout);

   virtual void startConnectTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction);

   virtual void stopConnectTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction);

   virtual bool isConnectTimerActive(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction) const;

private:
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   IServiceSearch* _serviceSearchIf; /**< service search interface */
   ISppPool* _sppPoolIf; /**< SPP pool interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IBasicConfiguration* _basicConfigurationIf; /**< basic configuration interface */
   IRestrictedPairingConnecting* _restrictionIf; /**< restriction interface */
   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData > _protocolList; /**< map containing protocol information */
   ::std::set< IProtocolObserver* > _observerList; /**< observer list */
   ::std::vector< IProtocolManagerRequest* > _protocolInstances; /**< protocol instance list */
   ProtocolSmPool _smPool; /**< SM pool */
   ::std::set< BTSDeviceName > _monitoredDevices; /**< active devices reported by udev monitor */
   const BTSTimeValue _waitForCharDevTimeout; /**< timeout in ms to wait for character device */
   BTSTimeValue _currentPageTimeout; /**< current page timeout */
   bool _pageTimeoutConfigured; /**< flag for configured page timeout */

   const BTSSppInstanceId _defaultSppInstanceId; /**< default SPP instance ID */
   const BTSMasInstanceId _defaultMasInstanceId; /**< default MAS instance ID */
   const BTSMasInstanceName _emptyMasInstanceName; /**< empty name for MAS instance */
   BTSTimeValue _retryTimeout; /**< timeout in ms before starting retry */
   unsigned int _defaultRetryMax; /**< default maximum number of retries */
   const BTSTimeValue _connectTimeout; /**< timeout in ms to measure time needed for connect */
   const BTSTimeValue _disconnectTimeout; /**< timeout in ms to monitor profile disconnect */

   bool _panViaConnMan; /**< flag indicating which interface shall be used to establish PAN connection */

   bool _testTriggerDisconnectWhileDisconnected; /**< flag for testing disconnect while disconnected */
   bool _testTriggerDisconnectWhileDisconnectedSetState; /**< flag for testing disconnect while disconnected */
   bool _testTriggerCancelDuringConnectResult; /**< flag for testing cancel connect during reception of connect result */
   bool _testTriggerCancelDuringAcceptRejectRemoteConnect; /**< flag for testing cancel remote connect after accepting or rejecting */
   bool _testDisableSearching; /**< flag to test connect without searching */
   bool _testDisableSetting; /**< flag to test connect without setting UUID */
   bool _testBlockSendingDisconnect; /**< test flag to block sending of disconnect request */
   bool _testTriggerLocalConnectWhileRemoteConnect; /**< flag for testing local connect while remote connection request */
   bool _testTriggerLocalDisconnectWhileRemoteConnect; /**< flag for testing local disconnect while remote connection request */
   bool _testTriggerLocalConnectBeforeRemoteConnect; /**< flag for testing local connect before remote connection request */
   bool _testTriggerLocalConnectedBeforeRemoteConnect; /**< flag for testing local connected before remote connection request */
   bool _testTriggerLocalConnectSameDevice; /**< flag for testing local connect before remote connection request: same / different device */

   void doConnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN ProtocolManagerData& entry);

   void doDisconnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN ProtocolManagerData& entry);

   void freeSmEntry(IN ProtocolSmEntry& sm);

   bool getSppSlot(OUT BTSSppInstanceId& instance, IN const BTSUuid& uuid);

   void freeSppSlot(IN const BTSProtocolId protocolId, IN const BTSSppInstanceId instance);

   void resetProtocolEntryControlData(IN ProtocolManagerData& entry);

   ProtocolManagerData& checkProtocolList(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator getProtocolEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator getProtocolEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance) const;

   void getSM(OUT ProtocolSmEntry& smEntry, IN const bool connect, IN IProtocolManagerRequest* requestIf);

   void getConfigureSM(OUT ProtocolSmEntry& smEntry);

   IProtocolManagerRequest* getRequestIf(IN const BTSProtocolId protocol) const;

   ProtocolManagerData::ProtocolInstance convertProtocol2Instance(IN const BTSProtocolId protocol, IN const bool panViaConnMan = true) const;

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   bool isDeviceDisconnectOngoing(IN const BTSBDAddress& address) const;

   bool isDeviceConnected(IN const BTSBDAddress& address) const;

   const BTSMasInstanceName& getVerifiedMasInstanceName(IN const BTSProtocolId protocol, IN const BTSMasInstanceName& masName) const;

   bool isSdpRecordAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const;

   /**
    * Get MAS instance id.
    *
    * @param[in] address: address
    * @param[in] protocol: protocol
    * @param[in] instanceName: MAS instance name
    *
    * @return = MAS instance id
    */
   BTSMasInstanceId getMasInstanceId(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceName& instanceName) const;

   /**
    * Get MAS instance name.
    *
    * @param[out] instanceName: MAS instance name
    * @param[in] address: address
    * @param[in] protocol: protocol
    * @param[in] instanceId: MAS instance id
    */
   void getMasInstanceName(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const;

   /**
    * Protocol Connection status of requested device address + protocol. Check if any related instance is connected.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    *
    * @return = true: connected,
    * @return = false: disconnected
    */
   bool isAnyProtocolInstanceConnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const;

   void createProtocolConnectionStatusUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN const ProtocolManagerData& entry, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const;

   void createProtocolConnectionStatusUpdate(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& instance, IN const ProtocolManagerData& entry, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const;

   void createConnectDisconnectProtocolResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance, IN const BTSRequestResult result, IN const bool isConnect, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const;

   void createConnectDisconnectProtocolResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& instance, IN const BTSRequestResult result, IN const bool isConnect, IN BtStackIfCallback* user, IN const BTSSessionHandle handle) const;

   App2Bts_BaseMessage* getWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item);

   void handleTestTriggerDisconnectWhileDisconnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance);

   void handleTestTriggerDisconnectWhileDisconnected(OUT BTSApp2BtsMessageCompareItem& compareItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance);

   void handleTestTriggerCancelDuringConnectResult(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId instance);

   void handleTestTriggerCancelDuringAcceptRejectRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& instance);

   void handleTestTriggerLocalConnectWhileRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry);

   void handleTestTriggerLocalDisconnectWhileRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry);

   void handleTestTriggerLocalConnectBeforeRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry, IN const bool sameDevice);

   void handleTestTriggerLocalConnectedBeforeRemoteConnect(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry);

   void internalExecuteTestAction(IN const unsigned int action);

   BTSProtocolId convertSearchType2Protocol(IN const BTSSearchType searchType) const;

   BTSSearchType convertProtocol2SearchType(IN const BTSProtocolId protocol) const;

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator getLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry);

   ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator getLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry) const;

   const ProtocolManagerData& getUpdateEntry(IN const ProtocolManagerData& mainEntry, IN const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator& linkageIt) const;

   ProtocolManagerData& getRemoteConnectEntry(IN ProtocolManagerData& mainEntry, IN ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator& linkageIt);

   ProtocolManagerData& getLocalDisconnectEntry(IN ProtocolManagerData& mainEntry, IN ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator& linkageIt);

   const BTSUuid& getLocalDisconnectUuid(IN const BTSUuid& mainUuid, IN const ProtocolManagerData& mainEntry, IN const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator& linkageIt) const;

   const BTSUuid& getMasterUuid(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) const;

   const BTSUuid& getMasterUuid(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry) const;

   inline bool isLinkageEntryConnected(IN const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::iterator& linkageIt) const { if(_protocolList.end() == linkageIt) { return false; } else { return linkageIt->second.connected; } }

   inline bool isLinkageEntryConnected(IN const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator& linkageIt) const { if(_protocolList.end() == linkageIt) { return false; } else { return linkageIt->second.connected; } }

   void setDisconnectReasonForLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const ProtocolManagerData& entry, IN const bool setStackReason = true);

   void addLinkageEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN ProtocolManagerData& entry);

   void checkResettingPageTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN ProtocolManagerData& entry);
};

} //btstackif

#endif //_PROTOCOL_MANAGER_H_
