/**
 * @file ProtocolDisconnect.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of generic protocol disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic protocol disconnect state machine.
 */

#ifndef _PROTOCOL_DISCONNECT_H_
#define _PROTOCOL_DISCONNECT_H_

#include "BtStackInternalTypes.h"
#include "IProtocolDisconnect.h"
#include "ProtocolDisconnectSm.h"

namespace btstackif {

/**
 * Protocol disconnect state machine class.
 */
class ProtocolDisconnect : public IProtocolDisconnect, public ProtocolDisconnectSm
{
public:
   /**
    * Default constructor.
    */
   ProtocolDisconnect();

   /**
    * Destructor.
    */
   virtual ~ProtocolDisconnect();

   //===================================================================================================================

   virtual void setProtocolManagerRequest(IN IProtocolManagerRequest* manager);

   virtual void setProtocolSmHelper(IN IProtocolSmHelper* helper);

   virtual void resetStateMachine(void);

   virtual void sendDisconnectEvent(void);

   virtual void sendCancelEvent(void);

   virtual void sendBusyEvent(void);

   virtual void sendDisconnectingEvent(void);

   virtual void sendDisconnectResultEvent(const BTSRequestResult result);

   virtual void sendFinalStateEvent(const BTSRequestResult result);

   virtual void setConnectionData(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId, IN const bool pauseBtStreaming);

   virtual bool doDisconnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* user = 0, IN const BTSSessionHandle handle = 0);

   virtual void setTestBlockSendingDisconnect(IN const bool enable);

   //===================================================================================================================

   virtual int entryBusy();

   virtual int entryCheckForRetry();

   virtual int exitBusy();

   virtual int handleBusyCanceled(const BTSRequestResult result);

   virtual int handleCancel();

   virtual int handleCancelDuringBusy();

   virtual int handleDisconnectCompleted(const BTSRequestResult result);

   virtual int handleDisconnectFinalState(const BTSRequestResult result);

   virtual int handleDisconnecting();

   virtual int handleDisconnectingCanceled();

   virtual int handleExitNoRetry();

   virtual int initSm();

   virtual int isCanceled();

   virtual int isFinalStateReached(const BTSRequestResult result);

   virtual int sendDisconnect();

   virtual int storeFinalState(const BTSRequestResult result);

   //===================================================================================================================

private:
   IProtocolManagerRequest* _protocolManagerRequest; /**< protocol manager request interface */
   IProtocolSmHelper* _protocolSmHelper; /**< protocol SM helper interface */
   BTSBDAddress _address; /**< device address */
   BTSProtocolId _protocol; /**< protocol */
   BTSSppInstanceId _sppInstance; /**< SPP instance id */
   BTSUuid _uuid; /**< SPP uuid */
   BTSMasInstanceName _masInstanceName; /**< MAS instance name */
   BTSMasInstanceId _masInstanceId; /**< MAS instance id */
   bool _pauseBtStreaming; /**< pause BT streaming*/
   bool _cancel; /**< cancel requested */
   bool _final; /**< final state received */
   bool _disconnecting; /**< disconnecting state set */
   BTSRequestResult _result; /**< disconnect result */
   BTSRequestResult _status; /**< disconnect status */
   bool _smProcessingActive; /**< flag indicating that SM processing is active */
   bool _testBlockSendingDisconnect; /**< test flag to block sending of disconnect request */

   void handleDisconnectFinished(void);

   void handleDisconnectFinishedNew(void);

   void sendExitNoRetryEvent(void);

   void sendRetryDisconnectEvent(void);
};

} //btstackif

#endif //_PROTOCOL_DISCONNECT_H_
