/**
 * @file IProtocolManagerRequest.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Request interface definition for protocol manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific protocol manager state machine.
 */

#ifndef _I_PROTOCOL_MANAGER_REQUEST_H_
#define _I_PROTOCOL_MANAGER_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerCallback;
class IBasicControl;
class ITimerPool;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IProtocolManagerRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolManagerRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN IProtocolManagerCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Prepare protocol connect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Set device availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] available: device availability
    */
   virtual void setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available) = 0;

   /**
    * Check if service is available. Service here means all needed communication end points (e.g. DBUS interfaces).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    *
    * @return = true: service available,
    * @return = false: service not available, wait is necessary
    */
   virtual bool isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const = 0;

   /**
    * Wait for service available. Service here means all needed communication end points (e.g. DBUS interfaces).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Check if service search is necessary. Service search here means service discovery.
    *
    * @param[out] searchType: search type
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    *
    * @return = true: service search needed,
    * @return = false: service search not needed
    */
   virtual bool isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const = 0;

   /**
    * Check if setting UUID is necessary. Normally this is only needed for SPP connect.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    *
    * @return = true: setting UUID needed,
    * @return = false: setting UUID not needed
    */
   virtual bool isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const = 0;

   /**
    * Wait for protocol available (level 2).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Connect protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Disconnect protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] pauseBtStreaming: Pause BT Streaming (AVP only)
    */
   virtual void disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming) = 0;

   /**
    * Abort/cancel connect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Accept/reject remote connection request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] accept: accept flag (true: accept, false: reject)
    */
   virtual void accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept) = 0;

   /**
    * Send virtual failed connect result result message (loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const = 0;

   /**
    * Send virtual failed disconnect result result message (loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const = 0;

   /**
    * Send virtual update message for profile connection status (loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] connected: connected flag
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode) = 0;
};

} //btstackif

#endif //_I_PROTOCOL_MANAGER_REQUEST_H_
