/**
 * @file IProtocolManager.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Interface definition for generic protocol manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for protocol manager state machine.
 */

#ifndef _I_PROTOCOL_MANAGER_H_
#define _I_PROTOCOL_MANAGER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerRequest;
class IBasicControl;
class ITimerPool;
class IDeviceManager;
class IServiceSearch;
class ISppPool;
class IConfiguration;
class IBasicConfiguration;
class IRestrictedPairingConnecting;
class IDeviceObserver;
class IServiceSearchObserver;
class ISppPoolObserver;
class IStateMachine;
class App2Bts_GetProtocolConnectionStatus;
class App2Bts_ConnectProtocol;
class App2Bts_AcceptRemoteProtocolConnect;
class App2Bts_RejectRemoteProtocolConnect;
class App2Bts_DisconnectProtocol;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IProtocolObserver;

/**
 * Interface definition.
 */
class IProtocolManager
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolManager() {}

   /**
    * Set specific instance (implementation). For each protocol a specific instance has to be registered.
    *
    * @param[in] instance: instance (implementation)
    * @param[in] protocol: protocol id
    * @param[in] panViaConnMan: flag for connect/disconnect PAN via ConnMan
    */
   virtual void setInstance(IN IProtocolManagerRequest* instance, IN const BTSProtocolId protocol, IN const bool panViaConnMan = true) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set service search interface.
    *
    * @param[in] serviceSearchIf: service search interface
    */
   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf) = 0;

   /**
    * Set SPP pool interface.
    *
    * @param[in] sppPool: SPP pool interface
    */
   virtual void setSppPoolIf(IN ISppPool* sppPool) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Set basic configuration interface.
    *
    * @param[in] basicConfigurationIf: basic configuration interface
    */
   virtual void setBasicConfigurationIf(IN IBasicConfiguration* basicConfigurationIf) = 0;

   /**
    * Set interface for restricted pairing and connecting.
    *
    * @param[in] restrictionIf: interface for restricted pairing and connecting
    */
   virtual void setRestrictionIf(IN IRestrictedPairingConnecting* restrictionIf) = 0;

   /**
    * Returns device observer.
    *
    * @return = device observer
    */
   virtual IDeviceObserver* getDeviceObserver(void) = 0;

   /**
    * Returns service search observer.
    *
    * @return = service search observer
    */
   virtual IServiceSearchObserver* getServiceSearchObserver(void) = 0;

   /**
    * Returns SPP pool observer.
    *
    * @return = SPP pool observer
    */
   virtual ISppPoolObserver* getSppPoolObserver(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetProtocolConnectionStatus& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    * @param[in] user: user
    * @param[in] handle: handle
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidGetRequest(IN const App2Bts_GetProtocolConnectionStatus& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidGetRequest(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidConnectRequest(IN const App2Bts_ConnectProtocol& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidAcceptRequest(IN const App2Bts_AcceptRemoteProtocolConnect& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidRejectRequest(IN const App2Bts_RejectRemoteProtocolConnect& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidDisconnectRequest(IN const App2Bts_DisconnectProtocol& request) const = 0;

   /**
    * Check if given protocol is enabled on local side.
    *
    * @param[in] protocol: protocol
    *
    * @return = true: protocol is enabled on local side,
    * @return = false: protocol is disabled on local side
    */
   virtual bool isProtocolEnabled(IN const BTSProtocolId protocol) const = 0;

   /**
    * Connect protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool connectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request) = 0;

   /**
    * Accept remote protocol connect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool acceptRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_AcceptRemoteProtocolConnect& request) = 0;

   /**
    * Reject remote protocol connect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool rejectRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_RejectRemoteProtocolConnect& request) = 0;

   /**
    * Disconnect protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool disconnectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request) = 0;

   /**
    * Set requested protocol connection state and check for accept/reject of open remote connect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    * @param[in] connectRequested: flag for connect or disconnect requested
    * @param[in] user: user
    * @param[in] handle: session handle
    * @param[in] remote: flag for remote/local connection request
    */
   virtual void setRequestedConnectionState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool connectRequested, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool remote = false) = 0;

   /**
    * Set requested protocol connection state and check for accept/reject of open remote connect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] connectRequested: flag for connect or disconnect requested
    */
   virtual void setRequestedConnectionState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool connectRequested) = 0;

   /**
    * Get list of connected protocols.
    *
    * @param[out] connectedProtocols: list of connected protocols
    * @param[in] address: device address
    */
   virtual void getConnectedProtocols(OUT ::std::vector< BTSProtocolId >& connectedProtocols, IN const BTSBDAddress& address) const = 0;

   /**
    * Check if protocol is connected.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    *
    * @return = true: connected,
    * @return = false: disconnected
    */
   virtual bool isProtocolConnected(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const = 0;

   /**
    * Check for any ongoing connect or disconnect activity.
    *
    * @return = true: any connect/disconnect ongoing,
    * @return = false: no connect/disconnect ongoing
    */
   virtual bool isAnyProtocolConnectDisconnectOngoing(void) const = 0;

   /**
    * Check for ongoing connect or disconnect activity of requested device address.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID
    * @param[in] masInstance: MAS instance name
    *
    * @return = true: connect/disconnect ongoing,
    * @return = false: no connect/disconnect ongoing
    */
   virtual bool isProtocolConnectDisconnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const = 0;

   /**
    * Check for ongoing connect activity of requested device address.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID
    * @param[in] masInstance: MAS instance name
    *
    * @return = true: connect ongoing,
    * @return = false: no connect ongoing
    */
   virtual bool isProtocolConnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const = 0;

   /**
    * Check for ongoing disconnect activity of requested device address.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID
    * @param[in] masInstance: MAS instance name
    *
    * @return = true: disconnect ongoing,
    * @return = false: no disconnect ongoing
    */
   virtual bool isProtocolDisconnectOngoing(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const = 0;

   /**
    * Check if protocol is part of internal list.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID
    * @param[in] masInstance: MAS instance name
    *
    * @return = true: protocol is added,
    * @return = false: protocol is not added
    */
   virtual bool isProtocolAdded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) const = 0;

   /**
    * Check if protocol is available (ready to use).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID
    * @param[in] masInstance: MAS instance name
    * @param[in] level2: flag for level 1/2
    *
    * @return = true: protocol is available,
    * @return = false: protocol is not available
    */
   virtual bool isProtocolAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2 = true) const = 0;

   /**
    * Check for free SPP slot.
    *
    * @param[in] uuid: SPP UUID to be used (will be compared with reserved ones)
    *
    * @return = true: free SPP slot available,
    * @return = false: no free SPP slot available
    */
   virtual bool isSppSlotAvailable(IN const BTSUuid& uuid) const = 0;

   /**
    * Set maximum number of possible SPP connections. To be used for testing purpose. Have to be called before any protocol entry is added.
    *
    * @param[in] limit: maximum number of possible SPP connections
    */
   virtual void setMaxSppConnections(IN const unsigned int limit) = 0;

   /**
    * Set mode to connect/disconnect PAN. To be used for testing purpose. Have to be called before any protocol entry is added.
    *
    * @param[in] mode: mode (true: via ConnMan, false: via Evolution (testing purpose))
    */
   virtual void setPanConnectionMode(IN const bool mode) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerDeviceConnectedUpdate(IN const bool enable) = 0;

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   virtual bool getTestTriggerDeviceConnectedUpdate(void) const = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerRemoteConnectDuringDeviceDisconnect(IN const bool enable) = 0;

   /**
    * Get test flag.
    *
    * @return = test flag
    */
   virtual bool getTestTriggerRemoteConnectDuringDeviceDisconnect(void) const = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setSimulateFailedDisconnectResult(IN const bool enable) = 0;

   /**
    * Register observer.
    *
    * @param[in] observer: observer
    */
   virtual void registerObserver(IN IProtocolObserver* observer) = 0;

   /**
    * Set disconnect reason directly.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] reason: disconnect reason
    */
   virtual void setDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerDisconnectWhileDisconnected(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerDisconnectWhileDisconnectedSetState(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTriggerCancelDuringConnectResult(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTriggerCancelDuringAcceptRejectRemoteConnect(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestDisableSearching(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestDisableSetting(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestBlockSendingDisconnect(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerLocalConnectWhileRemoteConnect(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerLocalDisconnectWhileRemoteConnect(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerLocalConnectBeforeRemoteConnect(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerLocalConnectedBeforeRemoteConnect(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerLocalConnectSameDevice(IN const bool enable) = 0;

   /**
    * Execute test action.
    *
    * @param[in] action: test action
    */
   virtual void executeTestAction(IN const unsigned int action) = 0;

   /**
    * Set max retry number.
    *
    * @param[in] number: max retry number
    */
   virtual void setTestMaxRetryNumber(IN const unsigned int number) = 0;

   /**
    * Set retry timeout.
    *
    * @param[in] timeout: retry timeout
    */
   virtual void setTestRetryTimeout(IN const unsigned int timeout) = 0;

   /**
    * Get SPP UUID that is connected. Return first entry.
    * To be used only for testing purpose.
    *
    * @param[in] address: address
    *
    * @return = found/not found
    */
   virtual bool getFirstConnectedSppUuid(OUT BTSUuid& uuid, IN const BTSBDAddress& address) const = 0;
};

} //btstackif

#endif //_I_PROTOCOL_MANAGER_H_
