/**
 * @file IProtocolDisconnect.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Interface definition for generic protocol disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for protocol disconnect state machine.
 */

#ifndef _I_PROTOCOL_DISCONNECT_H_
#define _I_PROTOCOL_DISCONNECT_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManagerRequest;
class IProtocolSmHelper;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IProtocolDisconnect
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolDisconnect() {}

   /**
    * Set protocol manager request interface.
    *
    * @param[in] manager: protocol manager request interface
    */
   virtual void setProtocolManagerRequest(IN IProtocolManagerRequest* manager) = 0;

   /**
    * Set protocol SM helper.
    *
    * @param[in] helper: protocol SM helper interface
    */
   virtual void setProtocolSmHelper(IN IProtocolSmHelper* helper) = 0;

   /**
    * Reset state machine.
    */
   virtual void resetStateMachine(void) = 0;

   /**
    * Send disconnect event.
    */
   virtual void sendDisconnectEvent(void) = 0;

   /**
    * Send cancel event.
    */
   virtual void sendCancelEvent(void) = 0;

   /**
    * Send busy event.
    */
   virtual void sendBusyEvent(void) = 0;

   /**
    * Send disconnecting event.
    */
   virtual void sendDisconnectingEvent(void) = 0;

   /**
    * Send disconnect result event.
    *
    * @param[in] result: result
    */
   virtual void sendDisconnectResultEvent(const BTSRequestResult result) = 0;

   /**
    * Send final state event.
    *
    * @param[in] result: result
    */
   virtual void sendFinalStateEvent(const BTSRequestResult result) = 0;

   /**
    * Set connection data.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstanceName: MAS instance name (MAP only)
    * @param[in] masInstanceId: MAS instance id (MAP only)
    * @param[in] pauseBtStreaming: Pause BT Streaming (AVP Only)
    */
   virtual void setConnectionData(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId, IN const bool pauseBtStreaming) = 0;

   /**
    * Do state machine processing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] user: user reference
    * @param[in] handle: session handle
    *
    * @return = true: SM finished,
    * @return = false: SM not finished
    */
   virtual bool doDisconnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* user = 0, IN const BTSSessionHandle handle = 0) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestBlockSendingDisconnect(IN const bool enable) = 0;
};

} //btstackif

#endif //_I_PROTOCOL_DISCONNECT_H_
