/**
 * @file Pairing.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Implementation of generic pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic pairing state machine.
 */

#ifndef _PAIRING_H_
#define _PAIRING_H_

#include "StateMachine.h"
#include "IPairing.h"
#include "IPairingCallback.h"
#include "IDeviceObserver.h"
#include "PairingData.h"

namespace btstackif {

/**
 * Pairing class.
 */
class Pairing : public StateMachine, public IPairing, public IPairingCallback, public IDeviceObserver
{
public:
   /**
    * Default constructor.
    */
   Pairing();

   /**
    * Destructor.
    */
   virtual ~Pairing();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IPairingRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setDeviceManager(IN IDeviceManager* deviceManager);

   virtual void setRestrictionIf(IN IRestrictedPairingConnecting* restrictionIf);

   virtual void setConfigurationIf(IN IConfiguration* configurationIf);

   virtual void setInquiryIf(IN IInquiry* inquiryIf);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidStartRequest(IN const App2Bts_StartPairing& request) const;

   virtual bool isValidCancelRequest(IN const App2Bts_CancelPairing& request) const;

   virtual bool isValidSetPinCodeRequest(IN const App2Bts_SetPinCode& request) const;

   virtual bool isValidConfirmSecureSimplePairingRequest(IN const App2Bts_ConfirmSecureSimplePairing& request) const;

   virtual bool startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request);

   virtual bool cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelPairing& request);

   virtual void cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual bool setPinCode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetPinCode& request);

   virtual bool confirmSecureSimplePairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConfirmSecureSimplePairing& request);

   virtual bool isPairingOngoing(void) const;

   virtual void setTestTriggerIgnoreReportedLinkKey(IN const bool enable);

   virtual void startPairingResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void indicateLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey);

   virtual void handleEndOfSuccessfulIncomingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void indicatePinCodeRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void indicateSecureSimplePairingRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address);

   virtual PairingData& getEntry(IN const BTSBDAddress& address);

   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected);

   virtual void deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

private:
   IPairingRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IDeviceManager* _deviceManager; /**< device manager interface */
   IRestrictedPairingConnecting* _restrictionIf; /**< interface for restricted pairing and connecting */
   IConfiguration* _configurationIf; /**< configuration interface */
   IInquiry* _inquiryIf; /**< inquiry interface */
   ::std::map< BTSBDAddress, PairingData > _deviceList; /**< map containing device information */
   bool _testTriggerIgnoreReportedLinkKey; /**< flag for ignoring reported link key */

   PairingData& checkPairingList(IN const BTSBDAddress& address);

   void removeDevice(IN const BTSBDAddress& address);

   void createPinCodeRequestedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSDeviceName& deviceName, IN const BTSPairConnectOriginator originator) const;

   void createSecureSimplePairingRequestedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSDeviceName& deviceName, IN const BTSPairConnectOriginator originator, IN const BTSSspMode mode, IN const BTSNumericValue& value) const;

   void createPairingFinishedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSRequestResult result,
         IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSMajorServiceClass& majorServiceClass, IN const BTSMajorDeviceClass majorDeviceClass, IN const BTSMinorDeviceClass minorDeviceClass, IN const BTSDeviceName& deviceName) const;

   void createStartPairingResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const BTSRequestResult result) const;

   void checkForPairingCompleted(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool rejected, IN const BTSBDAddress& address, IN const PairingData& entry, IN const BTSRequestResult result, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey);

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSApp2BtsMessageCompareItem& item);

   App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item);

   void stopInquiry(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   void checkForOngoingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   void handleEndOfFailedOutgoingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool* connectFailed, INOUT PairingData& entry, IN const BTSBDAddress& address);
};

} //btstackif

#endif //_PAIRING_H_
