/**
 * @file IPairingRequest.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Request interface definition for pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific pairing state machine.
 */

#ifndef _I_PAIRING_REQUEST_H_
#define _I_PAIRING_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IPairingCallback;
class IBasicControl;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IPairingRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~IPairingRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN IPairingCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Start pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: remote device address
    */
   virtual void startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Cancel pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: remote device address
    */
   virtual void cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Set pin code.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: remote device address
    * @param[in] pinCode: pin code
    */
   virtual void setPinCode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSPinCode& pinCode) = 0;

   /**
    * Confirm secure simple pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: remote device address
    * @param[in] mode: SSP mode
    * @param[in] numValue: numeric value
    * @param[in] confirm: accept/reject
    */
   virtual void confirmSecureSimplePairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSspMode mode, IN const BTSNumericValue& numValue, IN const BTSConfirmationMode confirm) = 0;
};

} //btstackif

#endif //_I_PAIRING_REQUEST_H_
